/*
 * TSLIB based touchscreen driver
 * Porting to Xorg by Daniel Stone.
 * Derived from ts.c by Keith Packard
 * Derived from ps2.c by Jim Gettys
 *
 * Copyright © 1999 Keith Packard
 * Copyright © 2000 Compaq Computer Corporation
 * Copyright © 2002 MontaVista Software Inc.
 * Copyright © 2005 OpenedHand Ltd.
 * Copyright © 2006-2008 Nokia Corporation
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and/or
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *
 * THE AUTHORS AND/OR COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD
 * TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE AUTHORS AND/OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *
 * Partially derived from xf86-input-evdev's evdev.c:
 * Copyright © 2004-2008 Red Hat, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation, and that the name of Red Hat
 * not be used in advertising or publicity pertaining to distribution
 * of the software without specific, written prior permission.  Red
 * Hat makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Authors:
 *      Kristian Høgsberg (krh@redhat.com)
 *      Adam Jackson (ajax@redhat.com)
 *      Peter Hutterer (peter.hutterer@redhat.com)
 */

#include "config.h"
#include "xorg-server.h"

#include "misc.h"
#include "xf86.h"
#include "xf86str.h"
#include "xf86Xinput.h"
#include "xf86Module.h"
#include "xorgVersion.h"

#include <sys/ioctl.h>
#include <tslib.h>

struct tslib_private {
    const char *path;
    struct tsdev *ts_dev;

    int screen; /* currently unused */

    int last_x, last_y, last_pressure;
    int is_down;
};

static void
tslib_read(InputInfoPtr info)
{
    struct tslib_private *private = info->private;
    struct ts_sample event;
    int is_down;

    while (ts_read(private->ts_dev, &event, 1) == 1) {
        if (event.x != private->last_x || event.y != private->last_y ||
            event.pressure != private->last_pressure) {
            xf86PostMotionEvent(info->dev, TRUE, 0, 3, event.x, event.y,
                                event.pressure);
            private->last_x = event.x;
            private->last_y = event.y;
            private->last_pressure = event.pressure;
        }

        is_down = !!event.pressure;
        if (is_down != private->is_down) {
            xf86PostButtonEvent(info->dev, 0, 1, is_down, 0, 0);
            private->is_down = is_down;
        }
    }
}

static void
tslib_ptr_ctrl(DeviceIntPtr device, PtrCtrl *ctrl)
{
    /* Nothing to do, dix handles all settings */
}

static int
tslib_add_abs_class(DeviceIntPtr device)
{
    InputInfoPtr info = device->public.devicePrivate;
    struct tslib_private *private = info->private;

    if (!InitValuatorClassDeviceStruct(device, 3, GetMotionHistorySize(),
                                       Absolute))
        return BadAlloc;

    xf86InitValuatorAxisStruct(device, 0, -1, -1, 10000, 0, 10000);
    xf86InitValuatorDefaults(device, 0);
    xf86InitValuatorAxisStruct(device, 1, -1, -1, 10000, 0, 10000);
    xf86InitValuatorDefaults(device, 1);
    xf86InitValuatorAxisStruct(device, 2, 0, 255, 10000, 0, 10000);
    xf86InitValuatorDefaults(device, 2);

    xf86MotionHistoryAllocate(info);

    if (!InitPtrFeedbackClassDeviceStruct(device, tslib_ptr_ctrl))
        return BadAlloc;

    info->flags |= XI86_POINTER_CAPABLE;

    return Success;
}

static int
tslib_add_button_class(DeviceIntPtr device)
{
    CARD8 btn_map[32];

    memset(btn_map, 0, sizeof(btn_map));
    btn_map[1] = 1;

    if (!InitButtonClassDeviceStruct(device, 8, btn_map))
        return BadMatch;

    return Success;
}

static int
tslib_init(DeviceIntPtr device)
{
    int ret;

    ret = tslib_add_abs_class(device);
    if (ret != Success)
        return ret;

    ret = tslib_add_button_class(device);
    if (ret != Success)
        return ret;

    return Success;
}

static int
tslib_disable(DeviceIntPtr device)
{
    InputInfoPtr info = device->public.devicePrivate;
    struct tslib_private *private = info->private;

    xf86RemoveEnabledDevice(info);
    device->public.on = FALSE;

    ts_close(private->ts_dev);
    private->ts_dev = NULL;

    return Success;
}


static int
tslib_enable(DeviceIntPtr device)
{
    InputInfoPtr info = device->public.devicePrivate;
    struct tslib_private *private = info->private;

    private->ts_dev = ts_open(private->path, 1);
    if (!private->ts_dev) {
        xf86Msg(X_ERROR, "Unable to open tslib device \"%s\".\n",
                private->path);
        return BadMatch;
    }

    info->fd = ts_fd(private->ts_dev);
    if (info->fd < 0 || ts_config(private->ts_dev)) {
        xf86Msg(X_ERROR, "Unable to enable tslib device \"%s\".\n",
                private->path);
        ts_close(private->ts_dev);
        return BadMatch;
    }

    private->last_x = -1;
    private->last_y = -1;
    private->last_pressure = -1;
    private->is_down = 0;

    xf86AddEnabledDevice(info);
    device->public.on = TRUE;

    return Success;
}

static int
tslib_uninit(DeviceIntPtr device)
{
    return Success;
}

static int
tslib_device_proc(DeviceIntPtr device, int what)
{
    InputInfoPtr info = device->public.devicePrivate;
    struct tslib_private *private = info->private;

    switch (what) {
    case DEVICE_INIT:
	return tslib_init(device);

    case DEVICE_ON:
        return tslib_enable(device);

    case DEVICE_OFF:
        return tslib_disable(device);

    case DEVICE_CLOSE:
        return tslib_uninit(device);
    }

    return BadMatch;
}


static InputInfoPtr
tslib_preinit(InputDriverPtr drv, IDevPtr dev, int flags)
{
    InputInfoPtr info;
    struct tslib_private *private;

    info = xf86AllocateInput(drv, 0);
    if (!info)
	goto err;

    /* Initialise the InputInfoRec. */
    info->name = dev->identifier;
    info->flags = 0;
    info->type_name = XI_TOUCHSCREEN;
    info->device_control = tslib_device_proc;
    info->read_input = tslib_read;
    info->history_size = 0;
    info->control_proc = NULL;
    info->close_proc = NULL;
    info->switch_mode = NULL;
    info->conversion_proc = NULL;
    info->reverse_conversion_proc = NULL;
    info->dev = NULL;
    info->private_flags = 0;
    info->always_core_feedback = 0;
    info->conf_idev = dev;
    info->fd = -1;

    private = xcalloc(sizeof(*private), 1);
    if (!private)
        goto err;

    info->private = private;

    xf86CollectInputOptions(info, NULL, NULL);
    xf86ProcessCommonOptions(info, info->options);

    private->screen = xf86SetIntOption(info->options, "ScreenNumber", 0);

    private->path = xf86CheckStrOption(dev->commonOptions, "Path", NULL);
    if (!private->path)
        private->path = xf86CheckStrOption(dev->commonOptions, "Device", NULL);
    if (!private->path) {
        xf86Msg(X_ERROR, "%s: No path specified.\n", info->name);
        goto err;
    }

    xf86Msg(X_CONFIG, "%s: Device path: \"%s\"\n", info->name, private->path);
    info->flags |= XI86_CONFIGURED;

    return info;

err:
    if (info)
        xf86DeleteInput(info, 0);
    return NULL;
}

_X_EXPORT InputDriverRec tslib_driver = {
    1,
    "tslib",
    NULL,
    tslib_preinit,
    NULL,
    NULL,
    0
};

static void
tslib_module_unplug(void *p)
{
}

static void *
tslib_module_plug(void *module, void *options, int *errmaj, int *errmin)
{
    xf86AddInputDriver(&tslib_driver, module, 0);
    return module;
}

static XF86ModuleVersionInfo tslib_version =
{
    "tslib",
    MODULEVENDORSTRING,
    MODINFOSTRING1,
    MODINFOSTRING2,
    XORG_VERSION_CURRENT,
    PACKAGE_VERSION_MAJOR, PACKAGE_VERSION_MINOR, PACKAGE_VERSION_PATCHLEVEL,
    ABI_CLASS_XINPUT,
    ABI_XINPUT_VERSION,
    MOD_CLASS_XINPUT,
    {0, 0, 0, 0}
};

_X_EXPORT XF86ModuleData tslibModuleData =
{
    &tslib_version,
    tslib_module_plug,
    tslib_module_unplug
};
