/*
 * xresponse - Interaction latency tester,
 *
 * Written by Ross Burton & Matthew Allum  
 *              <info@openedhand.com> 
 *
 * Copyright (C) 2005 Nokia
 *
 * Licensed under the GPL v2 or greater.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/time.h>

#include <wchar.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/XKBlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/extensions/XTest.h>
#include <X11/extensions/Xdamage.h>
#include <X11/extensions/record.h>

#include "kbd.h"

/* 
 * defs
 */

#define streq(a,b)      (strcmp(a,b) == 0)
#define CMD_STRING_MAXLEN 256
typedef struct Rectangle { int x,y,width,height; } Rectangle;

#define ASIZE(a)	(sizeof(a) / sizeof(a[0]))

/* flags used for optional functionality requiring specific X extensions */
enum {
	XT_FALSE = 0,				
	XT_TRUE = 1,			
	XT_ERROR = -1,	
};

/*
 * Global variables
 */

static FILE      *LogFile = NULL;       	/* The file to output the log output too */
static int        DamageEventNum;       	/* Damage Ext Event ID */
static Atom       AtomTimestamp;        	/* Atom for getting server time */
static int        DamageWaitSecs = -1;   	/* Max time to collect damamge */
static Rectangle  InterestedDamageRect; 	/* Damage rect to monitor */
static Time       LastEventTime;      		/* When last last event was started */
static unsigned long DefaultDelay = 100; 	/* Default delay for key synthesis */
static int								DamageReportLevel = XDamageReportBoundingBox;
static int				BreakOnDamage = 0;			/* break on the specified damage event */
static int				EnableInputMonitoring = XT_FALSE;   /* mouse/keyboard input monitoring */
Display*					RecordDisplay = NULL; 	/* display connection for event recording */
XRecordContext 		RecordContext =  0;			/* XRecord context */

static Bool       SuppressReport = False;  	/* suppresses the standard event reports. Used with --response option */
static char				LastUserAction[256] = {0};	/* the last user action in user friendly format */

/* response time support */
static unsigned int ResponseTimeout = 0; 					/* the timeout for application response checking */
static Time LastUserActionTime = 0; 							/* the last user user action timestamp */


/* region filtering rules */
enum {
	EXCLUDE_NONE,				/* region filtering is not set */
	EXCLUDE_LESS,				/* exclude regions that are less or equal than specified region/size */
	EXCLUDE_GREATER			/* exclude regions that are greater than specified region/size */
};

static Rectangle 		ExcludeRect;									/* Damage rect for filtering rules */
static unsigned int ExcludeSize = -1;							/* Damage rect size for filtering rules */
static unsigned int ExcludeRules = EXCLUDE_NONE;	/* filtering rules */


enum { /* for 'dragging' */
  XR_BUTTON_STATE_NONE,
  XR_BUTTON_STATE_PRESS,
  XR_BUTTON_STATE_RELEASE
};

/* wait timer resolution (msecs) */
#define WAIT_RESOLUTION		100

static unsigned int BreakTimeout = 0;

/*
 * Maximum number of active damage handlers.
 * Increase the size if necessary
 */


/* dynamic application/window monitoring */
#define MAX_APPLICATION_NAME	32
#define MAX_APPLICATIONS			50

/* resource alias for the root window */
#define ROOT_WINDOW_RESOURCE	"*SCREEN*"

/* flag specifying if all applications must be monitored */
Bool MonitorAllApplications = False;

/**
 * Application data structure
 */
typedef struct _application_t {
	
	char name[MAX_APPLICATION_NAME];				/* name of the application resource(binary) file */
	XDamageNotifyEvent firstDamageEvent; 		/* the first damage event after user action */
	XDamageNotifyEvent lastDamageEvent;			/* the last damage event after user action */
	Bool isDynamic;													/* true if the application was added dynamically 
																						 during window creation */
	
} application_t;

application_t* Applications[MAX_APPLICATIONS];  	/* application list */
int ApplicationIndex = 0;												/* first free index in application list */


#define MAX_WINDOWS			100

/**
 * Window data structure
 */
typedef struct _window_t {
	Window window;								/* the window id */
	Damage damage;								/* the damage context */
  application_t* application;   /* the owner appliation */
} window_t;

window_t Windows[MAX_WINDOWS];	/* monitored window list */
int WindowIndex = 0;						/* first free index in monitored window list */



/* function declarations */
static Time get_server_time(Display *dpy);
static void log_action(Time time, int is_stamp, const char *format, ...);


application_t* application_find(const char* name);
application_t* application_add(const char* name, Bool isDynamic);
void application_remove(application_t* app);
/**
 *
 * Window related functionality
 *
 */


/**
 * Adds window to monitored window list.
 * 
 * @param window[in] 	the window being monitored.
 * @return						a reference to the window data structure.
 */
window_t* window_add(Window window, application_t* application) {
	if (WindowIndex <= MAX_WINDOWS) {
		window_t* win = &Windows[WindowIndex++];
		win->window = window;
		win->damage = 0;
		win->application = application;
		return win;
	}
	fprintf(stderr, "[window_add] Failed to add window (exceeding limit of %d windows)\n", MAX_WINDOWS);
	return NULL;
}

/**
 * Clears the monitored window list.
 * 
 * @return
 */
void window_clear(Display* dpy) {
	window_t* win = NULL;
	for (win = Windows; win - Windows < WindowIndex; win++) {
		if (win->damage) XDamageDestroy(dpy, win->damage);
	}
	WindowIndex = 0;
}

/**
 * Searches monitored window list for the specified window.
 *
 * @param window[in]		the window to search.
 * @return							a reference to the located window or NULL otherwise.
 */
window_t* window_find(Window window) {
	window_t* win = NULL;
	for (win = Windows; win - Windows < WindowIndex; win++) {
		if (window == win->window) return win;
	}
	return NULL;
}


/** 
 * Removes window from monitoring list.
 * 
 * @param window[in]		the window to remove.
 * @return
 */
void window_remove(window_t* win, Display* dpy) {
	if (win) {
		application_t* app = win->application;
		window_t* tail = &Windows[--WindowIndex];
		XDamageDestroy(dpy, win->damage);
		if (win != tail) *win = *tail;
		application_remove(app);
	}
}

/**
 * Dump the contents of the monitored windows list.
 * 
 * Used for debug purposes.
 * @return
 */
void window_dump() {
	window_t* win;
	for (win = Windows; win - Windows < WindowIndex; win++) {
		printf("[window] 0x%lx (%s)\n", win->window, win->application->name);
	}
}

/** 
 * Retrieves resource (application) name associated with the window.
 *
 * The root windows is aliased as '*'.
 * @param window[in]		the window which resource name should be retrieved.
 * @param dpy[in]				the default display.
 * @return							a reference to the resource name or NULL. This name will be changed
 *                      by another window_get_resource_name call, so don't store the reference
 *                      but copy contents if needed.
 */
const char* window_get_resource_name(Window window, Display* dpy) {
	static char resourceName[MAX_APPLICATION_NAME];
	XClassHint classhint;
	
	/* alias the root window to '*' resource */
	if (window == DefaultRootWindow(dpy)) {
		strcpy(resourceName, ROOT_WINDOW_RESOURCE);
		return resourceName;
	}
	if(XGetClassHint(dpy, window, &classhint)) {
		strncpy(resourceName, classhint.res_name, MAX_APPLICATION_NAME);
		resourceName[MAX_APPLICATION_NAME - 1] = '\0';
		XFree(classhint.res_name);
		XFree(classhint.res_class);
		return resourceName;
	}
	return NULL;
}

/** 
 * Checks if monitored window list contains windows belonging to 
 * the specified application.
 *
 * @param application[in]		the owner application.
 * @return				         	True if monitored widnow list contains a window belonging
 *                          to the specified application. False otherwise.
 */
Bool window_is_application_monitored(application_t* application) {
	window_t* win = NULL;
	for (win = Windows; win - Windows < WindowIndex; win++) {
		if (application == win->application) {
			return True;
		}
	}
	return False;
}


/**
 * Start monitoring the specified window.
 *
 * @param app[in]				the owner application (can be NULL for windows monitored with --id option).
 * @param dpy[in]				the default display.
 * @return							true if the window monitoring started successfully.
 */
Bool window_start_monitoring(window_t* win, Display* dpy) {
	win->damage = XDamageCreate(dpy, win->window, DamageReportLevel);
	if (!win->damage) {
		fprintf(stderr, "XDamageCreate failed for window %lx, application %s\n", win->window, win->application ? win->application->name : "(null)");
		return False;
	}
	if (win->window != DefaultRootWindow(dpy)) XSelectInput(dpy, win->window, StructureNotifyMask);	
	return True;
}

/**
 * Try to monitor the specified window.
 * 
 * This method checks if the specified window must/can be monitored and 
 * will start monitoring it.
 * @param window[in]			the window to monitor.
 * @param dpy[in]					the default display.
 * @return								a reference to the window data structure if monitoring 
 *                        started successfully. NULL otherwise.
 */
window_t* window_try_monitoring(Window window, Display* dpy) {
	if (!window_find(window)) {
		const char* resource = window_get_resource_name(window, dpy);
		if (resource) {
			application_t* app = application_find(resource);
			if (!app && MonitorAllApplications) {
				app = application_add(resource, True);
			}
			if (app) {
				window_t* win = window_add(window, app);
				if (win && window_start_monitoring(win, dpy)) {
					return win;
				}
				window_remove(win, dpy);
			}
		}
	}
	return NULL;
}


/**
 * Start to monitor all windows in list.
 *
 * This function is called in the begining, to start monitoring all windows
 * which was added manually with --id option.
 * @param dpy[in]		the default display.
 * @return 
 */
void window_monitor_all(Display* dpy) {
	window_t* win = Windows;
	while (win - Windows < WindowIndex) {
		if (window_start_monitoring(win, dpy)) {
			win++;
		}
		else {
			window_remove(win, dpy);
		}
	}
}

/**
 * Tries to monitor children of the specified window.
 * 
 * This function will try to monitor every child of the specified window.
 * If monitoring a child fails, it will try to monitor its children.
 * @param window[in] 	the window which children to monitor.
 * @param dpy[in]			the default display.
 * @return
 */
void window_try_monitoring_children(Window window, Display* dpy) {
	Window root_win, parent_win;
	Window *child_list;
	unsigned int num_children;
	int iWin;
	
	if (!XQueryTree(dpy, window, &root_win, &parent_win, &child_list, &num_children)) {
		fprintf(stderr, "Cannot query window tree (%lx)\n", window);
		return;
	}
	for (iWin = 0; iWin < num_children; iWin++) {	
		if (!window_try_monitoring(child_list[iWin], dpy)) {
			window_try_monitoring_children(child_list[iWin], dpy);
		}
	}
	if (child_list) XFree(child_list);
}

/**
 *
 * Application related functionality
 */

/**
 * Resets application damage events.
 * 
 * @param app[in]		the application.
 */
void application_reset_events(application_t* app) {
	app->firstDamageEvent.timestamp = 0;
	app->lastDamageEvent.timestamp = 0;
}


/** 
 * Resets all application damage events.
 *
 * @return
 */
void application_all_reset_events() {
	application_t** app = NULL;
	for (app = Applications; app - Applications < ApplicationIndex; app++) {
		application_reset_events(*app);
	}
}

/**
 * Adds a new application to application list.
 *
 * When a new window is created xresponse checks application list if it should start 
 * monitoring the created window.
 * @param name[in] 				the application name.
 * @param isDynamic[in]		specifies if application is added dynamically and can be removed
 *                        when all of its windows are destroyed.
 * @return								a reference to the application data structure.
 */
application_t* application_add(const char* name, int isDynamic)  {
	if (ApplicationIndex <= MAX_APPLICATIONS) {
		application_t* app = malloc(sizeof(application_t));
		if (app) {
			Applications[ApplicationIndex++] = app;
			strncpy(app->name, name, MAX_APPLICATION_NAME);
			app->name[MAX_APPLICATION_NAME - 1] = '\0';
			app->isDynamic = isDynamic;
			application_reset_events(app);
			return app;
		}
		fprintf(stderr, "[application_add] failed to allocated memory for the application\n");
		return NULL;
	}
	fprintf(stderr, "[application_add] Failed to add application (exceeding limit of %d applications)\n", MAX_APPLICATIONS);
	return NULL;
}


/**
 * Clears the application list.
 *
 * @return
 */
void application_clear() {
	application_t** app = NULL;
	for (app = Applications; app - Applications < ApplicationIndex; app++) {
		free(*app);
	}
	ApplicationIndex = 0;
}

/**
 * Searches application list for the specified application.
 *
 * @param name[in]		the application name.
 * @return 						a reference to the located application or NULL otherwise.
 */
application_t* application_find(const char* name) {
	application_t** app = NULL;
  
	for (app = Applications; app - Applications < ApplicationIndex; app++) {
		if (!strncmp(name, (*app)->name, MAX_APPLICATION_NAME)) return *app;
	}
	return NULL;
}

/**
 * Removes application from the list.
 *
 * The application can be removed from list if it satisfies the following:
 * 1) application is added dynamically (when all applications are monitored)
 * 2) no pending response time reports (damage event timestamps are 0)
 * 3) no windows referring to this application is monitored.
 *
 * @param name[in] 		the application name.
 * @return
 */
void application_remove(application_t* app) {
	if (app && app->isDynamic && !app->firstDamageEvent.timestamp) {
		if (!window_is_application_monitored(app)) {
			application_t** papp = NULL;
      --ApplicationIndex;
			for (papp = Applications; papp - Applications < ApplicationIndex; papp++) { 
	      if (*papp == app) {
  				*papp = Applications[ApplicationIndex];
					break;
				}
			}
			free(app);
		}
	}
}


/**
 * Logs the response data of applications.
 *
 * @param timestamp		the log event timestamp.
 * @return
 */
 void application_log_response_data(Time timestamp) {
	application_t** app = NULL;
	for (app = Applications; app - Applications < ApplicationIndex; app++) {
		if ((*app)->firstDamageEvent.timestamp) {
			log_action(timestamp, 0, "\t%32s updates: first %5ims, last %5ims\n", (*app)->name ? (*app)->name : "(unknown)",
											(*app)->firstDamageEvent.timestamp - LastUserActionTime,  (*app)->lastDamageEvent.timestamp - LastUserActionTime);
			(*app)->firstDamageEvent.timestamp = 0;
      /* remove any pending applications */
      application_remove(*app);
		}
	}
 }


/**
 * Starts monitoring all windows belonging to the applications in list.
 *
 * @param dpy[in]		the default display.
 * @return
 */
void application_monitor_all(Display* dpy) {
	Window win = DefaultRootWindow(dpy);
	window_try_monitoring(win, dpy);
	window_try_monitoring_children(win, dpy);
}


/**
 * Stores last user action description.
 *
 * The last user action description is used for reporting application response times
 * @param format[in]
 * @param ...
 */
void register_user_action(const char* format, ...) {
	va_list ap;
  va_start(ap, format);
  vsnprintf(LastUserAction, sizeof(LastUserAction), format, ap);
  va_end(ap);
}


/* */
static int 
handle_xerror(Display *dpy, XErrorEvent *e)
{
  /* Really only here for debugging, for gdb backtrace */
  char msg[255];
  XGetErrorText(dpy, e->error_code, msg, sizeof msg);
  fprintf(stderr, "X error (%#lx): %s (opcode: %i:%i)",
				e->resourceid, msg, e->request_code, e->minor_code);

	/* ignore BadWindow errors - when monitoring application a window might 
	   get destroyed before it's being monitored, which would result in BadWindow 
	   error when xresponse tries to monitor it */
	if (e->error_code == BadWindow ||
	/* Unknown error, result of closing and opening a monitored application.
		 Ignore for now. TODO: find out what exactly the erorr means
	*/
		  e->error_code == 151) {
			fprintf(stderr, " - ignored\n");
				return 0;
		}
	fprintf(stderr, "\n");
  exit(1);
}


/** 
 * Perform simple logging with timestamp and diff from last log
 */
static void
log_action(Time time, int is_stamp, const char *format, ...)
{
  va_list     ap;
  char       *tmp = NULL;
  static int  displayed_header;

  va_start(ap,format);
  vasprintf(&tmp, format, ap);
  va_end(ap);

	if (!SuppressReport) {
		if (!displayed_header) { 		/* Header */
			fprintf(LogFile, "\n"
				" Server Time : Diff    : Info\n"
				"-----------------------------\n");
			displayed_header = 1;
		}
		if (is_stamp) {
			fprintf(LogFile, "%s\n", tmp);
		}
		else 	{
			fprintf(LogFile, "%10lums : %5lums : %s",
				time, 
				(LastEventTime > 0 && time > 0 && time > LastEventTime) ? time - LastEventTime : 0,
				tmp);
		}
	}
	else {
		if (!time) fprintf(LogFile, tmp);
	}
	fflush(LogFile);
  if (tmp) free(tmp);
}

/**
 * Get the current timestamp from the X server.
 */
static Time 
get_server_time(Display *dpy) 
{
  XChangeProperty (dpy, DefaultRootWindow (dpy), 
		   AtomTimestamp, AtomTimestamp, 8, 
		   PropModeReplace, (unsigned char*)"a", 1);
  for (;;) {
		XEvent xevent;

		XMaskEvent (dpy, PropertyChangeMask, &xevent);
		if (xevent.xproperty.atom == AtomTimestamp)	return xevent.xproperty.time;
    }
}

/**  
 * Get an X event with a timeout ( in secs ). The timeout is
 * updated for the number of secs left.
 */
static Bool
get_xevent_timed(Display        *dpy, XEvent         *event_return, struct timeval *tv) {

  if (tv == NULL) {
		XRecordProcessReplies(RecordDisplay); 
		XNextEvent(dpy, event_return);
		return True;
	}

  XFlush(dpy);

  if (XPending(dpy) == 0) {
		int    fd = ConnectionNumber(dpy);
		fd_set readset;

		FD_ZERO(&readset);
		FD_SET(fd, &readset);

		if (select(fd+1, &readset, NULL, NULL, tv) == 0) {
			if (RecordDisplay) XRecordProcessReplies(RecordDisplay);
			return False;
		}
	}
	if (RecordDisplay) { /* force input event processing */
			XRecordProcessReplies(RecordDisplay); 
	}
	XNextEvent(dpy, event_return);

	return True;
}

/** 
 * Set up Display connection, required extensions and req other X bits
 */
static Display*
setup_display(char *dpy_name) 
{
  Display *dpy;
  int      unused;

  if ((dpy = XOpenDisplay(dpy_name)) == NULL)
    {
      fprintf (stderr, "Unable to connect to DISPLAY.\n");
      return NULL;
    }

  /* Check the extensions we need are available */

  if (!XTestQueryExtension (dpy, &unused, &unused, &unused, &unused)) {
    fprintf (stderr, "No XTest extension found\n");
    return NULL;
  }

  if (!XDamageQueryExtension (dpy, &DamageEventNum, &unused)) {
    fprintf (stderr, "No DAMAGE extension found\n");
    return NULL;
  }
	
	{
		int major = 0, minor = 0;
		if (!XRecordQueryVersion(dpy, &major, &minor)) {
			fprintf(stderr, "No Record extension found\n");
			EnableInputMonitoring = XT_ERROR;
		}
	}

  XSetErrorHandler(handle_xerror); 

  /* Needed for get_server_time */
  AtomTimestamp = XInternAtom (dpy, "_X_LATENCY_TIMESTAMP", False);  
  XSelectInput(dpy, DefaultRootWindow(dpy), PropertyChangeMask | SubstructureNotifyMask);

  return dpy;
}


/**
 * Eat all Damage events in the X event queue.
 */
static void 
eat_damage(Display *dpy)
{
  while (XPending(dpy)) 
    {
      XEvent              xev;
      XDamageNotifyEvent *dev;

      XNextEvent(dpy, &xev);

      if (xev.type == DamageEventNum + XDamageNotify) 
	{
	  dev = (XDamageNotifyEvent*)&xev;
	  XDamageSubtract(dpy, dev->damage, None, None);
	}
    }
}

/** 
 * 'Fakes' a mouse click, returning time sent.
 */
static Time
fake_event(Display *dpy, int x, int y, int delay)
{
  Time start;

  XTestFakeMotionEvent(dpy, DefaultScreen(dpy), x, y, CurrentTime);
	
  /* Eat up any damage caused by above pointer move */
  eat_damage(dpy);

  start = get_server_time(dpy);

  /* 'click' mouse */
  XTestFakeButtonEvent(dpy, Button1, True, CurrentTime);
  XTestFakeButtonEvent(dpy, Button1, False, delay ? delay : CurrentTime);

  return start;
}

static Time
drag_event(Display *dpy, int x, int y, int button_state, int delay)
{
  Time start;

  start = get_server_time(dpy);

  XTestFakeMotionEvent(dpy, DefaultScreen(dpy), x, y, delay ? delay : CurrentTime);

  if (button_state == XR_BUTTON_STATE_PRESS)
    {
      eat_damage(dpy); 	/* ignore damage from first drag */
      XTestFakeButtonEvent(dpy, Button1, True, CurrentTime);
    }

  if (button_state == XR_BUTTON_STATE_RELEASE)
    XTestFakeButtonEvent(dpy, Button1, False, CurrentTime);

  return start;
}

/** 
 * Waits for a damage 'response' to above click / keypress(es)
 */
static int
wait_response(Display *dpy)
{
  XEvent e;
	int elapsed = 0;
	int lastEvent = 0;
	Time responseStart = 0;
	int lastResponse = 0;

	while ((!DamageWaitSecs || elapsed < DamageWaitSecs * 1000)) {
		/* check if break timeout is specified and elapsed */
		if (BreakTimeout && elapsed - lastEvent > BreakTimeout) break;
		if (RecordDisplay) XRecordProcessReplies(RecordDisplay);
		struct timeval tv = {0, WAIT_RESOLUTION * 1000};
		
		if (get_xevent_timed(dpy, &e, &tv)) {
			if (RecordDisplay) XRecordProcessReplies(RecordDisplay);
			if (e.type == DamageEventNum + XDamageNotify) {
				XDamageNotifyEvent *dev = (XDamageNotifyEvent*)&e;
				int xpos = dev->area.x + dev->geometry.x;
				int ypos = dev->area.y + dev->geometry.y;
				/* check if the damage are is in the monitoring area */
				if (xpos + dev->area.width >= InterestedDamageRect.x &&
						xpos <= (InterestedDamageRect.x +	InterestedDamageRect.width) &&
						ypos + dev->area.height >= InterestedDamageRect.y &&
						ypos <= (InterestedDamageRect.y + InterestedDamageRect.height)) {
					/* check if the damage area satisfies filtering rules */
					if (ExcludeRules == EXCLUDE_NONE ||
							(ExcludeRules == EXCLUDE_LESS && (
												(ExcludeSize && ExcludeSize < dev->area.width * dev->area.height) || 
												(!ExcludeSize && (dev->area.width > ExcludeRect.width || dev->area.height > ExcludeRect.height)) ) ) ||
							(ExcludeRules == EXCLUDE_GREATER && (
												(ExcludeSize &&	ExcludeSize >= dev->area.width * dev->area.height) ||
												(!ExcludeSize && (dev->area.width <= ExcludeRect.width || dev->area.height <= ExcludeRect.height)) ) )  ){
						window_t* win = window_find(dev->drawable);							
						lastEvent = elapsed;
						
						log_action(dev->timestamp, 0, "Got damage event %dx%d+%d+%d from 0x%lx (%s)\n",
																dev->area.width, dev->area.height, xpos, ypos, dev->drawable,
																win && win->application ? win->application->name : "unknown");
						LastEventTime = dev->timestamp;

						if (LastUserActionTime) {
							window_t* win = window_find(dev->drawable);
							if (win && win->application) {
								if (win->application->firstDamageEvent.timestamp < LastUserActionTime) win->application->firstDamageEvent = *dev;
								win->application->lastDamageEvent = *dev;
							}
							if (!responseStart) {
								responseStart = dev->timestamp;
								lastResponse = elapsed;
							}
						}
						if (BreakOnDamage && !(--BreakOnDamage)) return 1;
					}
				}
				XDamageSubtract(dpy, dev->damage, None, None);
			} 
			else if (e.type == CreateNotify) {
				/* check new windows if we have to monitor them */
				XCreateWindowEvent* ev = (XCreateWindowEvent*)&e;
				/* TODO: Check if we really must monitor only main windows.
				   Probably done to avoid double reporting. We could avoid that by 
				   going through monitored window list, checking if this window
				   is in the parent chain of any monitored window. If so, remove the child.
				   Might be expensive though, but worth a try.
				*/
				if (ev->parent == DefaultRootWindow(dpy)) {
					window_t* win = window_try_monitoring(ev->window, dpy);
					if (win) {
						Time start = get_server_time(dpy);
						log_action(start, 0, "Craeted window 0x%lx (%s)\n", ev->window, win->application ? win->application->name : "unknown");
						LastEventTime = start;
					}
				}
			}
			else if (e.type == UnmapNotify) {
				XUnmapEvent* ev = (XUnmapEvent*)&e;
				window_t* win = window_find(ev->window);
				if (win) {
					Time start = get_server_time(dpy);
					log_action(start, 0, "Unmapped window 0x%lx (%s)\n", ev->window, win->application ? win->application->name : "unknown");
					LastEventTime = start;
				}
			}
			else if (e.type == MapNotify) {
				XMapEvent* ev = (XMapEvent*)&e;
				window_t* win = window_find(ev->window);
				if (win) {
					Time start = get_server_time(dpy);
					log_action(start, 0, "Mapped window 0x%lx (%s)\n", ev->window, win->application ? win->application->name : "unknown");
					LastEventTime = start;
				}
			}
			else if (e.type == DestroyNotify) {
				XDestroyWindowEvent* ev = (XDestroyWindowEvent*)&e;
				window_t* win = window_find(ev->window);
				if (win) {
					Time start = get_server_time(dpy);
					log_action(start, 0, "Destroyed window 0x%lx (%s)\n", ev->window, win->application ? win->application->name : "unknown");
					window_remove(win, dpy);
					LastEventTime = start;
				}
			}
			else {
				/* remove to avoid reporting unwanted even types ?
				   with window creation monitoring there are more unhandled event types */
				/* fprintf(stderr, "Got unwanted event type %d\n", e.type); */
			}
		}
		elapsed += WAIT_RESOLUTION - tv.tv_usec / 1000;
		
		if (LastUserActionTime && responseStart + elapsed - lastResponse > LastUserActionTime + ResponseTimeout) {
			log_action(0, 0, "Device response time to %s:\n", LastUserAction);
			application_log_response_data(0);
			LastUserActionTime = 0;
			responseStart = 0;
			lastResponse = 0;
		}
	}
	
  return 0;
}

void
usage(char *progname)
{
  fprintf(stderr, "%s: usage, %s <-o|--logfile output> [commands..]\n" 
	          "Commands are any combination/order of;\n"
	          "-c|--click <XxY[,delay]>            Send click and await damage response\n" 
	          "                                    Delay is in milliseconds.\n"
						"                                    If not specified no delay is used\n"
	          "-d|--drag <delay|XxY,delay|XxY,...> Simulate mouse drag and collect damage\n"
	          "                                    Optionally add delay between drag points\n"
	          "-k|--key <keysym[,delay]>           Simulate pressing and releasing a key\n"
	          "                                    Delay is in milliseconds.\n"
						    "                                If not specified, default of %lu ms is used\n"
	          "-m|--monitor <WIDTHxHEIGHT+X+Y>     Watch area for damage ( default fullscreen )\n"
	          "-w|--wait <seconds>                 Max time to wait for damage, set to 0 to\n"
	          "                                    monitor for ever.\n"
	          "                                    ( default 5 secs)\n"
	          "-s|--stamp <string>                 Write 'string' to log file\n"
	          "-t|--type <string>                  Simulate typing a string\n"
	          "-i|--inspect                        Just display damage events\n"
	          "-id|--id <id>                       Resource id of window to examine\n"                            
	          "-v|--verbose                        Output response to all command line options\n"
						"-a|--application <name>             Monitor windows related to the specified application.\n"
						"                                    Use '*' to monitor all applications.\n"
						"-x|--exclude <XxY|S[,less|greater]> Exclude regions from damage reports based on their size.\n"
						"                                    Specify either region dimensions XxY or size S.\n"
						"                                    less - exclude regions less or equal than specified (default).\n" 
						"                                    greater - exclude regions graeter than specified.\n"
						"-b|--break <msec>|damage[,<number>] Break the wait if no damage was registered in <msec> period,\n"
						"                                    or after the <number> damage event if 'damage' was specified.\n"
						"-l|--level <raw|delta|box|nonempty> Specify the damage reporting level.\n"
						"-u|--user                           Enable user input monitoring.\n"
						"-r|--response <timeout[,verbose]>   Enable application response monitoring (timeout given in msecs).\n"
						"                                    If verbose is not specified the damage reporting will be suppresed.\n"
						"\n",
	  progname, progname, DefaultDelay);
  exit(1);
}

/* Code below this comment has been copied from xautomation / vte.c and
 * modified minimally as required to co-operate with xresponse. */

/* All key events need to go through here because it handles the lookup of
 * keysyms like Num_Lock, etc.  Thing should be something that represents a
 * single key on the keyboard, like KP_PLUS or Num_Lock or just A */
static KeyCode 
thing_to_keycode(Display *dpy, char *thing ) 
{
  KeyCode kc;
  KeySym ks;
  
  ks = XStringToKeysym(thing);
  if (ks == NoSymbol){
    fprintf(stderr, "Unable to resolve keysym for '%s'\n", thing);
    return(thing_to_keycode(dpy, "space" ));
  }

  kc = XKeysymToKeycode(dpy, ks);
  return(kc);
}

/* Simulate pressed key(s) to generate thing character
 * Only characters where the KeySym corresponds to the Unicode
 * character code and KeySym < MAX_KEYSYM are supported,
 * except the special character 'Tab'. */
static Time send_string( Display *dpy, char *thing_in ) 
{
  KeyCode wrap_key;
  int i = 0;

  KeyCode keycode;
  KeySym keysym;
  Time start;

  wchar_t thing[ CMD_STRING_MAXLEN ];
  wchar_t wc_singlechar_str[2];
  wmemset( thing, L'\0', CMD_STRING_MAXLEN );
  mbstowcs( thing, thing_in, CMD_STRING_MAXLEN );
  wc_singlechar_str[ 1 ] = L'\0';

  eat_damage(dpy);
  start = get_server_time(dpy);

  while( ( thing[ i ] != L'\0' ) && ( i < CMD_STRING_MAXLEN ) ) {

    wc_singlechar_str[ 0 ] = thing[ i ];

    /* keysym = wchar value */

    keysym = wc_singlechar_str[ 0 ];
		
		/* Keyboard modifier and KeyCode lookup */
		wrap_key = keysym_to_modifier_map[ keysym ];
		keycode = keysym_to_keycode_map[keysym];

    if( keysym >= MAX_KEYSYM || !keycode) {
			fprintf( stderr, "Special character '%ls' is currently not supported.\n", wc_singlechar_str );
		}
		else {
			if( wrap_key ) XTestFakeKeyEvent( dpy, wrap_key, True, CurrentTime );
			XTestFakeKeyEvent( dpy, keycode, True, CurrentTime );
			XTestFakeKeyEvent( dpy, keycode, False, CurrentTime );
			if( wrap_key ) XTestFakeKeyEvent( dpy, wrap_key, False, CurrentTime );

			/* Not flushing after every key like we need to, thanks
			 * thorsten@staerk.de */
			XFlush( dpy );
    }
    
    i++;

  }
  return start;
}

#define MAX_SHIFT_LEVELS		256

/* Load keycodes and modifiers of current keyboard mapping into arrays,
 * this is needed by the send_string function */
void 
load_keycodes( Display *dpy ) 
{
  char *str;
  KeySym keysym;
  KeyCode keycode;
	KeyCode mask_to_modifier_map[MAX_SHIFT_LEVELS] = {0};
	int ikey, ishift, igroup;

	/* reset mapping tables */
	memset(keysym_to_modifier_map, 0, sizeof(keysym_to_modifier_map));
	memset(keysym_to_keycode_map, 0, sizeof(keysym_to_keycode_map));
	
	/* initialize [modifier mask: modifier keycode] mapping table */
	XModifierKeymap* modkeys = XGetModifierMapping(dpy);
	for (ishift = 0; ishift < 8 * modkeys->max_keypermod; ishift += modkeys->max_keypermod) {
		if (modkeys->modifiermap[ishift]) {
			mask_to_modifier_map[1 << (ishift / modkeys->max_keypermod)] = modkeys->modifiermap[ishift];
		}
	}
	XFreeModifiermap(modkeys);
	
	/* acquire keycode:keysyms mapping */
	XkbDescPtr xkb = XkbGetMap(dpy, XkbAllClientInfoMask, XkbUseCoreKbd);
	
	/* initialize [keysym:keycode] and [keysym:modifier keycode] tables */
	for (ikey = xkb->min_key_code; ikey <= xkb->max_key_code; ikey++) {
		XkbSymMapPtr mkeys = &xkb->map->key_sym_map[ikey];
		for (igroup = 0; igroup < XkbKeyNumGroups(xkb, ikey); igroup++) {
			XkbKeyTypePtr keytype = &xkb->map->types[mkeys->kt_index[igroup]];
			unsigned char levels[MAX_SHIFT_LEVELS] = {0};
			/*  make temporary [shift level: modifier mask] mapping */
			for (ishift = 0; ishift < keytype->map_count; ishift++) {
				if (!levels[keytype->map[ishift].level]) levels[keytype->map[ishift].level] = keytype->map[ishift].mods.mask;
			}
			
			for (ishift = 0; ishift < keytype->num_levels; ishift++) {
				str = XKeysymToString(XkbKeySymsPtr(xkb, ikey)[ishift]);
				
				if( str != NULL ) {
					keysym = XStringToKeysym( str );
					keycode = XKeysymToKeycode( dpy, keysym );

					if( ( keysym < MAX_KEYSYM ) && ( !keysym_to_keycode_map[ keysym ]) ) {
						keysym_to_modifier_map[ keysym ] = mask_to_modifier_map[levels[ishift]];
						keysym_to_keycode_map[ keysym ] = keycode;
					}
				}
			}
		}
	}
  XkbFreeClientMap(xkb, XkbAllClientInfoMask, True);
}


static Time 
send_key(Display *dpy, char *thing, unsigned long delay) 
{
  Time start;

  /* It seems that we need to eat the initial 800x480 - sized damage
     event here, which has earlier timestamp than anything caused
     by the keypress/-release pair, where does it come from? And does
     this make us miss any other events? */

  eat_damage(dpy);
  start = get_server_time(dpy);
  XTestFakeKeyEvent(dpy, thing_to_keycode( dpy, thing ), True, CurrentTime);
  XTestFakeKeyEvent(dpy, thing_to_keycode( dpy, thing ), False, delay);
  return start;
}


void reset_application_response_monitoring(Time timestamp) {
	if (ResponseTimeout) {
		if (LastUserActionTime) {
			log_action(0, 0, "Warning, new user event received in the middle of update. "
															 "It is possible that update time is not correct.\n");
			application_all_reset_events();
		}
		LastUserActionTime = timestamp;
	}
}


void record_callback(XPointer closure, XRecordInterceptData* data) {
	if (XRecordFromServer == data->category) {
		Display* dpy = (Display*)closure;
		xEvent* xev = (xEvent*)data->data;
		int type = xev->u.u.type;
		static int x,y;
				
		switch (type) {
			case ButtonPress:
				log_action(xev->u.keyButtonPointer.time, 0, "Button %x pressed at %dx%d\n", xev->u.u.detail, x, y);
				LastEventTime = xev->u.keyButtonPointer.time;
				break;
			
			case ButtonRelease:
				log_action(xev->u.keyButtonPointer.time, 0, "Button %x released\n", xev->u.u.detail);
				LastEventTime = xev->u.keyButtonPointer.time;
				register_user_action("click (%dx%d)", x, y);
				reset_application_response_monitoring(xev->u.keyButtonPointer.time);
				break;
			
			case KeyPress:
				log_action(xev->u.keyButtonPointer.time, 0, "Key %s pressed\n",
											XKeysymToString(XKeycodeToKeysym(dpy, xev->u.u.detail, 0)));
				LastEventTime = xev->u.keyButtonPointer.time;
				break;
				
			case KeyRelease:
				log_action(xev->u.keyButtonPointer.time, 0, "Key %s released\n",
											XKeysymToString(XKeycodeToKeysym(dpy, xev->u.u.detail, 0)));
				LastEventTime = xev->u.keyButtonPointer.time;
				break;
			
			case MotionNotify:
				/*
				log_action(xev->u.keyButtonPointer.time, 0, "Pointer moved to %dx%d\n",
											xev->u.keyButtonPointer.rootX, xev->u.keyButtonPointer.rootY);
				LastEventTime = xev->u.keyButtonPointer.time;
				*/
				x = xev->u.keyButtonPointer.rootX;
				y = xev->u.keyButtonPointer.rootY;
				break;
			
			default:
				fprintf(stderr, "Unknown device event type %d\n", type);
				break;
		}
		
	}
	
	XRecordFreeData(data);
}

void start_user_input_monitoring(Display* dpy) {
	if (EnableInputMonitoring == XT_ERROR) {
		fprintf(stderr, "Can't monitor user input wihout xrecord extension\n");
		exit(-1);
	}
	
	if (EnableInputMonitoring == XT_FALSE) {
		XRecordClientSpec clients = XRecordAllClients;
		int numRanges = 3;
		int iRange = 0;
		XRecordRange** recRange = 0;
		
		RecordDisplay = XOpenDisplay(getenv("DISPLAY"));
		if (!RecordDisplay) {
			fprintf(stderr, "Failed to open event recording display connection\n");
			exit(-1);
		}
		/* prepare event range data */
		recRange = malloc(sizeof(XRecordRange*) * numRanges);
		if (!recRange) {
			fprintf(stderr, "Failed to allocate record range array\n");
			exit(-1);
		}
		XRecordRange* range;
		XRecordRange** ptrRange = recRange;

		range = XRecordAllocRange();
		range->device_events.first = KeyPress;
		range->device_events.last = KeyPress;
		*ptrRange++ = range;
		
		range = XRecordAllocRange();
		range->device_events.first = ButtonPress;
		range->device_events.last = ButtonRelease;
		*ptrRange++ = range;

		range = XRecordAllocRange();
		range->device_events.first = MotionNotify;
		range->device_events.last = MotionNotify;
		*ptrRange++ = range;
		
		/* */
		RecordContext = XRecordCreateContext(dpy, XRecordFromServerTime, &clients, 1, recRange, numRanges);
		XFlush(dpy);
		XFlush(RecordDisplay);
		XRecordEnableContextAsync(RecordDisplay, RecordContext, record_callback, (XPointer)dpy);

		/* release range data */
		for (iRange = 0; iRange < numRanges; iRange++)	{
			free(recRange[iRange]);
		}
		free(recRange);
	
		EnableInputMonitoring = XT_TRUE;
	}
}

/* Code copy from xautomation / vte.c ends */


int 
main(int argc, char **argv) 
{
  Display *dpy = NULL;
	int      cnt, x, y, i = 0, verbose = 0;
  Window win = 0;
	Bool keysymMappingInitialized = False;
	int rc = 0;
	int inputEvents[100];
	int inputEventsIndex = 0;
	int iEvent = 0;

  if (argc == 1)
    usage(argv[0]);

  if (streq(argv[1],"-o") || streq(argv[1],"--logfile"))
    {
      i++;

      if (++i > argc) usage (argv[0]);

      if ((LogFile = fopen(argv[i], "w")) == NULL)
	fprintf(stderr, "Failed to create logfile '%s'\n", argv[i]);
    }

  if (LogFile == NULL) 
    LogFile = stdout;

  if ((dpy = setup_display(getenv("DISPLAY"))) == NULL)
    exit(1);

	LastEventTime = get_server_time(dpy);
	log_action(LastEventTime, 0, "Startup\n");
	
	/*
	 * Process the command line options.
	 * Skip emulation options (--click, --drag, --key, --type), but remember they index
	 * and process them later.
	 */
  while (++i < argc) {

    if (streq(argv[i],"-v") || streq(argv[i],"--verbose")) {
			verbose = 1;
			continue;
		}
      
		if (streq(argv[i], "-id") || streq(argv[i], "--id")) {
			char name[MAX_APPLICATION_NAME];
			if (++i>=argc) usage (argv[0]);
	  
			cnt = sscanf(argv[i], "0x%lx", &win);
			if (cnt < 1) {
	      cnt = sscanf(argv[i], "%lu", &win);
	    }
			if (cnt < 1) {
	      fprintf(stderr, "*** invalid window id '%s'\n", argv[i]);
	      usage(argv[0]);
	    }
			sprintf(name, "0x%lx", win);
			if (!window_add(win, application_add(name, False))) {
				fprintf(stderr, "Could not setup damage monitoring for window 0x%lx!\n",	win);
				exit(1);
			}
			if (verbose) log_action(LastEventTime, 0, "Monitoring window 0x%lx\n", win);

			continue;
		}
	
		if (streq(argv[i], "-a") || streq(argv[i], "--application")) {
			if (++i >= argc) usage (argv[0]);
		
			
			if (application_add(argv[i], False) && verbose) {
				log_action(LastEventTime,0, "Monitoring application '%s'\n", argv[i]);
			}
			if (!strcmp(argv[i], "*")) {
				MonitorAllApplications = True;
			}
			continue;
		}
	
		if (streq("-c", argv[i]) || streq("--click", argv[i])) {
			if (inputEventsIndex == ASIZE(inputEvents)) {
				fprintf(stderr, "Too many input events specified\n");
				exit(-1);
			}
			inputEvents[inputEventsIndex++] = i;
			if (++i>=argc) usage (argv[0]);
	  
			continue;
		}

		if (streq("-l", argv[i]) || streq("--level", argv[i])) {
			if (++i>=argc) usage (argv[0]);
			
			if (!strcmp(argv[i], "raw")) {
				DamageReportLevel = XDamageReportRawRectangles;
			}
			else if (!strcmp(argv[i], "delta")) {
				DamageReportLevel = XDamageReportDeltaRectangles;
			}
			else if (!strcmp(argv[i], "box")) {
				DamageReportLevel = XDamageReportDeltaRectangles;
			}
			else if (!strcmp(argv[i], "nonempty")) {
				DamageReportLevel = XDamageReportNonEmpty;
			}
			else {
				fprintf(stderr, "Unrecongnized damage level: %s\n", argv[i]);
				usage(argv[0]);
			}
			if (verbose) log_action(LastEventTime, 0, "Setting damage report level to %s\n", argv[i]);
			continue;
		}
		
		/* probably deprecated since moving from command sequence approach */
		if (streq("-s", argv[i]) || streq("--stamp", argv[i])) {
			if (++i>=argc) usage (argv[0]);
			log_action(LastEventTime, 1, argv[i]);
			continue;
		}

		if (streq("-x", argv[i]) || streq("--exclude", argv[i])) {
			char* exclude[] = {"none", "less", "greater"};
			
			if (ExcludeRules != EXCLUDE_NONE) {
				fprintf(stderr, "Duplicated --exclude parameter detected. Aborting\n");
				exit(-1);
			}
			
			if (++i>=argc) usage (argv[0]);
			char rules[32] = "";
			if ((cnt = sscanf(argv[i], "%ux%u,%s", &ExcludeRect.width, &ExcludeRect.height, rules)) >= 2) {
				ExcludeSize = 0;
			}
			else if ((cnt = sscanf(argv[i], "%u,%s", &ExcludeSize, rules)) >= 1) {
				ExcludeRect.width = 0;
				ExcludeRect.height = 0;
			}
			else {
				fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
				usage(argv[0]);
			}
			ExcludeRules = *rules && !strcmp(rules, "greater") ? EXCLUDE_GREATER : EXCLUDE_LESS;
			if (verbose) {
				if (ExcludeSize) {
					log_action(LastEventTime, 0, "Excluding damage areas %s than %d pixels\n", exclude[ExcludeRules], ExcludeSize);
				}
				else {
					log_action(LastEventTime, 0, "Excluding damage areas %s than (%dx%d)\n", exclude[ExcludeRules], ExcludeRect.width, ExcludeRect.height);
				}
			}
			continue;
		}
      
		if (streq("-m", argv[i]) || streq("--monitor", argv[i])) {
			if (InterestedDamageRect.width || InterestedDamageRect.height ||
							InterestedDamageRect.x || InterestedDamageRect.y) {
				fprintf(stderr, "Duplicated --monitor parameter detected. Aborting\n");
				exit(-1);
			}
			if (++i>=argc) usage (argv[0]);
				
			if ((cnt = sscanf(argv[i], "%ux%u+%u+%u", 
						&InterestedDamageRect.width,
						&InterestedDamageRect.height,
						&InterestedDamageRect.x,
						&InterestedDamageRect.y)) != 4) {
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      usage(argv[0]);
	    }
			if (verbose) {
				log_action(LastEventTime, 0, "Set monitor rect to %ix%i+%i+%i\n",
						InterestedDamageRect.width,InterestedDamageRect.height,
						InterestedDamageRect.x,InterestedDamageRect.y);
			}
			continue;
		}

		if (streq("-w", argv[i]) || streq("--wait", argv[i])) {
			if (++i>=argc) usage (argv[0]);
	  
			if (DamageWaitSecs >= 0) {
				fprintf(stderr, "Duplicate -w(--wait) option detected. Discarding the previous value\n");
			}
			if ((DamageWaitSecs = atoi(argv[i])) < 0) {
				fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
				usage(argv[0]);
			}
			if (verbose) log_action(LastEventTime, 0, "Set event timeout to %isecs\n", DamageWaitSecs);

			continue;
		}
	
		if (streq("-b", argv[i]) || streq("--break", argv[i])) {
			if (BreakTimeout || BreakOnDamage) {
				fprintf(stderr, "Duplicate -b(--break)option detected. Discarding the previous value\n");
				BreakTimeout = 0;
				BreakOnDamage = 0;
			}
			if (++i>=argc) usage (argv[0]);
			
			if (!strncmp(argv[i], "damage", 6)) {
				sscanf(argv[i] + 6, ",%d", &BreakOnDamage);
				if (!BreakOnDamage) BreakOnDamage = 1;
				if (verbose) log_action(LastEventTime, 0, "Break wait on the %d damage event\n", BreakOnDamage);
			}
			else {
				if ((BreakTimeout = atoi(argv[i])) < 0) {
					fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
					usage(argv[0]);
				}
				if (verbose) log_action(LastEventTime, 0, "Set break timout to %imsecs\n", BreakTimeout);
			}
			continue;
		}

		if (streq("-d", argv[i]) || streq("--drag", argv[i])) {
			if (inputEventsIndex == ASIZE(inputEvents)) {
				fprintf(stderr, "Too many input events specified\n");
				exit(-1);
			}
			inputEvents[inputEventsIndex++] = i;
			
			if (++i>=argc) usage (argv[0]);
			continue;
		}

    if (streq("-k", argv[i]) || streq("--key", argv[i])) {
			if (inputEventsIndex == ASIZE(inputEvents)) {
				fprintf(stderr, "Too many input events specified\n");
				exit(-1);
			}
			inputEvents[inputEventsIndex++] = i;
			if (++i>=argc) usage (argv[0]);
			
			continue;
		}
      
		if (streq("-t", argv[i]) || streq("--type", argv[i])) {
			if (inputEventsIndex == ASIZE(inputEvents)) {
				fprintf(stderr, "Too many input events specified\n");
				exit(-1);
			}
			inputEvents[inputEventsIndex++] = i;
			if (++i>=argc) usage (argv[0]);
			
			if (!keysymMappingInitialized) {
				load_keycodes(dpy);
				keysymMappingInitialized = True;
			}
			
  		continue;
		}

		/* since moving from command sequence approach the inspect parameter is deprecated */
		if (streq("-i", argv[i]) || streq("--inspect", argv[i])) {
			if (verbose)
				log_action(LastEventTime, 0, "Just displaying damage events until timeout\n");
			continue;
		}
	
		/* */
		if (streq("-u", argv[i]) || streq("--user", argv[i])) {
			start_user_input_monitoring(dpy);
			if (verbose)
				log_action(LastEventTime, 0, "Reporting user input events\n");

			continue;
		}
		
		if (streq(argv[i], "-r") || streq(argv[i], "--response")) {
			if (++i>=argc) usage (argv[0]);
			char option[500];
			cnt = sscanf(argv[i], "%u,%s", &ResponseTimeout, option);
			if (cnt < 1) {
	      fprintf(stderr, "*** invalid response timeout value '%s'\n", argv[i]);
	      usage(argv[0]);
	    }
			if (cnt < 2) {
				SuppressReport = True;
			}
			else {
				if (strcmp(option, "verbose")) {
					fprintf(stderr, "*** invalid response option '%s'\n", argv[i]);
					usage(argv[0]);
				}
			}
			application_add(ROOT_WINDOW_RESOURCE	, False);
			start_user_input_monitoring(dpy);
			if (verbose) log_action(LastEventTime, 0, "Monitoring application response time\n");

			continue;
		}
		
		fprintf(stderr, "*** Dont understand  %s\n", argv[i]);
		usage(argv[0]);
	}

	/* Run */
	
	window_monitor_all(dpy);
	application_monitor_all(dpy);

	/* start monitoring the root window if no targets are specified */	
	if ((!WindowIndex && !ApplicationIndex) || ResponseTimeout) {
		window_try_monitoring(DefaultRootWindow(dpy), dpy);
	}

	/* eat first damage event when BreakOnDamage set */
	if (BreakOnDamage) eat_damage(dpy);
	
	/* monitor the whole screen of no area is specified */
	if (!InterestedDamageRect.width && !InterestedDamageRect.height &&
				  !InterestedDamageRect.x && !InterestedDamageRect.y) {
		InterestedDamageRect.x = 0;
		InterestedDamageRect.y = 0;
		InterestedDamageRect.width  = DisplayWidth(dpy, DefaultScreen(dpy));
		InterestedDamageRect.height = DisplayHeight(dpy, DefaultScreen(dpy));
	}
	
	/* emulate user input */
	
	for (iEvent = 0; iEvent < inputEventsIndex; iEvent++) {
		i = inputEvents[iEvent];
		
		if (!strcmp("-c", argv[i]) || !strcmp("--click", argv[i])) {
			unsigned long delay = 0;
			Time start = 0;
			cnt = sscanf(argv[++i], "%ux%u,%lu", &x, &y, &delay);
			if (cnt == 2) {
				log_action(LastEventTime, 0, "Using no delay between press/release\n");
				delay = 0;
			}
			else if (cnt != 3) {
				fprintf(stderr, "cnt: %d\n", cnt);
				fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
				usage(argv[0]);
			}
			/* Send the event */
			start = fake_event(dpy, x, y, delay);
			log_action(start, 0, "Clicked %ix%i\n", x, y);
			LastEventTime = start;
			
			continue;
		}
		
		if (!strcmp("-d", argv[i]) || !strcmp("--drag", argv[i])) {
			Time drag_time;
			char *s = NULL, *p = NULL;
			unsigned long delay = 0;
			int first_drag = 1, button_state = XR_BUTTON_STATE_PRESS;
			
			s = p = argv[++i];
		  while (1) {
	      if (*p == ',' || *p == '\0') {
					Bool end = False;

					if (*p == '\0') {
						if (button_state == XR_BUTTON_STATE_PRESS) {
							fprintf(stderr, "*** Need at least 2 drag points!\n");
							usage(argv[0]);
						}
						
						/* last passed point so make sure button released */
						 button_state = XR_BUTTON_STATE_RELEASE;
						 end = True;
					}
					else *p = '\0';

					cnt = sscanf(s, "%ux%u", &x, &y);
					if (cnt == 2) {
						/* Send the event */
						drag_time = drag_event(dpy, x, y, button_state, delay);
						if (first_drag) {
							first_drag = 0;
						}
						log_action(drag_time, 0, "Dragged to %ix%i\n", x, y);
						LastEventTime = drag_time;

						/* Make sure button state set to none after first point */
						button_state = XR_BUTTON_STATE_NONE;
					}
					else if (cnt == 1) {
						delay = x;
					}
					else {
						fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
						usage(argv[0]);
					}

					if (end) break;
					s = p+1;
				}
				p++;
			}
			continue;
		}
		
    if (!strcmp("-k", argv[i]) || !strcmp("--key", argv[i])) {
			char *key = NULL;
			char separator;
			unsigned long delay = 0;
			Time start = 0;

			cnt = sscanf(argv[++i], "%a[^,]%c%lu", &key, &separator, &delay);
			if (cnt == 1) {
	      log_action(LastEventTime, 0, "Using default delay between press/release\n", delay);
	      delay = DefaultDelay;
			}
			else if (cnt != 3 || separator != ',') {
	      fprintf(stderr, "cnt: %d\n", cnt);
	      fprintf(stderr, "*** failed to parse '%s'\n", argv[i]);
	      if (key != NULL) free(key);
	      usage(argv[0]);
	    }
			start = send_key(dpy, key, delay);
			log_action(start, 0, "Simulating keypress/-release pair (keycode '%s')\n",  key);
			LastEventTime = start;
			free(key);
			
			continue;
		}
      
		if (!strcmp("-t", argv[i]) || !strcmp("--type", argv[i])) {
			Time start = send_string(dpy, argv[++i]);
			log_action(start, 0, "Simulated keys for '%s'\n", argv[i]);
			LastEventTime = start;
			
  		continue;
		}
		
	}

	/* setting the default wait period */
	if (DamageWaitSecs < 0) {
		DamageWaitSecs = 5;
	}
	
	/* wait for damage events */
  rc = wait_response(dpy);

	if (EnableInputMonitoring == XT_TRUE) {
		XRecordDisableContext(dpy, RecordContext);
		XRecordFreeContext(dpy, RecordContext);
		XFlush(dpy);
		XCloseDisplay(RecordDisplay); 
	}
	
  /* Clean Up */
	window_clear(dpy);
	application_clear();
	
  XCloseDisplay(dpy);
  fclose(LogFile);

  return rc;
}
