/*
 * Program to visualize what parts of the screen are updated
 * according to the xresponse output. Requires a screenshot
 * of the logged activity.
 *
 * Uses:
 *   - SDL for graphics.
 *   - SDL_ttf for text rendering.
 *   - SDL_image for loading the screenshot PNG file.
 *   - Fontconfig for obtaining a good default font that we
 *     use with SDL_ttf.
 *   - POSIX regex for regular expression matching.
 * 
 * This file is part of xresponse-visualize
 * 
 * Copyright (C) 2009 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <sys/types.h>
#include <stdbool.h>
#include <stdlib.h>
#include <limits.h>
#include <regex.h>

#include <SDL.h>
#include <SDL_image.h>
#include <SDL_ttf.h>

#include <fontconfig/fontconfig.h>

static const char progname[] = "xresponse-visualize";
static unsigned verbose = 0;
static char* font_name = NULL;

#define FONT_PT_STATE 96u
#define FONT_PT_NUMBER 32u
#define FRAME_COLOR SDL_MapRGB(screen->format, 255, 0, 0)
#define MINDELAY 200u
#define MAXDELAY 2000u

#define Verbose(args...) do { if (verbose) fprintf(stderr, args); } while(0)
#define VVerbose(args...) do { if (verbose>1) fprintf(stderr, args); } while(0)
#define die(format, args...) do { fprintf (stderr, "ERROR: " format , ##args); exit(1); } while(0)

static void
usage(void)
{
	fprintf(stderr,
		"%s -- a program to visualize screen updates recorded\n"
		"by xresponse. You cat get latest upstream xresponse from:\n"
		"    http://labs.o-hand.com/xresponse/\n"
		"\n"
		"The visualization is done so that the updates are shown at\n"
		"the same intervals as they originally happened.  The time\n"
		"taken by these intervals is however clamped between\n"
		"0.2s - 2.0s for usability reasons (also in slow mode).\n"
		"\n"
		"Keys (PC/N8x0):\n"
		"    ESC, Back, Q      - Quit\n"
		"    Enter, Select     - Pause/resume\n"
		"    Space, Menu       - Repeat from start\n"
		"    Up, Left          - Pause and go back one frame.\n"
		"    Down, Right       - Pause and go forward one frame.\n"
		"    Shift-Up/Left     - Pause and go back ten frames.\n"
		"    Shift-Down/Right  - Pause and go forward ten frames.\n"
		"    F, Fullscreen     - toggle fullscreen\n"
		"\n"
		"    Keypad also supported in addition to arrow keys.\n"
		"\n"
		"usage:\n"
		"    %s [options] <screenshot PNG file> <xresponse log file>\n"
		"options:\n"
		"    -f  fullscreen\n"
		"    -s  slow (20x delay between updates)\n"
		"    -v  verbose (use twice for greater effect)\n"
		"example:\n"
		"    %s -f -s use-case.png use-case.log\n",
		progname, progname, progname);
}

static const char*
unsigned2str(unsigned k)
{
	static char buf[32];
	snprintf(buf, 32, "%u", k);
	buf[sizeof(buf)-1] = 0;
	return buf;
}

static int max_int(const int a, const int b)
{ if (b > a) return b; return a; }
static int min_int(const int a, const int b)
{ if (b > a) return a; return b; }

static unsigned max_unsigned(const unsigned a, const unsigned b)
{ if (b > a) return b; return a; }
static unsigned min_unsigned(const unsigned a, const unsigned b)
{ if (b > a) return a; return b; }

static int
clamp_int(int value, const int min, const int max)
{
	value = max_int(value, min);
	value = min_int(value, max);
	return value;
}

static unsigned
clamp_unsigned(unsigned value, const unsigned min, const unsigned max)
{
	value = max_unsigned(value, min);
	value = min_unsigned(value, max);
	return value;
}

typedef struct xresponse_event
{
	unsigned time;
	unsigned delay;
	SDL_Rect rect;
} xresponse_event_t;

static void
parse_xresponse_log(const char* filename,
                    unsigned maxw,
                    unsigned maxh,
                    unsigned delayfactor,
                    xresponse_event_t** x,
                    int* n)
{
	VVerbose("Parsing xresponse log '%s' ...\n", filename);
	regex_t re_event, re_update;
	if (regcomp(&re_event, "^ *([0-9]+)ms.* : *(Clicked|keypress)", REG_EXTENDED)) {
		die("Regex compilation failed (1).\n");
	}
	if (regcomp(&re_update,
		"^ *([0-9]+)ms.* damage event ([0-9]+)x([0-9]+)\\+([0-9]+)\\+([0-9]+)",
			REG_EXTENDED)) {
		die("Regex compilation failed (2).\n");
	}
	FILE* xlog = fopen(filename, "r");
	if (!xlog) {
		die("Could not open xresponse log '%s'.\n", filename);
	}
	unsigned timer = 0;
	unsigned starttime = 0;
	unsigned delay = 0;
	xresponse_event_t* updates = NULL;
	unsigned num = 0, cap = 0;
	char* line = NULL;
	size_t line_n = 0;
	// First entry contains the whole string after matching, so we need 6
	// entries for @re_update, which has 5 submatches.
	regmatch_t regex_results[6];
	while (true) {
		int ret = getline(&line, &line_n, xlog);
		if (ret == -1) break;
		if (regexec(&re_event, line,
		            sizeof(regex_results)/sizeof(regmatch_t),
		            regex_results, 0) == 0) {
			sscanf(&line[regex_results[1].rm_so], "%u", &starttime);
			continue;
		}
		if (regexec(&re_update, line,
		            sizeof(regex_results)/sizeof(regmatch_t),
		            regex_results, 0) == 0) {
			unsigned t,w,h,x,y;
			sscanf(&line[regex_results[1].rm_so], "%u", &t);
			sscanf(&line[regex_results[2].rm_so], "%u", &w);
			sscanf(&line[regex_results[3].rm_so], "%u", &h);
			sscanf(&line[regex_results[4].rm_so], "%u", &x);
			sscanf(&line[regex_results[5].rm_so], "%u", &y);
			if ((x+w) > maxw || (y+h) > maxh) {
				fprintf(stderr,
					"WARNING: ignored out-of-screen update, screenshot is %ux%u:\n"
					"\t%s\n",
					maxw, maxh, line);
				continue;
			}
			if (timer > 0) {
				delay = (t - timer) * delayfactor;
				delay = clamp_unsigned(delay, MINDELAY, MAXDELAY);
			} else {
				if (starttime == 0) {
					starttime = t;
				}
				delay = MINDELAY;
			}
			timer = t;
			if (num >= cap) {
				cap = max_unsigned(2*cap, 8u);
				updates = realloc(updates, cap*sizeof(xresponse_event_t));
				if (!updates) die("Memory allocation failed.\n");
			}
			updates[num].time = timer - starttime;
			updates[num].delay = delay;
			updates[num].rect.x = x;
			updates[num].rect.y = y;
			updates[num].rect.w = w;
			updates[num].rect.h = h;
			++num;
			continue;
		}
	}
	fclose(xlog);
	if (line) free(line);
	regfree(&re_event);
	regfree(&re_update);
	Verbose("Log parsing done, got %u entries.\n", num);
	if (!num) {
		die("Unable to parse any entry from xresponse log file.\n");
	}
	if (num > INT_MAX) {
		die("Too many log entries for my little mind.\n");
	}
	*x = updates;
	*n = (int)num;
}

static unsigned
line_width(const SDL_Rect* rect)
{
	unsigned minsize = min_unsigned(rect->w, rect->h);
	if (minsize < 4) {
		return 1;
	} else if (minsize < 10) {
		return (minsize - 2) / 2;
	}
	return 4;
}

/* Draw a frame around the given rect. Since SDL does not seem to have API call
 * for doing this, we manually draw the frame in four pieces.
 */
static SDL_Rect
rect_frame(SDL_Surface* screen, SDL_Rect* rect)
{
	const Uint32 color = FRAME_COLOR;
	const unsigned lw = line_width(rect);
	//
	// ------------------------------
	// |             1              |
	// ------------------------------
	// |   |                    |   |
	// |   |                    |   |
	// | 3 |                    | 4 |
	// |   |                    |   |
	// |   |                    |   |
	// ------------------------------
	// |             2              |
	// ------------------------------
	//
	SDL_Rect r1 = { rect->x - lw/2, rect->y - lw/2,           rect->w + lw, lw };
	SDL_Rect r2 = { rect->x - lw/2, rect->y + rect->h - lw/2, rect->w + lw, lw };
	SDL_Rect r3 = { rect->x - lw/2,           rect->y + lw/2, lw, rect->h - lw };
	SDL_Rect r4 = { rect->x + rect->w - lw/2, rect->y + lw/2, lw, rect->h - lw };
	SDL_FillRect(screen, &r1, color);
	SDL_FillRect(screen, &r2, color);
	SDL_FillRect(screen, &r3, color);
	SDL_FillRect(screen, &r4, color);
	SDL_Rect dirtied = { rect->x - lw/2, rect->y - lw/2, rect->w + lw, rect->h + lw };
	return dirtied;
}

static void
rect2center(const SDL_Surface* screen, SDL_Rect* rect)
{
	rect->x = screen->w/2 - rect->w/2;
	rect->y = screen->h/2 - rect->h/2;
}

/* Use Fontconfig to query us a nice default font. All we need is the absolute
 * path to the .ttf file. This should give the same font as /usr/bin/fc-match
 * gives as the best default match.
 */
static bool
font_init(void)
{
	if (!FcInit()) return false;
	bool ret = false;
	FcResult result;
	FcFontSet* set = NULL;
	FcPattern* def = NULL;
	FcPattern* m = NULL;
	if ((def = FcPatternCreate()) == NULL) goto error;
	// According to FcFontMatch() documentation, these should be called for
	// our FcPattern first.
	FcConfigSubstitute(NULL, def, FcMatchPattern);
	FcDefaultSubstitute(def);
	// Look for the best match.
	// Dont check @result, valgrind says its uninitialized.
	if ((m = FcFontMatch(NULL, def, &result)) == NULL) goto error;
	// Now that we have the match, we need to add it to a FontSet to
	// actually manipulate it.
	if ((set = FcFontSetCreate()) == NULL) goto error;
	if (!FcFontSetAdd(set, m)) goto error;
	if (set->nfont < 1) goto error;
	// Now get the filename.
	FcValue file;
	if (FcPatternGet(set->fonts[0], FC_FILE, 0, &file)) goto error;
	font_name = strdup((char*) file.u.s);
	ret = true;
error:
	if (def) FcPatternDestroy(def);
	if (set) FcFontSetDestroy(set);
	FcFini();
	return ret;
}

static void
font_fini(void)
{
	free(font_name);
	font_name = NULL;
}

/* Render black background for the text of given width and height. Return the
 * area that we modified.
 */
static SDL_Rect
draw_text_background(int textw, int texth, SDL_Surface* screen)
{
	SDL_Rect rect = {0, 0, textw+6, texth+6};
	rect2center(screen, &rect);
	SDL_FillRect(screen, &rect, SDL_MapRGB(screen->format, 0, 0, 0));
	return rect;
}

static SDL_Rect
show_centered_text(const char* msg, unsigned font_size, SDL_Surface* screen)
{
	TTF_Font* font = font_name ? TTF_OpenFont(font_name, font_size) : NULL;
	if (!font) {
		fprintf(stderr, "TTF_OpenFont: %s\n", TTF_GetError());
		SDL_Rect ret = {0,0,0,0};
		return ret;
	}
	// White text.
	SDL_Color fontcolor = {255,255,255,0};
	int textw=0, texth=0;
	TTF_SizeText(font, msg, &textw, &texth);
	SDL_Surface* text_surface = TTF_RenderText_Blended(font, msg, fontcolor);
	TTF_CloseFont(font);
	if (!text_surface) {
		die("TTF_RenderText_Solid: %s\n", TTF_GetError());
	}
	// Black background for the text.
	SDL_Rect ret = draw_text_background(textw, texth, screen);
	// Blit the text. Add +2 to what we got from TTF_SizeText(), in some
	// cases the text got slightly cropped.
	SDL_Rect rect = {0, 0, textw+2, texth+2};
	rect2center(screen, &rect);
	SDL_BlitSurface(text_surface, NULL, screen, &rect);
	SDL_FreeSurface(text_surface);
	return ret;
}

static void
verbose_updates(unsigned index, xresponse_event_t* ev)
{
	static char buf[128];
	snprintf(buf, 128, "update %u after %ums: %dx%d+%d+%d",
	         index+1, ev->time,
	         ev->rect.w, ev->rect.h, ev->rect.x, ev->rect.y);
	buf[sizeof(buf)-1] = 0;
	SDL_WM_SetCaption(buf, 0);
	printf("%s\n", buf);
}

/* Callback from the SDL timer. Push one event so that the main thread wakes up
 * from SDL_WaitEvent().
 */
static Uint32 sleep_callback(Uint32 interval)
{
	static SDL_Event user_event;
	(void) interval;
	user_event.type = SDL_USEREVENT;
	SDL_PushEvent(&user_event);
	return INT_MAX;
}

/* Track state of playback.
 *
 *     @Start    Startup state, show message for short period of time, and then
 *               move to Play state.
 *     @End      The last frame was reached during Play. Pause and show a
 *               message.
 *     @Paused   Paused due to input from user. Dont show any message.
 *     @Play     Normal playback: display updates based on given log file.
 */
typedef enum State
{
	Start,
	End,
	Paused,
	Play,
} State_t;

int main(int argc, char** argv)
{
	bool fullscreen = false;
	unsigned delayfactor = 1;
	int opt;
	while ((opt = getopt(argc, argv, "fsv")) != -1) {
		switch (opt) {
		case 'f':
			fullscreen = true;
			break;
		case 's':
			delayfactor = 20;
			break;
		case 'v':
			++verbose;
			break;
		default:
			usage();
			exit(1);
			break;
		}
	}
	if (optind >= argc-1) {
		usage();
		exit(1);
	}
	const char* png_filename = argv[optind];
	const char* log_filename = argv[optind+1];
	SDL_Surface* screenshot = IMG_Load(png_filename);
	if (!screenshot) {
		die("%s: %s\n", png_filename, IMG_GetError());
	}
	xresponse_event_t* events = NULL;
	int events_cnt = 0;
	parse_xresponse_log(log_filename,
	                    screenshot->w, screenshot->h,
	                    delayfactor, &events, &events_cnt);
	if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_TIMER)==-1) {
		die("could not initialize SDL: %s.\n", SDL_GetError());
	}
	if (font_init()) {
		Verbose("Using font %s\n", font_name);
	} else {
		fprintf(stderr, "WARNING: fontconfig initialization failed.\n");
	}
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	if (TTF_Init()==-1) {
		die("TTF_Init: %s\n", TTF_GetError());
	}
	SDL_Surface* screen = SDL_SetVideoMode(screenshot->w, screenshot->h, 0,
			fullscreen ? SDL_FULLSCREEN : 0);
	if (!screen) {
		die("could not initialize SDL: %s.\n", SDL_GetError());
	}
	SDL_WM_SetCaption(progname, 0);
	SDL_BlitSurface(screenshot, NULL, screen, NULL);
	State_t state = Start;
	int index = -1;
	while (true) {
		unsigned sleep=0, slept=0;
		SDL_Rect dirty[2] = { {0,0,0,0}, {0,0,0,0} };
		switch (state) {
		case Start:
			VVerbose("State: Start\n");
			state = Play;
			index = -1;
			sleep = 1200;
			dirty[0] = show_centered_text("start", FONT_PT_STATE, screen);
			break;
		case End:
			VVerbose("State: End\n");
			sleep = INT_MAX;
			dirty[0] = show_centered_text("paused", FONT_PT_STATE, screen);
			break;
		case Paused:
			index = clamp_int(index, 0, events_cnt-1);
			sleep = INT_MAX;
			VVerbose("State: Paused, frame=%d\n", index+1);
			dirty[0] = rect_frame(screen, &(events[index].rect));
			dirty[1] = show_centered_text(unsigned2str(index+1),
					FONT_PT_NUMBER, screen);
			break;
		case Play:
			index = clamp_int(index, 0, events_cnt-1);
			sleep = events[index].delay;
			VVerbose("State: Play, frame=%d, sleep=%ums\n", index+1, sleep);
			dirty[0] = rect_frame(screen, &(events[index].rect));
			dirty[1] = show_centered_text(unsigned2str(index+1),
					FONT_PT_NUMBER, screen);
			break;
		default:
			fprintf(stderr, "ERROR: Unknown state %d, exiting...", (int)(state));
			goto out;
		}
		SDL_Flip(screen);
		SDL_BlitSurface(screenshot, &dirty[0], screen, &dirty[0]);
		SDL_BlitSurface(screenshot, &dirty[1], screen, &dirty[1]);
		if (verbose > 0 && index >= 0 && index < events_cnt)
			verbose_updates(index, &events[index]);
		while (true) {
			SDL_Event event;
			bool stop_sleeping = false;
			// Set up a timer to wake us up from SDL_WaitEvent()
			// unless we get other events.
			SDL_SetTimer(max_unsigned(10u, sleep-slept), sleep_callback);
			Uint32 ticks = SDL_GetTicks();
			if (SDL_WaitEvent(&event) == 0) {
				die("SDL_WaitEvent() failed.\n");
			}
			slept += SDL_GetTicks() - ticks;
			SDL_SetTimer(0, NULL);
			switch (event.type) {
			case SDL_QUIT:
				Verbose("Got SDL_QUIT event, quitting.\n");
				goto out;
				break;
			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_UP ||
				    event.key.keysym.sym == SDLK_LEFT ||
				    event.key.keysym.sym == SDLK_KP4 ||
				    event.key.keysym.sym == SDLK_KP8) {
					state = Paused;
					if (event.key.keysym.mod & KMOD_SHIFT) {
						index = max_int(0, index-10);
					} else {
						index = max_int(0, index-1);
					}
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_DOWN ||
					   event.key.keysym.sym == SDLK_RIGHT ||
					   event.key.keysym.sym == SDLK_KP2 ||
					   event.key.keysym.sym == SDLK_KP5 ||
					   event.key.keysym.sym == SDLK_KP6) {
					state = Paused;
					if (event.key.keysym.mod & KMOD_SHIFT) {
						index = min_int(events_cnt-1, index+10);
					} else {
						index = min_int(events_cnt-1, index+1);
					}
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_PAGEUP) {
					state = Paused;
					index = max_int(0, index-10);
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_PAGEDOWN) {
					state = Paused;
					index = min_int(events_cnt-1, index+10);
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_HOME) {
					state = Paused;
					index = 0;
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_END) {
					state = Paused;
					index = events_cnt-1;
					stop_sleeping = true;
				}
				break;
			case SDL_KEYUP:
				if (event.key.keysym.sym == SDLK_ESCAPE ||
				    event.key.keysym.sym == SDLK_q) {
					Verbose("Quitting.\n");
					goto out;
				} else if (event.key.keysym.sym == SDLK_f ||
					   event.key.keysym.sym == SDLK_F6) {
					Verbose("Toggling fullscreen.\n");
					SDL_WM_ToggleFullScreen(screen);
					// We need to redraw, so break sleep.
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_SPACE ||
					   event.key.keysym.sym == SDLK_F4) {
					state = Start;
					stop_sleeping = true;
				} else if (event.key.keysym.sym == SDLK_RETURN ||
					   event.key.keysym.sym == SDLK_KP_ENTER) {
					if (state == Paused) {
						state = Play;
						stop_sleeping = true;
					} else if (state == Play) {
						state = Paused;
						stop_sleeping = true;
					}
				}
				break;
			default:
				break;
			}
			if (slept >= sleep || stop_sleeping) {
				break;
			}
		}
		if (state == Play) {
			++index;
			if (index >= events_cnt) {
				state = End;
			}
		}
	}
out:
	free(events);
	SDL_FreeSurface(screenshot);
	TTF_Quit();
	SDL_Quit();
	font_fini();
	return 0;
}
