/*
 * This file is part of TpSession
 *
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * Contact Kate Alhola  kate.alholanokia.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "tpsessionchannel.h"


/**
 * \class TpSessionChannel
 * \headerfile <tpsessionchannel.h>
 *
 *
 * When you start chat session or call with your buddy, channel is established with your buddy.
 * TpSessionChannel represents this connection. TpSession account makes automatically channel when
 * you send message to your buddy's address. If you send successive messages to same buddy with
 * TpSessionAccount, it automatically reuses existing connection.
 */
/**
 * \fn void TpSessionChannel::channelReady(TpSessionChannel *);
 *
 * Emitted when the channel becomes ready
 *
 * \param  TpSessionChannel  pointer to channel become ready
 */
/**
 * \fn void TpSessionChannel::channelDestroyed(TpSessionChannel *);
 *
 * Emitted when the channel is destroyed
 *
 * \param  TpSessionChannel  pointer to channel destroyed. The pointer is only for referenc to remove
 * it from some possible places where it could be stored. It is not guaranteed to point any more valid TpSessionChannel object
 */
/**
 * \fn void TpSessionChannel::messageReceived(const Tp::ReceivedMessage &,TpSessionConnection *);
 *
 * Emitted when any of Account Managers recived message
 *
 * \param  Tp::ReceivedMessage  Message received
 * \param  TpSessionChannel  pointer to channel received message
 */
/**
  * \fn void TpSessionChannel::messageSent(const Tp::Message &,Tp::MessageSendingFlags, const QString &,TpSessionChannel *);
  *
  * \param Tp::Message message sent
  */

/**
 * Construct a new TpSessionChannel object. This constructor is called by TpSessionAccount class when
 * new channel is created . It is not inended to be used stand alone
 * This varient with connection and contact as parameter is intented for creationg new connection from origginator side to your peer
 *
 *
 * \param conn connection where this channel is created
 * \param contact  Contacto to your peer to establish channel
 */


TpSessionChannel::TpSessionChannel(Tp::ConnectionPtr conn,const Tp::ContactPtr &contact)
{
    QVariantMap request;
    // qDebug() << "TpSessionChannel::TpSessionChannel" <<"contact.id() " << contact->id() << contact->id().toLocal8Bit().toHex();
    request.insert(QLatin1String(TELEPATHY_INTERFACE_CHANNEL ".ChannelType"),
                   TELEPATHY_INTERFACE_CHANNEL_TYPE_TEXT);
    request.insert(QLatin1String(TELEPATHY_INTERFACE_CHANNEL ".TargetHandleType"),
                   (uint) Tp::HandleTypeContact);
    request.insert(QLatin1String(TELEPATHY_INTERFACE_CHANNEL ".TargetHandle"),
                   contact->handle()[0]);

    connect(conn->ensureChannel(request),
            SIGNAL(finished(Tp::PendingOperation*)),
            SLOT(onChannelCreated(Tp::PendingOperation*)));
    peerContact=contact;
}

/**
 * Construct a new TpSessionChannel object. This constructor is called by TpSessionAccount class when
 * new channel is created . It is not inended to be used stand alone
 * This varient with connection only parameter is intented for receiving new connection from your peer
 *
 *
 * \param conn connection where this channel is created
 */


TpSessionChannel::TpSessionChannel(Tp::TextChannelPtr ch)
{
  if(TpSession::tpsDebug()) qDebug() << "TpSessionChannel::TpSessionChannel" <<"path " << ch->objectPath();
     channel=ch;
     connect(channel->becomeReady(Tp::TextChannel::FeatureMessageQueue|Tp::TextChannel::FeatureMessageSentSignal),
                    SIGNAL(finished(Tp::PendingOperation *)),
                    SLOT(onChannelReady(Tp::PendingOperation *)));
     connect(channel.data(),
             SIGNAL(invalidated(Tp::DBusProxy *, const QString &, const QString &)),
             SLOT(onChannelDestroyed()));

}

void TpSessionChannel::onChannelCreated(Tp::PendingOperation *op)
{
  if(TpSession::tpsDebug())qDebug() << "TpSessionChannel::onOutgoingChannelCreated" ;
   if (op->isError()) {
     qWarning() << "Connection cannot become connected" ;
     return;
   }
   Tp::PendingChannel *pc = qobject_cast<Tp::PendingChannel *>(op);

   channel = Tp::TextChannel::create(pc->connection(),pc->objectPath(), pc->immutableProperties());

    connect(channel->becomeReady(Tp::TextChannel::FeatureMessageQueue | Tp::TextChannel::FeatureMessageSentSignal),
            SIGNAL(finished(Tp::PendingOperation*)),
            SLOT(onChannelReady(Tp::PendingOperation*)));
    connect(channel.data(),
            SIGNAL(invalidated(Tp::DBusProxy *, const QString &, const QString &)),
            SLOT(onChannelDestroyed()));
}

QString TpSessionChannel::objectPath()
{
        return channel->objectPath();
}
QString TpSessionChannel::type()
{
        return channel->channelType();
}
void TpSessionChannel::onChannelReady(Tp::PendingOperation *op)
{
  Q_UNUSED(op);
  if(TpSession::tpsDebug()) qDebug() << "TpSessionChannel::onChannelReady type=" << channel->channelType() <<"path " << channel->objectPath() <<
            "initiatorContact=" << (channel->initiatorContact() ? channel->initiatorContact()->id():"NULL");
         ;
 connect(channel.data(),
         SIGNAL(messageReceived(const Tp::ReceivedMessage &)),
         SLOT(onMessageReceived(const Tp::ReceivedMessage &)));
 connect(channel.data(),
         SIGNAL(messageSent(const Tp::Message &,Tp::MessageSendingFlags, const QString &)),
         SLOT(onMessageSent(const Tp::Message &,Tp::MessageSendingFlags, const QString &)));
 connect(channel.data(),SIGNAL(destroyed(QObject *)),SLOT(onChannelDestroyed()));
 if(peerId().isEmpty()) peerContact=channel->initiatorContact(); //  If this is incoming channel
 emit channelReady(this);
 QList<Tp::ReceivedMessage> queuedMessages = channel->messageQueue();
 foreach(Tp::ReceivedMessage message, queuedMessages) {
   //      qDebug()  << "received " << message.text();
           emit messageReceived(message,this);
    }
}
/**
 *  Send message to to ths channel
 *
 *
 * \param message   message to send
 */

void TpSessionChannel::sendMessage(QString message)
{
    channel->send(message);
}
void TpSessionChannel::onMessageReceived(const Tp::ReceivedMessage &msg)
{
  //    qDebug() << "TpSessionChannel::onMessageReceived " << msg.text();
    emit messageReceived(msg,this);
};
void TpSessionChannel::onMessageSent(const Tp::Message &msg,Tp::MessageSendingFlags sflags, const QString &flags)
{
 //   qDebug() << "TpSessionChannel::onMessageSen" << peerId() <<"txt:" << msg.text();;
    emit messageSent(msg,sflags,flags,this);
};
/**
 *  Get id ( address of your peer )
 *
 *
 * \returns your peer id/address ir empty QString
 */
QString TpSessionChannel::peerId()
{
    return peerContact ? peerContact->id():QString();
}

void TpSessionChannel::onChannelDestroyed()
{
   if(TpSession::tpsDebug()) qDebug() << "TpSessionChannel::onChannelDestroyed" << peerId() << objectPath();
    //TpSessionChannel *call = (TpSessionChannel *) obj;
     emit channelDestroyed(this);
}

