/*
 * stream.c - Source for TpStreamEngineVideoStream
 * Copyright (C) 2006-2008 Collabora Ltd.
 * Copyright (C) 2006-2008 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "videostream.h"

#include <string.h>

#include <telepathy-glib/errors.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/proxy-subclass.h>

#include <gst/interfaces/xoverlay.h>
#include <gst/farsight/fs-element-added-notifier.h>

#include "tp-stream-engine.h"
#include "tp-stream-engine-signals-marshal.h"
#include "util.h"

#include "api/api.h"

G_DEFINE_TYPE (TpStreamEngineVideoStream, tp_stream_engine_video_stream,
    TP_STREAM_ENGINE_TYPE_VIDEO_SINK);

#define DEBUG(self, format, ...)                \
  g_debug ("stream %d (video) %s: " format,     \
      tf_stream_get_id ((self)->priv->stream),  \
      G_STRFUNC,                                \
      ##__VA_ARGS__)

struct _TpStreamEngineVideoStreamPrivate
{
  TfStream *stream;

  gulong src_pad_added_handler_id;

  GError *construction_error;

  GstPad *pad;

  GstElement *bin;

  GstElement *sink;

  gboolean manual_keyframes;
  volatile gint bitrate;
  volatile gint mtu;

  FsElementAddedNotifier *notifier;

  GMutex *mutex;

  /* Everything below this line is protected by the mutex */
  guint error_idle_id;

  GstPad *sinkpad;
  gulong receiving_probe_id;
  guint receiving_idle_id;
};


enum
{
  RECEIVING,
  RESOLUTION_CHANGED,
  SINK_ADDED_SIGNAL,
  SIGNAL_COUNT
};

static guint signals[SIGNAL_COUNT] = {0};


/* properties */
enum
{
  PROP_0,
  PROP_STREAM,
  PROP_BIN,
  PROP_PAD,
};


static GstElement *
tp_stream_engine_video_stream_make_sink (TpStreamEngineVideoStream *stream,
                                         GstPad **pad);

static void src_pad_added_cb (TfStream *stream, GstPad *pad,
    FsCodec *codec, gpointer user_data);



static void tp_stream_engine_video_stream_set_property  (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec);

static void tp_stream_engine_video_stream_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec);

static void _element_added (FsElementAddedNotifier *notifier,
    GstBin *conference,
    GstElement *element,
    TpStreamEngineVideoStream *self);


static void
tp_stream_engine_video_stream_init (TpStreamEngineVideoStream *self)
{
  TpStreamEngineVideoStreamPrivate *priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TP_STREAM_ENGINE_TYPE_VIDEO_STREAM, TpStreamEngineVideoStreamPrivate);

  self->width = G_MAXUINT;
  self->height = G_MAXUINT;

  self->priv = priv;

  self->priv->mutex = g_mutex_new ();

  self->priv->notifier = fs_element_added_notifier_new ();

  g_signal_connect (self->priv->notifier, "element-added",
      G_CALLBACK (_element_added), self);
}


static GstElement *
tp_stream_engine_video_stream_make_sink (TpStreamEngineVideoStream *self,
                                         GstPad **pad)
{
  GstElement *bin = gst_bin_new (NULL);
  GstElement *sink = NULL;
  GstElement *funnel = NULL;

  g_object_get (self, "sink", &sink, NULL);

  if (!sink)
    {
      g_warning ("Could not make sink");
      goto error;
    }

  if (!gst_bin_add (GST_BIN (bin), sink))
    {
      gst_object_unref (sink);
      g_warning ("Could not add sink to bin");
      goto error;
    }

  funnel = gst_element_factory_make ("fsfunnel", "funnel");
  if (!funnel)
    {
      g_warning ("Could not make funnel");
      goto error;
    }

  if (!gst_bin_add (GST_BIN (bin), funnel))
    {
      gst_object_unref (funnel);
      g_warning ("Could not add funnel to bin");
      goto error;
    }

  if (!gst_element_link (funnel, sink))
    {
      g_warning ("Could not link funnel and sink");
      goto error;
    }

  *pad = gst_element_get_static_pad (sink, "sink");
  gst_object_unref (*pad);

  return bin;
error:
  gst_object_unref (bin);
  return NULL;
}

static gboolean
receiving_idle (gpointer user_data)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (user_data);

  g_mutex_lock (self->priv->mutex);
  self->priv->receiving_idle_id = 0;
  g_mutex_unlock (self->priv->mutex);

  g_signal_emit (self, signals[RECEIVING], 0);

  return FALSE;
}

static gboolean
receiving_data (GstPad *pad, GstMiniObject *obj, gpointer user_data)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (user_data);

  g_mutex_lock (self->priv->mutex);
  self->priv->receiving_idle_id = g_idle_add (receiving_idle, self);
  gst_pad_remove_buffer_probe (pad, self->priv->receiving_probe_id);
  self->priv->receiving_probe_id = 0;
  g_mutex_unlock (self->priv->mutex);

  return TRUE;
}

static void
request_keyframe (TpProxy *proxy, gpointer user_data, GObject *weak_object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (weak_object);

  DEBUG (self, "Keyframe requested by connection manager");

  if (self->priv->pad)
    gst_pad_push_event (self->priv->pad,
        gst_event_new_custom (GST_EVENT_CUSTOM_DOWNSTREAM,
            gst_structure_new ("GstForceKeyUnit",
                "all-headers", G_TYPE_BOOLEAN, TRUE,
                NULL)));
}

static gboolean
is_in_send_codecbin (GstElement *element, GstElement *ultimate_parent,
    guint session_id)
{
  gint tmp_session_id, tmp_codec_id;
  gchar *name;
  gboolean valid = FALSE;
  GstElement *parent;
  gboolean rv;

  if (element == ultimate_parent)
    return FALSE;

  name = gst_element_get_name (element);
  if (name && sscanf (name, "send_%d_%d", &tmp_session_id, &tmp_codec_id) == 2)
    valid = TRUE;
  g_free (name);

  if (valid && session_id == tmp_session_id)
    return TRUE;

  parent = (GstElement*) gst_element_get_parent (element);
  if (!parent)
    return FALSE;
  rv = is_in_send_codecbin (parent, ultimate_parent, session_id);

  gst_object_unref (parent);

  return rv;
}

static void
_element_added (FsElementAddedNotifier *notifier,
    GstBin *bin, GstElement *element, TpStreamEngineVideoStream *self)
{
  gint bitrate = g_atomic_int_get (&self->priv->bitrate);
  gint mtu = g_atomic_int_get (&self->priv->mtu);

  if (G_UNLIKELY (
          (bitrate && g_object_has_property (G_OBJECT (element), "bitrate")) ||
          (mtu && g_object_has_property (G_OBJECT (element), "mtu")) ||
          (self->priv->manual_keyframes &&
              g_object_has_property (G_OBJECT (element), "keyframe-interval"))))
    {
      FsSession *fssession;
      GstElement *conf;
      guint id;

      g_object_get (self->priv->stream, "farsight-session", &fssession,
          "farsight-conference", &conf, NULL);

      g_object_get (fssession, "id", &id, NULL);

      if (is_in_send_codecbin (element, GST_ELEMENT (conf), id))
        {
          if (bitrate && g_object_has_property (G_OBJECT (element), "bitrate"))
            {
              g_object_set (element, "bitrate", bitrate, NULL);
              DEBUG (self, "Set bitrate %u on encoder", bitrate);
            }
          if (mtu && g_object_has_property (G_OBJECT (element), "mtu"))
            {
              g_object_set (element, "mtu", mtu, NULL);
              DEBUG (self, "Set MTU %u on payloader", mtu);
            }
          if (self->priv->manual_keyframes &&
              g_object_has_property (G_OBJECT (element), "keyframe-interval"))
            {
              g_object_set (element, "keyframe-interval", 600, NULL);
              DEBUG (self, "Disabled automatic keyframes");
            }
        }

      g_object_unref (fssession);
    }
}

static void
tp_stream_engine_video_stream_update_element_properties (
    TpStreamEngineVideoStream *self)
{
  GstBin *conf;

  g_object_get (self->priv->stream, "farsight-conference", &conf, NULL);

  fs_element_added_notifier_remove (self->priv->notifier, conf);

  if (g_atomic_int_get (&self->priv->bitrate) ||
      g_atomic_int_get (&self->priv->mtu) ||
      self->priv->manual_keyframes)
    fs_element_added_notifier_add (self->priv->notifier, conf);

  gst_object_unref (conf);
}

static void
tp_stream_engine_video_stream_set_bitrate (TpStreamEngineVideoStream *self,
    guint bitrate)
{
  if (g_atomic_int_get (&self->priv->bitrate) == bitrate)
    return;

  g_atomic_int_set (&self->priv->bitrate, bitrate);

  tp_stream_engine_video_stream_update_element_properties (self);
}


static void
bitrate_changed (TpProxy *proxy, guint bitrate, gpointer user_data,
    GObject *weak_object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (weak_object);

  DEBUG (self, "BitrateChanged: %u", bitrate);

  tp_stream_engine_video_stream_set_bitrate (self, bitrate);
}

static void
tp_stream_engine_video_stream_set_mtu (TpStreamEngineVideoStream *self,
    guint mtu)
{
  if (g_atomic_int_get (&self->priv->mtu) == mtu)
    return;

  g_atomic_int_set (&self->priv->mtu, mtu);

  tp_stream_engine_video_stream_update_element_properties (self);
}


static void
mtu_changed (TpProxy *proxy, guint mtu, gpointer user_data,
    GObject *weak_object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (weak_object);

  DEBUG (self, "MTUChanged: %u", mtu);

  tp_stream_engine_video_stream_set_mtu (self, mtu);
}

static void
tp_stream_engine_video_stream_set_manual_keyframes (
    TpStreamEngineVideoStream *self)
{
  self->priv->manual_keyframes = TRUE;

  tp_stream_engine_video_stream_update_element_properties (self);
}


static void
video_resolution_changed_real (TpStreamEngineVideoStream *self,
    guint width, guint height)
{
  DEBUG (self, "ResolutionChanged: %u x %u", width, height);

  if (width != self->width || height != self->height)
    {
      self->width = width;
      self->height = height;
      g_signal_emit (self, signals[RESOLUTION_CHANGED], 0);
    }
}

static void
video_resolution_changed (TpProxy *proxy, const GValueArray *arg_resolution,
    gpointer user_data, GObject *weak_object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (weak_object);
  GValue *val_width, *val_height;

  if (arg_resolution->n_values != 2)
    {
      DEBUG (self,
          "VideoResolutionChanged argument does not have two members");
      return;
    }

  val_width = g_value_array_get_nth (arg_resolution, 0);
  val_height = g_value_array_get_nth (arg_resolution, 1);

  if (!G_VALUE_HOLDS_UINT (val_width) || !G_VALUE_HOLDS_UINT (val_height))
    {
      DEBUG (self, "VideoResolutionChanged arguments are not uints");
      return;
    }

  video_resolution_changed_real (self, g_value_get_uint (val_width),
      g_value_get_uint (val_height));

}
static void
videocontrol_get_all_properties_cb (TpProxy *proxy, GHashTable *out_properties,
    const GError *error, gpointer user_data, GObject *weak_object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (weak_object);
  guint32 bitrate;
  guint32 mtu;
  gboolean manual_keyframes;
  gboolean valid;
  GValueArray *array;

  if (error)
    return;

  bitrate = tp_asv_get_uint32 (out_properties, "Bitrate", &valid);
  if (valid)
    {
      DEBUG (self, "Got initial bitrate: %u", bitrate);
      tp_stream_engine_video_stream_set_bitrate (self, bitrate);
    }

  mtu = tp_asv_get_uint32 (out_properties, "MTU", &valid);
  if (valid)
    {
      DEBUG (self, "Got initial mtu: %u", mtu);
      tp_stream_engine_video_stream_set_mtu (self, mtu);
    }

  array = tp_asv_get_boxed (out_properties, "VideoResolution",
      STREAM_ENGINE_STRUCT_TYPE_VIDEO_RESOLUTION);
  if (array)
    {
      int width = g_value_get_uint (g_value_array_get_nth (array, 0));
      int height = g_value_get_uint (g_value_array_get_nth (array, 1));

      DEBUG (self, "Got initial resolution: %u x %u", width, height);

      video_resolution_changed_real (self, width, height);
    }

  manual_keyframes = tp_asv_get_boolean (out_properties, "ManualKeyframes", &valid);
  if (valid)
    {
      DEBUG (self, "Keyframes will only come when manually requested");
      self->priv->manual_keyframes = TRUE;
      tp_stream_engine_video_stream_update_element_properties (self);
    }

}


static GObject *
tp_stream_engine_video_stream_constructor (GType type,
    guint n_props,
    GObjectConstructParam *props)
{
  GObject *obj;
  TpStreamEngineVideoStream *self = NULL;
  GstPad *sinkpad;
  TpProxy *proxy;
  GError *error = NULL;

  obj = G_OBJECT_CLASS (tp_stream_engine_video_stream_parent_class)->constructor (type, n_props, props);

  self = (TpStreamEngineVideoStream *) obj;

  g_object_get (self->priv->stream, "proxy", &proxy, NULL);

  tp_proxy_add_interface_by_id (proxy,
      STREAM_ENGINE_IFACE_QUARK_STREAMHANDLER_INTERFACE_VIDEO_CONTROL);
  stream_engine_cli_streamhandler_interface_video_control_connect_to_request_keyframe (proxy, request_keyframe, NULL, NULL, G_OBJECT (self), &error);
  g_assert (error == NULL);
  stream_engine_cli_streamhandler_interface_video_control_connect_to_bitrate_changed (proxy, bitrate_changed, NULL, NULL, G_OBJECT (self), &error);
  g_assert (error == NULL);
  stream_engine_cli_streamhandler_interface_video_control_connect_to_mtu_changed (proxy, mtu_changed, NULL, NULL, G_OBJECT (self), &error);
  g_assert (error == NULL);
  stream_engine_cli_streamhandler_interface_video_control_connect_to_video_resolution_changed (proxy, video_resolution_changed, NULL, NULL, G_OBJECT (self), &error);
  g_assert (error == NULL);
  tp_cli_dbus_properties_call_get_all (proxy, -1,
      STREAM_ENGINE_IFACE_STREAMHANDLER_INTERFACE_VIDEO_CONTROL,
      videocontrol_get_all_properties_cb, NULL, NULL, G_OBJECT (self));
  g_object_unref (proxy);

  g_object_get (self->priv->stream, "sink-pad", &sinkpad, NULL);

  if (!sinkpad)
    {
      g_warning ("Could not get stream's sinkpad");
      return obj;
    }

  if (GST_PAD_LINK_FAILED (gst_pad_link (self->priv->pad, sinkpad)))
    {
      g_warning ("Could not link sink to queue");
      gst_object_unref (sinkpad);
      return obj;
    }

  gst_object_unref (sinkpad);

  self->priv->src_pad_added_handler_id = g_signal_connect_object (
      self->priv->stream, "src-pad-added", G_CALLBACK (src_pad_added_cb), self,
      0);

  return obj;
}

static void
tp_stream_engine_video_stream_dispose (GObject *object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (object);


  if (self->priv->src_pad_added_handler_id)
    {
      g_signal_handler_disconnect (self->priv->stream,
          self->priv->src_pad_added_handler_id);
      self->priv->src_pad_added_handler_id = 0;
    }

  if (self->priv->notifier)
    {
      g_object_unref (self->priv->notifier);
      self->priv->notifier = NULL;
    }

  g_mutex_lock (self->priv->mutex);
  if (self->priv->receiving_probe_id)
    {
      gst_pad_remove_buffer_probe (self->priv->sinkpad,
          self->priv->receiving_probe_id);
      self->priv->receiving_probe_id = 0;
    }
  if (self->priv->error_idle_id)
    {
      g_source_remove (self->priv->error_idle_id);
      self->priv->error_idle_id = 0;
    }
  if (self->priv->receiving_idle_id)
    {
      g_source_remove (self->priv->receiving_idle_id);
      self->priv->error_idle_id = 0;
    }
  g_mutex_unlock (self->priv->mutex);

  if (self->priv->sink)
    {
      gst_element_set_locked_state (self->priv->sink, TRUE);
      gst_element_set_state (self->priv->sink, GST_STATE_NULL);

      gst_bin_remove (GST_BIN (self->priv->bin), self->priv->sink);
      self->priv->sink = NULL;
      self->priv->sinkpad = NULL;
    }

  if (self->priv->bin)
    {
      gst_object_unref (self->priv->bin);
      self->priv->bin = NULL;
    }

  if (self->priv->stream)
    {
      g_object_unref (self->priv->stream);
      self->priv->stream = NULL;
    }

  if (G_OBJECT_CLASS (tp_stream_engine_video_stream_parent_class)->dispose)
    G_OBJECT_CLASS (tp_stream_engine_video_stream_parent_class)->dispose (object);
}

static void
tp_stream_engine_video_stream_finalize (GObject *object)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (object);

  g_mutex_free (self->priv->mutex);

  if (G_OBJECT_CLASS (tp_stream_engine_video_stream_parent_class)->finalize)
    G_OBJECT_CLASS (tp_stream_engine_video_stream_parent_class)->finalize (object);
}

static void
tp_stream_engine_video_stream_set_property  (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (object);

    switch (property_id)
    {
    case PROP_STREAM:
      self->priv->stream = g_value_dup_object (value);
      break;
    case PROP_BIN:
      self->priv->bin = g_value_dup_object (value);
      break;
    case PROP_PAD:
      self->priv->pad = g_value_dup_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
tp_stream_engine_video_stream_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (object);

    switch (property_id)
    {
    case PROP_STREAM:
      g_value_set_object (value, self->priv->stream);
      break;
    case PROP_BIN:
      g_value_set_object (value, self->priv->bin);
      break;
   case PROP_PAD:
      g_value_set_object (value, self->priv->pad);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
tp_stream_engine_video_stream_class_init (TpStreamEngineVideoStreamClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (TpStreamEngineVideoStreamPrivate));
  object_class->dispose = tp_stream_engine_video_stream_dispose;
  object_class->finalize = tp_stream_engine_video_stream_finalize;
  object_class->constructor = tp_stream_engine_video_stream_constructor;
  object_class->set_property = tp_stream_engine_video_stream_set_property;
  object_class->get_property = tp_stream_engine_video_stream_get_property;

  g_object_class_install_property (object_class, PROP_STREAM,
      g_param_spec_object ("stream",
          "Tp StreamEngine Stream",
          "The Telepathy Stream Engine Stream",
          TF_TYPE_STREAM,
          G_PARAM_CONSTRUCT_ONLY |
          G_PARAM_READWRITE |
          G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_BIN,
      g_param_spec_object ("bin",
          "The Bin to add stuff to",
          "The Bin to add the elements to",
          GST_TYPE_BIN,
          G_PARAM_CONSTRUCT_ONLY |
          G_PARAM_READWRITE |
          G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_PAD,
      g_param_spec_object ("pad",
          "The pad to get the data from",
          "the GstPad the data comes from",
          GST_TYPE_PAD,
          G_PARAM_CONSTRUCT_ONLY |
          G_PARAM_READWRITE |
          G_PARAM_STATIC_STRINGS));

  signals[RECEIVING] =
      g_signal_new ("receiving",
          G_OBJECT_CLASS_TYPE (klass),
          G_SIGNAL_RUN_LAST,
          0,
          NULL, NULL,
          g_cclosure_marshal_VOID__VOID,
          G_TYPE_NONE, 0);

  signals[RESOLUTION_CHANGED] =
      g_signal_new ("resolution-changed",
          G_OBJECT_CLASS_TYPE (klass),
          G_SIGNAL_RUN_LAST,
          0,
          NULL, NULL,
          g_cclosure_marshal_VOID__VOID,
          G_TYPE_NONE, 0);

  signals[SINK_ADDED_SIGNAL] = g_signal_new ("sink-added",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL, NULL,
      g_cclosure_marshal_VOID__OBJECT,
      G_TYPE_NONE, 1, G_TYPE_OBJECT);
}


static gboolean
src_pad_added_idle_error (gpointer user_data)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (user_data);

  tf_stream_error (self->priv->stream, TP_MEDIA_STREAM_ERROR_MEDIA_ERROR,
      "Error setting up video reception");

  g_mutex_lock (self->priv->mutex);
  self->priv->error_idle_id = 0;
  g_mutex_unlock (self->priv->mutex);

  return FALSE;
}

static void
src_pad_added_cb (TfStream *stream, GstPad *pad, FsCodec *codec,
    gpointer user_data)
{
  TpStreamEngineVideoStream *self = TP_STREAM_ENGINE_VIDEO_STREAM (user_data);
  GstPad *sinkpad;
  GstPad *ghost;
  GstElement *funnel;

  g_mutex_lock (self->priv->mutex);
  if (!self->priv->sink)
    {
      GstElement *sink = NULL;

      sink = tp_stream_engine_video_stream_make_sink (self,
          &self->priv->sinkpad);

      if (!sink)
        {
          g_mutex_unlock (self->priv->mutex);
          return;
        }

      if (!gst_bin_add (GST_BIN (self->priv->bin), sink))
        {
          g_mutex_unlock (self->priv->mutex);
          g_warning ("Could not add sink to bin");
          return;
        }

      self->priv->sink = sink;

      self->priv->receiving_probe_id =
          gst_pad_add_buffer_probe (self->priv->sinkpad,
              G_CALLBACK (receiving_data), self);

      if (gst_element_set_state (sink, GST_STATE_PLAYING) ==
          GST_STATE_CHANGE_FAILURE)
        {
          g_mutex_unlock (self->priv->mutex);
          g_warning ("Could not start sink");
          return;
        }

    }

  funnel = gst_bin_get_by_name (GST_BIN (self->priv->sink), "funnel");

  g_mutex_unlock (self->priv->mutex);


  if (!funnel)
    {
      g_warning ("Could not get funnel");
      goto error;
    }

  g_signal_emit (self, signals[SINK_ADDED_SIGNAL], 0, funnel);

  sinkpad = gst_element_get_request_pad (funnel, "sink%d");
  if (!sinkpad)
    {
      gst_object_unref (funnel);
      g_warning ("Could not get funnel sink pad");
      goto error;
    }

  gst_object_unref (funnel);

  ghost = gst_ghost_pad_new (NULL, sinkpad);

  gst_object_unref (sinkpad);

  gst_pad_set_active (ghost, TRUE);

  if (!gst_element_add_pad (self->priv->sink, ghost))
    {
      g_warning ("Could not add ghost pad to sink bin");
      gst_object_unref (ghost);
      goto error;
    }

  if (GST_PAD_LINK_FAILED (gst_pad_link (pad, ghost)))
    {
      g_warning ("Could not link pad to ghost pad");
      goto error;
    }

  return;


 error:

  g_mutex_lock (self->priv->mutex);
  if (!self->priv->error_idle_id)
    self->priv->error_idle_id =
        g_idle_add (src_pad_added_idle_error, self);
  g_mutex_unlock (self->priv->mutex);
}



TpStreamEngineVideoStream *
tp_stream_engine_video_stream_new (
  TfStream *stream,
  GstBin *bin,
  GstPad *pad,
  GError **error)
{
  TpStreamEngineVideoStream *self = NULL;


  self = g_object_new (TP_STREAM_ENGINE_TYPE_VIDEO_STREAM,
      "stream", stream,
      "bin", bin,
      "pad", pad,
      "is-preview", FALSE,
      NULL);

  if (self->priv->construction_error)
    {
      g_propagate_error (error, self->priv->construction_error);
      g_object_unref (self);
      return NULL;
    }

  return self;
}
