/*
 * tp-stream-engine.c - Source for TpStreamEngine
 * Copyright (C) 2005-2007 Collabora Ltd.
 * Copyright (C) 2005-2007 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <dbus/dbus-glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <netinet/ip.h>

#include <telepathy-glib/dbus.h>
#include <telepathy-glib/errors.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/gtypes.h>

#include "tp-stream-engine.h"
#include "tp-stream-engine-signals-marshal.h"

#include <gst/farsight/fs-element-added-notifier.h>

#include <telepathy-farsight/channel.h>
#include <telepathy-farsight/stream.h>

#include "api/api.h"
#include "audiostream.h"
#include "videosink.h"
#include "videostream.h"
#include "videopreview.h"
#include "util.h"

#define BUS_NAME        "org.maemo.Telepathy.StreamEngine"
#define OBJECT_PATH     "/org/maemo/Telepathy/StreamEngine"

static void
_create_pipeline (TpStreamEngine *self);

static void
setup_realtime_thread (pthread_t thread);

static void
register_dbus_signal_marshallers()
{
  /*register a marshaller for the NewMediaStreamHandler signal*/
  dbus_g_object_register_marshaller
    (tp_stream_engine_marshal_VOID__BOXED_UINT_UINT_UINT, G_TYPE_NONE,
     DBUS_TYPE_G_OBJECT_PATH, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT,
     G_TYPE_INVALID);

  /*register a marshaller for the NewMediaSessionHandler signal*/
  dbus_g_object_register_marshaller
    (tp_stream_engine_marshal_VOID__BOXED_STRING, G_TYPE_NONE,
     DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING, G_TYPE_INVALID);

  /*register a marshaller for the AddRemoteCandidate signal*/
  dbus_g_object_register_marshaller
    (tp_stream_engine_marshal_VOID__STRING_BOXED, G_TYPE_NONE,
     G_TYPE_STRING, TP_ARRAY_TYPE_MEDIA_STREAM_HANDLER_TRANSPORT_LIST, G_TYPE_INVALID);

  /*register a marshaller for the SetActiveCandidatePair signal*/
  dbus_g_object_register_marshaller
    (tp_stream_engine_marshal_VOID__STRING_STRING, G_TYPE_NONE,
     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

  /*register a marshaller for the SetRemoteCandidateList signal*/
  dbus_g_object_register_marshaller
    (g_cclosure_marshal_VOID__BOXED, G_TYPE_NONE,
     TP_ARRAY_TYPE_MEDIA_STREAM_HANDLER_CANDIDATE_LIST, G_TYPE_INVALID);

  /*register a marshaller for the SetRemoteCodecs signal*/
  dbus_g_object_register_marshaller
    (g_cclosure_marshal_VOID__BOXED, G_TYPE_NONE,
     TP_ARRAY_TYPE_MEDIA_STREAM_HANDLER_CODEC_LIST, G_TYPE_INVALID);
}

GQuark
tp_stream_engine_error_quark (void)
{
  return g_quark_from_static_string ("telepathy-stream-engine-error");
}

static void se_iface_init (gpointer, gpointer);

G_DEFINE_TYPE_WITH_CODE (TpStreamEngine, tp_stream_engine, G_TYPE_OBJECT,
    G_IMPLEMENT_INTERFACE (STREAM_ENGINE_TYPE_SVC_STREAM_ENGINE,
      se_iface_init))

/* properties */
enum
{
  PROP_0,
  PROP_PIPELINE
};

/* signal enum */
enum
{
  HANDLING_CHANNEL,
  NO_MORE_CHANNELS,
  SHUTDOWN_REQUESTED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* private structure */
struct _TpStreamEnginePrivate
{
  gboolean dispose_has_run;

  TpDBusDaemon *dbus_daemon;

  GstElement *pipeline;

  GstElement *capsfilter;
  GstElement *videosrc;
  GstElement *videotee;

  GPtrArray *channels;
  GHashTable *channels_by_path;

  GMutex *mutex;

  FsElementAddedNotifier *notifier;

  gint video_source_use_count;

  gboolean failcount;
  guint reset_failcount_id;
  guint restart_pipeline_id;

  /* Everything below this is protected by the mutex */
  GList *output_sinks;
  GList *preview_sinks;

  GList *audio_streams;

  TpStreamEngineVideoPreview *preview;

  guint bus_async_source_id;

  GPtrArray *audio_objects;

  GHashTable *object_threads;
};

static void
set_element_props (FsElementAddedNotifier *notifier,
    GstBin *bin,
    GstElement *element,
    gpointer user_data)
{
  if (g_object_has_property ((GObject *) element, "min-ptime"))
    g_object_set (element, "min-ptime", GST_MSECOND * 20, NULL);

  if (g_object_has_property ((GObject *) element, "max-ptime"))
    g_object_set (element, "max-ptime", GST_MSECOND * 50, NULL);
}

static gboolean
load_keyfile (GKeyFile *keyfile, const gchar *path, const gchar *subdir)
{
  gchar *filename;
  gboolean loaded = FALSE;
  GError *error = NULL;


  filename = g_build_filename (path, subdir, "gstelements.conf", NULL);

  if (!g_key_file_load_from_file (keyfile, filename, G_KEY_FILE_NONE, &error))
    {
      g_debug ("Could not read element properties config at %s: %s", filename,
          error ? error->message : "(no error message set)");
    }
  else
    {
      loaded = TRUE;
    }
  g_free (filename);
  g_clear_error (&error);

  return loaded;
}

static void
add_gstelements_conf_to_notifier (FsElementAddedNotifier *notifier)
{
  GKeyFile *keyfile = g_key_file_new ();
  const gchar *home = NULL;

  home = g_getenv ("HOME");
  if (!home)
    home = g_get_home_dir ();

  if (!load_keyfile (keyfile, home, ".stream-engine"))
    {
      if (!load_keyfile (keyfile, SYSCONFDIR, "stream-engine"))
        {
          g_key_file_free (keyfile);
          return;
        }
    }

  fs_element_added_notifier_set_properties_from_keyfile (notifier,
      keyfile);
}

static void
tp_stream_engine_init (TpStreamEngine *self)
{
  TpStreamEnginePrivate *priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TP_TYPE_STREAM_ENGINE, TpStreamEnginePrivate);

  self->priv = priv;

  priv->channels = g_ptr_array_new ();

  priv->channels_by_path = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, NULL);

  priv->audio_objects = g_ptr_array_new ();
  priv->object_threads = g_hash_table_new (g_direct_hash, g_direct_equal);

  priv->mutex = g_mutex_new ();

  priv->notifier = fs_element_added_notifier_new ();
  g_signal_connect (priv->notifier, "element-added",
      G_CALLBACK (set_element_props), self);
  add_gstelements_conf_to_notifier (priv->notifier);

  _create_pipeline (self);
}

static void tp_stream_engine_dispose (GObject *object);
static void tp_stream_engine_finalize (GObject *object);



static void
tp_stream_engine_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (object);

  switch (property_id)
    {
    case PROP_PIPELINE:
      g_value_set_object (value, self->priv->pipeline);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}



static void
tp_stream_engine_class_init (TpStreamEngineClass *tp_stream_engine_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (tp_stream_engine_class);

  g_type_class_add_private (tp_stream_engine_class,
      sizeof (TpStreamEnginePrivate));


  object_class->get_property = tp_stream_engine_get_property;
  object_class->dispose = tp_stream_engine_dispose;
  object_class->finalize = tp_stream_engine_finalize;

  /**
   * TpStreamEngine::handling-channel:
   *
   * Emitted whenever this object starts handling a channel
   */
  signals[HANDLING_CHANNEL] =
  g_signal_new ("handling-channel",
                G_OBJECT_CLASS_TYPE (tp_stream_engine_class),
                G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                0,
                NULL, NULL,
                g_cclosure_marshal_VOID__VOID,
                G_TYPE_NONE, 0);

  /**
   * TpStreamEngine::no-more-channels:
   *
   * Emitted whenever this object is handling no channels
   */
  signals[NO_MORE_CHANNELS] =
  g_signal_new ("no-more-channels",
                G_OBJECT_CLASS_TYPE (tp_stream_engine_class),
                G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                0,
                NULL, NULL,
                g_cclosure_marshal_VOID__VOID,
                G_TYPE_NONE, 0);

  /**
   * TpStreamEngine::shutdown:
   *
   * Emitted whenever stream engine needs to be shutdown
   */
  signals[SHUTDOWN_REQUESTED] =
    g_signal_new ("shutdown-requested",
        G_OBJECT_CLASS_TYPE (tp_stream_engine_class),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE, 0);

  g_object_class_install_property (object_class, PROP_PIPELINE,
      g_param_spec_object ("pipeline",
          "The GstPipeline",
          "The GstPipeline that all the objects here live in",
          GST_TYPE_PIPELINE,
          G_PARAM_READABLE |
          G_PARAM_STATIC_STRINGS));
}

static void
tp_stream_engine_dispose (GObject *object)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (object);
  TpStreamEnginePrivate *priv = self->priv;

  if (priv->dispose_has_run)
    return;


  if (self->priv->reset_failcount_id)
    {
      g_source_remove (self->priv->reset_failcount_id);
      self->priv->reset_failcount_id = 0;
    }


  if (self->priv->restart_pipeline_id)
    {
      g_source_remove (self->priv->restart_pipeline_id);
      self->priv->restart_pipeline_id = 0;
    }

  if (priv->channels)
    {
      guint i;

      for (i = 0; i < priv->channels->len; i++)
        g_object_unref (g_ptr_array_index (priv->channels, i));

      g_ptr_array_free (priv->channels, TRUE);
      priv->channels = NULL;
    }

  if (priv->channels_by_path)
    {
      g_hash_table_destroy (priv->channels_by_path);
      priv->channels_by_path = NULL;
    }

  g_list_foreach (priv->preview_sinks, (GFunc) g_object_unref, NULL);
  g_list_free (priv->preview_sinks);
  priv->preview_sinks = NULL;

  if (priv->preview)
    g_object_unref (priv->preview);
  priv->preview = NULL;

  if (priv->pipeline)
    {
      /*
       * Lets not check the return values
       * if it fails or is async, it will crash on the following
       * unref anyways
       */
      gst_element_set_state (priv->videosrc, GST_STATE_NULL);
      gst_element_set_state (priv->pipeline, GST_STATE_NULL);
      gst_object_unref (priv->videosrc);
      gst_object_unref (priv->pipeline);
      priv->pipeline = NULL;
      priv->videosrc = NULL;
    }

  if (priv->notifier)
    {
      g_object_unref (priv->notifier);
      priv->notifier = NULL;
    }

  priv->dispose_has_run = TRUE;

  if (G_OBJECT_CLASS (tp_stream_engine_parent_class)->dispose)
    G_OBJECT_CLASS (tp_stream_engine_parent_class)->dispose (object);
}

static void
tp_stream_engine_finalize (GObject *object)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (object);

  g_mutex_free (self->priv->mutex);

  g_ptr_array_free (self->priv->audio_objects, TRUE);
  g_hash_table_destroy (self->priv->object_threads);

  G_OBJECT_CLASS (tp_stream_engine_parent_class)->finalize (object);
}

/**
 * tp_stream_engine_error
 *
 * Used to inform the stream engine than an exceptional situation has ocurred.
 *
 * @error:   The error ID, as per the
 *           org.freedesktop.Telepathy.Media.StreamHandler::Error signal.
 * @message: The human-readable error message.
 */
void
tp_stream_engine_error (TpStreamEngine *self, int error, const char *message)
{
  guint i;

  for (i = 0; i < self->priv->channels->len; i++)
    tf_channel_error (
      g_ptr_array_index (self->priv->channels, i), error, message);
}

static gboolean
tp_stream_engine_start_video_source (TpStreamEngine *self)
{
  GstStateChangeReturn state_ret;


  g_debug ("Starting video source");

  if (!gst_bin_add (GST_BIN (self->priv->pipeline), self->priv->videosrc))
    {
      g_warning ("Could not add videosrc to pipeline");
      return FALSE;
    }

  if (!gst_element_link (self->priv->videosrc, self->priv->capsfilter))
    {
      g_warning ("Could not link videosrc to capsfilter");
      return FALSE;
    }

  self->priv->video_source_use_count++;

  state_ret = gst_element_set_state (self->priv->videosrc, GST_STATE_PLAYING);

  if (state_ret == GST_STATE_CHANGE_FAILURE)
    {
      g_warning ("Error starting the video source");
      return FALSE;
    }
  else
    {
      return TRUE;
    }
}


static void
tp_stream_engine_stop_video_source (TpStreamEngine *self)
{
  GstStateChangeReturn state_ret;

  self->priv->video_source_use_count--;

  if (self->priv->video_source_use_count > 0)
    return;

  g_debug ("Stopping video source");

  if (!gst_bin_remove (GST_BIN (self->priv->pipeline), self->priv->videosrc))
      g_warning ("Could not remove video src from pipeline");

  state_ret = gst_element_set_state (self->priv->videosrc, GST_STATE_NULL);

  if (state_ret == GST_STATE_CHANGE_FAILURE)
    g_error ("Error stopping the video source");
  else if (state_ret == GST_STATE_CHANGE_ASYNC)
    g_debug ("Stopping video src async??");
}

static void
video_stream_free_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);

  if (!(dir & TP_MEDIA_STREAM_DIRECTION_SEND))
    return;

  tp_stream_engine_stop_video_source (self);
}

static gboolean
video_stream_request_resource (TfStream *stream,
    TpMediaStreamDirection dir,
    gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);

  if (!(dir & TP_MEDIA_STREAM_DIRECTION_SEND))
    return TRUE;

  return tp_stream_engine_start_video_source (self);
}

static void
channel_session_created (TfChannel *chan G_GNUC_UNUSED,
    FsConference *conference, FsParticipant *participant G_GNUC_UNUSED,
    gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);

  g_return_if_fail (conference);

  if (!gst_bin_add (GST_BIN (self->priv->pipeline), GST_ELEMENT (conference)))
    g_error ("Could not add conference to pipeline");

  gst_element_set_state (GST_ELEMENT (conference), GST_STATE_PLAYING);
}

static void
channel_session_invalidated (TfChannel *channel G_GNUC_UNUSED,
    FsConference *conference, FsParticipant *participant G_GNUC_UNUSED,
    gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);

  g_return_if_fail (conference);

  gst_element_set_locked_state (GST_ELEMENT (conference), TRUE);
  gst_element_set_state (GST_ELEMENT (conference), GST_STATE_NULL);

  gst_bin_remove (GST_BIN (self->priv->pipeline), GST_ELEMENT (conference));
}


static void
stream_closed (TfStream *stream, gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);
  GObject *sestream = NULL;

  sestream = g_object_get_data (G_OBJECT (stream), "se-stream");
  if (!sestream)
    return;

  if (TP_STREAM_ENGINE_IS_VIDEO_STREAM (sestream))
    {
      GstPad *pad, *peer;
      TpStreamEngineVideoStream *videostream =
          (TpStreamEngineVideoStream *) sestream;

      g_mutex_lock (self->priv->mutex);
      self->priv->output_sinks = g_list_remove (self->priv->output_sinks,
          videostream);
      g_mutex_unlock (self->priv->mutex);

      g_object_get (sestream, "pad", &pad, NULL);


      /* Take the stream lock to make sure nothing is flowing through the
       * pad
       * We can only do that because we have no blocking elements before
       * a queue in our pipeline after the pads.
       */
      peer = gst_pad_get_peer (pad);
      GST_PAD_STREAM_LOCK(pad);
      if (peer)
        {
          gst_pad_unlink (pad, peer);
          gst_object_unref (peer);
        }
      /*
       * Releasing request pads currently fail cause stuff like
       * alloc_buffer() does take any lock and there is no way to prevent it
       * ??? or something like that

       gst_element_release_request_pad (self->priv->videotee, pad);
      */
      GST_PAD_STREAM_UNLOCK(pad);

      gst_object_unref (pad);
    }
  else if (TP_STREAM_ENGINE_IS_AUDIO_STREAM (sestream))
    {
      g_mutex_lock (self->priv->mutex);
      self->priv->audio_streams = g_list_remove (self->priv->audio_streams,
          sestream);
      g_mutex_unlock (self->priv->mutex);
    }

  g_object_unref (sestream);
  g_object_set_data (G_OBJECT (stream), "se-stream", NULL);
}

static void
stream_receiving (TpStreamEngineVideoStream *videostream, gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);
  TfStream *stream = NULL;
  TfChannel *chan = NULL;
  guint stream_id;
  gchar *channel_path;

  g_object_get (videostream, "stream", &stream, NULL);

  g_object_get (stream, "channel", &chan, "stream-id", &stream_id, NULL);

  g_object_get (chan, "object-path", &channel_path, NULL);

  stream_engine_svc_stream_engine_emit_receiving (self,
      channel_path, stream_id, TRUE);

  g_object_unref (chan);
  g_object_unref (stream);
  g_free (channel_path);
}

static void
remove_audio_object (gpointer data, GObject *where_the_obj_was)
{
  TpStreamEngine *self = data;

  g_mutex_lock (self->priv->mutex);
  g_ptr_array_remove (self->priv->audio_objects, where_the_obj_was);
  g_mutex_unlock (self->priv->mutex);
}

static void
audio_pad_linked (GstPad *pad, GstPad *peer, TpStreamEngine *self);

static gboolean
insert_audio_object_locked (TpStreamEngine *self, gpointer object)
{
  guint i;
  pthread_t thread;

  for (i = 0; i < self->priv->audio_objects->len; i++)
    if (g_ptr_array_index (self->priv->audio_objects, i) == object)
      return FALSE;

  g_ptr_array_add (self->priv->audio_objects, object);
  g_object_weak_ref (G_OBJECT (object), remove_audio_object, self);

  thread = GPOINTER_TO_UINT (g_hash_table_lookup (self->priv->object_threads,
          object));
  if (thread)
    setup_realtime_thread (thread);

  return TRUE;
}

static void
add_audio_pad (TpStreamEngine *self, GstPad *pad)
{
  GstElement *el;

  insert_audio_object_locked (self, pad);

  el = gst_pad_get_parent_element (pad);

  if (el)
    {
      insert_audio_object_locked (self, el);
      gst_object_unref (el);
    }

  if (gst_pad_get_direction(pad) == GST_PAD_SINK)
    g_signal_connect (pad, "linked", G_CALLBACK (audio_pad_linked), self);
}

static void
audio_pad_iter (gpointer data, gpointer user_data)
{
  TpStreamEngine *self = user_data;
  GstPad *pad = data;
  GstPad *peerpad = gst_pad_get_peer (pad);

  add_audio_pad (self, pad);

  if (peerpad)
    {
      GstIterator *iter;
      add_audio_pad (self, peerpad);

      iter = gst_pad_iterate_internal_links (peerpad);
      while (gst_iterator_foreach (iter, audio_pad_iter, self) ==
          GST_ITERATOR_RESYNC)
        gst_iterator_resync (iter);
      gst_iterator_free (iter);

      gst_object_unref (peerpad);
    }
}

static void
audio_pad_linked (GstPad *pad, GstPad *peer, TpStreamEngine *self)
{
  GstIterator *iter = gst_pad_iterate_internal_links (pad);

  g_mutex_lock (self->priv->mutex);
  add_audio_pad (self, peer);

  while (gst_iterator_foreach (iter, audio_pad_iter, self) ==
      GST_ITERATOR_RESYNC)
    gst_iterator_resync (iter);

  gst_iterator_free (iter);
  g_mutex_unlock (self->priv->mutex);
}

static void
audio_pad_internal_links_iter (gpointer data, gpointer user_data)
{
  TpStreamEngine *self = user_data;
  GstPad *pad = data;
  GstIterator *iter;

  iter = gst_pad_iterate_internal_links (pad);
  while (gst_iterator_foreach (iter, audio_pad_iter, self) ==
      GST_ITERATOR_RESYNC)
    gst_iterator_resync (iter);
  gst_iterator_free (iter);
}

static void
audio_sink_added_cb (TpStreamEngineAudioStream *audiostream,
    GstElement *sink, TpStreamEngine *self)
{
  GstIterator *iter;

  g_mutex_lock (self->priv->mutex);

  iter = gst_element_iterate_sink_pads (sink);
  while (gst_iterator_foreach (iter, audio_pad_iter, self) ==
      GST_ITERATOR_RESYNC)
    gst_iterator_resync (iter);

  gst_iterator_resync (iter);
  while (gst_iterator_foreach (iter, audio_pad_internal_links_iter, self) ==
      GST_ITERATOR_RESYNC)
    gst_iterator_resync (iter);
  gst_iterator_free (iter);

  g_mutex_unlock (self->priv->mutex);
}

static void
channel_stream_created (TfChannel *chan G_GNUC_UNUSED,
    TfStream *stream, gpointer user_data)
{
  guint media_type;
  GError *error = NULL;
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);

  g_object_get (stream, "media-type", &media_type, NULL);

  if (media_type == TP_MEDIA_STREAM_TYPE_AUDIO)
    {
      TpStreamEngineAudioStream *audiostream;
      GstIterator *iter;
      GstElement *src;

      g_object_set (stream, "tos", IPTOS_LOWDELAY, NULL);

      audiostream = tp_stream_engine_audio_stream_new (stream,
          GST_BIN (self->priv->pipeline), &error);

      if (!audiostream)
        {
          tf_stream_error (stream, TP_MEDIA_STREAM_ERROR_UNKNOWN,
              "Could not create audio stream");
          g_warning ("Could not create audio stream: %s", error->message);
          return;
        }
      g_clear_error (&error);
      g_object_set_data (G_OBJECT (stream), "se-stream", audiostream);

      g_mutex_lock (self->priv->mutex);
      self->priv->audio_streams = g_list_prepend (self->priv->audio_streams,
          audiostream);
      g_mutex_unlock (self->priv->mutex);

      g_object_get (audiostream, "src", &src, NULL);

      g_mutex_lock (self->priv->mutex);
      iter = gst_element_iterate_src_pads (src);
      while (gst_iterator_foreach (iter, audio_pad_iter, self) ==
          GST_ITERATOR_RESYNC)
        gst_iterator_resync (iter);

      gst_iterator_resync (iter);
      while (gst_iterator_foreach (iter, audio_pad_internal_links_iter, self) ==
          GST_ITERATOR_RESYNC)
        gst_iterator_resync (iter);
      gst_iterator_free (iter);
      g_mutex_unlock (self->priv->mutex);

      gst_object_unref (src);

      g_signal_connect (audiostream, "sink-added",
          G_CALLBACK (audio_sink_added_cb), self);
    }
  else if (media_type == TP_MEDIA_STREAM_TYPE_VIDEO)
    {
      TpStreamEngineVideoStream *videostream = NULL;
      GstPad *pad;

      g_object_set (stream, "tos", IPTOS_THROUGHPUT, NULL);

      pad = gst_element_get_request_pad (self->priv->videotee, "src%d");

      videostream = tp_stream_engine_video_stream_new (stream,
          GST_BIN (self->priv->pipeline), pad, &error);

      if (!videostream)
        {
          g_warning ("Could not create video stream: %s", error->message);
          tf_stream_error (stream, TP_MEDIA_STREAM_ERROR_UNKNOWN,
              "Could not create video stream");
          gst_element_release_request_pad (self->priv->videotee, pad);
          return;
        }
      g_clear_error (&error);
      g_object_set_data (G_OBJECT (stream), "se-stream", videostream);

      g_mutex_lock (self->priv->mutex);
      self->priv->output_sinks = g_list_prepend (self->priv->output_sinks,
          videostream);
      g_mutex_unlock (self->priv->mutex);

      g_signal_connect (videostream, "receiving",
          G_CALLBACK (stream_receiving), self);

      g_signal_connect (stream, "request-resource",
          G_CALLBACK (video_stream_request_resource), self);
      g_signal_connect (stream, "free-resource",
          G_CALLBACK (video_stream_free_resource), self);
    }

  g_signal_connect (stream, "closed", G_CALLBACK (stream_closed), self);
}

static GList *
load_codecs_config (const gchar *path, const gchar *subdir)
{
  gchar *filename;
  GError *error = NULL;
  GList *codec_config = NULL;

  filename = g_build_filename (path, subdir, "gstcodecs.conf", NULL);

  codec_config = fs_codec_list_from_keyfile (filename, &error);

  if (!codec_config)
      g_debug ("Could not read local codecs config at %s: %s", filename,
          error ? error->message : "(no error message set)");

  g_clear_error (&error);
  g_free (filename);

  return codec_config;
}

static GList *
stream_get_codec_config (TfChannel *chan,
    guint stream_id,
    TpMediaStreamType media_type,
    TpMediaStreamDirection direction,
    TpStreamEngine *self)
{
  GList *codec_config = NULL;
  const gchar *home = NULL;

  home = g_getenv ("HOME");
  if (!home)
    home = g_get_home_dir ();

  codec_config = load_codecs_config (home, ".stream-engine");
  if (codec_config)
    return codec_config;

  codec_config = load_codecs_config (SYSCONFDIR, "stream-engine");

  return codec_config;
}

static void
check_if_busy (TpStreamEngine *self)
{
  guint num_previews;

  g_mutex_lock (self->priv->mutex);
  num_previews = g_list_length (self->priv->preview_sinks);
  g_mutex_unlock (self->priv->mutex);

  if (self->priv->channels->len == 0 && num_previews == 0)
    {
      g_debug ("no channels or previews remaining; emitting no-more-channels");
      g_signal_emit (self, signals[NO_MORE_CHANNELS], 0);
    }
  else
    {
      g_debug ("channels remaining: %d", self->priv->channels->len);
      g_debug ("preview windows remaining: %d", num_previews);
    }
}


static void
channel_closed (TfChannel *chan, gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);
  gchar *object_path;

  g_object_get (chan, "object-path", &object_path, NULL);

  g_debug ("channel %s (%p) closed", object_path, chan);

  g_ptr_array_remove_fast (self->priv->channels, chan);
  g_object_unref (chan);

  g_hash_table_remove (self->priv->channels_by_path, object_path);
  g_free (object_path);

  check_if_busy (self);
}

static void
error_one_stream (TfChannel *chan G_GNUC_UNUSED,
    guint stream_id G_GNUC_UNUSED,
    TfStream *stream,
    gpointer user_data)
{
  const gchar *message = (const gchar *) user_data;

  tf_stream_error (stream, TP_MEDIA_STREAM_ERROR_UNKNOWN,
      message);
}


static void
error_all_streams (TpStreamEngine *self, const gchar *message)
{
  guint i;

  g_debug ("Closing all streams");

  for (i = 0; i < self->priv->channels->len; i++)
    {
      TfChannel *channel = g_ptr_array_index (self->priv->channels,
          i);
      tf_channel_foreach_stream (channel,
          error_one_stream, (gpointer) message);
    }
}

static gboolean
reset_failcount (gpointer data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (data);

  self->priv->failcount = 0;
  self->priv->reset_failcount_id = 0;

  return FALSE;
}

static gboolean
restart_pipeline (gpointer data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (data);
  GstStateChangeReturn ret;
  GList *item;

  if (self->priv->reset_failcount_id)
    {
      g_source_remove (self->priv->reset_failcount_id);
      self->priv->reset_failcount_id = 0;
    }

  self->priv->failcount++;

  ret = gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
  g_assert (ret != GST_STATE_CHANGE_FAILURE);
  ret = gst_element_set_state (self->priv->videosrc, GST_STATE_NULL);
  g_assert (ret != GST_STATE_CHANGE_FAILURE);

  g_mutex_lock (self->priv->mutex);
  for (item = self->priv->audio_streams; item ; item = item->next)
    {
      if (!tp_stream_engine_audio_stream_set_playing (
              TP_STREAM_ENGINE_AUDIO_STREAM (item->data), FALSE))
        {
          g_mutex_unlock (self->priv->mutex);
          goto fail;
        }
    }
  g_mutex_unlock (self->priv->mutex);

  ret = gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto fail;
  if (self->priv->video_source_use_count)
    {
      gst_element_set_state (self->priv->videosrc, GST_STATE_PLAYING);
      if (ret == GST_STATE_CHANGE_FAILURE)
        goto fail;
    }

  g_mutex_lock (self->priv->mutex);
  for (item = self->priv->audio_streams; item ; item = item->next)
    {
      if (!tp_stream_engine_audio_stream_set_playing (
              TP_STREAM_ENGINE_AUDIO_STREAM (item->data), TRUE))
        {
          g_mutex_unlock (self->priv->mutex);
          goto fail;
        }
    }
  g_mutex_unlock (self->priv->mutex);

  self->priv->restart_pipeline_id = 0;
  self->priv->reset_failcount_id =
      g_timeout_add_seconds (4, reset_failcount, self);


  return FALSE;

 fail:

  g_warning ("Failed to restart the pipeline after an error");

  if (self->priv->failcount > 5)
    {
      error_all_streams (self, "Could not restart the pipeline after an error");
      g_error ("Failed five times to restart the pipeline after an error");
      return FALSE;
    }

  if (!self->priv->restart_pipeline_id)
    self->priv->restart_pipeline_id = g_timeout_add (200, restart_pipeline,
        self);


  return TRUE;
}

static gboolean
bus_async_handler (GstBus *bus G_GNUC_UNUSED,
    GstMessage *message,
    gpointer data)
{
  TpStreamEngine *engine = TP_STREAM_ENGINE (data);
  TpStreamEnginePrivate *priv = engine->priv;
  GError *error = NULL;
  gchar *error_string;
  guint i;
  GstObject *source = GST_MESSAGE_SRC (message);
  gchar *name = NULL;

  for (i = 0; i < priv->channels->len; i++)
    if (tf_channel_bus_message (
            g_ptr_array_index (priv->channels, i), message))
      return TRUE;

  name = gst_object_get_name (source);

  switch (GST_MESSAGE_TYPE (message))
    {
      case GST_MESSAGE_ERROR:
        gst_message_parse_error (message, &error, &error_string);


        //error_all_streams (engine, error->message);

        g_warning ("%s: got error from %s: %s: %s (%d %d), stopping pipeline",
            G_STRFUNC, name, error->message, error_string,
            error->domain, error->code);

        if (priv->restart_pipeline_id == 0)
          restart_pipeline (engine);

        g_free (error_string);
        g_error_free (error);
        break;
      case GST_MESSAGE_WARNING:
        {
          gchar *debug_msg = NULL;
          gst_message_parse_warning (message, &error, &debug_msg);
          g_warning ("%s: got warning: %s (%s)", G_STRFUNC, error->message,
              debug_msg);
          g_free (debug_msg);
          g_error_free (error);
          break;
        }
      case GST_MESSAGE_LATENCY:
        gst_bin_recalculate_latency (GST_BIN (engine->priv->pipeline));
        break;
      default:
        break;
    }

  g_free (name);
  return TRUE;
}

static void
setup_realtime_thread (pthread_t thread)
{
  struct sched_param param;
  gchar buf[128];

  param.sched_priority = 5;
  if (pthread_setschedparam (thread, SCHED_FIFO, &param))
    {
      strerror_r (errno, buf, 128);
      g_warning ("Could not set sched_param: %s", buf);
    }
}

static void
enter_thread (TpStreamEngine *self, GstObject *src, GstElement *owner)
{
  pthread_t thread = pthread_self ();
  guint i;

  g_mutex_lock (self->priv->mutex);
  g_hash_table_insert (self->priv->object_threads, src,
      GINT_TO_POINTER (thread));

  for (i = 0; i < self->priv->audio_objects->len; i++)
    if (g_ptr_array_index (self->priv->audio_objects, i) == src)
      {
        setup_realtime_thread (thread);
        break;
      }

  g_mutex_unlock (self->priv->mutex);
}

static void
leave_thread (TpStreamEngine *self, GstObject *src, GstElement *owner)
{
  pthread_t thread = pthread_self ();
  struct sched_param param;

  g_mutex_lock (self->priv->mutex);
  g_hash_table_remove (self->priv->object_threads, src);
  g_mutex_unlock (self->priv->mutex);

  param.sched_priority = 0;
  pthread_setschedparam (thread, SCHED_OTHER, &param);
}

static GstBusSyncReply
bus_sync_handler (GstBus *bus G_GNUC_UNUSED, GstMessage *message, gpointer data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (data);

  switch (GST_MESSAGE_TYPE (message))
    {
    case GST_MESSAGE_ELEMENT:
      {
        GList *item;
        gboolean handled = FALSE;

        if (!gst_structure_has_name (message->structure, "prepare-xwindow-id"))
          return GST_BUS_PASS;

        g_mutex_lock (self->priv->mutex);

        if (self->priv->preview)
          handled = tp_stream_engine_video_sink_bus_sync_message (
              TP_STREAM_ENGINE_VIDEO_SINK (self->priv->preview), message);
        if (handled)
          goto done;

        for (item = g_list_first (self->priv->preview_sinks);
             item && !handled;
             item = g_list_next (item))
          {
            TpStreamEngineVideoSink *preview = item->data;

            handled = tp_stream_engine_video_sink_bus_sync_message (preview, message);
            if (handled)
              break;
          }

        if (!handled)
          {
            for (item = g_list_first (self->priv->output_sinks);
                 item && !handled;
                 item = g_list_next (item))
              {
                TpStreamEngineVideoSink *output = item->data;

                handled = tp_stream_engine_video_sink_bus_sync_message (output,
                    message);
                if (handled)
                  break;
              }
          }
      done:
        g_mutex_unlock (self->priv->mutex);

        if (handled)
          {
            gst_message_unref (message);
            return GST_BUS_DROP;
          }
      }
      break;
    case GST_MESSAGE_STREAM_STATUS:
      {
        GstStreamStatusType type;
        GstElement *owner;

        gst_message_parse_stream_status (message, &type, &owner);

        switch (type)
          {
          case GST_STREAM_STATUS_TYPE_ENTER:
            enter_thread (self, GST_MESSAGE_SRC (message), owner);
            break;
          case GST_STREAM_STATUS_TYPE_LEAVE:
            leave_thread (self, GST_MESSAGE_SRC (message), owner);
            break;
          default:
            break;
          }
      }
      break;
    default:
      break;
    }
  return GST_BUS_PASS;
}

static void
_build_base_video_elements (TpStreamEngine *self)
{
  TpStreamEnginePrivate *priv = self->priv;
  GstElement *videosrc = NULL;
  GstElement *tee;
  GstElement *capsfilter;
  const gchar *elem;
  const gchar *caps_str;
  gboolean ret;
  GstCaps *filter = NULL;

  if ((elem = getenv ("FS_VIDEO_SRC")) || (elem = getenv ("FS_VIDEOSRC")))
    {
      g_debug ("making video src with pipeline \"%s\"", elem);
      videosrc = gst_parse_bin_from_description (elem, TRUE, NULL);
      g_assert (videosrc);
      gst_element_set_name (videosrc, "videosrc");
    }
  else
    {
      if (videosrc == NULL)
        videosrc = gst_element_factory_make ("gconfvideosrc", NULL);

      if (videosrc == NULL)
        videosrc = gst_element_factory_make ("v4l2camsrc", NULL);

      if (videosrc == NULL)
        videosrc = gst_element_factory_make ("v4l2src", NULL);

      if (videosrc == NULL)
        videosrc = gst_element_factory_make ("v4lsrc", NULL);

      if (videosrc != NULL)
        {
          g_debug ("using %s as video source", GST_OBJECT_NAME (videosrc));
        }
      else
        {
          videosrc = gst_element_factory_make ("videotestsrc", NULL);
          if (videosrc == NULL)
            g_error ("failed to create any video source");
          g_object_set (videosrc, "is-live", TRUE, NULL);
        }
    }

  if ((caps_str = getenv ("FS_VIDEO_SRC_CAPS")) != NULL ||
      (caps_str = getenv ("FS_VIDEOSRC_CAPS")) != NULL)
    {
      filter = gst_caps_from_string (caps_str);
      g_debug ("applying custom caps '%s' on the video source", caps_str);
    }
  else
    {
      filter = gst_caps_new_simple ("video/x-raw-yuv",
#if 0
          "width", G_TYPE_INT, 352,
          "height", G_TYPE_INT, 288,
#else
          "width", G_TYPE_INT, 176,
          "height", G_TYPE_INT, 144,
#endif
          "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('I', '4', '2', '0'),
          "framerate", GST_TYPE_FRACTION_RANGE, 5, 1, 20, 1,
          NULL);
    }

  gst_element_set_locked_state (videosrc, TRUE);

  tee = gst_element_factory_make ("tee", "videotee");
  g_assert (tee);
  if (!gst_bin_add (GST_BIN (priv->pipeline), tee))
    g_error ("Could not add tee to pipeline");

#if 0
  /* <wtay> Tester_, yes, videorate does not play nice with live pipelines */
  tmp = gst_element_factory_make ("videorate", NULL);
  if (tmp != NULL)
    {
      g_debug ("linking videorate");
      gst_bin_add (GST_BIN (priv->pipeline), tmp);
      gst_element_link (videosrc, tmp);
      videosrc = tmp;
    }
#endif

  capsfilter = gst_element_factory_make ("capsfilter", NULL);
  g_assert (capsfilter);
  ret = gst_bin_add (GST_BIN (priv->pipeline), capsfilter);
  g_assert (ret);

  g_object_set (capsfilter, "caps", filter, NULL);
  gst_caps_unref (filter);

  ret = gst_element_link (capsfilter, tee);
  g_assert (ret);

  priv->capsfilter = capsfilter;
  priv->videosrc = gst_object_ref (videosrc);
  priv->videotee = tee;

}


static void
_create_pipeline (TpStreamEngine *self)
{
  TpStreamEnginePrivate *priv = self->priv;
  GstBus *bus;
  GstStateChangeReturn state_ret;

  priv->pipeline = gst_pipeline_new (NULL);

  fs_element_added_notifier_add (priv->notifier, GST_BIN (priv->pipeline));

  _build_base_video_elements (self);

  /* connect a callback to the stream bus so that we can set X window IDs
   * at the right time, and detect when sinks have gone away */
  bus = gst_element_get_bus (priv->pipeline);
  gst_bus_set_sync_handler (bus, bus_sync_handler, self);
  priv->bus_async_source_id =
    gst_bus_add_watch (bus, bus_async_handler, self);
  gst_object_unref (bus);

  state_ret = gst_element_set_state (priv->pipeline, GST_STATE_PLAYING);
  g_assert (state_ret != GST_STATE_CHANGE_FAILURE);
}


static void
_preview_window_plug_deleted (TpStreamEngineVideoPreview *preview,
    gpointer user_data)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (user_data);
  GstPad *pad = NULL;

  while (g_signal_handlers_disconnect_by_func(preview,
          _preview_window_plug_deleted,
          user_data)) {}

  g_mutex_lock (self->priv->mutex);
  self->priv->preview_sinks = g_list_remove (self->priv->preview_sinks,
      preview);
  g_mutex_unlock (self->priv->mutex);

  tp_stream_engine_stop_video_source (self);

  g_object_get (preview, "pad", &pad, NULL);

  if (pad)
    {
      GstPad *peer;
      /* Take the stream lock to make sure nothing is flowing through the
       * pad
       * We can only do that because we have no blocking elements before
       * a queue in our pipeline after the pads.
       */
      peer = gst_pad_get_peer (pad);
      GST_PAD_STREAM_LOCK(pad);
      if (peer)
        {
          gst_pad_unlink (pad, peer);
          gst_object_unref (peer);
        }
      //gst_element_release_request_pad (self->priv->videotee, pad);
      GST_PAD_STREAM_UNLOCK(pad);

      gst_object_unref (pad);
    }

  gst_object_unref (preview);
}

/**
 * tp_stream_engine_create_preview_window
 *
 * Implements DBus method CreatePreviewWindow
 * on interface org.maemo.Telepathy.StreamEngine
 */
static void
tp_stream_engine_create_preview_window (StreamEngineSvcStreamEngine *iface,
    DBusGMethodInvocation *context)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (iface);
  GError *error = NULL;
  GstPad *pad;
  TpStreamEngineVideoPreview *preview;
  guint window_id;

  preview = tp_stream_engine_video_preview_new (GST_BIN (self->priv->pipeline),
      &error);

  if (!preview)
    {
      dbus_g_method_return_error (context, error);
      g_clear_error (&error);
      return;
    }

  g_mutex_lock (self->priv->mutex);
  self->priv->preview_sinks = g_list_prepend (self->priv->preview_sinks,
      preview);
  g_mutex_unlock (self->priv->mutex);

  pad = gst_element_get_request_pad (self->priv->videotee, "src%d");

  g_object_set (preview, "pad", pad, NULL);

  g_signal_connect (preview, "plug-deleted",
      G_CALLBACK (_preview_window_plug_deleted), self);

  g_object_get (preview, "window-id", &window_id, NULL);

  stream_engine_svc_stream_engine_return_from_create_preview_window (context,
      window_id);

  tp_stream_engine_start_video_source (self);

  g_signal_emit (self, signals[HANDLING_CHANNEL], 0);
}


/**
 * tp_stream_engine_get_preview_window
 *
 * Implements DBus method GetPreviewWindow
 * on interface org.maemo.Telepathy.StreamEngine
 */
static void
tp_stream_engine_get_preview_window (StreamEngineSvcStreamEngine *iface,
    DBusGMethodInvocation *context)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (iface);
  GError *error = NULL;
  GstPad *pad;
  TpStreamEngineVideoPreview *preview;
  guint window_id;

  g_mutex_lock (self->priv->mutex);
  if (self->priv->preview)
    {
      g_object_get (self->priv->preview, "window-id", &window_id, NULL);
      g_mutex_unlock (self->priv->mutex);
      stream_engine_svc_stream_engine_return_from_create_preview_window (
          context,
          window_id);
      return;
    }
  g_mutex_unlock (self->priv->mutex);

  preview = tp_stream_engine_video_preview_new (GST_BIN (self->priv->pipeline),
      &error);

  if (!preview)
    {
      dbus_g_method_return_error (context, error);
      g_clear_error (&error);
      return;
    }

  g_mutex_lock (self->priv->mutex);
  if (self->priv->preview)
    {
      g_object_get (self->priv->preview, "window-id", &window_id, NULL);
      g_mutex_unlock (self->priv->mutex);
      stream_engine_svc_stream_engine_return_from_create_preview_window (
          context,
          window_id);
      g_object_unref (preview);
      return;
    }
  self->priv->preview = preview;
  g_mutex_unlock (self->priv->mutex);

  pad = gst_element_get_request_pad (self->priv->videotee, "src%d");

  g_object_set (preview, "pad", pad, NULL);

  g_object_get (preview, "window-id", &window_id, NULL);

  stream_engine_svc_stream_engine_return_from_create_preview_window (context,
      window_id);
}


static void
handler_result (TfChannel *chan G_GNUC_UNUSED,
    GError *error,
    DBusGMethodInvocation *context)
{
  if (error == NULL)
    stream_engine_svc_stream_engine_return_from_attach_to_channel (context);
  else
    dbus_g_method_return_error (context, error);
}


/**
 * _channel_new:
 * @dbus_daemon: a #TpDBusDaemon
 * @bus_name: the name of the bus to connect to
 * @connection_path: the connection path to connect to
 * @channel_path: the path of the channel to connect to
 * @handle_type: the type of handle
 * @handle: the handle
 * @error: The location of a %GError or %NULL
 *
 * Creates a new #TfChannel by connecting to the D-Bus bus and finding
 * an already existing channel object. This API would normally be used with the
 * HandleChannel method.
 *
 * Returns: a new #TfChannel
 */

static TfChannel *
_channel_new (TpDBusDaemon *dbus_daemon,
    const gchar *bus_name,
    const gchar *connection_path,
    const gchar *channel_path,
    guint handle_type,
    guint handle,
    GError **error)
{
  TpConnection *connection;
  TpChannel *channel_proxy;
  TfChannel *ret;

  g_return_val_if_fail (bus_name != NULL, NULL);
  g_return_val_if_fail (connection_path != NULL, NULL);
  g_return_val_if_fail (channel_path != NULL, NULL);

  connection = tp_connection_new (dbus_daemon,
      bus_name, connection_path, error);

  if (connection == NULL)
    return NULL;

  channel_proxy = tp_channel_new (connection, channel_path,
      TP_IFACE_CHANNEL_TYPE_STREAMED_MEDIA, handle_type, handle, error);

  if (channel_proxy == NULL)
    return NULL;

  g_object_unref (connection);

  ret = tf_channel_new (channel_proxy);

  g_object_unref (channel_proxy);

  return ret;
}


/**
 * tp_stream_engine_attach_to_channel
 *
 * Implements DBus method AttachToChannel
 * on interface org.maemo.Telepathy.StreamEngine
 */
static void
tp_stream_engine_attach_to_channel (StreamEngineSvcStreamEngine *iface,
    const gchar *bus_name,
    const gchar *connection,
    const gchar *channel_type,
    const gchar *channel,
    guint handle_type,
    guint handle,
    DBusGMethodInvocation *context)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (iface);
  TfChannel *chan = NULL;
  GError *error = NULL;

  g_debug("HandleChannel called");

  if (strcmp (channel_type, TP_IFACE_CHANNEL_TYPE_STREAMED_MEDIA) != 0)
    {
      GError e = { TP_ERRORS, TP_ERROR_INVALID_ARGUMENT,
        "Stream Engine was passed a channel that was not a "
        TP_IFACE_CHANNEL_TYPE_STREAMED_MEDIA };

      g_message ("%s", e.message);
      dbus_g_method_return_error (context, &e);
      return;
     }

  chan = _channel_new (self->priv->dbus_daemon, bus_name,
      connection, channel, handle_type, handle, &error);

  if (chan == NULL)
    {
      dbus_g_method_return_error (context, error);
      g_error_free (error);
      return;
    }

  g_ptr_array_add (self->priv->channels, chan);
  g_hash_table_insert (self->priv->channels_by_path, g_strdup (channel), chan);

  g_signal_connect (chan, "handler-result", G_CALLBACK (handler_result),
      context);
  g_signal_connect (chan, "closed", G_CALLBACK (channel_closed), self);
  g_signal_connect (chan, "session-created",
      G_CALLBACK (channel_session_created), self);
  g_signal_connect (chan, "session-invalidated",
      G_CALLBACK (channel_session_invalidated), self);
  g_signal_connect (chan, "stream-created",
      G_CALLBACK (channel_stream_created), self);
  g_signal_connect (chan, "stream-get-codec-config",
      G_CALLBACK (stream_get_codec_config), self);

  g_signal_emit (self, signals[HANDLING_CHANNEL], 0);
}

void
tp_stream_engine_register (TpStreamEngine *self)
{
  DBusGConnection *bus;
  GError *error = NULL;
  guint request_name_result;

  g_assert (TP_IS_STREAM_ENGINE (self));

  bus = tp_get_bus ();
  self->priv->dbus_daemon = tp_dbus_daemon_new (bus);

  g_debug("registering StreamEngine at " OBJECT_PATH);
  dbus_g_connection_register_g_object (bus, OBJECT_PATH, G_OBJECT (self));

  register_dbus_signal_marshallers();

  g_debug("Requesting " BUS_NAME);

  if (!tp_cli_dbus_daemon_run_request_name (self->priv->dbus_daemon, -1,
        BUS_NAME, DBUS_NAME_FLAG_DO_NOT_QUEUE, &request_name_result, &error,
        NULL))
    g_error ("Failed to request bus name: %s", error->message);

  if (request_name_result == DBUS_REQUEST_NAME_REPLY_EXISTS)
    g_error ("Failed to acquire bus name, stream engine already running?");
}

static TfStream *
_lookup_stream (TpStreamEngine *self,
    const gchar *path,
    guint stream_id,
    GError **error)
{
  TfChannel *channel;
  TfStream *stream;

  channel = g_hash_table_lookup (self->priv->channels_by_path, path);
  if (channel == NULL)
    {
      g_set_error (error, TP_ERRORS, TP_ERROR_NOT_AVAILABLE,
        "stream-engine is not handling the channel %s", path);

      return NULL;
    }

  stream = tf_channel_lookup_stream (channel, stream_id);
  if (stream == NULL)
    {
      g_set_error (error, TP_ERRORS, TP_ERROR_NOT_AVAILABLE,
          "the channel %s has no stream with id %d", path, stream_id);

      return NULL;
    }

  return stream;
}

/**
 * tp_stream_engine_get_output_window
 *
 * Implements DBus method SetOutputWindow
 * on interface org.maemo.Telepathy.StreamEngine
 */
static void
tp_stream_engine_get_output_window (StreamEngineSvcStreamEngine *iface,
                                    const gchar *channel_path,
                                    guint stream_id,
                                    DBusGMethodInvocation *context)
{
  TpStreamEngine *self = TP_STREAM_ENGINE (iface);
  TfStream *stream;
  TpMediaStreamType media_type;
  GError *error = NULL;

  g_debug ("%s: channel_path=%s, stream_id=%u", G_STRFUNC, channel_path,
      stream_id);

  stream = _lookup_stream (self, channel_path, stream_id, &error);

  if (stream == NULL)
    {
      error = g_error_new (TP_ERRORS, TP_ERROR_INVALID_ARGUMENT,
          "Stream does not exist");
      dbus_g_method_return_error (context, error);
      g_error_free (error);
      return;
    }

  g_object_get (stream, "media-type", &media_type, NULL);

  if (media_type != TP_MEDIA_STREAM_TYPE_VIDEO)
    {
      error = g_error_new (TP_ERRORS, TP_ERROR_INVALID_ARGUMENT,
          "GetOutputWindow can only be called on video streams");
      dbus_g_method_return_error (context, error);
      g_error_free (error);
      return;
    }

  if (stream)
    {
      guint window_id;
      TpStreamEngineVideoSink *videosink;

      videosink = TP_STREAM_ENGINE_VIDEO_SINK (
          g_object_get_data ((GObject*) stream, "se-stream"));

      g_object_get (videosink, "window-id", &window_id, NULL);

      g_debug ("Returning window id %u", window_id);

      stream_engine_svc_stream_engine_return_from_get_output_window (context,
          window_id);
    }
}

/*
 * tp_stream_engine_get
 *
 * Return the stream engine singleton. Caller does not own a reference to the
 * stream engine.
 */

TpStreamEngine *
tp_stream_engine_get ()
{
  static TpStreamEngine *engine = NULL;

  if (NULL == engine)
    {
      engine = g_object_new (TP_TYPE_STREAM_ENGINE, NULL);
      g_object_add_weak_pointer (G_OBJECT (engine), (gpointer) &engine);
    }

  return engine;
}

/**
 * tp_stream_engine_shutdown
 *
 * Implements DBus method Shutdown
 * on interface org.maemo.Telepathy.StreamEngine
 */
static void
tp_stream_engine_shutdown (StreamEngineSvcStreamEngine *iface,
    DBusGMethodInvocation *context)
{
  g_debug ("%s: Emitting shutdown signal", G_STRFUNC);
  g_signal_emit (iface, signals[SHUTDOWN_REQUESTED], 0);
  stream_engine_svc_stream_engine_return_from_shutdown (context);
}

static void
se_iface_init (gpointer iface, gpointer data G_GNUC_UNUSED)
{
  StreamEngineSvcStreamEngineClass *klass = iface;

#define IMPLEMENT(x) stream_engine_svc_stream_engine_implement_##x (\
    klass, tp_stream_engine_##x)
  IMPLEMENT (get_output_window);
  IMPLEMENT (create_preview_window);
  IMPLEMENT (get_preview_window);
  IMPLEMENT (shutdown);
  IMPLEMENT (attach_to_channel);
#undef IMPLEMENT
}
