/*
 * This file is part of sp-rich-core
 *
 * Copyright (C) 2006-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */  

#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#define RICHCORE_HEADER "[---rich-core: "

const char *usage = "%s <input filename> [<output directory>]\n";

int main(int argc, char *argv[])
{
	char *input_fn;
	char *output_dir;
	struct stat stat_s;
	FILE *input_file;
	FILE *output_file;
	char buf[4096];
	int i;

	if ( argc < 2 ) {
		fprintf(stderr, usage, argv[0]);
		exit(1);
	}

	input_fn = argv[1];
	if ( argc < 3 ) {
		/* check if filename ends with .rcore or .rcore.lzo */
		char *suffix = strstr(input_fn, ".rcore");

		if ( !suffix || (strcmp(suffix, ".rcore.lzo") && 
		     strcmp(suffix, ".rcore") ) ) {
			fprintf(stderr, "please specify output directory\n");
			exit(1);
		}
		if ( !strcmp(suffix, ".rcore") ){
			output_dir = strndup( input_fn, strlen(input_fn)-6 );
		} else { 
			output_dir = strndup( input_fn, strlen(input_fn)-10 );
		}
		
	} else {
		output_dir = argv[2];
	}
#ifdef DEBUG
	fprintf(stderr, "input: '%s'\n", input_fn);
	fprintf(stderr, "output: '%s'\n", output_dir);
#endif

	if ( stat(input_fn, &stat_s) ) {
		fprintf(stderr, "input file error: %s\n", strerror(errno));
		exit(1);
	}

	if ( S_ISDIR(stat_s.st_mode) ) {
		fprintf(stderr, "%s is a directory\n", input_fn);
		exit(1);
	}

	if ( stat(output_dir, &stat_s) ) {
		if ( errno != ENOENT ) {
			fprintf(stderr, "error testing output: %s\n",
					strerror(errno));
			exit(1);
		}
	} else {
		fprintf(stderr, "%s exists, aborting\n", output_dir);
		exit(1);
	}

	if ( mkdir(output_dir, 0777) ) {
		fprintf(stderr, "error creating %s: %s\n",
				output_dir,
				strerror(errno));
		exit(1);
	}

	snprintf(buf, 255, "lzop -d -c %s", input_fn);
	input_file = popen(buf, "r");
	output_file = fopen("/dev/null", "w");
	if ( !input_file ) {
		fprintf(stderr, "error forking lzop: %s\n",
				strerror(errno));
		exit(1);
	}

	while ( fgets(buf, sizeof(buf), input_file) ) {
		if ( !strncmp( buf, RICHCORE_HEADER, sizeof(RICHCORE_HEADER)-1 ) ) {
			char *start_of_filename = buf + sizeof(RICHCORE_HEADER)-1;
			char *end_of_filename = strstr(start_of_filename, "---]\n");
			char *c;
			char fn[128];

			if ( !end_of_filename ) {
				fprintf(stderr, "skipping invalid rich core header\n");
				continue;
			}
			*end_of_filename = '\0';

			c = basename(start_of_filename);

			snprintf(fn, sizeof(fn), "%s/%s", output_dir, c);
#ifdef DEBUG			
			puts(fn);
#endif
			output_file = fopen(fn, "w");
			if ( !strcmp(c, "coredump") ||
			     !strcmp(c, "wimax-device-trace") ) {
#ifdef DEBUG			
				fprintf(stderr, "dump binary core\n");
#endif
				break;
			}
		} else {
			fputs(buf, output_file);
		}
	}
	do {
		i = fread(buf, 1, sizeof(buf), input_file);
		fwrite(buf, 1, i, output_file);
	} while ( i > 0 );

	fclose(output_file);
	pclose(input_file);
	exit(0);
}
