/**
 * @file ui_itemlist.c Item list/view handling
 *  
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <langinfo.h>
#include <string.h>
#include <osso-log.h>

#include <osso-rss-feed-reader/settings.h>

#include "htmlview.h"
#include "ui_itemlist.h"

/* tvh: IMPORTANT NOTE:
 * Many of ui_itemlist calls here are relavent only iin the 
 * 3 paned mode in liferea. Therefore in newsreader, most of these are not needed
 */
extern gboolean itemlist_mode;

extern AppData *app_data;

/* displayed_fp should almost always be the same as selected_fp in ui_feedlist */
nodePtr displayed_node = NULL;

void ui_itemlist_load_feed(feedPtr fp);

/* tvh: Return the time_t in string format 
 * Used in the 3 pane mode, most likely.
 * A TODO: Either remove this, or move it to somewhere else, definitely not here
 */
gchar *ui_itemlist_format_date(time_t t)
{
    gchar *tmp = FALSE;
    gchar *timestr = FALSE;
    gchar *time_string = NULL;
    gchar *date_string = NULL;
    time_t local_time = t;

    time_string = get_localized_time(local_time);
    date_string = get_localized_date(local_time, TRUE);

    tmp = g_strconcat(date_string, " ", time_string, NULL);

    g_free(time_string);
    g_free(date_string);

    timestr = g_locale_to_utf8(tmp, -1, NULL, NULL, NULL);
    g_free(tmp);

    return timestr;
}

void ui_itemlist_clear(void)
{
    AppUIData *app_ui_data = NULL;

    displayed_node = NULL;
    gtkhtml_stop();

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;

    free_find_string();         /*TODO: it needs to be cached doesn't it? */
    app_ui_data->feed_displayed = FALSE;
}

/* Function which is called when the contents of currently
 * selected object (feed or item) is updated or if the 
 * selection has changed and initial display is requested. 
 * 
 * What the function prints to the HTML view depends on the
 * focussed widget. If feedlist is selected feed info is printed.
 * If itemlist is selected the selected items content is shown. 
 * If anything other is focussed nothing is printed to avoid
 * disturbing the user. */

void ui_itemlist_display(nodePtr node)
{
    gtkhtml_stop();

    displayed_node = node;
    ui_htmlview_write(app_data->app_ui_data->html_scrollpane, NULL, NULL, node);

    ui_feedlist_update();
}

void ui_itemlist_load_feed(feedPtr fp)
{
    g_assert(fp != NULL);
    if (!conf_keep_feeds_in_memory) {
        ui_feedlist_do_for_all(NULL,
                               ACTION_FILTER_FEED |
                               ACTION_FILTER_DIRECTORY, feed_unload);
    }
    feed_load(fp);
}

void ui_itemlist_search_append(itemPtr ip)
{

    if (app_data->app_ui_data->search_mode != SFM_SEARCH)
        return;
    ui_htmlview_render_item(ip);
    if (!item_get_read_status(ip)) {
        item_set_read(ip);
    }
    ui_mainwindow_construct_search_view();
    ui_update();
}

void ui_itemlist_search_init()
{
    app_data->app_ui_data->rss_search_found = 0;
    if (!conf_keep_feeds_in_memory) {
        ui_feedlist_do_for_all(NULL,
                               ACTION_FILTER_FEED |
                               ACTION_FILTER_DIRECTORY, feed_unload);
    }
}

/* Load and display a node. Can be a normal feed.
 * Can be a searchFeed
 */

void ui_itemlist_load(nodePtr node)
{
    g_assert(NULL != node);

    /* free old and instruct getItemStore() to create new itemstore */
    /* ui_itemlist_clear() resets this */
    displayed_node = node;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_data->app_ui_data->feed_displayed = TRUE;

    /* Add the new items */
    if ((node == NULL) || (FST_FOLDER == node->type)) {
        ui_feedlist_do_for_all(node,
                               ACTION_FILTER_FEED |
                               ACTION_FILTER_DIRECTORY,
                               ui_itemlist_load_feed);
    } else if ((FST_FEED == node->type) || (FST_VFOLDER == node->type)) {
        ui_itemlist_load_feed((feedPtr) node);
    }

    ui_itemlist_display(node);
}
