/**
 * @file ui_dnd.c everything concerning DnD
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* For strsep */
#define _BSD_SOURCE 1

#include <string.h>
#include "callbacks.h"
#include "conf.h"
#include "ui_dnd.h"

#include <osso-log.h>
#include <gdk/gdk.h>

extern GtkTreeStore *feedstore;
extern GtkTreeModel *feedmodel;

extern AppData *app_data;
extern nodePtr displayed_node;

static gboolean(*old_drop_possible) (GtkTreeDragDest * drag_dest,
                                     GtkTreePath * dest_path,
                                     GtkSelectionData * selection_data);

/* ---------------------------------------------------------------------------- */
/* GtkTreeDragSource/GtkTreeDragDest implementation             */
/* ---------------------------------------------------------------------------- */

/** decides wether a feed cannot be dragged or not 
  *
  * @param drag_source drag source
  * @param path path
  * @return TRUE if a feed can be dragged, FALSE otherwise */
static gboolean
ui_dnd_feed_draggable(GtkTreeDragSource * drag_source, GtkTreePath * path)
{
    GtkTreeModel *cur_model = NULL;
    GtkTreeIter iter;
    nodePtr ptr = NULL;

    if (GTK_IS_TREE_MODEL(drag_source)) {
        cur_model = GTK_TREE_MODEL(drag_source);
    } else {
        g_warning("ui_dnd_feed_draggable: could not get model");
        return FALSE;
    }

    g_assert(NULL != cur_model);
    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(cur_model), &iter, path)) {
        gtk_tree_model_get(GTK_TREE_MODEL(cur_model), &iter, FS_PTR, &ptr,
                           -1);

        /* everything besides "empty", root entries and factory feeds/folders may be dragged */
        if (ptr == NULL) {
            return FALSE;
        }
        if (ptr->type == FST_FOLDER &&  /*((folderPtr) ptr)->root && */
            folder_get_nonremovable((folderPtr) ptr)) {
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           dgettext
                                           (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                            "sfil_ib_unable_to_move_selected_folder"));
            return FALSE;
        } else if (ptr->type == FST_FEED && ((feedPtr) ptr)->nonremovable) {
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           dgettext
                                           (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                            "sfil_ib_unable_to_move_selected_items"));
            return FALSE;
        }
        return TRUE;
    } else {
        g_warning("fatal error! could not resolve tree path!");
        return FALSE;
    }
}

/** decides wether a feed cannot be dropped onto a user selection tree position or not 
  *
  * @param drag_dest drag destination
  * @param dest_path destination path
  * @param selection_data selection data
  * @return TRUE if a feed can be dropped onto a user selected tree position,
  *         FALSE otherwise
  */
static gboolean
ui_dnd_feed_drop_possible(GtkTreeDragDest * drag_dest,
                          GtkTreePath * dest_path,
                          GtkSelectionData * selection_data)
{
    GtkTreeModel *tree_model = NULL;
    GtkTreeIter iter;

    /* The only situation when we don't want to drop is when a
     * feed was selected (note you can select drop targets between
     * feeds/folders, a folder or a feed). Dropping onto a feed
     * is not possible with GTK 2.0-2.2 because it disallows to
     * drops as a children but its possible since GTK 2.4 */

    tree_model = GTK_TREE_MODEL(drag_dest);

    if (((old_drop_possible) (drag_dest, dest_path, selection_data)) == FALSE)
        return FALSE;

    if (gtk_tree_model_get_iter(tree_model, &iter, dest_path)) {
        /* if we get an iterator its either a folder or the feed 
         * iterator after the insertion point */
        nodePtr ptr;

        gtk_tree_model_get(tree_model, &iter, FS_PTR, &ptr, -1);

        if (ptr == NULL) 
            return TRUE;
        
        folderPtr fptr = ui_feedlist_get_target_folder_ptr(ptr);

        if (fptr && (fptr != ui_feedlist_get_root_folder()) &&
            folder_get_nonremovable(fptr))
            return FALSE;

        if (fptr && fptr->root == TRUE ) 
            return FALSE;


    } else {
        /* we come here if a drop on a feed happens */
        return FALSE;
    }
    return TRUE;
}


/** method to receive URLs which were dropped anywhere in the main window 
  *
  * @param mainwindow main window
  * @param context Gdk drag context
  * @param x not used
  * @param y not used
  * @param data selection data
  * @param info not used
  * @param time time 
  */
static void ui_dnd_URL_received(GtkWidget * mainwindow,
                                GdkDragContext * context, gint x, gint y,
                                GtkSelectionData * data, guint info,
                                guint time)
{
    if (data->data == NULL)
        return;

    if ((data->length >= 0) && (data->format == 8)) {
        gtk_drag_finish(context, TRUE, FALSE, time);
    } else {
        gtk_drag_finish(context, FALSE, FALSE, time);
    }
}

/* ---------------------------------------------------------------------------- */
/* DnD callbacks                                */
/* ---------------------------------------------------------------------------- */

void on_feedlist_drag_end(GtkWidget * widget,
                          GdkDragContext * drag_context, gpointer user_data)
{
    ui_feedlist_update();
    ui_folder_check_if_empty();
    conf_feedlist_save_config();

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_data->app_ui_data->draggednode = NULL;
}

void on_feedlist_drag_begin(GtkWidget * widget,
                            GdkDragContext * drag_context, gpointer user_data)
{
    GdkPixbuf *pixbuf = NULL;
    gint w, h;
    feedPtr fp;
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_data->app_ui_data->draggednode = ui_feedlist_get_selected();

    /* create pixbuf here so that it will use this pixbuf icon
     * in stead of the whole text line
     */

    if (app_data->app_ui_data->draggednode->type == FST_FOLDER)
        pixbuf = icons[ICON_FOLDER_CLOSED];
    else if (app_data->app_ui_data->draggednode->type == FST_FEED) {
        fp = (feedPtr) app_data->app_ui_data->draggednode;
        pixbuf = fp->icon;
        if (pixbuf == NULL) {
            pixbuf = icons[ICON_RSS_NEWS_FEED];
        }
    }

    if (pixbuf != NULL) {
        w = gdk_pixbuf_get_width(pixbuf);
        h = gdk_pixbuf_get_height(pixbuf);

        gtk_drag_set_icon_pixbuf(drag_context, pixbuf, w, h);
        // No need for this as pixbuf is just a reference to the feed icon
        // g_object_unref(pixbuf);
        pixbuf = NULL;
    }
    g_signal_stop_emission_by_name(widget, "drag-begin");
}

void rss_feed_tree_drag_init(GtkTreeDragDestIface * iface, gpointer user_data)
{
    iface->row_drop_possible = ui_dnd_feed_drop_possible;
}

void ui_dnd_init(void)
{
    GtkTreeDragSourceIface *drag_source_iface = NULL;
    GtkTreeDragDestIface *drag_dest_iface = NULL;

    g_assert(NULL != feedstore);

    if (NULL !=
        (drag_source_iface =
         GTK_TREE_DRAG_SOURCE_GET_IFACE(GTK_TREE_MODEL(feedstore)))) {
        drag_source_iface->row_draggable = ui_dnd_feed_draggable;
    }
    if (NULL !=
        (drag_dest_iface =
         GTK_TREE_DRAG_DEST_GET_IFACE(GTK_TREE_MODEL(feedstore)))) {
        old_drop_possible = drag_dest_iface->row_drop_possible;
        drag_dest_iface->row_drop_possible = ui_dnd_feed_drop_possible;
    }
}

/* ---------------------------------------------------------------------------- */
/* receiving URLs                               */
/* ---------------------------------------------------------------------------- */

/* sets up URL receiving */

void ui_dnd_setup_URL_receiver(GtkWidget * mainwindow)
{
    GtkTargetEntry target_table[] = {
        {"STRING", 0, 0},
        {"text/plain", 0, 0},
        {"text/uri-list", 0, 1},
        {"_NETSCAPE_URL", 0, 1},
        {"application/x-rootwin-drop", 0, 2}
    };

    // doesn't work with GTK_DEST_DEFAULT_DROP... 
    gtk_drag_dest_set(mainwindow, GTK_DEST_DEFAULT_ALL,
                      target_table,
                      sizeof(target_table) / sizeof(target_table[0]),
                      GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);

    gtk_signal_connect(GTK_OBJECT(mainwindow), "drag_data_received",
                       G_CALLBACK(ui_dnd_URL_received), NULL);
}


gboolean ui_dnd_drop_possible(GtkTreeModel * model, GtkTreePath * path)
{

    return TRUE;
}


void ui_dnd_receive_drag_data(GtkWidget * widget,
                              GdkDragContext * drag_context,
                              gint x,
                              gint y,
                              GtkSelectionData * data,
                              guint info, guint time, gpointer user_data)
{
    GdkDragAction suggested_action = GDK_ACTION_DEFAULT;
    GtkTreeView *tree_view = NULL;
    GtkTreeModel *model = NULL;

    tree_view = GTK_TREE_VIEW(widget);

    g_assert(tree_view != NULL);

    model = gtk_tree_view_get_model(tree_view);

    g_assert(model != NULL);

    g_signal_stop_emission_by_name(widget, "drag_data_received");

    suggested_action = GPOINTER_TO_INT(g_object_get_data
                                       (G_OBJECT(drag_context),
                                        "gtk-tree-view-status-pending"));

    if (suggested_action != 0) {
        gdk_drag_status(drag_context, suggested_action, time);
    } else {
        if (data->length >= 0) {
            gtk_drag_finish(drag_context, TRUE, TRUE, time);
        } else {
            gtk_drag_finish(drag_context, FALSE, FALSE, time);
        }
    }
}

/** Handles a drop of a feed
  *
  * @param path the destination to drop the feed
  */
static gboolean ui_dnd_handle_feed_drop(GtkTreePath * path, nodePtr selected)
{
    GtkTreeModel *model = NULL;
    GtkTreeIter iter;
    gboolean valid = FALSE;
    gboolean success = FALSE;
   gboolean retval = FALSE;
    nodePtr ptr = NULL;
    gchar *str = NULL;

    model = GTK_TREE_MODEL(feedmodel);

    if (path != NULL) {
        str = gtk_tree_path_to_string(path);
        valid = gtk_tree_model_get_iter_from_string(model, &iter, str);
    } else {
        ptr = (nodePtr) ui_feedlist_get_root_folder();
        valid = TRUE;
    }

    if (valid) {
        if (ptr == NULL) {
            gtk_tree_model_get(model, &iter, FS_PTR, &ptr, -1);

            if (ptr == NULL) {
                GtkTreeIter parent;

                retval = gtk_tree_model_iter_parent(model, &parent, &iter);
		  g_assert(retval);
                gtk_tree_model_get(model, &parent, FS_PTR, &ptr, -1);
            }
        }

        if (ptr->type == FST_FEED) {
            int *indices = NULL;
            int depth = 0;
            int position = 0;

            indices = gtk_tree_path_get_indices(path);
            depth = gtk_tree_path_get_depth(path);
            position = indices[depth - 1];

            ptr = (nodePtr) ui_feedlist_get_parent(ptr);

            if (displayed_node == (nodePtr) selected)
                ui_mainwindow_show_statistics();
            ui_feedlist_remove(selected);
            ui_feedlist_add((folderPtr) ptr, selected, position);
            success = TRUE;
        } else if (ptr != (nodePtr) ui_feedlist_get_parent(selected)) {
            if (displayed_node == (nodePtr) selected)
                ui_mainwindow_show_statistics();
            ui_feedlist_remove(selected);
            ui_feedlist_add((folderPtr) ptr, selected, -1);
            success = TRUE;
        }
    }

    if (str != NULL)
        g_free(str);

    return success;
}

void ui_dnd_reparent(GtkTreeIter * from, nodePtr selected, nodePtr fptr,
                     gboolean copy, folderPtr new_folder)
{
    GtkTreeModel *model;
    GtkTreeIter iter;
    gboolean valid = FALSE;
    nodePtr node = NULL;

    g_assert(NULL != feedstore);

    model = GTK_TREE_MODEL(feedstore);

    if (from != NULL) {
        gtk_tree_model_get(model, from, FS_PTR, &node, -1);
        valid = gtk_tree_model_iter_children(model, &iter, from);
    } else {
        valid = gtk_tree_model_get_iter_first(model, &iter);
    }

    while (valid) {
        gtk_tree_model_get(model, &iter, FS_PTR, &node, -1);

        if (new_folder != NULL) {
            gchar *title = folder_get_title(new_folder);
            folderPtr new_fptr =
                restore_folder(NULL, title, NULL, FST_FOLDER);
            ui_feedlist_add((folderPtr) fptr, (nodePtr) new_fptr, -1);
            fptr = (nodePtr) new_fptr;
            new_folder = NULL;
        }

        if (NULL != node) {
            if (FST_FEED == feed_get_type((feedPtr) node) && copy == TRUE) {
                valid = gtk_tree_model_iter_next(model, &iter);
                ui_feedlist_remove(node);
                ui_feedlist_add((folderPtr) fptr, node, -1);
                continue;
            } else if (FST_FOLDER == feed_get_type((feedPtr) node)) {
                if (copy == TRUE)
                    ui_dnd_reparent(&iter, selected, fptr, TRUE,
                                    (folderPtr) node);
                else if (node == selected) {
                    ui_dnd_reparent(&iter, selected, fptr, TRUE, NULL);
                    break;
                } else
                    ui_dnd_reparent(&iter, selected, fptr, FALSE, NULL);
            }
        }
        valid = gtk_tree_model_iter_next(model, &iter);
    }
}

/** Handles a drop of a folder
  *
  * @param path the destination to drop the folder
  */
static gboolean ui_dnd_handle_folder_drop(GtkTreePath * path,
                                          nodePtr selected)
{
    GtkTreeModel *model = NULL;
    GtkTreeIter iter;
    gboolean valid = FALSE;
    gboolean success = FALSE;
    gboolean retval = FALSE;
    nodePtr ptr = NULL;
    gchar *str = NULL;

    if (path != NULL) {
        str = gtk_tree_path_to_string(path);

        model = GTK_TREE_MODEL(feedmodel);
        valid = gtk_tree_model_get_iter_from_string(model, &iter, str);
    } else {
        ptr = (nodePtr) ui_feedlist_get_root_folder();
        valid = TRUE;
    }

    if (valid) {
        if (ptr == NULL) {
            gtk_tree_model_get(model, &iter, FS_PTR, &ptr, -1);

            if (ptr == NULL) {
                GtkTreeIter parent;

                retval = gtk_tree_model_iter_parent(model, &parent, &iter);
                g_assert(retval);				
                gtk_tree_model_get(model, &parent, FS_PTR, &ptr, -1);
            }
        }

        if (ptr->type != FST_FOLDER)
            ptr = (nodePtr) ui_feedlist_get_parent(ptr);

        if (ptr != selected &&
            ptr != (nodePtr) ui_feedlist_get_parent(selected)) {
            if (!ui_feedlist_is_ancestor
                (ptr, &((ui_data *) (selected->ui_data))->row)) {
                gchar *title = folder_get_title((folderPtr) selected);

                folderPtr fptr =
                    restore_folder(NULL, title, NULL, FST_FOLDER);
                ui_feedlist_add((folderPtr) ptr, (nodePtr) fptr, -1);
                ui_dnd_reparent(NULL, selected, (nodePtr) fptr, FALSE, NULL);
                ui_feedlist_delete_((nodePtr) selected);
            }
        }
    }

    if (str != NULL)
        g_free(str);

    return success;
}

gboolean ui_dnd_drag_drop(GtkWidget * widget,
                          GdkDragContext * drag_context,
                          gint x, gint y, guint time, gpointer user_data)
{
    GtkTreeView *tree_view = NULL;
    GtkTreeModel *model = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    GtkTreeViewDropPosition pos = GTK_TREE_VIEW_DROP_BEFORE;
    gboolean success = FALSE;
    nodePtr selected = NULL;

    tree_view = GTK_TREE_VIEW(widget);

    g_assert(tree_view != NULL);

    model = gtk_tree_view_get_model(tree_view);

    g_assert(model != NULL);

    g_signal_stop_emission_by_name(widget, "drag_drop");

    gtk_tree_view_get_dest_row_at_pos(tree_view, x, y, &path, &pos);

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    selected = app_data->app_ui_data->draggednode;

    if (selected == NULL) {
        if (path != NULL)
            gtk_tree_path_free(path);
        return FALSE;
    }

    if (path && gtk_tree_model_get_iter(model, &iter, path)) {
        nodePtr ptr;

        gtk_tree_model_get(model, &iter, FS_PTR, &ptr, -1);
        folderPtr fptr = ui_feedlist_get_target_folder_ptr(ptr);
        ULOG_DEBUG("Folder: %p, %d", fptr, folder_get_nonremovable(fptr));
        if (fptr && (fptr != ui_feedlist_get_root_folder()) &&
            folder_get_nonremovable(fptr)) {
            gtk_tree_path_free(path);
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _
                                           ("rss_ib_unable_to_move_items_here"));
            return FALSE;
        }
    }

    if (selected->type == FST_FEED)
        success = ui_dnd_handle_feed_drop(path, selected);
    else if (selected->type == FST_FOLDER)
        success = ui_dnd_handle_folder_drop(path, selected);

    if (!success) {
        if (path != NULL)
            gtk_tree_path_free(path);
        return FALSE;
    }

    if (path != NULL) {
        g_object_set_data(G_OBJECT(drag_context),
                          "gtk-tree-view-status-pending", GINT_TO_POINTER(0));

        gtk_drag_get_data(widget, drag_context,
                          gdk_atom_intern("GTK_TREE_MODEL_ROW", FALSE), time);

        gtk_tree_path_free(path);
        path = NULL;
        return TRUE;
    } else {
        return FALSE;
    }
}
