/**
 * @file main.c Liferea main program
 *
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 * 
 * Some code like the command line handling was inspired by 
 *
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "main.h"

#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/dbus.h>
#include <dbus/dbus-glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <signal.h>

static int sigpipe[2];

AppData *app_data;

Iphb_cb_data iphb_data;	// Rama added

extern GtkWidget *newsbox;
extern time_t rss_updated;
extern gboolean quit_reset_state;
gboolean lifereaStarted = FALSE;
extern gboolean just_empty_items;

static void initialize_app_ui_data()
{
    app_data->app_ui_data->start_on_background = FALSE;
    app_data->app_ui_data->start_alarmdel_on_background = FALSE;
    app_data->app_ui_data->find_string = "";
    app_data->app_ui_data->rss_search_found = 0;
    app_data->app_ui_data->automatic_update_source = 0;
    app_data->app_ui_data->download_dequeue_source = 0;
    app_data->app_ui_data->iap_action = OSSO_IAP_NO_ACTION;
    app_data->app_ui_data->errors_in_refreshing = FALSE;
    app_data->app_ui_data->add_feed_dialog = NULL;
    app_data->app_ui_data->ui_data = NULL;
    app_data->app_ui_data->source = NULL;
    app_data->app_ui_data->filter = NULL;
    app_data->app_ui_data->draggednode = NULL;
    app_data->app_ui_data->feed_directory_cancelled = FALSE;
    app_data->app_ui_data->network_iap = NULL;
    app_data->app_ui_data->application_topped = TRUE;
    app_data->app_ui_data->update_feed_list_needed = FALSE;  
    app_data->app_ui_data->update_progress_needed = FALSE;		
    app_data->app_ui_data->call_ui_mainwindow_finish_needed = FALSE;
    app_data->app_ui_data->refresh_dialog = NULL;
    app_data->app_ui_data->dialog_closed = FALSE;
    app_data->gconf_client = NULL;
}

static void sig_handler(int _sig)
{
     uint8_t sig = _sig;
     /* write to the sigpipe if SIGTERM is caught*/
     write(sigpipe[1], &sig, sizeof(sig));
}

static void quit_cleanup(void)
{
    download_done();
    gtkhtml_deinit();
    if (app_data && app_data->app_ui_data) {
        g_object_unref(app_data->app_ui_data->button_image_search);
        g_object_unref(app_data->app_ui_data->button_image_stop);
        g_object_unref(app_data->app_ui_data->search_field);
        g_object_unref(app_data->app_ui_data->button_image_refresh);
        g_object_unref(app_data->app_ui_data->progress_bar);

        if (app_data->app_ui_data->actions) {
            hildon_uri_free_actions(app_data->app_ui_data->actions);
            app_data->app_ui_data->actions = NULL;
        }
    }
    g_assert(app_data != NULL);
    
    if (app_data->app_ui_data != NULL)
        g_object_unref(G_OBJECT(app_data->app_ui_data->img_cache));

    if (quit_reset_state)
        reset_state_file(app_data->osso);

    deinit_osso(app_data);

    if (app_data) {

        if (app_data->app_ui_data) {
            g_free(app_data->app_ui_data);
            app_data->app_ui_data = NULL;
        }

        g_free(app_data);
        app_data = NULL;
    }
}

static gboolean sig_event(GIOChannel *channel, GIOCondition condition, gpointer data)
{
     uint8_t sig;
     if (read(sigpipe[0], &sig, sizeof(sig)) == sizeof(sig)) {

	 save_state(app_data);
	 on_quit(FALSE);

         //quit_cleanup();
         gtk_main_quit();
         return TRUE;
	}
     return FALSE;
}

static void setup_signals()
{
     GIOChannel *channel;

     /* create the pipe */
     pipe(sigpipe);

     channel = g_io_channel_unix_new(sigpipe[0]);

     /* Adds the GIOChannel into the default main loop context with the default priority.
      * G_IO_IN There is data to read.
      * G_IO_HUP Hung up
      * G_IO_ERR Error condition.
      */
     g_io_add_watch(channel, G_IO_IN|G_IO_HUP|G_IO_ERR, sig_event, NULL);

     /* installs a new sig_handler for the SIGTERM with number 15 */
     signal(SIGTERM, sig_handler);
}

int main(int argc, char *argv[])
{
    gulong debug_flags = 0;
    gboolean startIconified = FALSE;
    gboolean headline_clicked = FALSE;
    const char *arg = NULL;
    gchar *filename = NULL;
    gint i = 0;
    if (!g_thread_supported()) {
        g_thread_init(NULL);
    }
    dbus_g_thread_init();

    /* Allocate application data */
    app_data = g_new0(AppData, 1);
    app_data->app_ui_data = g_new0(AppUIData, 1);

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    initialize_app_ui_data();

    /*Register the SIGTERM handler for state save and clean exit*/
    setup_signals();
    

    /* Use this safety patch from bug 25738 */

    init_osso(app_data);

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "background", g_get_home_dir());
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
        startIconified = TRUE;
        app_data->app_ui_data->start_on_background = TRUE;
        g_unlink(filename);
    }
    g_free(filename);

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "headline_clicked",
                        g_get_home_dir());
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
        headline_clicked = TRUE;
        g_unlink(filename);
    }
    g_free(filename);

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "alarmdelcheck",
                        g_get_home_dir());
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
       app_data->app_ui_data->start_alarmdel_on_background= TRUE;
        g_unlink(filename);
    }
    g_free(filename);

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif
    gnome_vfs_init();
    /* parse arguments  */
    debug_flags = 0;
    for (i = 1; i < argc; ++i) {
        arg = argv[i];
        if (!strcmp(arg, "--debug-cache"))
            debug_flags |= DEBUG_CACHE;
        else if (!strcmp(arg, "--debug-conf"))
            debug_flags |= DEBUG_CONF;
        else if (!strcmp(arg, "--debug-update"))
            debug_flags |= DEBUG_UPDATE;
        else if (!strcmp(arg, "--debug-parsing"))
            debug_flags |= DEBUG_PARSING;
        else if (!strcmp(arg, "--debug-gui"))
            debug_flags |= DEBUG_GUI;
        else if (!strcmp(arg, "--debug-trace"))
            debug_flags |= DEBUG_TRACE;
        else if (!strcmp(arg, "--debug-all"))
            debug_flags |=
                DEBUG_TRACE | DEBUG_CACHE | DEBUG_CONF | DEBUG_UPDATE |
                DEBUG_PARSING | DEBUG_GUI;
        else if (!strcmp(arg, "--debug-verbose"))
            debug_flags |= DEBUG_VERBOSE;
        else if (!strcmp(arg, "--empty"))
            just_empty_items = TRUE;
        else if (!strcmp(arg, "--iconify")) {
            startIconified = TRUE;
            app_data->app_ui_data->start_on_background = TRUE;
        }
    }
    set_debug_level(debug_flags);

    gtk_set_locale();

    hildon_gtk_init (&argc, &argv);

    /* order is important! */

    conf_init();                /* initialize gconf */

    download_init();            /* Initialize the download subsystem */

    gtkhtml_init();

    metadata_init();

    feed_init();                /* register feed types */

    /* Create application UI */
    ui_create_main_window(app_data);

    ui_mainwindow_load_search_entries();

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ui_init(startIconified, app_data->app_ui_data); /* setup mainwindow and initialize gconf configured GUI behaviour */

	gchar *testfilename = NULL;
	testfilename = g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "testfile",
                        g_get_home_dir());

   if (g_file_test(testfilename, G_FILE_TEST_EXISTS))
    	{
		if (read_testfile(testfilename) == 0)
	    	{
				app_data->app_ui_data->iap_action = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
				request_iap_connect();
	    	}
				g_free(testfilename);	
    	}


    g_assert(app_data != NULL);

    if (!startIconified) {
        ui_feedlist_delete_read_items(NULL);
        /* Refresh the applet since the feed contents might have been changed */
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL);
        restore_state(app_data);
        app_data->app_ui_data->auto_refresh_is_ongoing = FALSE;
    }

    ui_folder_handle_empty_root();

    if( (!app_data->app_ui_data->start_alarmdel_on_background) && (!app_data->app_ui_data->start_on_background) ) {
	    iphb_data.iphb_callback = (GSourceFunc)(poll_iphb_fd);
	    iphb_data.source_id = 0;
	    setup_alarm_event_now(TRUE, &iphb_data);	// Rama - commented
	    iphb_data.source_id = setup_iphb_callback(&iphb_data);
    }

    lifereaStarted = TRUE;
    
    gtk_main();

    quit_cleanup();
    return 0;
}

void save_application_data()
{
    conf_feedlist_save();       /* should save feedlist and folder states */

    /* should save all feeds still in memory */
    ui_feedlist_do_for_all(NULL,
                           ACTION_FILTER_FEED | ACTION_FILTER_DIRECTORY,
                           feed_unload);

    /* save itemlist properties */
    setBooleanConfValue(LAST_ITEMLIST_MODE, !itemlist_mode);

    /* save pane proportions */
    if (GTK_IS_WIDGET(newsbox)) {
        gint x = gtk_paned_get_position(GTK_PANED(GTK_PANED(newsbox)));
        setNumericConfValue(LAST_VPANE_POS, x);
    }

}

int read_testfile(gchar *testfilename)
{
	FILE * hFile;
	int nNumber=0;

	hFile = fopen( testfilename, "r+");
	if (hFile == NULL)
	{
		return -1;
	}
	else
	{
		// Read the file
		fread(&nNumber, sizeof(int), 1, hFile);
		//Reset the file pointer at initial point
		fseek ( hFile , 0 , SEEK_SET );
		nNumber++;
		fwrite(&nNumber, sizeof(int), 1, hFile);
		fclose(hFile);
		return (nNumber-1);
	}
}
