/**
 * @file item.c common item handling
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *	      
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <time.h>
#include <string.h>             /* For memset() */
#include <stdlib.h>

#include "vfolder.h"
#include "item.h"
#include "support.h"
#include "common.h"
#include "htmlview.h"

#include "metadata.h"
#include "debug.h"
#include "cache.h"
#include "html.h"

extern AppData *app_data;

/* function to create a new feed structure */
itemPtr item_new(void)
{
    itemPtr ip = NULL;

    ip = g_new0(struct item, 1);
    ip->newStatus = TRUE;

    return ip;
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

void item_copy(itemPtr from, itemPtr to)
{
    g_assert(from != NULL);

    item_set_title(to, from->title);
    item_set_source(to, from->source);
    item_set_real_source_url(to, from->real_source_url);
    item_set_real_source_title(to, from->real_source_title);
    item_set_description(to, from->description);
    item_set_id(to, from->id);

    g_assert(to != NULL);

    to->readStatus = from->readStatus;
    to->newStatus = FALSE;
    to->marked = from->marked;
    to->time = from->time;
    to->nr = from->nr;

    /* the following line enables state propagation in item.c */
    to->sourceFeed = from->fp;

    /* this copies metadata */
    metadata_list_free(to->metadata);
    to->metadata = metadata_list_copy(from->metadata, NULL);

    /* copy on_server flag */
    to->on_server = from->on_server;

}

inline void item_set_title(itemPtr ip, const gchar * title)
{
    g_assert(ip != NULL);
    if (NULL != ip->title)
        g_free(ip->title);
    ip->title = g_strdup(title);
}

inline void item_set_description(itemPtr ip, const gchar * description)
{
    g_assert(ip != NULL);
    if (NULL != ip->description)
        g_free(ip->description);
    ip->description = g_strdup(description);
}

inline void item_set_source(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (ip->source != NULL)
        g_free(ip->source);
    ip->source = g_strdup(source);
}

inline void item_set_real_source_url(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (NULL != ip->real_source_url)
        g_free(ip->real_source_url);
    ip->real_source_url = g_strdup(source);
}

inline void item_set_real_source_title(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (NULL != ip->real_source_title)
        g_free(ip->real_source_title);
    ip->real_source_title = g_strdup(source);
}

inline void item_set_time(itemPtr ip, time_t t)
{
    g_assert(ip != NULL);
    ip->time = t;
}

inline void item_set_read_status(itemPtr ip, gboolean readStatus)
{
    g_assert(ip != NULL);
    ip->readStatus = readStatus;
}

inline void item_set_id(itemPtr ip, const gchar * id)
{
    /* NOTE: should remove white spaces? */
    g_assert(ip != NULL);
    if (NULL != ip->id)
        g_free(ip->id);
    ip->id = g_strdup(id);
}

inline void item_set_hidden(itemPtr ip, gboolean hidden)
{
    g_assert(ip != NULL);
    ip->hidden = hidden;
}

inline const gchar *item_get_id(itemPtr ip)
{
    return (ip != NULL ? ip->id : NULL);
}
inline const gchar *item_get_title(itemPtr ip)
{
    return (ip != NULL ? ip->title : NULL);
}
inline const gchar *item_get_description(itemPtr ip)
{
    return (ip != NULL ? ip->description : NULL);
}
inline const gchar *item_get_source(itemPtr ip)
{
    return (ip != NULL ? ip->source : NULL);
}
inline const gchar *item_get_real_source_url(itemPtr ip)
{
    return (ip != NULL ? ip->real_source_url : NULL);
}
inline const gchar *item_get_real_source_title(itemPtr ip)
{
    return (ip != NULL ? ip->real_source_title : NULL);
}
inline time_t item_get_time(itemPtr ip)
{
    return (ip != NULL ? ip->time : 0);
}
inline gboolean item_get_read_status(itemPtr ip)
{
    return (ip != NULL ? ip->readStatus : FALSE);
}

inline gboolean item_get_mark(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->marked;
}
inline gboolean item_get_hidden(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->hidden;
}
inline gboolean item_get_new_status(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->newStatus;
}

void item_set_mark(itemPtr ip, gboolean flag)
{
    itemPtr sourceItem = NULL;

    ip->marked = flag;
    if (ip->fp != NULL)
        ip->fp->needsCacheSave = TRUE;

    /* if this item belongs to a vfolder update the source feed */
    if (ip->sourceFeed != NULL) {
        feed_load(ip->sourceFeed);
        sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
        item_set_mark(sourceItem, flag);
        feed_unload(ip->sourceFeed);
    } else {
        vfolder_update_item(ip);    /* there might be vfolders using this
                                     * item */
        vfolder_check_item(ip); /* and check if now a rule matches */
    }
}

void item_set_new_status(itemPtr ip, gboolean newStatus)
{
    if (newStatus != ip->newStatus) {
        ip->newStatus = newStatus;

        if (TRUE == newStatus)
            feed_increase_new_counter((feedPtr) (ip->fp));
        else
            feed_decrease_new_counter((feedPtr) (ip->fp));
    }
}

/* why are two separate functions for setting the same variable?
 * this and next one. */
void item_set_unread(itemPtr ip)
{
    itemPtr sourceItem = NULL;

    if (TRUE == ip->readStatus) {
        feed_increase_unread_counter((feedPtr) (ip->fp));

        ip->readStatus = FALSE;
        if (ip->fp != NULL)
            ip->fp->needsCacheSave = TRUE;

        /* if this item belongs to a vfolder update the source feed */
        if (ip->sourceFeed != NULL) {
            feed_load(ip->sourceFeed);
            sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
            item_set_unread(sourceItem);
            feed_unload(ip->sourceFeed);
        } else {
            vfolder_update_item(ip);    /* there might be vfolders using this 
                                         * item */
            vfolder_check_item(ip); /* and check if now a rule matches */
        }
    }
}

void item_set_read(itemPtr ip)
{
    itemPtr sourceItem = NULL;
    if (FALSE == ip->readStatus) {
        feed_decrease_unread_counter((feedPtr) (ip->fp));
        ip->readStatus = TRUE;
        if (ip->fp != NULL) {
            ip->fp->needsCacheSave = TRUE;
        }
        /* if this item belongs to a vfolder update the source feed */
        if (ip->sourceFeed != NULL) {
            feed_load(ip->sourceFeed);
            sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
            item_set_read(sourceItem);
            feed_unload(ip->sourceFeed);
        } else {
            vfolder_update_item(ip);    /* there might be vfolders using this 
                                         * item */
            vfolder_check_item(ip); /* and check if now a rule matches */
        }
    }
}

void item_free(itemPtr ip)
{

    if (ip->fp) {
	if (FALSE == ip->readStatus)
	    feed_decrease_unread_counter(ip->fp);
	if (TRUE == ip->newStatus)
	    feed_decrease_new_counter(ip->fp);
    }

    g_free(ip->title);
    g_free(ip->source);
    g_free(ip->real_source_url);
    g_free(ip->real_source_title);
    g_free(ip->description);
    g_free(ip->id);
    g_assert(NULL == ip->tmpdata);  /* should be free after rendering */
    if (ip->metadata) {
	metadata_list_free(ip->metadata);
    }

    // only assert when ip is not in vfolder
    if (ip->sourceFeed == NULL)
        g_assert(ip->ui_data == NULL);

    g_free(ip);
}

static void remove_link_from_cache(gchar * link, gpointer user_data)
{
    file_cache_remove(FILE_CACHE(user_data), link);
}

void remove_links_from_cache(itemPtr ip)
{
    find_image_links((gchar *) item_get_description(ip),
                     ip->source,
                     remove_link_from_cache,
                     (gpointer) app_data->app_ui_data->img_cache);
}
