/**
 * @file callbacks.c misc UI stuff
 *
 * Most of the GUI code is distributed over the ui_*.c
 * files but what didn't fit somewhere else stayed here.
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Christophe Barbe <christophe.barbe@ufies.org>	
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <time.h>
#include <unistd.h>


#include <gdk/gdkkeysyms.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <hildon/hildon-banner.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-time-editor.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-helper.h>
#include <hildon/hildon-sort-dialog.h>
#include <hildon/hildon-file-details-dialog.h>
#include <hildon/hildon-file-chooser-dialog.h>

#include <osso-log.h>
#include <string.h>
#include <glib.h>

#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/dbus.h>
#include <gtkhtml/gtkhtml.h>

#include "debug.h"
#include "interface.h"
#include "support.h"
#include "folder.h"
#include "feed.h"
#include "item.h"
#include "conf.h"
#include "export.h"
#include "htmlview.h"
#include "common.h"
#include "callbacks.h"
#include "ui_mainwindow.h"
#include "ui_folder.h"
#include "ui_feedlist.h"
#include "ui_itemlist.h"
#include "ui_notification.h"
#include "ui_mainwindow.h"
#include "ui_prefs.h"
#include "state_save.h"
#include "update.h"
#include "dbus.h"
#include "main.h"
#include "gtkhtml/gtkhtml_plugin.h"
#include "gtkhtml/htmlimport.h"

#include "debug_new.h"

#include <libgnomevfs/gnome-vfs.h>

/* For Zoom buttons*/
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
/**/

#define L_(s) dgettext ("hildon-libs", s)

extern AppData *app_data;

extern time_t rss_updated;
extern long rss_unread;
extern gboolean adding_feeds_from_opml;

extern GHashTable *feedHandler;

extern GtkWidget *newsbox;
extern GtkWidget *feedlist_container;
extern feedPtr searchFeed;
extern GtkWidget *gtkhtml_htmlview;

extern Iphb_cb_data iphb_data; // Rama added

gchar *prev_find_text = NULL;
/** prev_text_has_a_match
    will be set to TRUE if prev_find_text appears in the page
    will be reset to FALSE if pre_find_text is also reset
*/
gboolean prev_text_has_a_match;
gboolean nomemdialog_visible = FALSE;
/* This enables, displaying not enough memory dialog if refresh
 * clicked within WAITTIME_BEFORE_SHOWN_NEW_OUTOFMEM_DIALOG more than once */
gboolean    refresh_started = FALSE;

progress_bar_mode progressbar_mode_prev = 0;


/* shoot myself in the food and feel the joy! */

gint refreshcounter = 0;
gint addfeedcounter = 0;



/* all used icons */
GdkPixbuf *icons[MAX_ICONS];

/* icon names */
static gchar *iconNames[] = 
{   "general_rss",  /* ICON_READ */	
    "rss_reader_feed_invalid",    /* invalid feed */
    "general_toolbar_folder",   /* ICON_VFOLDER */
    "general_toolbar_folder",   /* ICON_OPEN */
    "qgn_list_gene_fldr_clp",   /*  closed folder icon - plus sign */
    "qgn_list_gene_fldr_exp",   /*  expanded folder icon - minus sign */
    NULL
};

gint ui_show_replace_file_dialog(gchar * file);

void on_folder_cursor_changed(GtkTreeView * tv, GtkWidget * dlg);
void on_change_folder_action_response(GtkWidget * change_folder_dialog,
                                      gint responseId, gpointer user_data);
void on_manage_folders_action_response(GtkWidget * manage_folders_dialog,
                                       gint responseId, gpointer user_data);
void on_new_folder_action_response(GtkWidget * new_folder_dialog,
                                   gint responseId, gpointer user_data);
void on_refresh_action_response(GtkWidget * refresh_dialog,
                                gint responseId, gpointer user_data);
gboolean dialog_key_press(GtkWidget * widget, GdkEventKey * event,
                          gpointer data);
void populate_folders(GtkWidget * dialog, gboolean hide_factory);
void
dialog_folder_icon_render_func(GtkTreeViewColumn * col,
                               GtkCellRenderer * renderer,
                               GtkTreeModel * model,
                               GtkTreeIter * iter, gpointer user_data);
void free_icons(void);

/*------------------------------------------------------------------------------*/
/* generic GUI functions                            */
/*------------------------------------------------------------------------------*/

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/


static void change_manage_delete_button_status(GtkWidget *
                                               manage_folders_dialog)
{
    GtkWidget *TView = lookup_widget(manage_folders_dialog, "folder_view");
    GtkTreeSelection *curent_selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(TView));
    if (gtk_tree_selection_get_selected(curent_selection, NULL, NULL)) {
        gtk_widget_set_sensitive(lookup_widget
                                 (manage_folders_dialog, "mf_renamebtn"),
                                 TRUE);
        nodePtr node =
            ui_feedlist_get_selected_from_tree(GTK_TREE_VIEW(TView));
        if (app_data->app_ui_data->search_mode >= SFM_REFRESH) {
            if (ui_feedlist_check_if_folder_empty((folderPtr) node)) {
                gtk_widget_set_sensitive(lookup_widget
                                         (manage_folders_dialog,
                                          "mf_deletebtn"), TRUE);
            } else {
                gtk_widget_set_sensitive(lookup_widget
                                         (manage_folders_dialog,
                                          "mf_deletebtn"), FALSE);
            }
        } else {
		folderPtr abc = (folderPtr)node;
		if (abc == ui_feedlist_get_root_folder())
			{
				gtk_widget_set_sensitive(lookup_widget
                                 (manage_folders_dialog, "mf_deletebtn"),
                                 FALSE);
        		gtk_widget_set_sensitive(lookup_widget
                                 (manage_folders_dialog, "mf_renamebtn"),
                                 FALSE);
				 
			}
		else
			{
            gtk_widget_set_sensitive(lookup_widget
                                     (manage_folders_dialog, "mf_deletebtn"),
                                     TRUE);
			}
        }

    } else {
        gtk_widget_set_sensitive(lookup_widget
                                 (manage_folders_dialog, "mf_deletebtn"),
                                 FALSE);
        gtk_widget_set_sensitive(lookup_widget
                                 (manage_folders_dialog, "mf_renamebtn"),
                                 FALSE);
    }
}


/** Filters the tree model so that only folders are shown
  *
  * @param model the tree model to filter
  * @param iter a tree iter
  * @param data not used
  * @return TRUE if the node pointed at by the iterator is a folder, FALSE otherwise
  */
static gboolean filter_show_folders_only(GtkTreeModel * model,
                                         GtkTreeIter * iter, gpointer data)
{

    nodePtr ptr = NULL;

    if (model == NULL) {
        ULOG_WARN("filter_show_folders_only: WARNING: model is NULL.");
        return TRUE;
    }

    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);
    if (ptr == NULL) {
        return FALSE;
    }

    if (ptr->type == FST_FOLDER) {
        if ((gboolean) data && folder_get_nonremovable((folderPtr) ptr))
            return (folderPtr) ptr == ui_feedlist_get_root_folder();
        else
            return TRUE;
    } else {
        return FALSE;
    }

}

/* free the icons[] array
 */
void free_icons(void)
{
    gint i;
    ULOG_DEBUG("freeing the icons array");
    for (i = 0; i < MAX_ICONS; i++)
        if (icons[i] != NULL) {
            g_object_unref(icons[i]);
            icons[i] = NULL;
        }
}

extern GtkWidget *newsbox;

static gboolean set_pane_pos(gpointer data)
{
    gtk_paned_set_position(GTK_PANED(GTK_PANED(newsbox)), (gint) data);
    return FALSE;
}

static void setAtomFlagForX(AppUIData *app_ui_data)
{
	GdkDisplay *display;
	Atom atom;
	unsigned long val = 1;

	display = gdk_drawable_get_display(GTK_WIDGET(app_ui_data->main_view)->window);

	atom = gdk_x11_get_xatom_by_name_for_display(display, "_HILDON_ZOOM_KEY_ATOM");

	XChangeProperty (GDK_DISPLAY_XDISPLAY(display), GDK_WINDOW_XID(GTK_WIDGET(app_ui_data->main_view)->window), atom, XA_INTEGER, 32, PropModeReplace,
		                            (unsigned char *) &val, 1);
    
}  

void ui_init(gboolean startIconified, AppUIData * app_ui_data)
{
    gchar *path;
    gint size;
    GTimer *tm = g_timer_new();
    //    int i;

    g_assert(app_ui_data != NULL);
    g_assert(app_ui_data->main_view != NULL);
    app_ui_data->zoom_level = getNumericConfValue(ZOOM_LEVEL);
    if (app_ui_data->zoom_level == 0.0) {
        app_ui_data->zoom_level = 1.0;
        ULOG_DEBUG("Default Zoom level");
    } else {
        app_ui_data->zoom_level = app_data->app_ui_data->zoom_level / 100.0;
    }

    ULOG_DEBUG("UI_INIT %s: %f", "zoom", g_timer_elapsed(tm, NULL));
    conf_keep_feeds_in_memory = getBooleanConfValue(KEEP_FEEDS_IN_MEMORY);

    /* initialize cache */
    path = getStringConfValue(RSS_SETTINGS_IMAGE_CACHE_DIR);
    if (!*path) {
        path =
            g_strdup_printf("%s" G_DIR_SEPARATOR_S
                            ".osso_rss_feed_reader/cache/images",
                            g_get_home_dir());
        ULOG_DEBUG("Default image cache path %s", path);
    }
    size = getNumericConfValue(RSS_SETTINGS_IMAGE_CACHE_SIZE);
    app_ui_data->img_cache = file_cache_new(path, size * 1024 * 1024);
    g_free(path);
    ULOG_DEBUG("UI_INIT %s: %f", "cache", g_timer_elapsed(tm, NULL));

    mainwindow = GTK_WIDGET(app_ui_data->main_view);
    create_mainwindow(app_ui_data);

    ULOG_DEBUG("UI_INIT %s: %f", "mainwindow", g_timer_elapsed(tm, NULL));

    /* order important !!! */
    ui_feedlist_init(feedlist);
    //getItemStore();

    ULOG_DEBUG("UI_INIT %s: %f", "feedlist", g_timer_elapsed(tm, NULL));

    ui_folder_set_expansion(ui_feedlist_get_root_folder(), TRUE);
    //TODO: REMOVE THIS
    ui_mainwindow_set_mode(itemlist_mode);

    ULOG_DEBUG("UI_INIT %s: %f", "set_mode", g_timer_elapsed(tm, NULL));
    /*
     * for (i = 0; i < MAX_ICONS -2 ; i++)
     * icons[i] = create_pixbuf(iconNames[i]);
     */

    /* TODO: will Those icons will be unref'ed by the gtktree view ? */
    /* create rss feed icons */
    icons[0] = create_pixbuf(iconNames[0]);
    /* create rss invalid feed icons */
    icons[1] = create_pixbuf(iconNames[1]);
    /* create rss folder closed icon . This is sometimes used in dialog lable */
    icons[2] = create_pixbuf(iconNames[2]);

    /* create composite folder closed with emblem */
    icons[3] = create_composite_pixbuf(iconNames[2], iconNames[4]);
    /* create composite folder opened with emblem */
    icons[4] = create_composite_pixbuf(iconNames[3], iconNames[5]);

    ULOG_DEBUG("UI_INIT %s: %f", "pixbuf", g_timer_elapsed(tm, NULL));

    app_ui_data->search_mode = SFM_NORMAL;
    app_ui_data->feed_displayed = FALSE;

    /* set up icons that are build from stock */
    ui_dnd_setup_URL_receiver(mainwindow);  /* setup URL dropping support */
    ULOG_DEBUG("UI_INIT %s: %f", "dnd", g_timer_elapsed(tm, NULL));
    conf_load_subscriptions();
    ULOG_DEBUG("UI_INIT %s: %f", "subscriptions", g_timer_elapsed(tm, NULL));
    /* Set the interval for automatic updates */
    /*    if(getBooleanConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES))
     * set_update_timer(FALSE); */
    ui_mainwindow_finish();     /* Display the initial status info */
    ULOG_DEBUG("UI_INIT %s: %f", "finish", g_timer_elapsed(tm, NULL));
//    set_feed_properties_insens_messages(app_ui_data);

#ifdef REMOVED_REFERENCE_FULLSCREEN
    if (getBooleanConfValue(DISABLE_TOOLBAR)) {
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active = FALSE;
        gtk_widget_hide_all(app_ui_data->main_toolbar);
    } else
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active = TRUE;

    if (getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN))
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active = FALSE;
    else
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active = TRUE;
#endif

    ULOG_DEBUG("UI_INIT %s: %f", "hide", g_timer_elapsed(tm, NULL));

    /* initialize sort */
    int key = getNumericConfValue(RSS_SETTINGS_SORT_KEY);
    if (key < 0 || key > 1) {
        key = 0;
        setNumericConfValue(RSS_SETTINGS_SORT_KEY, key);
    }
    key = getNumericConfValueDef(RSS_SETTINGS_SORT_ORDER, 1);
    if (key < 0 || key > 1) {
        key = 0;
        setNumericConfValue(RSS_SETTINGS_SORT_ORDER, key);
    }

    /*    if (checkConfValueExist(RSS_SETTINGS_REUSE_BROWSER_WINDOW) == FALSE)
     * setBooleanConfValue(RSS_SETTINGS_REUSE_BROWSER_WINDOW, TRUE);
     * 
     * if (checkConfValueExist(RSS_SETTINGS_NEWEST_POSTS_FIRST)== FALSE)
     * setBooleanConfValue(RSS_SETTINGS_NEWEST_POSTS_FIRST, TRUE);
     * 
     * if (checkConfValueExist(RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL)== FALSE)
     * setNumericConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL, DEFAULT_INTERVAL); */
    ULOG_DEBUG("UI_INIT %s: %f", "default values", g_timer_elapsed(tm, NULL));

    /*    if (!getBooleanConfValue(LAST_SHOWFOLDERS))
     * toggle_folder_list(NULL, NULL); */
    /*    if (getBooleanConfValue(LAST_WINDOW_MAXIMIZED))
     * toggle_fullscreen_and_update_menu(NULL, app_data->app_ui_data); */

    if ( (startIconified) || (app_ui_data->start_alarmdel_on_background))
        gtk_widget_hide(GTK_WIDGET(app_ui_data->main_view));
    else {
        gtk_widget_show(GTK_WIDGET(app_ui_data->main_view));
	if(app_data->app_ui_data->main_view) {
		setAtomFlagForX(app_data->app_ui_data);
	}
    }
   
    ULOG_DEBUG("UI_INIT %s: %f", "finished", g_timer_elapsed(tm, NULL));

    if (getBooleanConfValue(ENABLE_FIND_ON_PAGE)) {
         gtk_widget_show_all(GTK_WIDGET(app_ui_data->find_toolbar));
    } else {
         gtk_widget_hide_all(GTK_WIDGET(app_ui_data->find_toolbar));
    }
    g_timer_destroy(tm);
    g_idle_add(set_pane_pos,
               (gpointer) gtk_paned_get_position(GTK_PANED(newsbox)));
    app_data->app_ui_data->cursor_changed_on_idle =
        g_idle_add(ui_feedlist_call_cursor_changed_on_idle,
                   (gpointer) app_data);
}

/*------------------------------------------------------------------------------*/
/* simple callbacks which don't belong to item or feed list             */
/*------------------------------------------------------------------------------*/
static gint quit_count = 0;
static gint quit_source = 0;
gboolean quit_reset_state = FALSE;

gboolean on_quit_cb(gpointer data)
{
	if( (!app_data->app_ui_data->start_alarmdel_on_background) && (!app_data->app_ui_data->start_on_background) ) {	
		close_iphb_handle(iphb_data);
		remove_iphb_callback(iphb_data.source_id);
		iphb_data.source_id = 0;
	}
    download_cancel_all(FALSE);
    free_icons();
    free_find_string();
    ui_mainwindow_save_search_entries();
    ui_itemlist_clear();        /* to free all item list related data */
    save_application_data();
    gtk_main_quit();
    return FALSE;
}

gboolean on_quit(gboolean reset_state)
{
    quit_reset_state = reset_state;

    if( (app_data->app_ui_data->start_alarmdel_on_background) || (!app_data->app_ui_data->start_on_background) ) {
	    delete_allalarm_events();
	    rss_dbus_signal_applet(OSSO_RSS_FEED_READER_ALARMDELETE_RSSSHUTDOWN_SIGNAL);
    }

    if (quit_count == 0) {
        quit_count++;

        if (app_data->app_ui_data->search_mode == SFM_REFRESH)
        {
            /* If refresh was in progress, notify the applet, that refresh was
             * prevented, to avoid "infinite refresh" in applet */
            rss_dbus_signal_applet(
                OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL);

            setNumericConfValue(REFRESH_STATE, 0);
        }

        if (app_data->app_ui_data->cursor_changed_on_idle) {
            g_source_remove(app_data->app_ui_data->cursor_changed_on_idle);
        }
        g_signal_handlers_block_by_func((gpointer) app_data->app_ui_data->
                                        main_view, G_CALLBACK(key_release),
                                        app_data->app_ui_data);

        g_signal_handlers_block_by_func((gpointer) app_data->app_ui_data->
                                        main_view, G_CALLBACK(key_press),
                                        app_data->app_ui_data);

        quit_source = g_idle_add(on_quit_cb, NULL);
        return (TRUE);
    }
    return (FALSE);
}

void abort_quit()
{
    if (quit_source) {
        g_source_remove(quit_source);
        quit_count = 0;
        quit_source = 0;
    }
}


void change_zoom_cb(GtkWidget * widget, gpointer data)
{
    AppUIData *app_ui_data;
    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);
    app_ui_data = app_data->app_ui_data;
    app_ui_data->zoom_level = (int) data / 100.0;
    setNumericConfValue(ZOOM_LEVEL, (int) data);
    gtk_html_set_magnification(GTK_HTML
                               (gtk_bin_get_child
                                (GTK_BIN(app_ui_data->html_scrollpane))),
                               app_ui_data->zoom_level);

    gchar tmp[60], tmp2[60];
    snprintf(tmp, sizeof(tmp), "rss_ib_zoom_%d", (int) data);
    snprintf(tmp2, sizeof(tmp2), _(tmp));
    hildon_banner_show_information(GTK_WIDGET
                                   (app_data->app_ui_data->main_view), NULL,
                                   tmp2);
}


/**tvh: Icon render for the Folder tree view in the dialog mode
 * (manage folder dialog and change folder dialog )
 * Must use the icon, just for the folder icons being changed
 * as the dialogs use a filtered, but different model, and treeviews

*/
void
dialog_folder_icon_render_func(GtkTreeViewColumn * col,
                               GtkCellRenderer * renderer,
                               GtkTreeModel * model,
                               GtkTreeIter * iter, gpointer user_data)
{
    folderPtr ptr = NULL;
    GtkTreeIter child;
    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);
    gboolean has_sub_folder =
        gtk_tree_model_iter_children(model, &child, iter);

    if (ptr->type == FST_FOLDER) {
        if (has_sub_folder == FALSE)
            g_object_set(renderer, "pixbuf", icons[ICON_FOLDER_CLOSED], NULL);
        else
            g_object_set(renderer, "pixbuf",
                         ui_folder_select_icon(ptr->dialog_mode_expanded),
                         NULL);
    }
}

void populate_folders(GtkWidget * dialog, gboolean hide_factory)
{
    GtkWidget *folder_view = lookup_widget(dialog, "folder_view");

    GtkTreeModel *model = NULL;
    GtkCellRenderer *textRenderer = NULL;
    GtkCellRenderer *iconRenderer = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkTreeModel *filter = NULL;
    folderPtr rootfolder;

    ULOG_DEBUG("populate_folders: starting...");

    filter =
        (GtkTreeModel *) gtk_tree_model_filter_new(GTK_TREE_MODEL(feedstore),
                                                   NULL);

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
                                           filter_show_folders_only,
                                           (gpointer) hide_factory, NULL);
    model = GTK_TREE_MODEL(filter);

    /*  model = GTK_TREE_MODEL(feedstore); */
    ULOG_DEBUG("populate_folders: ...1...");

    gtk_tree_view_set_model(GTK_TREE_VIEW(folder_view),
                            GTK_TREE_MODEL(model));

    /* Show all the folders. */
    gtk_tree_view_collapse_all(GTK_TREE_VIEW(folder_view));
    /* Collapse the root folder */
    rootfolder = (folderPtr) ui_feedlist_get_root_folder();
    rootfolder->dialog_mode_expanded = FALSE;

    /* folder_view should have its own handlers */
    g_signal_connect(G_OBJECT(folder_view), "row-expanded",
                     G_CALLBACK(ui_folder_mfdialog_collapse_expand_cb), NULL);
    g_signal_connect(G_OBJECT(folder_view), "row-collapsed",
                     G_CALLBACK(ui_folder_mfdialog_collapse_expand_cb), NULL);

    /* we only render the state and title */
    iconRenderer = gtk_cell_renderer_pixbuf_new();
    textRenderer = gtk_cell_renderer_text_new();

    g_object_set(G_OBJECT(textRenderer), "ellipsize", PANGO_ELLIPSIZE_END,
                  NULL);
	
    ULOG_DEBUG("populate_folders: ...2...");

    gtk_cell_renderer_set_fixed_size(textRenderer, CELL_RENDERER_WIDTH , -1);

    column = gtk_tree_view_column_new();

    gtk_tree_view_column_pack_start(column, iconRenderer, FALSE);
    gtk_tree_view_column_pack_start(column, textRenderer, TRUE);

    /* Must use the func to render correctly as we don't want to modify the
     * original model just because of expanding/collapsing the folders in dialogs
     */
    gtk_tree_view_column_set_cell_data_func(column, iconRenderer,
                                            dialog_folder_icon_render_func,
                                            NULL, NULL);
    /*
     * gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
     * FS_ICON);
     */

    gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
                                       FS_ICON);
    gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
                                       FS_LABEL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(folder_view), column);
}

/** Update the progress bar if needed
 * This is called in two cases , for now
    1. Refreshing
    2. Receiving images
    3. TODO: searching
*/
void update_progress_bar()
{
	if ((app_data->app_ui_data->application_topped == TRUE) && (app_data->app_ui_data->display_off == FALSE))
	{

	    GtkProgressBar *pb =
	        GTK_PROGRESS_BAR(app_data->app_ui_data->progress_bar);
	    switch (app_data->app_ui_data->progressbar_mode) {
	        case PROGRESSBAR_REFRESH_MODE:
	            gtk_progress_bar_set_fraction(pb,
	                                          (float) get_counter(C_RSS_CURRENT) /
	                                          get_counter(C_RSS_MAX));
			app_data->app_ui_data->search_mode = SFM_REFRESH; //Added MCH
		/*	hildon_helper_set_insensitive_message(GTK_WIDGET
				                                     (app_data->app_ui_data->refresh_button),
								    _("rss_ib_refresh_already_started")); */
	            break;
	        case PROGRESSBAR_IMAGE_MODE:
	        {
	            if (!get_counter(C_IMG_MAX))
	                gtk_progress_bar_set_text(pb, "");
	            else {
	                gchar *fmt = _("rss_pb_toolbar_receiving");
	                gchar *text = NULL;
	                text = g_strdup_printf(fmt,
	                                       get_counter(C_IMG_CURRENT),
	                                       get_counter(C_IMG_MAX));
	                gtk_progress_bar_set_text(pb, text);
			/*NB#117508*/
			app_data->app_ui_data->search_mode = SFM_REFRESH; 
		/*	hildon_helper_set_insensitive_message(GTK_WIDGET
				                                     (app_data->app_ui_data->refresh_button),
								    _("rss_ib_refresh_already_started")); */

	                g_free(text);
	            }
	            break;
	        }
	        case PROGRESSBAR_CONNECT_MODE:
	            gtk_progress_bar_pulse(pb);
	            gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_connecting"));
	            break;
	        case PROGRESSBAR_SEARCH_MODE:
	            //TODO
	            break;
	        default:
	            break;
	    }

	    gtk_widget_queue_draw(GTK_WIDGET(pb));
	}	
	else
	{
		ULOG_DEBUG("application is not on top %s %s\n",__FILE__,__FUNCTION__); 
		  return;
	}
		
}

/*
 * PROGRESSBAR_NORMAL_MODE,
 * PROGRESSBAR_SEARCH_MODE,
 * PROGRESSBAR_CONNECT_MODE,
 * PROGRESSBAR_REFRESH_MODE,
 * PROGRESSBAR_IMAGE_MODE,
 * PROGRESSBAR_RECEIVING_MODE
 */

static void pb_change_sensitive_buttons(AppUIData * app_ui_data,
                                        gboolean enable, gboolean nosearch)
{
    DMSG("Entering %s\n", __FUNCTION__);
    ULOG_DEBUG("\n\n\n\n\n@@@@@@@@@@@ Set sensitive buttons: %d, %d\n\n\n\n",
               enable, nosearch);
    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();
    gboolean feed_selected=TRUE;

    gchar *feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);	
    if ((NULL != ptr && ptr->type != FST_FEED)|| (ptr == NULL) )
        {
            feed_selected=FALSE;
        }
	    
    if (enable &&  strlen(feed_dir_source)) 
    {
	gtk_widget_show(app_ui_data->menuitem_feed_directory);
    } else {
	gtk_widget_hide(app_ui_data->menuitem_feed_directory);
    }
    
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->refresh_button), enable);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button), enable);
    ui_mainwindow_set_feed_deatils_sensitive(app_data, nosearch&&feed_selected);
    
    if (enable || nosearch) {
        gtk_widget_show(app_ui_data->menuitem_manage_folders);
    } else {
        gtk_widget_hide(app_ui_data->menuitem_manage_folders);
    }

    gtk_widget_set_sensitive(GTK_WIDGET
                             (app_ui_data->new_feed_button), enable);
    ui_mainwindow_set_unsubscribe_sensitive(app_data, enable);

    /*gtk_widget_set_sensitive(GTK_WIDGET
     * (app_ui_data->manage_folders_button), enable || nosearch); */

    if (enable)
        ui_feedlist_enable_def_buttons();
//    set_feed_properties_insens_messages(app_ui_data);
    unsubscribe_insens_messages(app_ui_data);
//    new_insens_messages(app_ui_data);
    refresh_insens_messages(app_ui_data);
}

static gboolean progress_update(gpointer data)
{

	if((app_data->app_ui_data->application_topped == TRUE) && (app_data->app_ui_data->display_off == FALSE))
	{

   		 if (app_data->app_ui_data->progressbar_mode != PROGRESSBAR_CONNECT_MODE)
   		{
        		app_data->app_ui_data->pidle = 0;
   		}
		else 
		{
		        update_progress_bar();
		}
	}
	else
	{
        	app_data->app_ui_data->pidle = 0;	
	}

    return app_data->app_ui_data->pidle;
}

void switch_progressbar(progress_bar_mode mode)
{
    g_assert(app_data);
    g_assert(app_data->app_ui_data);
    gint smode = 0;	

    AppUIData *app_ui_data = app_data->app_ui_data;
    if((app_data->app_ui_data->application_topped == TRUE) && (app_data->app_ui_data->display_off == FALSE))			
   {	
       GtkProgressBar *pb = GTK_PROGRESS_BAR(app_ui_data->progress_bar);
     
   
       g_assert(app_ui_data->search_button);
       g_assert(app_ui_data->button_image_stop);
   
       ULOG_DEBUG("<!><!> switch_progressbar: %d <!><!>", mode);
   
       //Filtering some invalid switch modes
       if (mode == app_ui_data->progressbar_mode)
           return;
   
       ULOG_DEBUG("<!><!> switch_progressbar: 1, %s", app_ui_data->network_iap);
       if ((mode == PROGRESSBAR_CONNECT_MODE) && app_ui_data->network_iap &&
           *app_ui_data->network_iap)
           return;
   
       ULOG_DEBUG("<!><!> switch_progressbar: 2, %d-%d, %d-%d",
                  app_ui_data->progressbar_mode,
                  PROGRESSBAR_RECEIVING_MODE, mode, PROGRESSBAR_REFRESH_MODE);
       if ((app_ui_data->progressbar_mode == PROGRESSBAR_RECEIVING_MODE) &&
           (mode == PROGRESSBAR_REFRESH_MODE))
           return;
   
       ULOG_DEBUG("<!><!> switch_progressbar: continue <!><!>");
       if (app_ui_data->progressbar_mode == PROGRESSBAR_NORMAL_MODE) {
           g_assert(app_ui_data->button_image_stop);
   
           //disable toolbar / menu items
           pb_change_sensitive_buttons(app_ui_data, FALSE,
                                       mode != PROGRESSBAR_SEARCH_MODE);
   
           //change toolbutton image
           gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                                           (app_ui_data->search_button),
                                           app_ui_data->button_image_stop);
   
           //remove from the container the searchfield and add progressbar
           gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
                                app_ui_data->search_field);
           gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
                             GTK_WIDGET(pb));
   
           //show the toolbar if it's hidden 
           gtk_widget_show_all(GTK_WIDGET(app_ui_data->main_toolbar));
           /*
            * if (pulse)
            * app_ui_data->search_mode = SFM_CONNECTING; */
       }
   
       if (app_ui_data->dlg_btn_unsubscribe) {
           gtk_widget_set_sensitive(app_ui_data->dlg_btn_unsubscribe,
                                    mode == PROGRESSBAR_NORMAL_MODE);
           unsubscribe_insens_messages(app_ui_data);
       }
   
       if (app_ui_data->pidle) {
           g_source_remove(app_ui_data->pidle);
           app_ui_data->pidle = 0;
       }
   
       app_ui_data->progressbar_mode = mode;
   
       switch (mode) {
           case PROGRESSBAR_NORMAL_MODE:
               DMSG("Pbar normal\n");
               app_ui_data->search_mode = SFM_NORMAL;
   
               //enable toolbar / menu items
               pb_change_sensitive_buttons(app_ui_data, TRUE, TRUE);
   
               //change toolbutton image
               gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                                               (app_ui_data->search_button),
                                               app_ui_data->button_image_search);
   
               //remove from the container the progressbar and add the searchfield
               gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
                                    GTK_WIDGET(pb));
               gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
                                 app_ui_data->search_field);
   
               //hide the toolbar if needed
               if ((getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN) &&
                    (app_ui_data->fullscreen)
                   ) ||
                   (getBooleanConfValue(DISABLE_TOOLBAR) &&
                    (FALSE == app_ui_data->fullscreen)
                   )
                  )               // then hide the toolbar
               {
                   app_ui_data->toolbar_temp_shown = FALSE;
                   gtk_widget_hide_all(GTK_WIDGET(app_ui_data->main_toolbar));
               }
   
   
   
               smode = SFM_NORMAL;
   
               break;
   
           case PROGRESSBAR_SEARCH_MODE:
               DMSG("Pbar search\n");
               gtk_progress_bar_set_text(pb, _("rss_ia_searching"));
               gtk_progress_bar_set_fraction(pb, 0);
   
               smode = SFM_SEARCH;
               app_ui_data->search_mode = SFM_SEARCH;
               pb_change_sensitive_buttons(app_ui_data, FALSE, FALSE);
               break;
   
           case PROGRESSBAR_CONNECT_MODE:
               //TODO Szs change id
               gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_connecting"));
               gtk_progress_bar_set_pulse_step(pb, 0.2);
   
               if (!app_ui_data->pidle)
                   app_ui_data->pidle =
                       g_timeout_add(200, progress_update, NULL);
   
               smode = SFM_CONNECTING;
   
               break;
   
           case PROGRESSBAR_REFRESH_MODE:
               app_ui_data->search_mode = SFM_REFRESH;
               gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_refreshing"));
               gtk_progress_bar_set_fraction(pb, 0);
   
               smode = SFM_REFRESH;
               pb_change_sensitive_buttons(app_ui_data, FALSE,
                                           mode != PROGRESSBAR_SEARCH_MODE);
               break;
   
           case PROGRESSBAR_IMAGE_MODE:
           {
               gchar *fmt = _("rss_pb_toolbar_receiving");
               gchar *text = g_strdup_printf(fmt, 0, get_counter(C_IMG_MAX));
               gtk_progress_bar_set_text(pb, text);
               g_free(text);
               gtk_progress_bar_set_fraction(pb, 1);
               smode = SFM_REFRESH;
   
               break;
           }
           case PROGRESSBAR_RECEIVING_MODE:
               gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_receiving_file"));
               gtk_progress_bar_set_fraction(pb, 1);
   
               smode = SFM_REFRESH;
   
               break;
       }
   }	
  else
    {
	
	switch (mode) {
	    case PROGRESSBAR_NORMAL_MODE: 
	       smode = SFM_NORMAL;
	        break;

	    case PROGRESSBAR_SEARCH_MODE:
	        smode = SFM_SEARCH;          
	        break;

	    case PROGRESSBAR_CONNECT_MODE:
	        smode = SFM_CONNECTING;
	        break;

	    case PROGRESSBAR_REFRESH_MODE:
	  case PROGRESSBAR_IMAGE_MODE:	
	 case PROGRESSBAR_RECEIVING_MODE:	
	        smode = SFM_REFRESH;
	        break;
	   
	}
	
    }	
	progressbar_mode_prev = mode;
	ULOG_DEBUG("mode = %d\n",mode);
	app_data->app_ui_data->update_progress_needed = TRUE;	

    ULOG_DEBUG("<!><!> switch_progressbar: new mode %d, %d <!><!>", smode,
               app_ui_data->search_mode);
    if (smode == SFM_CONNECTING)
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_CONNECTING_STARTED);
    else if (smode == SFM_REFRESH)
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_STARTED);
    else rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_INTERRUPTED_SIGNAL);

    app_ui_data->search_mode = smode;
}


gboolean dialog_key_press(GtkWidget * widget, GdkEventKey * event,
                          gpointer data)
{
    (void) data;

    if (event->state & (GDK_CONTROL_MASK |
                        GDK_SHIFT_MASK |
                        GDK_MOD1_MASK |
                        GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
        return FALSE;
    }

    switch (event->keyval) {
        case GDK_Escape:
            gtk_dialog_response(GTK_DIALOG(widget), GTK_RESPONSE_OK);
            break;
        default:
            break;
    }

    return FALSE;
}


/* returns true if enough time has gone since last dialog */
static gboolean checkifenoughtimeelapsed(time_t *old)
{
    time_t ts = 0;
    /* static time_t old = 0; */
    ts = time(NULL);

    if ((*old) == 0) {
        (*old) = ts;
        return FALSE;
    }

    if ((*old) + WAITTIME_BEFORE_SHOWN_NEW_OUTOFMEM_DIALOG >= ts) {
        return FALSE;
    } else {
        (*old) = ts;
        return TRUE;
    }
}


static
void shownomemdialog(void) {
    GtkWidget *parent = GTK_WIDGET(app_data->app_ui_data->main_view);

    if (app_data->app_ui_data->add_feed_dialog)
        parent = app_data->app_ui_data->add_feed_dialog;

    nomemdialog_visible = TRUE;
    HildonNote *note = HILDON_NOTE(hildon_note_new_information (GTK_WINDOW(parent), dgettext("ke-recv", "cerm_device_memory_full")));
    hildon_note_set_button_text(note, dgettext("hildon-common-strings", "sfil_ni_not_enough_memory_ok"));
    g_signal_connect(G_OBJECT(note), "key_press_event", G_CALLBACK(dialog_key_press), NULL);
    DMSG("Running nofreespace dialog");
    gtk_dialog_run(GTK_DIALOG(note));
    DMSG("Running nofreespace dialog ended");
    gtk_widget_destroy(GTK_WIDGET(note));
    nomemdialog_visible = FALSE;
}

void ui_show_save_nodevicemem_dialog(void)
{
    static time_t oldaddfeed = 0;
    static time_t oldunknown = 0;

    static gint unknowncounter = 0;

    
    DMSG("Entering %s\n", __FUNCTION__);
    if (app_data->app_ui_data->start_on_background || nomemdialog_visible == TRUE) {
        return;
    } else {
        /* IF we have active add feed dialog ! */
        if (app_data->app_ui_data->add_feed_dialog != NULL )  {
            if (addfeedcounter == 0 ) {
                addfeedcounter ++;
                shownomemdialog();
            } else {
                addfeedcounter ++;
                /* then we check if x amount of seconds has passed away and if yes,
                 * we reset the counter so new dialog can be shown..
                 * Actually, this might be completely irrelevant but.. hey! we dont
                 * want to rewrite anything!!! */
                if (checkifenoughtimeelapsed(&oldaddfeed)) {
                    addfeedcounter  = 0;
                }
            }
        /* If user explicitly wanted feed refresh */
        } else if (refresh_started == TRUE ) {
            if (refreshcounter == 0 ) {
                refreshcounter++;
                shownomemdialog();
            } else {
                refreshcounter++;
            }
        } else {

            if (unknowncounter == 0 ) {
                unknowncounter ++;
                shownomemdialog();
            } else {
                unknowncounter ++;
                if (checkifenoughtimeelapsed(&oldunknown)) {
                    unknowncounter = 0;
                }
            }
        }
    }

}

void on_menu_activate(GtkWidget * widget, gpointer user_data)
{
  on_menuitem_edit(NULL, user_data);
  on_menuitem_tools(NULL, user_data);
}

void on_refresh_finished(gboolean disable_banner,
                         gboolean do_not_signal_applet)
{
    AppUIData *app_ui_data;
    int search_mode = 0;
    static gboolean in_refresh_finished = FALSE;
    app_data->app_ui_data->auto_refresh_is_ongoing = FALSE;
    if (in_refresh_finished)
        return;

    in_refresh_finished = TRUE;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;
    app_ui_data->nospace = FALSE;

    g_assert(app_ui_data != NULL);

    conf_feedlist_save();

    search_mode = app_ui_data->search_mode;
    app_ui_data->search_mode = SFM_NORMAL;
    if (app_ui_data->manage_folders_displayed) {
        change_manage_delete_button_status(app_ui_data->dialog);
    }
    if (SFM_INTERRUPTED == search_mode) {

        g_debug("refresh interrupted");

        if (app_ui_data->add_feed_dialog != NULL)
            newdialog_return_to_normal();

        if (!do_not_signal_applet && app_ui_data->start_on_background &&
            app_ui_data->network_iap) {
            do_not_signal_applet = TRUE;
            rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_INTERRUPTED_SIGNAL);
        } else if ((!do_not_signal_applet &&
                    !app_ui_data->start_on_background) ||
                   (!do_not_signal_applet &&
                    app_ui_data->start_on_background &&
                    !app_ui_data->network_iap)) {
            do_not_signal_applet = TRUE;
            rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL);
        }

    }

    if (SFM_REFRESH == search_mode || SFM_INTERRUPTED == search_mode) {
        /* This happens if there are errors in refreshing, for example
         * the feed is unavailable. This is set in ui_feed:ui_feed_process_update_result */

        if (app_ui_data->errors_in_refreshing) {
            /* If the user has interrupted the refresh we dont
             * display banners */
            if (search_mode != SFM_INTERRUPTED) {
                if (app_ui_data->add_feed_dialog == NULL) {
                    if (app_ui_data->start_on_background) {
                        do_not_signal_applet = TRUE;
                        rss_dbus_signal_applet
                            (OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL);
                    }
                    else if (adding_feeds_from_opml) {
                        hildon_banner_show_information(GTK_WIDGET
                                                       (app_data->app_ui_data->
                                                        main_view), NULL,
                                                       _("rss_ni_directory_adding_failed"));
                    }
                    else if (app_data->app_ui_data->pending_requests_nr > 1)
                        hildon_banner_show_information(GTK_WIDGET
                                                       (app_data->
                                                        app_ui_data->
                                                        main_view), NULL,
                                                       _
                                                       ("rss_ni_error_refreshing_all_feeds"));
                    else if (app_data->app_ui_data->pending_requests_nr == 1)
                        hildon_banner_show_information(GTK_WIDGET
                                                       (app_data->
                                                        app_ui_data->
                                                        main_view), NULL,
                                                       _
                                                       ("rss_ni_error_resfreshing_one_feed"));
                } else {
                    hildon_banner_show_information(GTK_WIDGET
                                                   (app_data->app_ui_data->
                                                    main_view), NULL,
                                                   _("rss_ni_adding_failed"));
                    newdialog_return_to_normal();
                }
            }
            app_ui_data->errors_in_refreshing = FALSE;
        }

        switch_progressbar(PROGRESSBAR_NORMAL_MODE);

	/* if no feeds keep the refresh button dimmed*/
	/* nodePtr ptr = (nodePtr)ui_feedlist_get_selected();
	 if(NULL == ptr)
	 {
	       gtk_widget_set_sensitive(GTK_WIDGET(app_data->app_ui_data->refresh_button),FALSE);
	 } */
	 
    }

    if (!do_not_signal_applet) {
        g_debug("refresh completed");
        prev_find_text = NULL;
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL);
    }

    /*tvh: Hide toolbar again if needed,
     * according to gconf value and which fullscreen mode we're in */
    if ((getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN) &&
         (TRUE == app_ui_data->fullscreen)
        ) ||
        (getBooleanConfValue(DISABLE_TOOLBAR) &&
         (FALSE == app_ui_data->fullscreen)
        )
        )                       // then hide the toolbar
        gtk_widget_hide_all(GTK_WIDGET(app_ui_data->main_toolbar));

    if (app_ui_data->start_on_background) {
        g_debug("Calling quit!");
        on_quit(TRUE);
    }

    app_data->app_ui_data->pending_requests_nr = 0;

    in_refresh_finished = FALSE;
    adding_feeds_from_opml = FALSE;
}

void on_refreshbtn_clicked_cb(nodePtr ptr)
{
    refresh_started = TRUE;
    feed_schedule_update((feedPtr) ptr, FEED_REQ_DOWNLOAD_FAVICON);
}


void on_refresh_action_response(GtkWidget * refresh_dialog,
                                gint responseId, gpointer user_data)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (responseId == GTK_RESPONSE_CANCEL ||
        responseId == GTK_RESPONSE_DELETE_EVENT) {
        gtk_widget_destroy(GTK_WIDGET(refresh_dialog));
        app_data->app_ui_data->refresh_dialog = NULL;
        ULOG_INFO("Refresh was canceled");
        return;
    }

    GtkWidget *selected_feed_widget =
        lookup_widget(refresh_dialog, "radio_button_selected_feed");
    gboolean selectedFeedOnly =
        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(selected_feed_widget));

    AppUIData *app_ui_data = app_data->app_ui_data;

    ULOG_INFO("Refresh invoked...");
    /*TODO: Show toolbar if currently hidden and hide in on_refresh_finished again */
    if (selectedFeedOnly) {
        app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED;
        app_ui_data->iap_disconnect = OSSO_IAP_REFRESH_FEED;
        app_ui_data->new_subscription = (feedPtr) ui_feedlist_get_selected();
        app_ui_data->new_subscription_flags = 0;
    } else {
        app_ui_data->iap_action = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
        app_ui_data->iap_disconnect = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
    }
    refreshcounter = 0;
    ULOG_DEBUG("Requesting connection...");
    request_iap_connect();
    /*        ULOG_INFO("Selected feed only...");
     * nodePtr ptr = ui_feedlist_get_selected();
     * 
     * if (ptr != NULL && ptr->type == FST_FEED) {
     * ULOG_DEBUG("Updating the selected feed");
     * feed_schedule_update((feedPtr) ptr, FEED_REQ_DOWNLOAD_FAVICON);
     * }
     * } 
     * else {
     * ULOG_DEBUG("Updating all feeds");
     * res = ui_feedlist_do_for_all_with_int(NULL,
     * ACTION_FILTER_FEED,
     * update_node);
     * } */
    gtk_widget_destroy(GTK_WIDGET(refresh_dialog));
    app_data->app_ui_data->refresh_dialog = NULL;
}

void on_refresh_action(AppUIData * app_ui_data)
{
    GtkWidget *refreshdialog = NULL;

    g_assert(NULL != app_ui_data);

    /* A refresh dialog has already been created so lets not create
     * another one. This might happen when several iap requests are
     * sent in succession */

    if (app_ui_data->refresh_dialog == NULL) {

        app_ui_data->refresh_dialog = refreshdialog = create_refresh_dialog();
        g_signal_connect(G_OBJECT(refreshdialog), "response",
                         G_CALLBACK(on_refresh_action_response), NULL);
        gtk_widget_show(refreshdialog);
    }
}

void on_new_folder_action_response(GtkWidget * new_folder_dialog,
                                   gint responseId, gpointer user_data)
{
    GtkWidget *dialog = NULL;
    GtkWidget *folder_view = NULL;

    ULOG_DEBUG(__FUNCTION__);

    dialog = (GtkWidget *) user_data;
    if (dialog != NULL) {
        folder_view = lookup_widget(dialog, "folder_view");
        g_assert(folder_view != NULL);
        ULOG_DEBUG("attempting to grab focus");
        gtk_widget_grab_focus(folder_view);
    }
}

gboolean update_node(nodePtr ptr)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    feed_schedule_update((feedPtr) ptr, FEED_REQ_RESET_TITLE |
                         FEED_REQ_RESET_UPDATE_INT |
                         FEED_REQ_AUTO_DISCOVER | FEED_REQ_DOWNLOAD_FAVICON);
    /*ULOG_DEBUG("update_node: search_mode = %d",
     * app_data->app_ui_data->search_mode); */
    return app_data->app_ui_data->search_mode != SFM_INTERRUPTED;
}

GtkWidget *display_new_folder_dialog(GtkWidget * widget,
                                     GtkWindow * parent_window)
{
    GtkTreeView *folder_view = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    nodePtr node = NULL;
    folderPtr folder = NULL;
    GValue value;

    ULOG_DEBUG(__FUNCTION__);

    memset(&value, 0, sizeof(value));
    if (widget != NULL) {
        ULOG_DEBUG("widget was set on call");
        folder_view = GTK_TREE_VIEW(lookup_widget(widget, "folder_view"));
        g_assert(folder_view != NULL && GTK_IS_TREE_VIEW(folder_view));
        gtk_tree_view_get_cursor(folder_view, &path, &column);
        if (path != NULL) {
            ULOG_DEBUG("and we got path to cursor");
            gtk_tree_model_get_iter(gtk_tree_view_get_model
                                    (folder_view), &iter, path);
            gtk_tree_model_get_value(gtk_tree_view_get_model
                                     (folder_view), &iter, FS_PTR, &value);
            node = (nodePtr) g_value_peek_pointer(&value);
            if (node != NULL && node->type == FST_FOLDER) {
                folder = (folderPtr) node;
            }
            gtk_tree_path_free(path);
            path = NULL;
        }
    }
    GtkWidget *nfdialog =
        create_newfolderdialog(folder, folder_view, parent_window);
    g_value_unset(&value);
    g_signal_connect(G_OBJECT(nfdialog), "response",
                     G_CALLBACK(on_new_folder_action_response), widget);
    gtk_widget_show(nfdialog);

    return nfdialog;
}



/*TODO: this will be removed later as SAVE & SAVEAS is not used
 * Actually it can be used for exporting opml file
 */
enum e_file_chooser {
    FILE_CHOOSER_LOAD,
    FILE_CHOOSER_SAVE,
    FILE_CHOOSER_SAVEAS
};

static
HildonFileSystemModel *get_file_system_model(AppUIData * app_ui,
                                             GtkWidget * ref_widget)
{
    g_return_val_if_fail(app_ui != NULL, NULL);

    if (!app_ui->model) {
        app_ui->model = g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
                                     "ref_widget", ref_widget, NULL);
    }

    return app_ui->model;
}


/**
  Show a Hildon file chooser dialog

  @param app_ui Application UI data
  @param save #FILE_CHOOSER_LOAD, #FILE_CHOOSER_SAVE, or
              #FILE_CHOOSER_SAVEAS
  @return filename (g_free'd by the caller), or NULL if problems
*/
static
const gchar *file_chooser(AppUIData * app_ui, enum e_file_chooser mode)
{
    g_assert(app_ui);

    HildonFileSystemModel *model = NULL;
    GtkWidget *fs = NULL;
    GtkWidget *parent = NULL;
    
    if (app_ui->add_feed_dialog)
	    parent = app_ui->add_feed_dialog;
    else
	    parent = GTK_WIDGET(app_ui->main_view);

    model = get_file_system_model(app_ui, GTK_WIDGET(parent/*app_ui->add_feed_dialog*/));

    if (!model)
        return NULL;

    app_data->app_ui_data->fsdialog = fs =
        hildon_file_chooser_dialog_new_with_properties(GTK_WINDOW
                                                       (parent/*app_ui->add_feed_dialog*/),
                                                       "file_system_model",
                                                       model, "action",
                                                       GTK_FILE_CHOOSER_ACTION_OPEN,
                                                       NULL);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(fs), FALSE);
    gtk_window_set_modal(GTK_WINDOW(fs), TRUE);

    if (!fs) {
        ULOG_ERR("Failed creating HildonFileChooserDialog");
        return NULL;
    }

    /* RSS News Reader-specific dialog texts
     */
    gtk_window_set_title(GTK_WINDOW(fs), _("rss_ti_feed_select_opml_file"));

    /*what's this? 
     * g_object_set( G_OBJECT (fs), 
     * "empty-text", _("sfil_va_select_object_no_objects_sketches"),
     * NULL );
     */

    // Filtering .xml or .opml files only
    GtkFileFilter *filter = gtk_file_filter_new();

    gtk_file_filter_add_pattern(filter, "*." OPML_FILE_EXTENSION_1);    //.xml
    gtk_file_filter_add_pattern(filter, "*." OPML_FILE_EXTENSION_2);    //.opml

    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fs), filter);
    gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(fs), filter);
    /* Folder that the dialog opens in 
     */
    const gchar *uri = "/home/user/MyDocs";
    g_assert(uri);
    gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(fs), uri);


    if (!app_data->app_ui_data->add_feed_dialog) {
        gtk_widget_destroy(fs);
        app_data->app_ui_data->fsdialog = NULL;
	return NULL;
    }
	    
    /* Run the dialog
     */
    if (gtk_dialog_run(GTK_DIALOG(fs)) != GTK_RESPONSE_OK) {
        /* dialog cancelled, exit */
        gtk_widget_destroy(fs);
        app_data->app_ui_data->fsdialog = NULL;
        return NULL;
    }
    /* check that user did not just click OK
     * without selecting anything
     */
    const gchar *fn = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(fs));
    if (!fn) {
        gtk_widget_destroy(fs);
        app_data->app_ui_data->fsdialog = NULL;
        return NULL;
    }

    /* "Houston, we have selection.."
     */
    const gchar *uri_new =
        gtk_file_chooser_get_current_folder_uri(GTK_FILE_CHOOSER(fs));
    if (!uri_new) {
    }
    /*
     * else if ( strcmp( app_ui->uri_, uri_new ) != 0 ) {        // changed the folder
     * g_free(app_ui->uri_);
     * app_ui->uri_= uri_new;        
     * uri_new=NULL;
     * const gchar *cwd = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(fs));
     * if (!cwd) {
     * } else if (strcmp(app_ui->cwd_,cwd)!=0){
     * g_free(app_ui->cwd_);
     * app_ui->cwd_=cwd;
     * g_free(cwd), cwd=NULL; 
     * } 
     * }
     */
    app_data->app_ui_data->fsdialog = NULL;
    gtk_widget_destroy(fs), fs = NULL;

    // append opml file extension if necessary 
    // How could this happen? Didn't the pattern filter work?

    /*
     * if (!g_str_has_suffix( fn, "." OPML_FILE_EXTENSION_1 )) {
     * gchar *tmp= g_strconcat( fn, "." OPML_FILE_EXTENSION_1, NULL );
     * g_free(fn);
     * fn= tmp;
     * }
     * else if (!g_str_has_suffix( fn, "." OPML_FILE_EXTENSION_2 )) {
     * gchar *tmp= g_strconcat( fn, "." OPML_FILE_EXTENSION_2, NULL );
     * g_free(fn);
     * fn= tmp;
     * }
     */
    return fn;
}


void display_file_chooser_dialog(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG("display_file_chooser_dialog ...........\n");
    struct fp_prop_ui_data *data = (struct fp_prop_ui_data *) user_data;

    g_assert(app_data != NULL);

    const gchar *filename =
        file_chooser(app_data->app_ui_data, FILE_CHOOSER_LOAD);
    if (!filename)
    {
        if (!app_data->app_ui_data->add_feed_dialog)
        return;
	/* add the entry focus and cursor*/
        /*NB#110375 */
        gtk_widget_grab_focus(GTK_WIDGET(app_data->app_ui_data->ui_data->sourceEntry));
        gtk_entry_set_activates_default(GTK_ENTRY(app_data->app_ui_data->ui_data->sourceEntry), TRUE);
        return;
    }
		
    ULOG_DEBUG("display_file_chooser_dialog: Browsed file name = %s \n",
               filename);

    //now update the opmlAdressEntry field
    //    GtkWidget *entry = lookup_widget (GTK_WIDGET (app_ui_data->dialog),"opmlAddressEntry");
    //    gtk_entry_set_text(GTK_ENTRY(opmlAdressEntry),filename);
    if (filename) {
        gchar *name_for_display = gnome_vfs_unescape_string(filename, NULL);
        gtk_entry_set_text(GTK_ENTRY(data->sourceEntry), name_for_display);
        g_free(name_for_display);
        data->sourceURI = g_strdup(filename);
	g_free(filename);
    }
}


void on_manage_folders_action_response(GtkWidget * manage_folders_dialog,
                                       gint responseId, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);


    app_ui_data = app_data->app_ui_data;

    GtkTreeView *treeview =
        GTK_TREE_VIEW(lookup_widget(manage_folders_dialog, "folder_view"));
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    nodePtr node = ui_feedlist_get_selected_from_tree(treeview);

    ULOG_DEBUG(__FUNCTION__);

    switch (responseId) {
        case RSS_RESPONSE_NEW_FOLDER:
            app_ui_data->nfdialog =
                display_new_folder_dialog(manage_folders_dialog,
                                          GTK_WINDOW(user_data));
            break;
        case RSS_RESPONSE_RENAME_FOLDER:
            ULOG_INFO("Rename folder");

            if ((node != NULL) && (node->type == FST_FOLDER)) {
                folderPtr folder = (folderPtr) node;
                ULOG_INFO("Selected folder for rename: %s", folder->title);
                if (folder_get_nonremovable(folder))
                    /*
                     * hildon_banner_show_information(GTK_WIDGET(app_data->app_ui_data->main_view), NULL,
                     * _("rss_ib_renaming_folder_not_allowed"));
                     */
                    hildon_banner_show_information(GTK_WIDGET
                                                   (app_data->app_ui_data->
                                                    main_view), NULL,
                                                   dgettext
                                                   (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                                    "sfil_ib_unable_to_rename_selected_folder"));

                else
                    create_folder_rename_dialog(folder);
            } else {
                ULOG_INFO("No selection, or selection is not a folder!");
            }

            break;

        case RSS_RESPONSE_DELETE_FOLDER:
            ULOG_INFO("Delete folder");

            if ((node != NULL) && (node->type == FST_FOLDER)) {
                folderPtr folder = (folderPtr) node;
                GtkTreePath *path_to_select = NULL;
                path_to_select =
                    ui_feedlist_get_previous_after_delete_folder(treeview);
                if (ui_feedlist_confirm_and_delete_folder(folder)) {
                    if (path_to_select) {
                        ULOG_INFO("Selecting new folder");
                        gtk_widget_grab_focus(GTK_WIDGET(treeview));
                        gtk_tree_selection_select_path(selection,
                                                       path_to_select);
			   gtk_tree_view_set_cursor(treeview, path_to_select, NULL, FALSE);
                    }
                }
                if (path_to_select) {
                    gtk_tree_path_free(path_to_select);
                }
                change_manage_delete_button_status(manage_folders_dialog);
            } else {
                ULOG_INFO("No selection, or selection is not a folder!");
            }

            break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
            app_ui_data->dialog = NULL;
            gtk_widget_destroy(GTK_WIDGET(manage_folders_dialog));
            app_data->app_ui_data->manage_folders_displayed = FALSE;
            break;
    }
}

void on_change_folder_action_response(GtkWidget * change_folder_dialog,
                                      gint responseId, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;

    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;

    switch (responseId) {
        case RSS_RESPONSE_NEW_FOLDER:
            app_ui_data->nfdialog =
                display_new_folder_dialog(change_folder_dialog,
                                          GTK_WINDOW(ui_data->parent_window));
            break;
        case GTK_RESPONSE_OK:
            ULOG_INFO("change folder");
            GtkTreeView *treeview =
                GTK_TREE_VIEW(lookup_widget
                              (change_folder_dialog, "folder_view"));
            nodePtr node = ui_feedlist_get_selected_from_tree(treeview);
            if ((node != NULL) && (node->type == FST_FOLDER)) {
                folderPtr folder = (folderPtr) node;
                ULOG_INFO("Selected folder to change to: %s", folder->title);

                ui_data->parent_folder = folder;
                gtk_label_set_text(GTK_LABEL(ui_data->label_widget),
                                   folder->title);
            } else {
                ULOG_INFO
                    ("No selection, or selection is not a folder! Falling back to root folder.");
                folderPtr folder = ui_feedlist_get_root_folder();
                ULOG_INFO("Selected folder to change to: %s", folder->title);

                ui_data->parent_folder = folder;
                gtk_label_set_text(GTK_LABEL(ui_data->label_widget),
                                   folder->title);
            }
            gtk_widget_destroy(GTK_WIDGET(change_folder_dialog));
            //ui_data is destroyed by the parent dialog

            break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(GTK_WIDGET(change_folder_dialog));
            break;
    }
        /* add the entry focus and cursor*/
        /*NB#108697*/
        gtk_widget_grab_focus(GTK_WIDGET(ui_data->sourceEntry));
        gtk_entry_set_activates_default(GTK_ENTRY(ui_data->sourceEntry), TRUE);
    
}


void display_change_folder_dialog(GtkWidget * widget, gpointer user_data)
{
    GtkTreeView *folder_view = NULL;
    GtkTreeModel *folder_model = NULL;
    folderPtr current_folder = NULL;
    GtkTreePath *current_path = NULL;
    AppUIData *app_ui_data = NULL;
    struct fp_prop_ui_data *data = NULL;

    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;
    data = (struct fp_prop_ui_data *) user_data;

    if (data->parent_folder != NULL) {
        current_folder = data->parent_folder;
    } else {
        current_folder = ui_feedlist_get_root_folder();
    }
    g_assert(current_folder != NULL);

    GtkWidget *cfdialog = data->parent_window =
        create_change_folder_dialog(GTK_WINDOW(data->dialog));
    populate_folders(cfdialog, TRUE);
    app_ui_data->cfdialog = cfdialog;

    /* Select the root folder. */
    folder_view = GTK_TREE_VIEW(lookup_widget(cfdialog, "folder_view"));
    g_assert(folder_view != NULL);
    folder_model = gtk_tree_view_get_model(folder_view);
    g_assert(folder_model != NULL);

    ui_feedlist_search_model_by_ptr(folder_model,
                                    (nodePtr) current_folder,
                                    &current_path, NULL);
    if (current_folder != ui_feedlist_get_root_folder())  {
        gtk_tree_view_expand_to_path(folder_view,current_path);
    }
    gtk_tree_view_set_cursor(folder_view, current_path, NULL, FALSE);
    gtk_widget_grab_focus(GTK_WIDGET(folder_view));
    gtk_tree_path_free(current_path);
    current_path = NULL;

    g_signal_connect(G_OBJECT(cfdialog), "response",
                     G_CALLBACK(on_change_folder_action_response), user_data);
    ULOG_INFO("Change folder dialog created, showing ...\n");
    gtk_widget_show(cfdialog);
    ULOG_INFO("Change folder dialog created, on screen.\n");
}

void on_folder_cursor_changed(GtkTreeView * tv, GtkWidget * dlg)
{
    nodePtr node = ui_feedlist_get_selected_from_tree(tv);
    gboolean enable = FALSE, root = FALSE, can_enable = FALSE;

    if (node && (node->type == FST_FOLDER)) {
        enable = !folder_get_nonremovable((folderPtr) node);
        root = (folderPtr) node == ui_feedlist_get_root_folder();
        if (enable && (app_data->app_ui_data->search_mode >= SFM_REFRESH)) {
            can_enable = ui_feedlist_check_if_folder_empty((folderPtr) node);
        } else {
            can_enable = TRUE;
        }
    }

    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_newbtn"), enable || root);
    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_renamebtn"), enable
                             && can_enable);
    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_deletebtn"), enable
                             && can_enable);
}

void on_managefoldersbtn_clicked(GtkButton * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;

    ULOG_DEBUG(__FUNCTION__);

    if (SFM_SEARCH == app_ui_data->search_mode) {
        return;
    }

    ULOG_DEBUG("on_managefoldersbtn_clicked: creating mng folder");
    GtkWidget *mfdialog = app_ui_data->dialog =
        create_manage_folders_dialog();
    app_ui_data->mfdialog = mfdialog;

    GtkWidget *fld = lookup_widget(mfdialog, "folder_view");
    g_signal_connect(G_OBJECT(fld), "cursor_changed",
                     G_CALLBACK(on_folder_cursor_changed), mfdialog);

    ULOG_DEBUG("on_managefoldersbtn_clicked: populating mng folder");
    populate_folders(mfdialog, FALSE);
    ULOG_DEBUG("on_managefoldersbtn_clicked: DONE populating mng folder");

    g_signal_connect(G_OBJECT(mfdialog), "response",
                     G_CALLBACK(on_manage_folders_action_response), mfdialog);
    GtkWidget *btn = lookup_widget(mfdialog, "mf_newbtn");
    hildon_helper_set_insensitive_message(GTK_WIDGET(btn),
                                          _
                                          ("rss_ib_unable_to_create_folder_into"));
    btn = lookup_widget(mfdialog, "mf_renamebtn");
   /* hildon_helper_set_insensitive_message(GTK_WIDGET(btn),
                                          dgettext
                                          (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                           "sfil_ib_unable_to_rename_selected_folder")); */
    btn = lookup_widget(mfdialog, "mf_deletebtn");
  /*  hildon_helper_set_insensitive_message(GTK_WIDGET(btn),
                                          dgettext
                                          (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                           "sfil_ib_unable_to_delete_selected_folder")); */
    ULOG_DEBUG("on_managefoldersbtn_clicked: signal connected");
    gtk_widget_show(mfdialog);
}

void automatic_updates_toggled_cb(GtkToggleButton * togglebutton,
                                  gpointer user_data)
{
    GtkWidget *prefdialog = (GtkWidget *) user_data;
    GtkWidget *connTypePulldown =
        lookup_widget(prefdialog, "connTypePulldown");
    GtkWidget *connIntervalPulldown =
        lookup_widget(prefdialog, "connIntervalPulldown");
    gboolean active = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(connTypePulldown, active);
    gtk_widget_set_sensitive(connIntervalPulldown, active);
}

gboolean on_prefdialog_insensitive_press(GtkWidget * widget,
                                         GdkEventButton * event,
                                         gpointer user_data)
{
    /* CID: 17540 */	
    return FALSE;
}

void on_refreshbtn_clicked(GtkButton * button, gpointer user_data)
{
    on_refresh_menuitem_selected(NULL, user_data);
}

void on_refresh_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    nodePtr ptr = NULL;

    ULOG_DEBUG("***refresh menuitem selected");
    app_ui_data = (AppUIData *) user_data;
    g_assert(app_ui_data != NULL);


    //        ptr = ui_feedlist_get_selected();
    /* TODO: This is not the right way....gtkhtml can get the focus with a simple
    tap. Simply the selected items should be counted in the treeview, but currently
    it is a big change */
    if (displayed_node&&gtk_widget_is_focus
        (gtk_bin_get_child(GTK_BIN(app_data->app_ui_data->html_scrollpane))))
    {
        ptr = (nodePtr) displayed_node;

    } else {
        ptr = (nodePtr) ui_feedlist_get_selected();

    }

    //        ui_search_clear_search_results();
    if (displayed_node == (nodePtr) searchFeed) {
        ui_feedlist_select(app_ui_data->feedlist_lastnode);
        ptr = (nodePtr) app_ui_data->feedlist_lastnode;
        ui_mainwindow_show_statistics();
    }

    ULOG_DEBUG("refresh clicked: ...checking iap_action");

    if (ptr == NULL || ptr->type != FST_FEED) {
        app_ui_data->iap_action = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
        app_ui_data->iap_disconnect = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
	
    } else if (menuitem == (GtkWidget *) app_ui_data->context_menuitem_refresh) {
        ULOG_DEBUG("refresh clicked:...refreshing only one feed");
        app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED;
        app_ui_data->iap_disconnect = OSSO_IAP_REFRESH_FEED;
        app_ui_data->new_subscription = (feedPtr) ptr;
        app_ui_data->new_subscription_flags = 0;
    } else {
        on_refresh_action(app_data->app_ui_data);
        return;
    }
    //            app_ui_data->iap_action = OSSO_IAP_NORMAL;

    ULOG_DEBUG("refresh clicked:...requesting connection");
    request_iap_connect();
}

void toggle_folder_list(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;

    gboolean state = FALSE;

    ULOG_DEBUG("toggle_folder_list");

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;
    state = !(GTK_WIDGET_VISIBLE(feedlist_container) != 0);


    gboolean is_active =
        gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON
                                          (app_ui_data->
                                           folders_list_toggle_button));
    gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON
                                      (app_ui_data->
                                       folders_list_toggle_button),
                                      !is_active);
    ui_set_folder_list_state(app_ui_data, state, FALSE);

    /* Action for the folder_list_button */

    debug_exit("toggle_folder_list");
}


void on_folder_list_button_clicked(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;

    gboolean state = FALSE;

    debug_enter("on_folder_list_button_clicked");

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;

    state = !(GTK_WIDGET_VISIBLE(feedlist_container) != 0);

    ui_set_folder_list_state(app_ui_data, state, TRUE);

    debug_exit("on_folder_list_button_clicked");
}

gboolean key_press_with_hidden_toolbar(GtkWidget * widget,
                                       GdkEventKey * event,
                                       gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    GtkWidget *toolbar;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (GDK_F4 == event->keyval ||
        GDK_Control_L == event->keyval ||
        GDK_Control_R == event->keyval)
    {
        //ctrl or menu key pressed, don't need to show the toolbar
        return FALSE;
    }

    if ((event->state & GDK_CONTROL_MASK)) {
        //shortcut pressed, don't need to show the toolbar
        return FALSE;
    }


    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data->main_toolbar != NULL);

    toolbar = app_ui_data->main_toolbar;
    
    if (GDK_Up == event->keyval ||
        GDK_Down == event->keyval ||
        GDK_Left == event->keyval ||
        GDK_Right == event->keyval ||
        GDK_Return == event->keyval)
    {
        //navigation key pressed, don't need to show the toolbar
        return FALSE;
    }


    if (GTK_WIDGET_VISIBLE(app_ui_data->find_toolbar)) {
        return FALSE;
    }

    if ((GDK_BackSpace == event->keyval) && !GTK_WIDGET_HAS_FOCUS(app_ui_data->search_field))
        {
            return FALSE;
        }

    if( !GTK_WIDGET_VISIBLE( toolbar ) && !app_ui_data->toolbar_temp_shown ) {
        app_ui_data->toolbar_temp_shown = TRUE;
        gtk_widget_show_all(toolbar);
        gtk_widget_grab_focus(GTK_WIDGET(app_ui_data->search_field));
    }
    
    return FALSE;
}

gboolean hide_temp_shown_toolbar(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    GtkWidget *toolbar;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data->main_toolbar != NULL);

    toolbar = app_ui_data->main_toolbar;

    if (app_ui_data->toolbar_temp_shown) {
        app_ui_data->toolbar_temp_shown = FALSE;
        gtk_widget_hide(toolbar);
    }

    return FALSE;
}

#ifdef REMOVED_REFERENCE_FULLSCREEN
void on_toggle_fullscreen_mode(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    gboolean state = FALSE;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data != NULL);

    state = !(app_ui_data->fullscreen != FALSE);
    ui_set_fullscreen_state(app_ui_data, state, FALSE);
}


void toggle_fullscreen_and_update_menu(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    gboolean state = FALSE;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data != NULL);

    state = !(app_ui_data->fullscreen != FALSE);
    ui_set_fullscreen_state(app_ui_data, state, TRUE);

}
#endif

void on_cut(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);
}

void on_copy(GtkAction * action, gpointer data)
{

    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);
            return;
}

void on_paste(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);
            return;
        }

void on_selectall(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    ULOG_DEBUG("on_selectall");
    return;

}

void on_menuitem_edit(GtkWidget * widget, gpointer user_data)
{
    /* run this in idle (so return is fast) */
    g_idle_add( ui_update_edit_menu, user_data );
}

/* Insensitiveize feed directory if neeeded */
void on_menuitem_tools(GtkWidget * widget, gpointer user_data)
{

    gchar *feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
    if (strlen(feed_dir_source) == 0) {
        ULOG_DEBUG("on_menuitem_tools: setting feed dir to insensitive");
        gtk_widget_hide(GTK_WIDGET
                        (app_data->app_ui_data->menuitem_feed_directory));
    } else if (app_data->app_ui_data->search_mode == SFM_NORMAL)
        gtk_widget_show(GTK_WIDGET
                        (app_data->app_ui_data->menuitem_feed_directory));

    g_free(feed_dir_source);
    return;
}

void toggle_find_on_page(GtkWidget * widget, gpointer user_data)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(app_data->app_ui_data->find_toolbar))) {
        ULOG_DEBUG("toggle_find_on_page: show");
        fprintf(stderr, "toggle_find_on_page: show\n");
        gtk_widget_show_all(GTK_WIDGET(app_data->app_ui_data->find_toolbar));
        fprintf(stderr, "toggle_find_on_page: set focus\n");
        ULOG_DEBUG("toggle_find_on_page: set focus");
        hildon_find_toolbar_highlight_entry(app_data->app_ui_data->
                                            find_toolbar, TRUE);
        /*        gtk_widget_grab_focus(GTK_WIDGET(app_data->app_ui_data->find_toolbar));  */
    } else {
        ULOG_DEBUG("toggle_find_on_page: hide");
        fprintf(stderr, "toggle_find_on_page: hide\n");
        gtk_widget_hide_all(GTK_WIDGET(app_data->app_ui_data->find_toolbar));
    }
}

void free_find_string()
{
    AppUIData *app_ui_data = NULL;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_ui_data = app_data->app_ui_data;

    if (app_ui_data->find_string != NULL &&
        strcmp(app_ui_data->find_string, ""))
        g_free(app_ui_data->find_string);

    app_ui_data->find_string = "";
}

void find_toolbar_search(GtkWidget * widget, gpointer data)
{
    gchar *find_text = NULL;

    ULOG_DEBUG("find_toolbar_search");

    g_assert(widget != NULL);

    g_object_get(G_OBJECT(widget), "prefix", &find_text, NULL);
    ULOG_DEBUG("prefix: %s", find_text);

    if ((NULL == find_text) || (strcmp(find_text, "") == 0)) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext(HILDON_LIBS_L10N_PACKAGE,
                                                "ckct_ib_find_enter_the_text"));
        return;
    }
    /*TODO: when is rewind = FALSE/TRUE. Add a global find_text to compare
     * if this is first search or second search with the same str 
     * TODO: remove find() */

    gboolean rewind = FALSE;
    gboolean matched = FALSE;
    if (prev_find_text == NULL) {
        prev_find_text = g_strdup(find_text);
        prev_text_has_a_match = FALSE;
        rewind = TRUE;          //start from beginning if starting a new search
    } else if (strcmp(find_text, prev_find_text) != 0) {
        if (NULL != prev_find_text)
            g_free(prev_find_text);
        prev_find_text = g_strdup(find_text);
        prev_text_has_a_match = FALSE;
        rewind = TRUE;          //start from beginning if starting a new search
    } else if ((strcmp(find_text, prev_find_text) == 0)
               && (prev_text_has_a_match == FALSE))
        rewind = TRUE;


    gtkhtml_find_cb(GTK_WIDGET(app_data->app_ui_data->html_scrollpane),
                    find_text, rewind, &matched);

    if (matched)
        prev_text_has_a_match = TRUE;

	if(matched == FALSE && prev_text_has_a_match)
	{
              hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, L_("ckct_ib_find_search_complete"));
              hildon_find_toolbar_highlight_entry(app_data->app_ui_data->
                                            find_toolbar, TRUE);
	}
    if (matched == FALSE&& (!prev_text_has_a_match)) {

        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       dgettext(HILDON_LIBS_L10N_PACKAGE,
                                                "ckct_ib_find_no_matches"));

        hildon_find_toolbar_highlight_entry(app_data->app_ui_data->
                                            find_toolbar, TRUE);

    }
    g_free(find_text);
    find_text = NULL;

}

void find_toolbar_close(GtkWidget * widget, gpointer data)
{
    ULOG_DEBUG("find toolbar close");
    AppUIData *app_ui_data = NULL;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);

    if (app_ui_data->find_toolbar == NULL)
        return;

    setBooleanConfValue(ENABLE_FIND_ON_PAGE, FALSE);

    gtk_widget_hide_all(GTK_WIDGET(app_ui_data->find_toolbar));
}

/*------------------------------------------------------------------------------*/
/* status bar callback, error box function, GUI update function         */
/*------------------------------------------------------------------------------*/



void ui_show_error_box(const char *format, ...)
{
    GtkWidget *dialog = NULL;
    va_list args;
    gchar *msg = NULL;

    if (format == NULL)
        return;

    va_start(args, format);
    msg = g_strdup_vprintf(format, args);
    va_end(args);

    dialog = gtk_message_dialog_new(GTK_WINDOW(mainwindow),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR,
                                    GTK_BUTTONS_CLOSE, "%s", msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    g_free(msg);
}

void ui_show_info_box(const char *format, ...)
{
    GtkWidget *dialog = NULL;
    va_list args;
    gchar *msg = NULL;

    if (format == NULL)
        return;

    va_start(args, format);
    msg = g_strdup_vprintf(format, args);
    va_end(args);

    dialog = gtk_message_dialog_new(GTK_WINDOW(mainwindow),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_INFO,
                                    GTK_BUTTONS_CLOSE, "%s", msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    g_free(msg);
}

/*------------------------------------------------------------------------------*/
/* exit handler                                 */
/*------------------------------------------------------------------------------*/

void save_state(AppData * app_data)
{
    AppState *app_state = NULL;

    g_assert(app_data != NULL);

    reset_state_file(app_data->osso);

    app_state = create_app_state();
    construct_app_state(app_state);
    reader_save_state(app_data, app_state);
    destroy_app_state(app_state);
    app_state = NULL;
}

void restore_state(AppData * app_data)
{
    AppState *app_state = NULL;
    StateSaveResultCode stres = ST_SUCCESS;

    g_assert(app_data != NULL);

    app_state = create_app_state();
    stres = reader_load_state(app_data, app_state);

    ULOG_DEBUG("State save result code: %d", stres);
    if (stres == ST_SUCCESS) {
        ULOG_DEBUG("Updating app state");
        update_app_state(app_state);
    }
    destroy_app_state(app_state);
    app_state = NULL;
}

void top_changed(GObject * self,
                 GParamSpec * property_param, gpointer user_data)
{
    HildonProgram *program = HILDON_PROGRAM(self);
    if (hildon_program_get_is_topmost(program)) {
        AppData *app_data = NULL;

        app_data = (AppData *) user_data;

        DMSG("************************on_top*****************************\n");

        g_assert(app_data != NULL);

        //ui_mainwindow_empty_search_field();

        g_assert(app_data->app_ui_data != NULL);

        app_data->app_ui_data->start_on_background = FALSE;
        app_data->app_ui_data->application_topped = TRUE;
	if(app_data->app_ui_data->display_off == FALSE)
	{
		if(app_data->app_ui_data->update_feed_list_needed == TRUE)
		{
			ui_feedlist_update();
			app_data->app_ui_data->update_feed_list_needed = FALSE;
		}
               if( app_data->app_ui_data->call_ui_mainwindow_finish_needed == TRUE)
               {
			call_ui_mainwindow_finish(NULL);
			app_data->app_ui_data->call_ui_mainwindow_finish_needed = FALSE;				
		}
		switch_progressbar(progressbar_mode_prev);
	}		

        hildon_program_set_can_hibernate(app_data->app_ui_data->app, FALSE);
    } else {
        AppData *app_data = NULL;

        app_data = (AppData *) user_data;

        ULOG_DEBUG
            ("**************************on_untop**************************");

        g_assert(app_data != NULL);

        g_assert(app_data->app_ui_data != NULL);

        app_data->app_ui_data->application_topped = FALSE;

        hildon_program_set_can_hibernate(app_data->app_ui_data->app, TRUE);
    }
}


gboolean on_key_press(GtkWidget * widget, GdkEventKey * event,
                      gpointer user_data)
{
    AppData *app_data = (AppData *) user_data;

    switch (event->keyval) {
        case GDK_F3:           /* For debugging only. */
            ULOG_DEBUG("emulate task away\n");
            top_changed(G_OBJECT(app_data->app_ui_data->app), NULL, app_data);
            return TRUE;
        default:
            break;
    }
    return FALSE;
}

gboolean key_press_for_cancelling_dialog(GtkWidget * widget,
                                         GdkEventKey * event, gpointer data)
{
    (void) data;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (event == NULL)
        return FALSE;

    switch (event->keyval) {
        case GDK_Escape:
            app_data->app_ui_data->dialog_closed = TRUE;
            return FALSE;
        default:
            return FALSE;
    }
    return FALSE;
}

/*
void on_open_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_assert(NULL != app_ui_data);

    if (!app_ui_data->context_url || !*app_ui_data->context_url)
        on_menu_open(NULL, user_data);
    else {
        invoke_url(app_ui_data->context_url);
    }
}*/

void on_new_folder_cmenuitem_selected(GtkWidget * menuitem,
                                      gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    nodePtr node = NULL;

    g_assert(NULL != app_ui_data);

    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FEED)
        node = (nodePtr) ui_feedlist_get_parent(node);

    if (node) {
        GtkWidget *nfdialog = create_newfolderdialog((folderPtr) node,
                                                     GTK_TREE_VIEW(feedlist),
                                                     GTK_WINDOW(app_ui_data->
                                                                main_view));
        g_signal_connect(G_OBJECT(nfdialog), "response",
                         G_CALLBACK(on_new_folder_action_response), NULL);
        gtk_widget_show(nfdialog);
    }
}

void on_rename_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    nodePtr node = NULL;

    g_assert(NULL != app_ui_data);

    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FOLDER) {
        if (folder_get_nonremovable((folderPtr) node))
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           dgettext(HILDON_FM_L10N_PACKAGE,
                                                    "sfil_ib_renaming_not_allowed"));
        else
            create_folder_rename_dialog((folderPtr) node);
    }
}

void on_delete_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    nodePtr node = NULL;

    g_assert(NULL != app_ui_data);

    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FOLDER) {
        ui_feedlist_confirm_and_delete_folder((folderPtr) node);
    }
}

void on_copy_link_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    AppUIData *app_ui_data = (AppUIData *) user_data;
    gchar *url = NULL;

    g_assert(NULL != clipboard);
    g_assert(NULL != app_ui_data);

    clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    url = strip_url(app_ui_data->context_url);
    gtk_clipboard_set_text(clipboard, url, -1);
    hildon_banner_show_information(GTK_WIDGET(app_ui_data->main_view), NULL,
                                   dgettext
                                   (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                    "ecoc_ib_edwin_copied"));
    g_free(url);
}

void on_send_post_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    g_assert(NULL != app_data);
    g_assert(NULL != app_data->app_ui_data);

    post_url(app_data->app_ui_data->context_url, user_data != NULL);
}

void on_load_image_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_assert(NULL != app_ui_data);

    if (!app_ui_data->load_url
        && (app_ui_data->search_mode <= SFM_INTERRUPTED)
        && app_ui_data->image_object && app_ui_data->image_url) {
        app_ui_data->load_url = g_strdup(app_ui_data->image_url);
        file_cache_remove(app_ui_data->img_cache, app_ui_data->load_url);
        html_image_edit_set_url(HTML_IMAGE(app_ui_data->image_object),
                                app_ui_data->load_url);
    }
}

gint ui_show_replace_file_dialog(gchar * file)
{
   gint answer = GTK_RESPONSE_CANCEL;
    gchar *base = g_path_get_basename(file);
    gchar *fmt = dgettext(HILDON_FM_L10N_PACKAGE, "docm_nc_replace_file");
    gchar *tmp = g_strdup_printf("%s\n%s", fmt, base);

	HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation
					   (GTK_WINDOW(app_data->app_ui_data->main_view),
						tmp));
    answer = gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));

    g_free(tmp);
    g_free(base);
    return answer;
}

void on_save_image_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_assert(NULL != app_ui_data);

    if (app_ui_data->image_url) {

        size_t size;
        gpointer data = file_cache_find(app_ui_data->img_cache,
                                        app_ui_data->image_url, &size, FALSE);


        if (data) {
            GtkWidget *fs = NULL;

            fs = hildon_file_chooser_dialog_new(GTK_WINDOW
                                                (app_data->app_ui_data->
                                                 main_view),
                                                GTK_FILE_CHOOSER_ACTION_SAVE);

            gchar *buf = g_path_get_basename(app_ui_data->image_url);

            // Truncate the question mark and the content after it
            // because the filename is wrong (NB# 48234)
            if (strstr(buf, "?") != NULL) {

                buf = g_strndup(buf, strlen(buf) - strlen(strstr(buf, "?")));

            }

            gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(fs), buf);
            g_free(buf);

            gchar *images_folder;
            if ((images_folder = g_get_user_special_dir (G_USER_DIRECTORY_PICTURES)) == NULL) {
            /* fallback */
	            images_folder = "/home/user/MyDocs/.images";
            }
            images_folder = g_strconcat("file://", images_folder, NULL);
	
            gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(fs),
                                                   images_folder);
            g_free(images_folder);

            if (gtk_dialog_run(GTK_DIALOG(fs)) == GTK_RESPONSE_OK) {
                gchar *filename = NULL;

                gtk_widget_hide(fs);
                filename = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(fs));

                if (filename) {

                    GnomeVFSURI *uri = gnome_vfs_uri_new(filename);
                    GnomeVFSHandle *handle = NULL;

                    ULOG_DEBUG("filename: %s", filename);
                    if (gnome_vfs_uri_exists(uri) &&
                        ( (ui_show_replace_file_dialog(filename) != GTK_RESPONSE_OK))) {
                        gnome_vfs_uri_unref(uri);
                        g_free(filename);
                        gtk_widget_destroy(fs);
                        g_free(data);

                        return;
                    }

                    if (gnome_vfs_create_uri(&handle, uri,
                                             GNOME_VFS_OPEN_WRITE |
                                             GNOME_VFS_OPEN_TRUNCATE,
                                             TRUE, 0644) == GNOME_VFS_OK) {
                        gnome_vfs_write(handle, data, size, NULL);
                        gnome_vfs_close(handle);
                        ULOG_DEBUG("saved: %s", filename);
                    }

                    gnome_vfs_uri_unref(uri);
                    g_free(filename);
                }
            	gtk_widget_destroy(fs);
	    	fs = NULL;
            	g_free(data);
            }
	    else
	    {	    
            	gtk_widget_destroy(fs);
	    	fs = NULL;
            	g_free(data);
	    }
        }
    }
}

void on_clear_image_cache(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_assert(NULL != app_ui_data);

    gint answer = GTK_RESPONSE_CANCEL;

#ifdef UISPEC_CHANGES
    HildonNote *note =
        HILDON_NOTE(hildon_note_new_confirmation_add_buttons
                    (GTK_WINDOW(app_data->app_ui_data->main_view),
                     _("rss_nc_clear_image_cache"),
                     _("wdgt_bd_done"),
                     1,
                     NULL));
#endif

		HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation
						(GTK_WINDOW(app_data->app_ui_data->main_view),
						 _("rss_nc_clear_image_cache")));
	
    if (note) {
        answer = gtk_dialog_run(GTK_DIALOG(note));

        if (answer==GTK_RESPONSE_OK)
            file_cache_clear(app_ui_data->img_cache);
    } else
        ULOG_CRIT("Unable to create HildonNote");

    gtk_widget_destroy(GTK_WIDGET(note));
}

void on_menuitem_sort(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_assert(app_ui_data);

    HildonSortDialog *sort_dialog =
        HILDON_SORT_DIALOG(hildon_sort_dialog_new
                           (GTK_WINDOW(app_ui_data->main_view)));

    gint response_id, add_sort_index;

    add_sort_index = hildon_sort_dialog_add_sort_key(sort_dialog,
                                                     dgettext
                                                     (HILDON_LIBS_L10N_PACKAGE,
                                                      "ckdg_va_sort_date"));

    hildon_sort_dialog_add_sort_key(sort_dialog,
                                    dgettext(HILDON_LIBS_L10N_PACKAGE,
                                             "ckdg_va_sort_name"));

    int sort_key = getNumericConfValue(RSS_SETTINGS_SORT_KEY);
    int sort_order = getNumericConfValue(RSS_SETTINGS_SORT_ORDER);

    hildon_sort_dialog_set_sort_key(sort_dialog, sort_key);
    hildon_sort_dialog_set_sort_order(sort_dialog, sort_order);

    gtk_widget_show(GTK_WIDGET(sort_dialog));

    response_id = gtk_dialog_run(GTK_DIALOG(sort_dialog));

    if (response_id == GTK_RESPONSE_OK) {
        setNumericConfValue(RSS_SETTINGS_SORT_KEY,
                            hildon_sort_dialog_get_sort_key(sort_dialog));

        setNumericConfValue(RSS_SETTINGS_SORT_ORDER,
                            hildon_sort_dialog_get_sort_order(sort_dialog));

        if (displayed_node == (nodePtr) searchFeed && searchFeed)
            ui_itemlist_load((nodePtr) searchFeed);
        else {
            //      nodePtr ptr = (nodePtr) ui_feedlist_get_selected();
            nodePtr ptr;

            if (gtk_widget_is_focus
                (gtk_bin_get_child
                 (GTK_BIN(app_data->app_ui_data->html_scrollpane))))
            {

                ptr = (nodePtr) displayed_node;

            } else {

                ptr = (nodePtr) ui_feedlist_get_selected();

            }

            if (ptr && displayed_node
                && ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type))) {
                displayed_node = NULL;
                ui_feedlist_load_selected(ptr);
            }
        }
    }

    gtk_widget_destroy(GTK_WIDGET(sort_dialog));
}

/* handle for users pressing stop while searching
 */
void stop_search()
{
    ULOG_DEBUG("vvvvvvvvvvvvv....stop_search.....vvvvvvvvvvv......");
    //this is enough to stop the vfolder_apply_rules() for one feed
    app_data->app_ui_data->search_mode = SFM_INTERRUPTED;
    //Now we need to stop the html page too. TODO
    gtkhtml_finish_search_feed();
    ui_mainwindow_empty_search_field();

    if (SFM_INTERRUPTED == app_data->app_ui_data->search_mode) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, _("rss_ib_search_interrupted"));
    }

    switch_progressbar(PROGRESSBAR_NORMAL_MODE);
    //    ui_on_search_finished();
    return;
}

void
on_feedlist_focus_in()
{
    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();

    if (NULL != ptr && ptr->type == FST_FEED)
    {
        ui_mainwindow_set_feed_deatils_sensitive(app_data, TRUE);
	 if (app_data->app_ui_data->search_mode == SFM_NORMAL ) 
        ui_mainwindow_set_unsubscribe_sensitive(app_data, TRUE);
    }
}

void
on_html_scrollpane_focus_in(GtkWidget * widget,
                            GdkEventFocus * event, gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    ui_update_edit_menu(app_data->app_ui_data);

    if (displayed_node && displayed_node == (nodePtr) searchFeed)
    {
        ui_mainwindow_set_unsubscribe_sensitive(app_data, FALSE);
        ui_mainwindow_set_feed_deatils_sensitive(app_data, FALSE);

//        set_feed_properties_insens_messages(app_data->app_ui_data);
        unsubscribe_insens_messages(app_data->app_ui_data);
//        new_insens_messages(app_data->app_ui_data);
        refresh_insens_messages(app_data->app_ui_data);
    }
}


void on_insensitive_infoprint(GtkWidget * widget, gchar * info)
{
    if (info) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, info);
    }
}


gboolean on_gtkhtml_keyboard_event(GtkWidget *widget, GdkEventKey *event, gpointer user_data) {

    (void) user_data;
    (void) widget;

    switch(event->keyval) {
        case GDK_BackSpace:
            /* We dont want to scroll on backspace */
            ULOG_DEBUG("On the page, no scrolling!");
            return TRUE;
            break;
        default:
            return FALSE;
    }

}

gboolean on_gtkhtml_expose() {
    /* Copy item in edit menu are by default insensitive. It should be made
     * sensitive, when there's something selected. This callback is used to
     * update the state of copy item, when the user selects some text with
     * double-tap. It seems, the only event emitted after the selection
     * actually happened is the expose event. If you find somethink better,
     * feel free to replace it. (: (Edit menuitems are updated all the time,
     * when edit menu is opened. Though it won't let the shortcuts work when
     * the menu wasn't opened after the selection took place) */


    ULOG_DEBUG(__FUNCTION__);

    ui_update_edit_menu(app_data->app_ui_data);

    return FALSE;
}

gboolean on_search_field_key_press(GtkWidget * widget, GdkEventKey * event,
                                   gpointer user_data)
{

    AppUIData *app_ui_data = (AppUIData *) user_data;
    g_assert(app_ui_data != NULL);

    /* Mask enter to search button */
    if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter) {
        g_signal_emit_by_name(G_OBJECT(app_ui_data->search_button),
                              "clicked");
        return TRUE;
    }

    /* Rest can be handled by widget */
    return FALSE;

}


#ifdef REMOVED_REFERENCE_FULLSCREEN
void toggle_toolbar_cb(GtkWidget * widget, gpointer user_data) {
    AppUIData *app_ui_data = NULL;
    gboolean hidden;
    GtkWidget *toolbar = NULL;

    g_assert(user_data != NULL);
    app_ui_data = (AppUIData *) user_data;

    setBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN, !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)-> active);
    setBooleanConfValue(DISABLE_TOOLBAR, !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)-> active);


    toolbar  = app_ui_data->main_toolbar;

    g_assert(toolbar != NULL);

    /* unclear here is that what if user hides the toolbar *during* search,
     * what should happen ? so it will propably need fixing later but current
     * implementatin seems to work as specs say */
    if (app_ui_data->fullscreen) {
        hidden = !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active;
        if (hidden) {
            app_ui_data->toolbar_temp_shown = FALSE;
            if (app_ui_data->toolbar_keypress_handler_blocked == TRUE) {
                g_signal_handlers_unblock_by_func(G_OBJECT(app_ui_data->main_view),G_CALLBACK(key_press_with_hidden_toolbar), NULL);
                app_ui_data->toolbar_keypress_handler_blocked = FALSE;
            }
            gtk_widget_hide_all(GTK_WIDGET(toolbar));
        } else {
            if (app_ui_data->toolbar_keypress_handler_blocked == FALSE) {
                g_signal_handlers_block_by_func(G_OBJECT(app_ui_data->main_view),G_CALLBACK(key_press_with_hidden_toolbar), NULL);
                app_ui_data->toolbar_keypress_handler_blocked = TRUE;
            }
            gtk_widget_show_all(GTK_WIDGET(toolbar));
        }
    } else {
        hidden = !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active;
        if (hidden) {
            app_ui_data->toolbar_temp_shown = FALSE;
            if (app_ui_data->toolbar_keypress_handler_blocked == TRUE) {
                g_signal_handlers_unblock_by_func(G_OBJECT(app_ui_data->main_view),G_CALLBACK(key_press_with_hidden_toolbar), NULL);
                app_ui_data->toolbar_keypress_handler_blocked = FALSE;
            }
            gtk_widget_hide_all(GTK_WIDGET(toolbar));
        } else {
            if (app_ui_data->toolbar_keypress_handler_blocked == FALSE) {
                g_signal_handlers_block_by_func(G_OBJECT(app_ui_data->main_view),G_CALLBACK(key_press_with_hidden_toolbar), NULL);
                app_ui_data->toolbar_keypress_handler_blocked = TRUE;
            }
            gtk_widget_show_all(GTK_WIDGET(toolbar));
        }
    }

}
#endif


gboolean on_search_field_focus_out (GtkWidget *widget, GdkEventFocus *event, gpointer user_data)  {
    AppUIData *app_ui_data = NULL;
    GtkWidget *toolbar = NULL;
    gboolean hidden=FALSE;
    
    g_assert(user_data != NULL);
    app_ui_data = (AppUIData *) user_data;

    
    toolbar  = app_ui_data->main_toolbar;
#ifdef REMOVED_REFERENCE_FULLSCREEN
    if (app_ui_data->fullscreen) {
        hidden = !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active;
    } else {
        hidden = !GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active;
    }
#endif

    if (app_ui_data->toolbar_temp_shown == TRUE && hidden == TRUE) {
        gtk_widget_hide_all(GTK_WIDGET(toolbar));
        app_ui_data->toolbar_temp_shown = FALSE;
    }
    return FALSE;
}
