/**
 * @file ui_popup.c popup menus
 *
 * Copyright (C) 2003 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ui_popup.h"
#include "update.h"

/* from callbacks.c */
extern gboolean itemlist_mode;

#define TOGGLE_CONDENSED_VIEW	"/Condensed View"

#define TOGGLE_WORK_OFFLINE     "/Work Offline"
#define TOGGLE_SHOW_WINDOW      "/Show Window"

extern AppData *app_data;

void set_menuitem_visibility(GtkMenuItem * item, gboolean visible);

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

gboolean on_mainfeedlist_button_press_event(GtkWidget * widget,
                                            GdkEventButton * event,
                                            gpointer user_data)
{
    GdkEventButton *eb = NULL;
    GtkWidget *treeview = NULL;
    GtkTreeModel *model = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    gboolean selected = TRUE;
    nodePtr node = NULL;

    treeview = feedlist;
    g_assert(treeview);

    if (event->type != GDK_BUTTON_PRESS)
        return FALSE;

    eb = (GdkEventButton *) event;

    if (eb->button != 3)
        return FALSE;

    if (!gtk_tree_view_get_path_at_pos
        (GTK_TREE_VIEW(feedlist), event->x, event->y, &path, NULL, NULL,
         NULL)) {
        selected = FALSE;
    } else {
        model = gtk_tree_view_get_model(GTK_TREE_VIEW(feedlist));
        gtk_tree_model_get_iter(model, &iter, path);
        gtk_tree_model_get(model, &iter, FS_PTR, &node, -1);
        if (node != NULL)
            ui_feedlist_select(node);
        else                    /* This happens when an "empty" node is clicked */
            selected = FALSE;
        gtk_tree_path_free(path);
    }

    return TRUE;
}

void set_menuitem_visibility(GtkMenuItem * item, gboolean visible)
{
    if (visible)
        gtk_widget_show(GTK_WIDGET(item));
    else
        gtk_widget_hide(GTK_WIDGET(item));
}

static void remove_action_menu_item(GtkWidget * child, GtkWidget * container)
{
    if (g_object_get_data(G_OBJECT(child), "action-menu"))
        gtk_container_remove(GTK_CONTAINER(container), child);
}

static void
on_action_menu_activate(GtkMenuItem * item, HildonURIAction * action)
{
    if (app_data->app_ui_data->low_mem) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, dgettext("ke-recv",
                                                      "memr_ib_operation_disabled"));
        return;
    }

    gchar *link = strip_url(app_data->app_ui_data->context_url);

    ULOG_DEBUG("on_action_menu_activate: %s\n", link ? link : "(null)");
    if (!hildon_uri_open(link, action, NULL))
        g_warning("failed to open: %s", link);
}

/*
 * state:
 * 0 - folder
 * 1 - feed
 * 2 - title of feed
 * 3 - link
 * 4 - image with link
 * 5 - image
 * 6 - background
 */
gboolean setup_context_popup(int state)
{
    AppUIData *app_ui_data = app_data->app_ui_data;
    gboolean idle = app_ui_data->search_mode <= SFM_INTERRUPTED;
    gboolean searching = app_ui_data->search_mode == SFM_SEARCH;
    gboolean refreshing_img = app_ui_data->search_mode == SFM_REFRESH &&
        is_image_mode();
    gchar *url = strip_url(app_ui_data->context_url);
    gboolean valid_url = url && *url;
    gboolean netimg = app_ui_data->image_url
        && (strncasecmp(app_ui_data->image_url, "file", 4) != 0);
    gboolean factory_delete = !ui_feedlist_check_factory_delete(FALSE);
    gboolean show = FALSE;
    gint pos;

    ULOG_DEBUG("Tap-and-hold: %d, %s, %s\n%s", state,
               app_ui_data->context_url ? app_ui_data->context_url : "(null)",
               app_ui_data->image_url ? app_ui_data->image_url : "(null)",
               url);

    gtk_container_foreach(GTK_CONTAINER(app_ui_data->context_menu),
                          (GtkCallback) (remove_action_menu_item),
                          app_ui_data->context_menu);
    if (app_ui_data->actions) {
        hildon_uri_free_actions(app_ui_data->actions);
        app_ui_data->actions = NULL;
    }

    if (valid_url && (0 != state)) {
        GSList *cursor = app_ui_data->actions =
            hildon_uri_get_actions_by_uri(url, -1, NULL);
        pos = 0;
        while (cursor) {
            HildonURIAction *action = (HildonURIAction *) cursor->data;
            GtkWidget *menu_item = gtk_menu_item_new_with_label
                (dgettext(hildon_uri_action_get_translation_domain(action),
                          hildon_uri_action_get_name(action)));

            g_object_set_data(G_OBJECT(menu_item), "action-menu",
                              GINT_TO_POINTER(1));
            g_signal_connect(G_OBJECT(menu_item), "activate",
                             G_CALLBACK(on_action_menu_activate), action);

            ULOG_DEBUG("Menu: %s\n", hildon_uri_action_get_name(action));
            ULOG_DEBUG("Menu2: %s\n",
                       dgettext(hildon_uri_action_get_translation_domain
                                (action),
                                hildon_uri_action_get_name(action)));
            gtk_widget_show(menu_item);
            //            gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menu_item);
            gtk_menu_shell_insert(GTK_MENU_SHELL(app_ui_data->context_menu),
                                  menu_item, pos++);
            cursor = cursor->next;
        }
        set_menuitem_visibility(app_ui_data->context_sp,
                                app_ui_data->actions != NULL);
    } else
        set_menuitem_visibility(app_ui_data->context_sp, FALSE);

    g_free(url);

//    set_menuitem_visibility(app_ui_data->context_menuitem_open, (state == 1));
//    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_open);

/*    if (searching) {
        gtk_widget_set_sensitive(GTK_WIDGET
                                 (app_ui_data->context_menuitem_open), FALSE); 
        gtk_widget_set_sensitive(GTK_WIDGET
                                 (app_ui_data->context_menuitem_edit_details),
                                 FALSE);
    } else {
        gtk_widget_set_sensitive(GTK_WIDGET
                                 (app_ui_data->context_menuitem_open), TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET
                                 (app_ui_data->context_menuitem_edit_details),
                                 TRUE);
    }*/

    set_menuitem_visibility(app_ui_data->context_menuitem_refresh,
                            (state == 1 || state == 6) && idle);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_refresh);
    set_menuitem_visibility(app_ui_data->context_menuitem_new_folder,
                            (state >= 0 && state <= 1) && /*idle */ !searching
                            && !ui_feedlist_check_factory_add(FALSE));
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_new_folder);
    set_menuitem_visibility(app_ui_data->context_menuitem_sort,
                            (state == 6) /* && idle */ );
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_sort);
/*    set_menuitem_visibility(app_ui_data->context_menuitem_edit_details,
                            (state == 1 || (state == 6 && displayed_node)));
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_edit_details); */
    set_menuitem_visibility(app_ui_data->context_menuitem_rename,
                            (state == 0) && idle && factory_delete);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_rename);
/*    set_menuitem_visibility(app_ui_data->context_menuitem_unsubscribe,
                            (state == 1 || (state == 6 && displayed_node)) && idle
                            && factory_delete);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_unsubscribe); */
    set_menuitem_visibility(app_ui_data->context_menuitem_delete,
                            (state == 0) && idle && factory_delete);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_delete);
    set_menuitem_visibility(app_ui_data->context_menuitem_copy_link,
                            (state >= 2 && state <= 4) && valid_url);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_copy_link);
#ifdef DEL_SEND_VIA
    set_menuitem_visibility(app_ui_data->context_menuitem_send_post,
                            (state >= 2 && state <= 4) && (idle
                                                           || refreshing_img)
                            && app_ui_data->context_url
                            && *app_ui_data->context_url);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_send_post);
    set_menuitem_visibility(app_ui_data->context_menuitem_send_bt,
                            (GTK_WIDGET_VISIBLE
                             (app_ui_data->context_menuitem_send_post)
                             && bt_avail()));
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_send_bt);
    set_menuitem_visibility(app_ui_data->context_menuitem_send,
                            (GTK_WIDGET_VISIBLE
                             (app_ui_data->context_menuitem_send_post)));
#endif
    set_menuitem_visibility(app_ui_data->context_menuitem_load_image,
                            (state == 4) && (idle || refreshing_img)
                            && netimg);
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_load_image);
    set_menuitem_visibility(app_ui_data->context_menuitem_save_image,
                            (state == 4) && netimg &&
                            file_cache_exists(app_ui_data->img_cache,
                                              app_ui_data->image_url));
    show |= GTK_WIDGET_VISIBLE(app_ui_data->context_menuitem_save_image);

    return show;
}

void on_mainfeedlist_tap_and_hold_event(GtkWidget * widget)
{
    nodePtr node;
    gboolean show = FALSE;


    if (app_data->app_ui_data->bt_file != NULL)
        return;

    if ((node = ui_feedlist_get_selected())) {
        if (node->type == FST_FOLDER) {
            if (ui_feedlist_check_factory_delete(FALSE))
                return;
            else
                show = setup_context_popup(0);
        } else if (node->type == FST_FEED) {
            app_data->app_ui_data->context_url = NULL;
            show = setup_context_popup(1);
        } else
            return;

        app_data->app_ui_data->context_url = NULL;

        if (show) {
            ULOG_DEBUG("*** menu should appear");
            gtk_menu_popup(app_data->app_ui_data->context_menu, NULL, NULL,
                           NULL, NULL, 0, gtk_get_current_event_time());
        }
    }
}
