/**
 * @file favicon.c Liferea favicon handling
 * 
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <hildon/hildon-defines.h>

#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <osso-log.h>
#include <osso-rss-feed-reader/cache_handling.h>
#include "favicon_cache.h"
#include "favicon.h"
#include "support.h"
#include "feed.h"
#include "common.h"
#include "update.h"
#include "debug.h"
#include "html.h"
#include "ui_feedlist.h"
#include "ui_feed.h"
#include "callbacks.h"

AppUIData *app_ui_data = NULL;

static void favicon_download_html(feedPtr fp);

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/



static void favicon_download_request_favicon_cb(struct request *request)
{
    feedPtr fp = (feedPtr) request->user_data;
    gchar *tmp;
    gboolean success = FALSE;

    fp->otherRequests = g_slist_remove(fp->otherRequests, request);

    if (NULL != request->data && request->size > 0) {
        GdkPixbufLoader *loader = gdk_pixbuf_loader_new();
        GdkPixbuf *pixbuf;
        tmp =
            common_create_cache_filename("cache" G_DIR_SEPARATOR_S "favicons",
                                         feed_get_id(fp), "png");

        if (gdk_pixbuf_loader_write
            (loader, (const guchar *)request->data, request->size, NULL)) {
            gdk_pixbuf_loader_close(loader, NULL);
            if (NULL != (pixbuf = gdk_pixbuf_loader_get_pixbuf(loader))) {
                if (FALSE ==
                    (gdk_pixbuf_save(pixbuf, tmp, "png", NULL, NULL))) {
                    g_warning("favicon saving error!");
                    gchar *tempmsg = _("rss_ni_unable_increase_img_cache");
                    gchar *message = g_strdup_printf(tempmsg, "device");
                    hildon_banner_show_information(GTK_WIDGET
                                                   (app_ui_data->main_view),
                                                   NULL, message);
                    ui_feedlist_delete_((nodePtr) fp);
                    app_ui_data->search_mode = SFM_INTERRUPTED;
                    download_cancel_all(TRUE);
                    g_warning("cancel all downloads!");
                    return;
                }
                success = TRUE;
                favicon_load(fp);
            }
        }
        //                gdk_pixbuf_loader_close(loader, NULL);
        g_object_unref(loader);
        g_free(tmp);
        ui_feed_update(fp);
    }

    if (request->flags && !success)
        favicon_download_html(fp);

}

static void favicon_download_html_request_cb(struct request *request)
{
    gchar *iconUri;
    struct request *request2 = NULL;
    feedPtr fp = (feedPtr) request->user_data;

    if (request->size > 0 && request->data != NULL) {
        iconUri = html_discover_favicon(request->data, request->source);

        if (iconUri != NULL) {
            request2 = download_request_new();
            request2->source = iconUri;
            request2->callback = &favicon_download_request_favicon_cb;
            request2->user_data = fp;
            request2->feed = fp;
            request2->flags = 0;
            fp->otherRequests = g_slist_append(fp->otherRequests, request2);
            download_queue(request2);
        }
    }
}

static void favicon_download_html(feedPtr fp)
{
    gchar *htmlurl = NULL;
    gchar *tmp = NULL;
    struct request *request = NULL;

    if (feed_get_html_url(fp) != NULL)
        htmlurl = g_strdup(feed_get_html_url(fp));
    else
        return;

    if (NULL != (tmp = strstr(htmlurl, "://"))) {
        tmp += 3;

        if (NULL == strrchr(tmp, '/')) {
            tmp = g_strconcat(htmlurl, "/", NULL);
            g_free(htmlurl);
            htmlurl = tmp;
        }
    }

    request = download_request_new();
    request->source = htmlurl;
    request->callback = &favicon_download_html_request_cb;
    request->user_data = fp;
    request->feed = fp;
    request->flags = 0;
    fp->otherRequests = g_slist_append(fp->otherRequests, request);
    download_queue(request);
}

static void favicon_download_from_source(feedPtr fp, const gchar * source)
{
    gchar *baseurl, *tmp;
    struct request *request;

    if (source == NULL)
        return;

    baseurl = g_strdup(source);
    if (NULL != (tmp = strstr(baseurl, "://"))) {
        tmp += 3;
        if (NULL != (tmp = strchr(tmp, '/'))) {
            *tmp = 0;
            request = download_request_new();
            request->source = g_strdup_printf("%s/favicon.ico", baseurl);

            request->callback = &favicon_download_request_favicon_cb;
            request->user_data = fp;
            request->feed = fp;
            request->flags = 1;
            fp->otherRequests = g_slist_append(fp->otherRequests, request);

            download_queue(request);
        }
    }
    g_free(baseurl);
}

void favicon_download(feedPtr fp, gpointer user_data)
{

    app_ui_data = (AppUIData *) user_data;
    g_assert(app_ui_data != NULL);

    if (feed_get_favicon(fp) == NULL)
        favicon_download_from_source(fp, feed_get_source(fp));
}
