/**
 * @file conf.c Liferea configuration (gconf access and feedlist import)
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <libxml/uri.h>
#include <string.h>
#include <time.h>
#include <osso-log.h>

#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/cache_handling.h>

#include <libgnomevfs/gnome-vfs.h>

#include "conf.h"
#include "debug.h"
#include "htmlview.h"
#include "ui_mainwindow.h"
#include "cache.h"

#include "debug_new.h"

#define MAX_GCONF_PATHLEN	256
#define HOMEPAGE	"http://liferea.sf.net/"

#define FEEDLIST_CONFIG_FILE	"feedlist.opml"
#define FEEDLIST_STATE_FILE		"feedlist_state.opml"

extern AppData *app_data;
extern time_t rss_updated;

static GConfClient *gconfclient = NULL;

static guint feedlist_save_timer;
static guint feedlistLoading;

static gboolean feedlist_save_timer_with_config = FALSE;

/* configuration strings for the SnowNews HTTP code used from within netio.c */
char *useragent = NULL;
char *proxyname = NULL;
char *proxyusername = NULL;
char *proxypassword = NULL;
int proxyport = 0;

/* Function prototypes */
static void conf_proxy_reset_settings_cb(GConfClient * gconfclient,
                                         guint cnxn_id, GConfEntry * entry,
                                         gpointer user_data);


/************************************************************************/
/* PRIVATE FUNCTIONS */
/************************************************************************/

/** Checks for errors
  *
  * @param err an error structure
  * @return TRUE if an error has been found, ELSE otherwise
  */
static gboolean is_gconf_error(GError ** err)
{
    if (*err != NULL) {
        ULOG_ERR("%s\n", (*err)->message);
        g_error_free(*err);
        *err = NULL;
        return TRUE;
    }

    return FALSE;
}

/** Resets proxy settings
  * 
  * @param gconfclient the gconf client
  * @param cnxn_id not used
  * @param entry a gconf entry
  * @param user_data not used
  */
static void
conf_proxy_reset_settings_cb(GConfClient * gconfclient,
                             guint cnxn_id, GConfEntry * entry,
                             gpointer user_data)
{
    gchar *tmp = NULL;
    xmlURIPtr uri = NULL;

    g_free(proxyname);
    proxyname = NULL;
    proxyport = 0;

    g_free(proxyusername);
    proxyusername = NULL;
    g_free(proxypassword);
    proxypassword = NULL;

    /* first check for a configured GNOME proxy */
    if (getBooleanConfValue(USE_PROXY)) {
        proxyname = getStringConfValue(PROXY_HOST);
        proxyport = getNumericConfValue(PROXY_PORT);
        debug2(DEBUG_CONF,
               "using GNOME configured proxy: \"%s\" port \"%d\"",
               proxyname, proxyport);
        if (getBooleanConfValue(PROXY_USEAUTH)) {
            proxyusername = getStringConfValue(PROXY_USER);
            proxypassword = getStringConfValue(PROXY_PASSWD);
        }
    } else {
        /* otherwise there could be a proxy specified in the environment the 
         * following code was derived from SnowNews' setup.c */
        if (g_getenv("http_proxy") != NULL) {
            /* The pointer returned by getenv must not be altered. What about 
             * mentioning this in the manpage of getenv? */
            debug0(DEBUG_CONF, "using proxy from environment");
            do {
                uri = xmlParseURI((const char *)g_getenv("http_proxy"));
                if (uri == NULL)
                    break;
                if (uri->server == NULL) {
                    xmlFreeURI(uri);
                    break;
                }
                proxyname = g_strdup(uri->server);
                proxyport = (uri->port == 0) ? 3128 : uri->port;
                if (uri->user != NULL) {
                    tmp = strtok(uri->user, ":");
                    tmp = strtok(NULL, ":");
                    if (tmp != NULL) {
                        proxyusername = g_strdup(uri->user);
                        proxypassword = g_strdup(tmp);
                    }
                }
                xmlFreeURI(uri);
            } while (FALSE);
        }
    }

    ui_htmlview_set_proxy(proxyname, proxyport, proxyusername, proxypassword);
    debug4(DEBUG_CONF, "Proxy settings are now %s:%d %s:%s",
           proxyname != NULL ? proxyname : "NULL", proxyport,
           proxyusername != NULL ? proxyusername : "NULL",
           proxypassword != NULL ? proxypassword : "NULL");
}


static void
configuration_changed_cb(GConfClient * gconfclient,
                         guint cnxn_id, GConfEntry * entry,
                         gpointer user_data)
{
    ULOG_DEBUG(">>> Configuration changed ! <<<");

    AppUIData *app_ui_data = NULL;

    if (!app_data || !app_data->app_ui_data)
        return;

    app_ui_data = app_data->app_ui_data;

    GConfValue *gval = gconf_entry_get_value(entry);
    if (gval) {
        size_t size = gconf_value_get_int(gval);
        ULOG_DEBUG("Resizing image cache to: %d mb", size);
        file_cache_resize(app_ui_data->img_cache,
                          all_cache_size[find_cache_size(size)] * 1024 *
                          1024);
    }
}

/** Saves the feedlist into a file
  *
  * @param user_data not used
  * @return always FALSE
  */
static gboolean conf_feedlist_schedule_save_cb(gpointer user_data)
{
    gboolean save_config = (gboolean) user_data;
    feedlist_save_timer = 0;
    if (save_config) {
        conf_feedlist_save_config();
    } else {
        conf_feedlist_save();
    }
    feedlist_save_timer_with_config = FALSE;
    return FALSE;
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

/* called once on startup */
void conf_init()
{
    int ualength = 0;
    const char *lang = NULL;

    g_assert(app_data != NULL);

    /* have to be called for multithreaded programs */
    xmlInitParser();

    /* the following code was copied from SnowNews and adapted to build a
     * Liferea user agent... */

    /* Construct the User-Agent string of Liferea. This is done here in
     * program init, because we need to do it exactly once and it will never
     * change while the program is running. */
    if (g_getenv("LANG") != NULL) {
        lang = g_getenv("LANG");
        /* e.g. Liferea/0.3.8 (Linux; de_DE; (http://liferea.sf.net/) */
        ualength =
            strlen("Liferea/") + strlen(VERSION) + 2 + strlen(lang) + 2 +
            strlen(OSNAME) + 2 + strlen(HOMEPAGE) + 2;
        useragent = g_malloc(ualength);
        snprintf(useragent, ualength, "Liferea/%s (%s; %s; %s)", VERSION,
                 OSNAME, lang, HOMEPAGE);
    } else {
        /* "Liferea/" + VERSION + "(" OS + "; " + HOMEPAGE + ")" */
        ualength =
            strlen("Liferea/") + strlen(VERSION) + 2 + strlen(OSNAME) + 2 +
            strlen("http://liferea.sf.net/") + 2;
        useragent = g_malloc(ualength);
        snprintf(useragent, ualength, "Liferea/%s (%s; %s)", VERSION,
                 OSNAME, HOMEPAGE);
    }

    /* initialize GConf gconfclient */
    gconfclient = gconf_client_get_default();
    config_check_first_time(gconfclient);

    app_data->gconf_client = gconfclient;
    gconf_client_add_dir(gconfclient, "/system/http_proxy",
                         GCONF_CLIENT_PRELOAD_NONE, NULL);

    gconf_client_notify_add(gconfclient, "/system/http_proxy",
                            conf_proxy_reset_settings_cb, NULL, NULL, NULL);

    gconf_client_add_dir(gconfclient, RSS_SETTINGS_IMAGE_CACHE_SIZE,
                         GCONF_CLIENT_PRELOAD_NONE, NULL);

    gconf_client_notify_add(gconfclient, RSS_SETTINGS_IMAGE_CACHE_SIZE,
                            configuration_changed_cb, NULL, NULL, NULL);


    /* Load settings into static buffers */
    conf_proxy_reset_settings_cb(NULL, 0, NULL, NULL);
}

/*----------------------------------------------------------------------*/
/* feed entry handling */
/*----------------------------------------------------------------------*/

gchar *conf_new_id()
{
    int i = 0;
    gchar *id = NULL, *filename = NULL;
    gboolean already_used = FALSE;

    id = g_new0(gchar, 10);
    do {
        for (i = 0; i < 7; i++)
            id[i] = (char) g_random_int_range('a', 'z');
        id[7] = '\0';

        filename =
            common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                                         id, NULL);
        already_used = g_file_test(filename, G_FILE_TEST_EXISTS);
        g_free(filename);
    } while (already_used);

    return id;
}

/*----------------------------------------------------------------------*/
/* config loading on startup */
/*----------------------------------------------------------------------*/

#define PRE_INSTALLED_DIR "/pre-installed/osso_rss_feed_reader/"
#define CONF_DIR_CREATE_PERMS GNOME_VFS_PERM_USER_ALL|GNOME_VFS_PERM_GROUP_ALL|GNOME_VFS_PERM_OTHER_ALL
#define CONF_COPY_OPTS GNOME_VFS_XFER_DEFAULT|GNOME_VFS_XFER_TARGET_DEFAULT_PERMS

struct conf_dircopy {
    gchar *path_dest;
    gchar *path_src;
};

static gboolean conf_copy_dircontent(const gchar * rel_path,
                                     GnomeVFSFileInfo * info,
                                     gboolean recursing_will_loop,
                                     gpointer data, gboolean * recurse)
{
    struct conf_dircopy *dircopy_data = (struct conf_dircopy *) data;
    GnomeVFSURI *src_uri = NULL, *dest_uri = NULL;
    gchar *src_uri_full_txt = NULL;
    gchar *dest_uri_full_txt = NULL;
    src_uri_full_txt = g_strconcat(dircopy_data->path_src, rel_path, NULL);
    src_uri = gnome_vfs_uri_new(src_uri_full_txt);

    dest_uri_full_txt = g_strconcat(dircopy_data->path_dest, rel_path, NULL);
    dest_uri = gnome_vfs_uri_new(dest_uri_full_txt);
    if (src_uri && dest_uri) {
        if (gnome_vfs_xfer_uri(src_uri, dest_uri,
                               CONF_COPY_OPTS,
                               GNOME_VFS_XFER_ERROR_MODE_ABORT,
                               GNOME_VFS_XFER_ERROR_MODE_ABORT, NULL,
                               NULL) != GNOME_VFS_OK) {
            gnome_vfs_uri_unref(src_uri);
            return FALSE;
        }
    } else {
        return FALSE;
    }
    gnome_vfs_uri_unref(src_uri);
    return TRUE;
}


static gboolean conf_restore_original_data(void)
{
    gchar *filename = NULL, *filename_src_uri = NULL, *filename_dest_uri =
        NULL;
    gchar *temp_direct_uri = NULL;
    GnomeVFSURI *src_uri = NULL, *dest_uri = NULL, *temp_uri = NULL;
    GnomeVFSResult comm_res = 0;

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S PRE_INSTALLED_DIR "rootdir"
                        G_DIR_SEPARATOR_S "feedlist.opml", PACKAGE_DATA_DIR);
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
        struct conf_dircopy *dircopy_data = NULL;
        gchar *directory_list[] =
            { G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S,
            G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
                "cache" G_DIR_SEPARATOR_S,
            G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
                "cache" G_DIR_SEPARATOR_S "favicons" G_DIR_SEPARATOR_S,
            G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
                "cache" G_DIR_SEPARATOR_S "feeds" G_DIR_SEPARATOR_S,
            NULL
        };
        gchar *cache_list[][2] = {
            {"file://" PACKAGE_DATA_DIR G_DIR_SEPARATOR_S PRE_INSTALLED_DIR
             G_DIR_SEPARATOR_S "favicons" G_DIR_SEPARATOR_S,
             G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
             "cache" G_DIR_SEPARATOR_S "favicons" G_DIR_SEPARATOR_S},
            {"file://" PACKAGE_DATA_DIR G_DIR_SEPARATOR_S PRE_INSTALLED_DIR
             G_DIR_SEPARATOR_S "feeds" G_DIR_SEPARATOR_S,
             G_DIR_SEPARATOR_S ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
             "cache" G_DIR_SEPARATOR_S "feeds" G_DIR_SEPARATOR_S},
            {NULL, NULL}
        };
	
        gint cur_index = 0;

        while (directory_list[cur_index]) {
            temp_direct_uri =
                g_strdup_printf("file://%s%s", g_get_home_dir(),
                                directory_list[cur_index]);
            temp_uri = gnome_vfs_uri_new(temp_direct_uri);
            comm_res =
                gnome_vfs_make_directory_for_uri(temp_uri,
                                                 CONF_DIR_CREATE_PERMS);
            if (!
                ((comm_res == GNOME_VFS_OK)
                 || (comm_res == GNOME_VFS_ERROR_FILE_EXISTS))) {
                g_free(filename);
                g_free(temp_direct_uri);
                gnome_vfs_uri_unref(temp_uri);
                return TRUE;
            }
            gnome_vfs_uri_unref(temp_uri);
            g_free(temp_direct_uri);
            cur_index++;
        }



        /*Copy the feedlist.opml */
        filename_src_uri = g_strconcat("file://", filename, NULL);
        filename_dest_uri =
            g_strdup_printf("file://%s" G_DIR_SEPARATOR_S
                            ".osso_rss_feed_reader" G_DIR_SEPARATOR_S
                            FEEDLIST_CONFIG_FILE, g_get_home_dir());
        src_uri = gnome_vfs_uri_new(filename_src_uri);
        dest_uri = gnome_vfs_uri_new(filename_dest_uri);
        if (gnome_vfs_xfer_uri(src_uri, dest_uri,
                               CONF_COPY_OPTS,
                               GNOME_VFS_XFER_ERROR_MODE_ABORT,
                               GNOME_VFS_XFER_ERROR_MODE_ABORT, NULL,
                               NULL) != GNOME_VFS_OK) {
            gnome_vfs_uri_unref(src_uri);
            gnome_vfs_uri_unref(dest_uri);
            g_free(filename);
            g_free(filename_dest_uri);
            g_free(filename_src_uri);
            return TRUE;
        }
        gnome_vfs_uri_unref(src_uri);
        gnome_vfs_uri_unref(dest_uri);
        g_free(filename);
        g_free(filename_dest_uri);
        g_free(filename_src_uri);

        cur_index = 0;
        while (cache_list[cur_index][0]) {
            filename_src_uri = cache_list[cur_index][0];
            filename_dest_uri =
                g_strdup_printf("file://%s%s", g_get_home_dir(),
                                cache_list[cur_index][1]);
            dircopy_data = g_new0(struct conf_dircopy, 1);
            dircopy_data->path_src = filename_src_uri;
            dircopy_data->path_dest = filename_dest_uri;
            if (gnome_vfs_directory_visit
                (filename_src_uri, GNOME_VFS_FILE_INFO_DEFAULT,
                 GNOME_VFS_DIRECTORY_VISIT_DEFAULT, conf_copy_dircontent,
                 (gpointer) dircopy_data) != GNOME_VFS_OK) {
                g_free(filename_dest_uri);
                g_free(dircopy_data);
                return TRUE;
            }
            g_free(filename_dest_uri);
            g_free(dircopy_data);
            cur_index++;
        }
    }
    return FALSE;
}

void conf_load_subscriptions(void)
{
    gchar *filename = NULL;

    folderPtr root_folder = NULL;

    feedlistLoading = TRUE;
    root_folder = ui_feedlist_get_root_folder();
    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S FEEDLIST_CONFIG_FILE,
                        g_get_home_dir());
#if 0
    if (!g_file_test(filename, G_FILE_TEST_EXISTS)) {
        conf_restore_original_data();
    }
#endif    
    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
        import_OPML_feedlist(filename, root_folder, FALSE, TRUE,
                             RSS_CONFIG_MODE_FULL);
        g_free(filename);
        filename =
            g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                            G_DIR_SEPARATOR_S FEEDLIST_STATE_FILE,
                            g_get_home_dir());

        if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
            import_OPML_feedlist(filename, root_folder, FALSE, TRUE,
                                 RSS_CONFIG_MODE_STATE);
        }
        g_free(filename);
        filename = NULL;
    } else {
        g_message("Not exist %s!!!!", filename);


    }
    if (filename) {
        g_free(filename);
    }

    feedlistLoading = FALSE;
}

//return FALSE on error
gboolean conf_feedlist_save_config()
{
    gchar *filename = NULL;
    DMSG("Entering %s\n", __FUNCTION__);
    if (feedlistLoading)
        return TRUE;

    debug_enter("conf_feedlist_save_config");
    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S FEEDLIST_CONFIG_FILE,
                        common_get_cache_path());
    if (0 != export_OPML_feedlist(filename, TRUE)) {
        ui_show_save_nodevicemem_dialog();
        return FALSE;
    }
    /*
     * make the latest cache conform to latest config
     * (not sure - maybe that would happen without calling)
     * conf_feedlist_save here explicitly...
     */

    g_free(filename);
    DMSG("Leaving %s\n", __FUNCTION__);
    debug_exit("conf_feedlist_save");
    return conf_feedlist_save();
}

//returns FALSE on error
gboolean conf_feedlist_save()
{
    gchar *filename = NULL, *filename_real = NULL,*filename_test=NULL;
    gboolean res = FALSE;
    DMSG("Entering %s\n", __FUNCTION__);

    if (feedlistLoading) {
        return TRUE;
    }

    if (feedlist_save_timer) {

        g_source_remove(feedlist_save_timer);
        feedlist_save_timer = 0;
        if (feedlist_save_timer_with_config) {
            return conf_feedlist_save_config();
        }
    }

    debug_enter("conf_feedlist_save");
    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "feedlist.opml~",
                        common_get_cache_path());

    filename_test =
	        g_strdup_printf("%s" G_DIR_SEPARATOR_S FEEDLIST_CONFIG_FILE,
				                        common_get_cache_path());

    if (0 != export_OPML_feedlist(filename_test, TRUE)) {
        ui_show_save_nodevicemem_dialog();
	return FALSE;
    }
    
    if (0 == export_OPML_feedlist(filename, TRUE)) {
        filename_real =
            g_strdup_printf("%s" G_DIR_SEPARATOR_S FEEDLIST_STATE_FILE,
                            common_get_cache_path());
        if (rename(filename, filename_real) < 0)
            g_warning("Error renaming %s to %s\n", filename, filename_real);
        else
            res = TRUE;
        g_free(filename_real);
    } else {
        ui_show_save_nodevicemem_dialog();
    }
    g_free(filename);
    g_free(filename_test);	
    debug_exit("conf_feedlist_save");
    DMSG("Leaving %s\n", __FUNCTION__);
    return res;
}

void conf_feedlist_schedule_save(gboolean save_config)
{
    if (!feedlistLoading) {
        if (feedlist_save_timer && (save_config && !feedlist_save_timer_with_config)) { //if there is a timer, but without config saving, we delete that, and we add one, which saves it with the config
            g_source_remove(feedlist_save_timer);
            feedlist_save_timer = 0;
        }
        if (!feedlist_save_timer) {
            debug0(DEBUG_CONF, "Scheduling feedlist save");
            feedlist_save_timer_with_config = save_config;
            feedlist_save_timer =
                g_timeout_add(5000, conf_feedlist_schedule_save_cb,
                              (gpointer) save_config);
        }
    }
}

/* returns true if namespace is enabled in configuration */
gboolean getNameSpaceStatus(gchar * nsname)
{
    GConfValue *value = NULL;
    gchar *gconfpath = NULL;
    gboolean status = FALSE;

    g_assert(NULL != nsname);
    gconfpath = g_strdup_printf("%s/ns_%s", RSS_SETTINGS, nsname);
    value = gconf_client_get_without_default(gconfclient, gconfpath, NULL);
    if (NULL == value) {
        ULOG_INFO("RSS namespace %s not yet configured! Activating...",
                  nsname);
        setNameSpaceStatus(nsname, TRUE);
        status = TRUE;
    } else {
        status = gconf_value_get_bool(value);
    }
    g_free(gconfpath);
    if (value)
        gconf_value_free(value);
    return status;
}

/* used to enable/disable a namespace in configuration */
void setNameSpaceStatus(gchar * nsname, gboolean enable)
{
    gchar *gconfpath = NULL;

    g_assert(NULL != nsname);

    gconfpath = g_strdup_printf("%s/ns_%s", RSS_SETTINGS, nsname);
    setBooleanConfValue(gconfpath, enable);
    g_free(gconfpath);
}

/*----------------------------------------------------------------------*/
/* generic configuration access methods */
/*----------------------------------------------------------------------*/

gboolean checkConfValueExist(gchar * valuename)
{
    GConfValue *value = NULL;
    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL == value) {
        return FALSE;           // value not initialized
    } else {
        gconf_value_free(value);
        return TRUE;
    }
}

void setStringConfValue(gchar * valuename, gchar * value)
{
    GError *err = NULL;
    GConfValue *gcv = NULL;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    if ((gcv = gconf_value_new(GCONF_VALUE_STRING)))
    {
        gconf_value_set_string(gcv, value);
        gconf_client_set(gconfclient, valuename, gcv, &err);
        gconf_value_free(gcv);
        is_gconf_error(&err);
    }
}

gchar *getStringConfValue(gchar * valuename)
{
    GConfValue *value = NULL;
    gchar *result = NULL;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL == value) {
        result = g_strdup("");
    } else {
        result = (gchar *) g_strdup(gconf_value_get_string(value));
        gconf_value_free(value);
    }

    return result;
}

gint getNumericConfValueDef(gchar * valuename, gint def)
{
    GConfValue *value = NULL;
    gint result = def;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL != value) {
        result = gconf_value_get_int(value);
        gconf_value_free(value);
    }
    debug2(DEBUG_CONF, "Reading %s to %d", valuename, result);
    return result;
}

void setNumericConfValue(gchar * valuename, gint value)
{
    gconf_client_set_int(gconfclient, valuename, value, NULL);
}

gint getNumericConfValue(gchar * valuename)
{
    return gconf_client_get_int(gconfclient, valuename, NULL);
}

void setBooleanConfValue(gchar * valuename, gboolean value)
{
    gconf_client_set_bool(gconfclient, valuename, value, NULL);
}

gboolean getBooleanConfValue(gchar * valuename)
{
    return gconf_client_get_bool(gconfclient, valuename, NULL);
}
