/**
   @file rss-feed-applet-button.c

   A small button-like widget used on the home RSS applet.

   Based on fmp-image-button.c, part of the Fremantle Media Player, so
   probably it have more features that required, but works fine right
   now. Ie: the subtitle was removed. Probably the title could be
   removed as well.

   Copyright (c) 2004-2009 Nokia. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA
*/

#include <hildon/hildon.h>

#include "rss-feed-applet-button.h"
#include "debug.h"


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Prototypes                                                                 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
static void _set_image_visibility(RssFeedAppletButton *self);
static void _event_show(RssFeedAppletButton *self, gpointer user_data);
static gboolean _event_button_press(GtkWidget      *widget,
                                    GdkEventButton *event,
                                    RssFeedAppletButton *self);
static gboolean _event_button_release(GtkWidget      *widget,
                                      GdkEventButton *event,
                                      RssFeedAppletButton *self);
static gboolean _event_enter_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletButton   *self);
static gboolean _event_leave_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletButton   *self);
static gboolean _event_img_up_expose(GtkWidget *widget,
                                     GdkEventExpose *event,
                                     gpointer image_button);
static gboolean _event_img_down_expose(GtkWidget *widget,
                                       GdkEventExpose *event,
                                       gpointer image_button);



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Variables                                                                  */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

enum
    {
        SIGNAL_CLICKED,
        SIGNAL_PRESSED,
        SIGNAL_RELEASED,
        SIGNAL_COUNT
    };

static guint signals[SIGNAL_COUNT];



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Construction                                                               */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

G_DEFINE_TYPE(RssFeedAppletButton, rss_feed_applet_button, GTK_TYPE_ALIGNMENT);
//G_DEFINE_DYNAMIC_TYPE(RssFeedAppletButton, rss_feed_applet_button, GTK_TYPE_ALIGNMENT);

void rss_feed_applet_button_register_types(GTypeModule *module)
{
    //rss_feed_applet_button_register_type(module);
}

/*
  static void
  rss_feed_applet_button_class_finalize (RssFeedAppletButtonClass *klass)
  {
  }
*/
static void rss_feed_applet_button_finalize(GObject *object)
{
    ULOG_DEBUG(__FUNCTION__);
    if(RSS_FEED_APPLET_BUTTON(object)->icon_up)
        {
            g_object_unref(RSS_FEED_APPLET_BUTTON(object)->icon_up);
        }
    if(RSS_FEED_APPLET_BUTTON(object)->icon_down)
	{
            g_object_unref(RSS_FEED_APPLET_BUTTON(object)->icon_down);
	}

    /* The image ones are freed with the gtk_image */

    G_OBJECT_CLASS(rss_feed_applet_button_parent_class)->finalize(object);
}

static void rss_feed_applet_button_class_init(RssFeedAppletButtonClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
    /*GtkVBoxClass *parent_class = GTK_VBOX_CLASS(klass);*/
    object_class->finalize = rss_feed_applet_button_finalize;
    ULOG_DEBUG(__FUNCTION__);

    ULOG_INFO("RssFeedAppletButton: Signal: Register Clicked");
    signals[SIGNAL_CLICKED] = g_signal_new("clicked",
                                           G_OBJECT_CLASS_TYPE(object_class),
                                           G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                           0, NULL, NULL,
                                           g_cclosure_marshal_VOID__VOID,
                                           G_TYPE_NONE, 0);

    signals[SIGNAL_PRESSED] = g_signal_new("button-pressed",
                                           G_OBJECT_CLASS_TYPE(object_class),
                                           G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                           0, NULL, NULL,
                                           g_cclosure_marshal_VOID__VOID,
                                           G_TYPE_NONE, 0);

    signals[SIGNAL_RELEASED] = g_signal_new("button-released",
                                            G_OBJECT_CLASS_TYPE(object_class),
                                            G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                            0, NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
}

static void rss_feed_applet_button_init(RssFeedAppletButton *self)
{
    GtkWidget *vbox;
    GtkWidget *vbox_img;
    ULOG_DEBUG(__FUNCTION__);

    ULOG_INFO("RssFeedAppletButton: Init start");

    gtk_alignment_set(GTK_ALIGNMENT(self), 0.5, 0.5, 0, 0);

    vbox = gtk_vbox_new(0, 0);
    vbox_img = gtk_vbox_new(0, 0);
    self->img_up = gtk_image_new();
    self->img_down = gtk_image_new();
    self->title = gtk_label_new(NULL);
    self->align_title = gtk_alignment_new(0.5, 0.5, 1, 1);
    self->img_event_box = gtk_event_box_new();
    self->img_align = gtk_alignment_new(0.5, 0.5, 0, 0);
    self->icon_xalign = 0.5;
    self->icon_yalign = 0.5;
    self->icon_xpadding = 0;
    self->icon_ypadding = 0;

    /* Setup Image Eventbox. */
    ULOG_INFO("RssFeedAppletButton: Setup Eventbox");
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->img_event_box),
                                     FALSE);
    gtk_container_add(GTK_CONTAINER(self->img_event_box), self->img_align);
    gtk_container_add(GTK_CONTAINER(self->img_align), vbox_img);
    gtk_box_pack_start(GTK_BOX(vbox_img), self->img_up, 0, 0, 0);
    gtk_box_pack_start(GTK_BOX(vbox_img), self->img_down, 0, 0, 0);

    /* Setup title alignments.
       - The purpose of aligments here is to add 4px padding on top
       of both titles with the possibility of hiding both title widgets
       and their alignments if titles are not in use.
       FIXME: this in inherited from the FMPImageButton, consider remove it */
    ULOG_INFO("RssFeedAppletButton: Setup alignments");
    gtk_container_add(GTK_CONTAINER(self->align_title), self->title);
    gtk_alignment_set_padding(GTK_ALIGNMENT(self->align_title),
                              4, 0, 0, 0);
    /* Fill main VBox. */
    ULOG_INFO("RssFeedAppletButton: Fill VBox");
    gtk_box_pack_start(GTK_BOX(vbox), self->img_event_box, 0, 0, 0);
    gtk_box_pack_start(GTK_BOX(vbox), self->align_title, 0, 0, 0);
    gtk_container_add(GTK_CONTAINER(self), vbox);

    /* Attach signals. */
    ULOG_INFO("RssFeedAppletButton: Connecting signals");
    g_signal_connect(G_OBJECT(self), "show", G_CALLBACK(_event_show), NULL);
    g_signal_connect(G_OBJECT(self->img_event_box), "button-press-event",
                     G_CALLBACK(_event_button_press), self);
    g_signal_connect(G_OBJECT(self->img_event_box), "button-release-event",
                     G_CALLBACK(_event_button_release), self);
    g_signal_connect(G_OBJECT(self->img_event_box), "enter-notify-event",
                     G_CALLBACK(_event_enter_notify), self);
    g_signal_connect(G_OBJECT(self->img_event_box), "leave-notify-event",
                     G_CALLBACK(_event_leave_notify), self);
    g_signal_connect_after(G_OBJECT(self->img_up), "expose-event",
                           G_CALLBACK(_event_img_up_expose), self);
    g_signal_connect_after(G_OBJECT(self->img_down), "expose-event",
                           G_CALLBACK(_event_img_down_expose), self);

    /* Set visibility. */
    ULOG_INFO("RssFeedAppletButton: Setting widget visibility");
    gtk_widget_set_no_show_all(GTK_WIDGET(vbox), TRUE);
    gtk_widget_show(vbox);
    gtk_widget_show(vbox_img);
    gtk_widget_show(self->img_up);
    gtk_widget_hide(self->img_down);
    gtk_widget_show(self->title);
    gtk_widget_hide(self->align_title);
    gtk_widget_show(self->img_event_box);
    gtk_widget_show(self->img_align);
    ULOG_INFO("RssFeedAppletButton: Init end");
}

GtkWidget *rss_feed_applet_button_new()
{
    RssFeedAppletButton *self;
    ULOG_DEBUG(__FUNCTION__);

    self = g_object_new(RSS_TYPE_FEED_APPLET_BUTTON, NULL);
    return GTK_WIDGET(self);
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Private Functions                                                          */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

/**
 * Set visibility of image widgets.
 */
static void _set_image_visibility(RssFeedAppletButton *self)
{
    gboolean show_down_img = FALSE;
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    if (self->togglable && self->mouse_button_is_down &&
        self->button_is_toggled && self->cursor_inside_button)
        {
            show_down_img = FALSE;
        }
    else if (self->togglable && self->mouse_button_is_down &&
             self->button_is_toggled && !self->cursor_inside_button)
        {
            show_down_img = TRUE;
        }
    else if (self->togglable && self->button_is_toggled &&
             !self->mouse_button_is_down)
        {
            show_down_img = TRUE;
        }
    else if (self->mouse_button_is_down && self->cursor_inside_button)
        {
            show_down_img = TRUE;
        }

    if (show_down_img)
        {
            if (GTK_WIDGET_VISIBLE(self->img_up))
                {
                    gtk_widget_hide(self->img_up);
                }
            if (!GTK_WIDGET_VISIBLE(self->img_down))
                {
                    gtk_widget_show(self->img_down);
                }
        }
    else
        {
            if (GTK_WIDGET_VISIBLE(self->img_down))
                {
                    gtk_widget_hide(self->img_down);
                }
            if (!GTK_WIDGET_VISIBLE(self->img_up))
                {
                    gtk_widget_show(self->img_up);
                }
        }
}

/**
 * Set visibility of title widgets.
 */
static void _set_title_visibility(RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    if (self->title_is_set)
        {
            gtk_widget_show(self->align_title);
        }
    else
        {
            gtk_widget_hide(self->align_title);
        }
}

/**
 * Manage state of button_is_being_pressed variable. 
 * Sends signals when state changes. 
 */
static void _button_is_being_pressed(RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    gboolean is_pressed;
    gboolean state_changed;

    is_pressed = self->mouse_button_is_down && self->cursor_inside_button;
    state_changed = is_pressed != self->button_is_being_pressed;
    self->button_is_being_pressed = is_pressed;

    // Emit signals if button state has changed.
    if (state_changed)
        {
            if (is_pressed)
                {
                    g_signal_emit(self, signals[SIGNAL_PRESSED], 0);
                }
            else
                {
                    g_signal_emit(self, signals[SIGNAL_RELEASED], 0);
                }
        }
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Public Functions                                                           */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

/**
 * Align RssFeedAppletButton.
 *
 * By default RssFeedAppletButton will center it self if it is given larges
 * size allocation than its size request. You can use this funtion to
 * override that behaviour.
 *
 * Even though RssFeedAppletButton is based on GtkAlignment, please do not call
 * GtkAlignment alignment functions directly. Using those might break
 * RssFeedAppletButton.
 *
 * @param xalign The horizontal alignment of the child widget,
 *               from 0 (left) to 1 (right).
 * @param yalign The vertical alignment of the child widget,
 *               from 0 (top) to 1 (bottom).
 */
void rss_feed_applet_button_align(RssFeedAppletButton *self,
                            gfloat          xalign,
                            gfloat          yalign)
{
    gtk_alignment_set(GTK_ALIGNMENT(self), xalign, yalign, 0, 0);
}


/**
 * Expand the area which responds to user touch.
 *
 * By default the area which responds to user clicks is exactly the same size as
 * the image being shown. You can use this function to expand that area.
 */
void rss_feed_applet_button_expand_click_area(RssFeedAppletButton *self,
                                        guint top,
                                        guint bottom,
                                        guint left,
                                        guint right)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    gtk_alignment_set_padding(GTK_ALIGNMENT(self->img_align),
                              top, bottom, left, right);
}

/**
 * Set title of button using default font.
 */
void rss_feed_applet_button_set_title(RssFeedAppletButton *self,
                                const gchar    *title)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    if (title)
        {
            self->title_is_set = TRUE;
            gtk_label_set_text(GTK_LABEL(self->title), title);
        }
    else
        {
            self->title_is_set = FALSE;
        }
    _set_title_visibility(self);
}

/**
 * Set title using pango markup language.
 *
 * @b Example
 * @code
 * rss_feed_applet_button_set_title_markup(RSS_FEED_APPLET_BUTTON(button),
 *                                   "<span size='12000'>title</span>");
 * @endcode
 *
 * @param title Title to show, or NULL to hide title.
 */
void rss_feed_applet_button_set_title_markup(RssFeedAppletButton *self,
                                       const gchar    *title)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    if (title)
        {
            self->title_is_set = TRUE;
            gtk_label_set_markup(GTK_LABEL(self->title), title);
        }
    else
        {
            self->title_is_set = FALSE;
        }
    _set_title_visibility(self);
}

void rss_feed_applet_button_set_image_from_file(RssFeedAppletButton *self,
                                                const gchar    *image_up,
                                                const gchar    *image_down)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    g_assert(image_up != NULL);
    g_assert(image_down != NULL);

    gtk_image_set_from_file(GTK_IMAGE(self->img_up), image_up);
    gtk_image_set_from_file(GTK_IMAGE(self->img_down), image_down);
}

void rss_feed_applet_button_set_image_from_pixbuf(RssFeedAppletButton *self,
                                                  GdkPixbuf      *image_up,
                                                  GdkPixbuf      *image_down)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    g_return_if_fail (GDK_IS_PIXBUF(image_up));
    g_return_if_fail (GDK_IS_PIXBUF(image_down));

    gtk_image_set_from_pixbuf(GTK_IMAGE(self->img_up), image_up);
    gtk_image_set_from_pixbuf(GTK_IMAGE(self->img_down), image_down);
}

void rss_feed_applet_button_set_icon_from_pixbuf(RssFeedAppletButton *self,
                                                 GdkPixbuf      *icon_up,
                                                 GdkPixbuf      *icon_down)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    if(GDK_IS_PIXBUF(self->icon_up))
        g_object_unref(self->icon_up);
    if(GDK_IS_PIXBUF(self->icon_down))
        g_object_unref(self->icon_down);

    if (GDK_IS_PIXBUF (icon_up))
        g_object_ref(icon_up);

    if (GDK_IS_PIXBUF (icon_down))
        g_object_ref(icon_down);

    self->icon_up = icon_up;
    self->icon_down = icon_down;
    gtk_widget_queue_draw(GTK_WIDGET(self));
}

/**
 * Set minimum width of RssFeedAppletButton.
 *
 * By default the width of RssFeedAppletButton is defined by the images used, and
 * so the maximum width of the text titles used is also set by the images.
 *
 * Sometimes, however, it is necessary to use titles wider than the buttons, in
 * which case you can use this function to make RssFeedAppletButton larger.
 */
void rss_feed_applet_button_set_min_width(RssFeedAppletButton *self,
                                    gint            width)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->min_button_width = width;
}

/**
 * Enable or disable toggle button behaviour.
 *
 * @param toggled TRUE if togglable, FALSE if not.
 */
void rss_feed_applet_button_set_togglable(RssFeedAppletButton *self,
                                    gboolean togglable)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->togglable = togglable;
    if (togglable == FALSE)
        {
            self->button_is_toggled = FALSE;
        }
    _set_image_visibility(self);
}

/**
 * Get togglable state.
 *
 * @return TRUE if togglable, FALSE if not.
 */
gboolean rss_feed_applet_button_get_togglable(RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    return self->togglable;
}

/**
 * Set button toggle state.
 *
 * @param toggled TRUE if toggled, FALSE if not.
 */
void rss_feed_applet_button_set_toggle(RssFeedAppletButton *self,
                                 gboolean toggled)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    g_assert(self->togglable == TRUE);

    self->button_is_toggled = toggled;
    _set_image_visibility(self);
}

/**
 * Get button toggle state.
 *
 * @return TRUE if toggled, FALSE if not.
 */
gboolean rss_feed_applet_button_get_toggle(RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    g_assert(self->togglable == TRUE);
    return self->button_is_toggled;
}

/**
 * Get button pressed state.
 *
 * @return TRUE if pressed, FALSE if not.
 */
gboolean rss_feed_applet_button_is_beeing_pressed(RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));
    return self->button_is_being_pressed;
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Signals                                                                    */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

static void _event_show(RssFeedAppletButton *self, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    gtk_widget_show_all(GTK_WIDGET(self));
    _set_image_visibility(self);
}

static gboolean _event_button_press(GtkWidget      *widget,
                                    GdkEventButton *event,
                                    RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->mouse_button_is_down = TRUE;
    _set_image_visibility(self);
    _button_is_being_pressed(self);
    return FALSE;
}

static gboolean _event_button_release(GtkWidget      *widget,
                                      GdkEventButton *event,
                                      RssFeedAppletButton *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->mouse_button_is_down = FALSE;

    if (self->togglable && self->cursor_inside_button)
        {
            self->button_is_toggled = !(self->button_is_toggled);
        }

    _set_image_visibility(self);
    _button_is_being_pressed(self);

    if (self->cursor_inside_button)
        {
            ULOG_INFO("RssFeedAppletButton: Signal: Emitting Clicked");
            g_signal_emit(self, signals[SIGNAL_CLICKED], 0);
        }
    return FALSE;
}

static gboolean _event_enter_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletButton   *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->cursor_inside_button = TRUE;
    _set_image_visibility(self);
    _button_is_being_pressed(self);
    return FALSE;
}

static gboolean _event_leave_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletButton   *self)
{
    ULOG_DEBUG(__FUNCTION__);
    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    self->cursor_inside_button = FALSE;
    _set_image_visibility(self);
    _button_is_being_pressed(self);
    return FALSE;
}

static gboolean _event_img_up_expose(GtkWidget *widget,
                                     GdkEventExpose *event,
                                     gpointer self)
{
    ULOG_INFO("RssFeedAppletButton: img_up expose handler");
    RssFeedAppletButton *image_button;
    cairo_t *cr;

    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    image_button = RSS_FEED_APPLET_BUTTON(self);

    if(GDK_IS_PIXBUF(image_button->icon_up))
	{
            ULOG_INFO("RssFeedAppletButton: DRAWING ICON");
            cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
            gdk_cairo_region (cr, event->region);
            cairo_clip (cr);

            cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
            gdk_cairo_set_source_pixbuf(cr, image_button->icon_up,
                                        widget->allocation.x + image_button->icon_xpadding + (widget->allocation.width
                                                              -gdk_pixbuf_get_width(image_button->icon_up))*image_button->icon_xalign,
                                        widget->allocation.y + image_button->icon_ypadding + (widget->allocation.height    
                                                              -gdk_pixbuf_get_height(image_button->icon_up))*image_button->icon_yalign);

            cairo_paint (cr);
	   cairo_destroy(cr);		
	}
    return FALSE;
}

static gboolean _event_img_down_expose(GtkWidget *widget,
                                       GdkEventExpose *event,
                                       gpointer self)
{
    ULOG_INFO("RssFeedAppletButton: img_up expose handler");
    RssFeedAppletButton *image_button;
    cairo_t *cr;

    g_assert(RSS_IS_FEED_APPLET_BUTTON(self));

    image_button = RSS_FEED_APPLET_BUTTON(self);

    if(GDK_IS_PIXBUF(image_button->icon_down))
	{
            ULOG_INFO("RssFeedAppletButton: DRAWING ICON");
            cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
            gdk_cairo_region (cr, event->region);
            cairo_clip (cr);

            cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
            gdk_cairo_set_source_pixbuf(cr, image_button->icon_down,
                                        widget->allocation.x + image_button->icon_xpadding + (widget->allocation.width			    
                                                              -gdk_pixbuf_get_width(image_button->icon_down))*image_button->icon_xalign, 
                                        widget->allocation.y + image_button->icon_ypadding + (widget->allocation.height					
                                                              -gdk_pixbuf_get_height(image_button->icon_down))*image_button->icon_yalign);

            cairo_paint (cr);
	   cairo_destroy(cr);		
	}
    return FALSE;
}

void rss_feed_applet_button_set_icon_align(RssFeedAppletButton *self, gfloat xalign, gfloat yalign)
{
    g_return_if_fail(RSS_IS_FEED_APPLET_BUTTON(self));

    if(xalign < 0)
        self->icon_xalign = 0;
    else if(xalign > 1)
        self->icon_xalign = 1;
    else self->icon_xalign = xalign;

    if(yalign < 0)
        self->icon_yalign = 0;
    else if(yalign > 1)
        self->icon_yalign = 1;
    else self->icon_yalign = yalign;
}

void rss_feed_applet_button_set_icon_padding(RssFeedAppletButton *self, gint xpadding, gint ypadding)
{
    g_return_if_fail(RSS_IS_FEED_APPLET_BUTTON(self));
    self->icon_xpadding = xpadding;
    self->icon_ypadding = ypadding;
}
