/**
    @file applet_cache.c

    Applet cache functionality.

    Copyright (c) 2004-2005 Nokia. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#define DBUS_API_SUBJECT_TO_CHANGE

#include <stdlib.h>
#include <sys/stat.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <glib.h>
#include <string.h>
#include <hildon/hildon-defines.h>

#include <osso-rss-feed-reader/cache_handling.h>
#include <osso-rss-feed-reader/cache_favicon.h>

#include "debug.h"

#include "applet_cache.h"

#include "debug.h"


/** Loads a favicon for a feed
  *
  * @param hl a struct where a pointer to the favicon is stored
  * @param id the filename of the cache file. This is identical
  *           to the feed id, so it can be used to find
  *           the favicon
  * @return FALSE in case of errors
  */
static gboolean
favicon_load(HeadLine * hl, gchar * id)
{
    hl->icon=rsslib_favicon_load(id);

    if (hl->icon != NULL)
        return TRUE;
    else
        return FALSE;
}

/** A function for comparing headlines
  *
  * @param a the first headline to compare
  * @param b the second headline to compare
  * @return the difference between the times
  */
static gint
time_order(gconstpointer a, gconstpointer b)
{
    return ((HeadLine *) b)->time - ((HeadLine *) a)->time;
}


/** Read a feed item
  *
  * @param cache the cache file
  * @param feed the feed to read from
  */
static HeadLine *
cache_read_feed_item(rsslib_cache_item_data *item_data, gchar * feed)
{
    HeadLine *hl = g_new0(HeadLine, 1);
    hl->title =item_data->item_title;
    hl->description =item_data->item_description;
    hl->url = item_data->item_source;
    hl->nr =item_data->nr;
    hl->time = item_data->item_time;
    hl->feed =g_strdup(feed);
    return hl;
    
}

static void applet_free_unneeded_itemdata(rsslib_cache_item_data *item_data)
{
    
    GSList *enclosure_metadata=NULL,*enclosure_metadata_start=NULL;
    
    if (item_data->item_real_source_title)
        g_free(item_data->item_real_source_title);
    
    if (item_data->item_real_source_url)
        g_free(item_data->item_real_source_url);
    
    if (item_data->item_id)
        g_free(item_data->item_id);
    
    if (item_data->enclosure_metadata)
    {
        struct enclosure_attribute *cur_attr=NULL;
        enclosure_metadata_start=enclosure_metadata=item_data->enclosure_metadata;
        while(enclosure_metadata)
        {
            cur_attr=enclosure_metadata->data;
            if (cur_attr->ul)
            {
                g_free(cur_attr->ul);
            }
            if (cur_attr->type)
                g_free(cur_attr->type);
            g_free(cur_attr);
            enclosure_metadata=g_slist_next(enclosure_metadata);
        }
        g_slist_free(enclosure_metadata_start);
    }
    
}

/** Load a feed
  *
  * @param filename the file to read from
  * @return the list of items
  */
static GList *
applet_cache_feed_load(gchar * feed)
{
    rsslib_cache_init_data *cache_init_data=NULL;
    rsslib_cache_item_data *item_data=NULL;
    GList *items = NULL;
    GdkPixbuf *pixbuf = NULL;
    gboolean loading_succeeded = TRUE;

    /* ULOG_DEBUG("%s reading feed %s", __FUNCTION__, filename); */
    
    cache_init_data=rsslib_cache_open_file(feed);
    if (cache_init_data!=NULL) {
        while ((item_data=rsslib_cache_get_next_item(cache_init_data))!=NULL) {
            HeadLine *hl = cache_read_feed_item(item_data,feed);
            if (hl != NULL)
            {
                if (loading_succeeded)
                {
                    if (pixbuf != NULL)
                    {
                        hl->icon = pixbuf;
                        g_object_ref(hl->icon);
                    } else  {
                        loading_succeeded = favicon_load(hl, feed);
                    	pixbuf = hl->icon;
    				}
                }
                items = g_list_append(items, hl);
            }
            applet_free_unneeded_itemdata(item_data);
            g_free(item_data);
            item_data=NULL;
        }
        rsslib_cache_init_data_free(cache_init_data);
    } 
    return items;
}

/** Read the number of unread posts and the time of the last refresh
  *
  * @param lifereaUserPath the user path
  * @param applet the applet structure
  */
/* static void general_load(gchar * lifereaUserPath, AppletInfo * applet) {
 * gchar *filename; return; g_assert(applet != NULL); filename =
 * g_strdup_printf("%s" G_DIR_SEPARATOR_S "general", lifereaUserPath); FILE
 * *cache = fopen(filename, "rb"); g_free(filename); if (NULL == cache) {
 * applet->unread = 0; applet->updated = 0; return; } applet->unread =
 * cache_read_long(cache); applet->updated = cache_read_long(cache);
 * fclose(cache); } */

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

void
test_if_cache_directory_exists()
{
    gchar *lifereaUserPath = NULL;

    lifereaUserPath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
                        g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR))
    {
        if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR))
        {
            ULOG_ERR("Cannot create cache directory %s!", lifereaUserPath);
        }
    }
    g_free(lifereaUserPath);
}


GList *
feed_load_all( time_t *updated )
{
    
    GList *items = NULL;
    DIR *cacheDir = NULL;
    struct dirent *cacheFile = NULL;
    FILE *file = NULL;
    long int filesize;

    gchar *cachePath = NULL;
    gchar *feedCachePath = NULL;
    gchar *lifereaUserPath = NULL;
    gchar *feedlistPath = NULL;
    gchar *rss = NULL;
    gchar *slpt = NULL;
    long lpt;

    lifereaUserPath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
                        g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR))
    {
        if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR))
        {
            ULOG_ERR("Cannot create cache directory %s!", lifereaUserPath);
        }
    }

    cachePath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "cache", lifereaUserPath);
    if (!g_file_test(cachePath, G_FILE_TEST_IS_DIR))
    {
        if (0 != mkdir(cachePath, S_IRUSR | S_IWUSR | S_IXUSR))
        {
            ULOG_ERR("Cannot create cache directory %s!", cachePath);
        }
    }

    feedCachePath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "feeds", cachePath);
    if (!g_file_test(feedCachePath, G_FILE_TEST_IS_DIR))
    {
        if (0 != mkdir(feedCachePath, S_IRUSR | S_IWUSR | S_IXUSR))
        {
            ULOG_ERR("Cannot create cache directory %s!", feedCachePath);
        }
    }

    feedlistPath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "feedlist.opml",
                        lifereaUserPath);
    file = fopen(feedlistPath, "r");
    if (file)
    {
        fseek(file, 0, SEEK_END);
        filesize = ftell(file);
        fseek(file, 0, SEEK_SET);
        if (filesize > 0 && filesize < 1024 * 1024)
        {
            rss = g_malloc(filesize + 1);
            fread(rss, filesize, 1, file);
            rss[filesize] = 0;
        }
        fclose(file);
    }
    g_free(feedlistPath);
    if (!rss)
        rss = g_strdup("");

    // general_load(lifereaUserPath, applet);

    cacheDir = opendir(feedCachePath);
    if (NULL == cacheDir)
    {
        g_free(cachePath);
        g_free(feedCachePath);
        g_free(lifereaUserPath);
        g_free(rss);
        return NULL;
    }
    TDB("=========================");
    *updated = 0;
    while ((cacheFile = readdir(cacheDir)) != NULL)
    {
        gchar *path = g_strdup_printf("%s" G_DIR_SEPARATOR_S "%s",
                                      feedCachePath,
                                      cacheFile->d_name);
        if (g_file_test(path, G_FILE_TEST_IS_REGULAR))
        {
            if ((slpt = strstr(rss, cacheFile->d_name)))
            {
                if ((slpt = strstr(slpt, "lastPollTime"))
                    && (slpt = strchr(slpt, '\"')))
                {
                    lpt = atol(slpt + 1);

                    if ((*updated < (time_t) lpt))
                        *updated = (time_t) lpt;
                }

                TDB("Parsing feed: %s", cacheFile->d_name);
                items =
                    g_list_concat(items,
                                  applet_cache_feed_load(cacheFile->d_name));
            }
            else
            { // TODO: this cache cleanup should be in the RSS!
                unlink(path);
                gchar *imgpath =
                    g_strdup_printf("%s" G_DIR_SEPARATOR_S "favicons"
                                    G_DIR_SEPARATOR_S "%s.png",
                                    cachePath,
                                    cacheFile->d_name);
                TDB("Removing: %s", path);
                TDB("Removing image: %s", imgpath);
                unlink(imgpath);
                g_free(imgpath);
            }
        }

        g_free(path);
    }
    TDB("--------------------");
    g_free(cachePath);
    g_free(feedCachePath);
    g_free(lifereaUserPath);
    g_free(rss);
    closedir(cacheDir);

    items = g_list_sort(items, (GCompareFunc) time_order);

    return items;
}

void freeHeadLine(HeadLine *hl) {
	ULOG_DEBUG("Releasing HeadLine memory, no news is good news right?! =)");
	if (hl!=NULL) {
		if (hl->title != NULL) {
			g_free(hl->title);
			hl->title=NULL;		
		}
		if (hl->description!= NULL) {
			g_free(hl->description);
			hl->description=NULL;		
		}
		if (hl->url!= NULL) {
			g_free(hl->url);
			hl->url=NULL;		
		}
		if (hl->feed!= NULL) {
			g_free(hl->feed);
			hl->feed=NULL;		
		}
		if (hl->icon != NULL) {
			g_object_unref(hl->icon);
		}
		g_free(hl);
		hl = NULL;
	}
}
