/**
   @file rss-feed-applet-view.c

   A small button-like widget used to show a individual feed on the
   home RSS applet

   Based on fmp-image-button.c, part of the Fremantle Media Player

   Copyright (c) 2004-2009 Nokia. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA
*/

#include <hildon/hildon.h>
#include <stdlib.h>

#include "rss-feed-applet-view.h"
#include "debug.h"

static gfloat default_color [4]         = {0.0, 0.0, 0.0, 0.0};
static gfloat default_pressed_color [4] = {0.0, 0.8, 0.0, 0.8};
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Prototypes                                                                 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

static void     _set_image_visibility  (RssFeedAppletView *self);
static void     _event_show            (RssFeedAppletView *self, gpointer user_data);
static gboolean _event_button_press    (GtkWidget      *widget,
                                        GdkEventButton *event,
                                        RssFeedAppletView *self);
static gboolean _event_button_release  (GtkWidget      *widget,
                                        GdkEventButton *event,
                                        RssFeedAppletView *self);
static gboolean _event_enter_notify    (GtkWidget        *widget,
                                        GdkEventCrossing *event,
                                        RssFeedAppletView   *self);
static gboolean _event_leave_notify    (GtkWidget        *widget,
                                        GdkEventCrossing *event,
                                        RssFeedAppletView   *self);
static gboolean _event_img_up_expose   (GtkWidget *widget,
                                        GdkEventExpose *event,
                                        gpointer image_button);
static gboolean _event_img_down_expose (GtkWidget *widget,
                                        GdkEventExpose *event,
                                        gpointer image_button);
static gboolean _event_expose_cb       (GtkWidget *widget,
                                        GdkEventExpose *event,
                                        gpointer user_data);
static void     _event_realize_cb      (GtkWidget *widget,
                                        gpointer user_data);
static void     _copy_color            (gfloat dest[4],
                                        gfloat source [4]);
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Variables                                                                  */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

enum
  {
    SIGNAL_CLICKED,
    SIGNAL_PRESSED,
    SIGNAL_RELEASED,
    SIGNAL_COUNT
  };

static guint signals[SIGNAL_COUNT];


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Construction                                                               */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

G_DEFINE_TYPE(RssFeedAppletView, rss_feed_applet_view, GTK_TYPE_ALIGNMENT);
//G_DEFINE_DYNAMIC_TYPE(RssFeedAppletView, rss_feed_applet_view, GTK_TYPE_ALIGNMENT);


static void rss_feed_applet_view_finalize(GObject *object)
{
  if(RSS_FEED_APPLET_VIEW(object)->icon_up)
    {
      g_object_unref(RSS_FEED_APPLET_VIEW(object)->icon_up);
    }
  if(RSS_FEED_APPLET_VIEW(object)->icon_down)
    {
      g_object_unref(RSS_FEED_APPLET_VIEW(object)->icon_down);
    }

  G_OBJECT_CLASS(rss_feed_applet_view_parent_class)->finalize(object);
}

static void rss_feed_applet_view_class_init(RssFeedAppletViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);

  object_class->finalize = rss_feed_applet_view_finalize;

  ULOG_INFO("RssFeedAppletView: Signal: Register Clicked");
  signals[SIGNAL_CLICKED] = g_signal_new("clicked",
                                         G_OBJECT_CLASS_TYPE(object_class),
                                         G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                         0, NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);

  signals[SIGNAL_PRESSED] = g_signal_new("button-pressed",
                                         G_OBJECT_CLASS_TYPE(object_class),
                                         G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                         0, NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);

  signals[SIGNAL_RELEASED] = g_signal_new("button-released",
                                          G_OBJECT_CLASS_TYPE(object_class),
                                          G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                                          0, NULL, NULL,
                                          g_cclosure_marshal_VOID__VOID,
                                          G_TYPE_NONE, 0);
}


static void rss_feed_applet_view_init(RssFeedAppletView *self)
{
  GtkWidget *hbox;
  GtkWidget *vbox_img;

  ULOG_INFO("RssFeedAppletView: Init start");

  gtk_alignment_set(GTK_ALIGNMENT(self), 0.0, 0.5, 1.0, 1.0);

  hbox = gtk_hbox_new(0, 0);
  vbox_img = gtk_vbox_new(0, 0);
  self->img_up = gtk_image_new();
  self->img_down = gtk_image_new();
  self->title = gtk_label_new(NULL);
  self->align_title = gtk_alignment_new (0.0, 0.5, 1.0, 1.0);
  self->event_box = gtk_event_box_new ();
  self->img_align = gtk_alignment_new (0.5, 0.5, 0, 0);
  self->icon_xalign = 0.5;
  self->icon_yalign = 0.5;

  self->url = NULL;
  self->feed = NULL;
  self->nr = 0;

  _copy_color (self->normal_color, default_color);
  _copy_color (self->current_color, default_color);
  _copy_color (self->pressed_color, default_pressed_color);

  /* Setup Image Eventbox. */
  ULOG_INFO("RssFeedAppletView: Setup Eventbox");
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->event_box),
                                   FALSE);
  gtk_widget_set_app_paintable(self->event_box, TRUE);

  /* Move the icon 8px to the right, to make the applet look more balanced */
  gtk_alignment_set_padding (GTK_ALIGNMENT (self->img_align), 0, 0, 8, 0);
 
  gtk_container_add(GTK_CONTAINER(self->img_align), vbox_img);
  gtk_box_pack_start(GTK_BOX(vbox_img), self->img_up, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox_img), self->img_down, FALSE, FALSE, 0);

  /* Setup title alignments.
     - The purpose of aligments here is to add 4px padding on top
     of both titles with the possibility of hiding both title widgets
     and their alignments if titles are not in use.
     FIXME: this in inherited from the FMPImageButton, consider remove it*/
  ULOG_INFO("RssFeedAppletView: Setup alignments");
  gtk_container_add(GTK_CONTAINER(self->align_title), self->title);
  gtk_alignment_set_padding(GTK_ALIGNMENT(self->align_title),
                            4, 0, 0, 0);
  gtk_label_set_ellipsize (GTK_LABEL (self->title), PANGO_ELLIPSIZE_END);
  gtk_misc_set_alignment (GTK_MISC (self->title), 0.0, 0.5);

  /* Fill main HBox. */
  ULOG_INFO("RssFeedAppletView: Fill HBox");
  gtk_box_pack_start(GTK_BOX(hbox), self->img_align, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), self->align_title, TRUE, TRUE, HILDON_MARGIN_DEFAULT);

  gtk_container_add(GTK_CONTAINER(self->event_box), hbox);
  gtk_container_add(GTK_CONTAINER(self), self->event_box);

  /* Attach signals. */
  ULOG_INFO("RssFeedAppletView: Connecting signals");
  g_signal_connect (G_OBJECT(self), "show", G_CALLBACK(_event_show), NULL);
  g_signal_connect (G_OBJECT(self->event_box), "button-press-event",
                    G_CALLBACK(_event_button_press), self);
  g_signal_connect (G_OBJECT(self->event_box), "button-release-event",
                    G_CALLBACK(_event_button_release), self);
  g_signal_connect (G_OBJECT(self->event_box), "enter-notify-event",
                    G_CALLBACK(_event_enter_notify), self);
  g_signal_connect (G_OBJECT(self->event_box), "leave-notify-event",
                    G_CALLBACK(_event_leave_notify), self);
  g_signal_connect(G_OBJECT(self->event_box), "expose-event",
                   G_CALLBACK (_event_expose_cb), self);
  g_signal_connect_after(G_OBJECT(self->img_up), "expose-event",
                         G_CALLBACK(_event_img_up_expose), self);
  g_signal_connect_after(G_OBJECT(self->img_down), "expose-event",
                         G_CALLBACK(_event_img_down_expose), self);
  g_signal_connect (G_OBJECT (self), "realize",
                    G_CALLBACK (_event_realize_cb), self);

  /* Set visibility. */
  ULOG_INFO("RssFeedAppletView: Setting widget visibility");
  gtk_widget_set_no_show_all(GTK_WIDGET(hbox), TRUE);
  gtk_widget_show(hbox);
  gtk_widget_show(vbox_img);
  gtk_widget_show(self->img_up);
  gtk_widget_hide(self->img_down);
  gtk_widget_show(self->title);
  gtk_widget_hide(self->align_title);
  gtk_widget_show(self->event_box);
  gtk_widget_show(self->img_align);
  ULOG_INFO("RssFeedAppletView: Init end");
}

GtkWidget *rss_feed_applet_view_new()
{
  RssFeedAppletView *self;

  self = g_object_new(RSS_TYPE_FEED_APPLET_VIEW, NULL);
  return GTK_WIDGET(self);
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Private Functions                                                          */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

/**
 * Set visibility of image widgets.
 */
static void _set_image_visibility(RssFeedAppletView *self)
{
  gboolean show_down_img = FALSE;

  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  if (self->togglable && self->mouse_button_is_down &&
      self->button_is_toggled && self->cursor_inside_button)
    {
      show_down_img = FALSE;
    }
  else if (self->togglable && self->mouse_button_is_down &&
           self->button_is_toggled && !self->cursor_inside_button)
    {
      show_down_img = TRUE;
    }
  else if (self->togglable && self->button_is_toggled &&
           !self->mouse_button_is_down)
    {
      show_down_img = TRUE;
    }
  else if (self->mouse_button_is_down && self->cursor_inside_button)
    {
      show_down_img = TRUE;
    }

  if (show_down_img)
    {
      _copy_color (self->current_color, self->pressed_color);

      if (GTK_WIDGET_VISIBLE(self->img_up))
        {
          gtk_widget_hide(self->img_up);
        }
      if (!GTK_WIDGET_VISIBLE(self->img_down))
        {
          gtk_widget_show(self->img_down);
        }
    }
  else
    {
      _copy_color (self->current_color, self->normal_color);

      if (GTK_WIDGET_VISIBLE(self->img_down))
        {
          gtk_widget_hide(self->img_down);
        }
      if (!GTK_WIDGET_VISIBLE(self->img_up))
        {
          gtk_widget_show(self->img_up);
        }
    }

  gtk_widget_queue_draw(self->event_box); /* as the alpha was updated */
}

/**
 * Set visibility of title widgets.
 */
static void _set_title_visibility(RssFeedAppletView *self)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  if (self->title_is_set)
    {
      gtk_widget_show(self->align_title);
    }
  else
    {
      gtk_widget_hide(self->align_title);
    }
}

/**
 * Manage state of button_is_being_pressed variable. 
 * Sends signals when state changes. 
 */
static void _button_is_being_pressed(RssFeedAppletView *self)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  gboolean is_pressed;
  gboolean state_changed;

  is_pressed = self->mouse_button_is_down && self->cursor_inside_button;
  state_changed = is_pressed != self->button_is_being_pressed;
  self->button_is_being_pressed = is_pressed;

  // Emit signals if button state has changed.
  if (state_changed)
    {
      if (is_pressed)
        {
/*           gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->event_box), */
/*                                            TRUE); */

          g_signal_emit(self, signals[SIGNAL_PRESSED], 0);
        }
      else
        {
/*           gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->event_box), */
/*                                            FALSE); */

          g_signal_emit(self, signals[SIGNAL_RELEASED], 0);
        }
    }
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Public Functions                                                           */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

/**
 * Align RssFeedAppletView.
 *
 * By default RssFeedAppletView will center it self if it is given larges
 * size allocation than its size request. You can use this funtion to
 * override that behaviour.
 *
 * Even though RssFeedAppletView is based on GtkAlignment, please do not call
 * GtkAlignment alignment functions directly. Using those might break
 * RssFeedAppletView.
 *
 * @param xalign The horizontal alignment of the child widget,
 *               from 0 (left) to 1 (right).
 * @param yalign The vertical alignment of the child widget,
 *               from 0 (top) to 1 (bottom).
 */
void rss_feed_applet_view_align(RssFeedAppletView *self,
                                gfloat          xalign,
                                gfloat          yalign)
{
  gtk_alignment_set(GTK_ALIGNMENT(self), xalign, yalign, 1.0, 1.0);
}

/**
 * rss_feed_applet_view_set_title
 * @self: a #RssFeedAppletView
 * @title: a string with the new title
 *
 * Sets the feed title
 *
 */
void rss_feed_applet_view_set_title(RssFeedAppletView *self,
                                    const gchar    *title)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  if (title)
    {
      self->title_is_set = TRUE;
      gtk_label_set_text (GTK_LABEL(self->title), title);
    }
  else
    {
      self->title_is_set = FALSE;
    }
  _set_title_visibility(self);
}


/**
 * rss_feed_applet_view_set_title_markup
 * @self: a #RssFeedAppletView
 * @title: a markup string with the new title
 *
 * Sets the feed title using pango markup language.
 *
 * @b Example
 * @code
 * rss_feed_applet_view_set_title_markup(RSS_FEED_APPLET_VIEW(button),
 *                                   "<span size='12000'>title</span>");
 * @endcode
 *
 * @param title Title to show, or NULL to hide title.
 */
void rss_feed_applet_view_set_title_markup(RssFeedAppletView *self,
                                           const gchar    *title)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  if (title)
    {
      self->title_is_set = TRUE;
      gtk_label_set_markup(GTK_LABEL(self->title), title);
    }
  else
    {
      self->title_is_set = FALSE;
    }
  _set_title_visibility(self);
}

void rss_feed_applet_view_set_image_from_file(RssFeedAppletView *self,
                                              const gchar    *image_up,
                                              const gchar    *image_down)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  g_assert(image_up != NULL);
  g_assert(image_down != NULL);

  gtk_image_set_from_file(GTK_IMAGE(self->img_up), image_up);
  gtk_image_set_from_file(GTK_IMAGE(self->img_down), image_down);
}

void rss_feed_applet_view_set_image_from_pixbuf(RssFeedAppletView *self,
                                                GdkPixbuf      *image_up,
                                                GdkPixbuf      *image_down)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  g_assert(GDK_IS_PIXBUF(image_up));
  g_assert(GDK_IS_PIXBUF(image_down));

  gtk_image_set_from_pixbuf(GTK_IMAGE(self->img_up), image_up);
  gtk_image_set_from_pixbuf(GTK_IMAGE(self->img_down), image_down);
}
void rss_feed_applet_view_set_icon_from_pixbuf(RssFeedAppletView *self,
                                               GdkPixbuf      *icon_up,
                                               GdkPixbuf      *icon_down)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  g_assert(GDK_IS_PIXBUF(icon_up));
  g_assert(GDK_IS_PIXBUF(icon_down));

  if(GDK_IS_PIXBUF(self->icon_up))
    g_object_unref(self->icon_up);
  if(GDK_IS_PIXBUF(self->icon_down))
    g_object_unref(self->icon_down);

  g_object_ref(icon_up);
  g_object_ref(icon_down);

  self->icon_up = icon_up;
  self->icon_down = icon_down;
  gtk_widget_queue_draw(GTK_WIDGET(self));
}

/**
 * Enable or disable toggle button behaviour.
 *
 * @param toggled TRUE if togglable, FALSE if not.
 */
void rss_feed_applet_view_set_togglable(RssFeedAppletView *self,
                                        gboolean togglable)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  self->togglable = togglable;
  if (togglable == FALSE)
    {
      self->button_is_toggled = FALSE;
    }
  _set_image_visibility(self);
}

/**
 * Get togglable state.
 *
 * @return TRUE if togglable, FALSE if not.
 */
gboolean rss_feed_applet_view_get_togglable(RssFeedAppletView *self)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  return self->togglable;
}

/**
 * Set button toggle state.
 *
 * @param toggled TRUE if toggled, FALSE if not.
 */
void rss_feed_applet_view_set_toggle(RssFeedAppletView *self,
                                     gboolean toggled)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  g_assert(self->togglable == TRUE);

  self->button_is_toggled = toggled;
  _set_image_visibility(self);
}

/**
 * Get button toggle state.
 *
 * @return TRUE if toggled, FALSE if not.
 */
gboolean rss_feed_applet_view_get_toggle(RssFeedAppletView *self)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  g_assert(self->togglable == TRUE);
  return self->button_is_toggled;
}

/**
 * Get button pressed state.
 *
 * @return TRUE if pressed, FALSE if not.
 */
gboolean rss_feed_applet_view_is_beeing_pressed(RssFeedAppletView *self)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));
  return self->button_is_being_pressed;
}



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Signals                                                                    */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

static void _event_show(RssFeedAppletView *self, gpointer user_data)
{
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  gtk_widget_show_all(GTK_WIDGET(self));
  _set_image_visibility(self);
}

static gboolean _event_button_press(GtkWidget      *widget,
                                    GdkEventButton *event,
                                    RssFeedAppletView *self)
{
  ULOG_DEBUG(__FUNCTION__);
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  self->mouse_button_is_down = TRUE;
  _set_image_visibility(self);
  _button_is_being_pressed(self);

  return FALSE;
}

static gboolean _event_button_release(GtkWidget      *widget,
                                      GdkEventButton *event,
                                      RssFeedAppletView *self)
{
  ULOG_DEBUG(__FUNCTION__);
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  self->mouse_button_is_down = FALSE;

  if (self->togglable && self->cursor_inside_button)
    {
      self->button_is_toggled = !(self->button_is_toggled);
    }

  _set_image_visibility(self);
  _button_is_being_pressed(self);

  if (self->cursor_inside_button)
    {
      ULOG_INFO("RssFeedAppletView: Signal: Emitting Clicked");
      g_signal_emit(self, signals[SIGNAL_CLICKED], 0);
    }
  return FALSE;
}

static gboolean _event_enter_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletView   *self)
{
  ULOG_DEBUG(__FUNCTION__);
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  self->cursor_inside_button = TRUE;
  _set_image_visibility(self);
  _button_is_being_pressed(self);
  return FALSE;
}

static gboolean _event_leave_notify(GtkWidget        *widget,
                                    GdkEventCrossing *event,
                                    RssFeedAppletView   *self)
{
  ULOG_DEBUG(__FUNCTION__);
  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  self->cursor_inside_button = FALSE;
  _set_image_visibility(self);
  _button_is_being_pressed(self);
  return FALSE;
}


static gboolean _event_img_up_expose(GtkWidget *widget,
                                     GdkEventExpose *event,
                                     gpointer self)
{
  ULOG_INFO("RssFeedAppletView: img_up expose handler");
  RssFeedAppletView *image_button;
  cairo_t *cr;

  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  image_button = RSS_FEED_APPLET_VIEW(self);

  if(GDK_IS_PIXBUF(image_button->icon_up))
    {
      ULOG_INFO("RssFeedAppletView: DRAWING ICON");
      cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
      gdk_cairo_region (cr, event->region);
      cairo_clip (cr);

      cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
      gdk_cairo_set_source_pixbuf(cr, image_button->icon_up,
                                  widget->allocation.x+(widget->allocation.width
                                                        -gdk_pixbuf_get_width(image_button->icon_up))*image_button->icon_xalign,
                                  widget->allocation.y+(widget->allocation.height
                                                        -gdk_pixbuf_get_height(image_button->icon_up))*image_button->icon_yalign);

      cairo_paint (cr);
    }
  return FALSE;
}

static gboolean _event_img_down_expose(GtkWidget *widget,
                                       GdkEventExpose *event,
                                       gpointer self)
{
  ULOG_INFO("RssFeedAppletView: img_up expose handler");
  RssFeedAppletView *image_button;
  cairo_t *cr;

  g_assert(RSS_IS_FEED_APPLET_VIEW(self));

  image_button = RSS_FEED_APPLET_VIEW(self);

  if(GDK_IS_PIXBUF(image_button->icon_down))
    {
      ULOG_INFO("RssFeedAppletView: DRAWING ICON");
      cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
      gdk_cairo_region (cr, event->region);
      cairo_clip (cr);

      cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
      gdk_cairo_set_source_pixbuf(cr, image_button->icon_down,
                                  widget->allocation.x+(widget->allocation.width
                                                        -gdk_pixbuf_get_width(image_button->icon_down))*image_button->icon_xalign, 
                                  widget->allocation.y+(widget->allocation.height
                                                        -gdk_pixbuf_get_height(image_button->icon_down))*image_button->icon_yalign);

      cairo_paint (cr);
    }
  return FALSE;
}

static gboolean
_event_expose_cb (GtkWidget *widget,
                  GdkEventExpose *event,
                  gpointer user_data)
{
  cairo_t *cr;
  RssFeedAppletView *self = RSS_FEED_APPLET_VIEW (user_data);

  g_assert (RSS_IS_FEED_APPLET_VIEW (user_data));

  cr = gdk_cairo_create (widget->window);
  gdk_cairo_region (cr, event->region);
  cairo_clip (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

  cairo_set_source_rgba (cr,
                         self->current_color[0], self->current_color[1],
                         self->current_color[2], self->current_color[3]);

  cairo_paint (cr);

  cairo_destroy (cr);

  return FALSE;
}

static void
_event_realize_cb (GtkWidget *widget,
                   gpointer user_data)
{
  /* This is neccessary in order to be sure that the position of the
     label is correct after the custom wrapping made on the applet.

     FIXME: this is a good reason to move calc_breaking_position from
     applet.c to RssFeedAppletView */
  gtk_widget_queue_resize (widget);
}

static void
_copy_color (gfloat dest[4], gfloat source [4])
{
  gint i = 0;

  for (i = 0; i < 4; i++)
    {
      dest[i] = source[i];
    }
}

void rss_feed_applet_view_set_icon_align(RssFeedAppletView *self, gfloat xalign, gfloat yalign)
{
  g_return_if_fail(RSS_IS_FEED_APPLET_VIEW(self));

  if(xalign < 0)
    self->icon_xalign = 0;
  else if(xalign > 1)
    self->icon_xalign = 1;
  else self->icon_xalign = xalign;

  if(yalign < 0)
    self->icon_yalign = 0;
  else if(yalign > 1)
    self->icon_yalign = 1;
  else self->icon_yalign = yalign;
}

void
rss_feed_applet_view_set_url (RssFeedAppletView *self,
                              const gchar       *url)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  if (self->url)
    {
      g_free (self->url);
    }
  self->url = g_strdup (url);
}

const gchar*
rss_feed_applet_view_get_url (RssFeedAppletView *self)
{
  g_return_val_if_fail (RSS_IS_FEED_APPLET_VIEW (self), NULL);

  return self->url;
}

void
rss_feed_applet_view_set_feed (RssFeedAppletView *self,
                               const gchar       *feed)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  if (self->feed)
    {
      g_free (self->feed);
    }
  self->feed = g_strdup (feed);
}

const gchar*
rss_feed_applet_view_get_feed (RssFeedAppletView *self)
{
  g_return_val_if_fail (RSS_IS_FEED_APPLET_VIEW (self), NULL);

  return self->feed;
}

void
rss_feed_applet_view_set_nr (RssFeedAppletView *self,
                             gulong             nr)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  self->nr = nr;
}

gulong
rss_feed_applet_view_get_nr (RssFeedAppletView *self)
{
  g_return_val_if_fail (RSS_IS_FEED_APPLET_VIEW (self), 0);

  return self->nr;
}


void
rss_feed_applet_view_set_color (RssFeedAppletView *self,
                                gfloat r,
                                gfloat g,
                                gfloat b,
                                gfloat a)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  self->normal_color[0] = r;
  self->normal_color[1] = g;
  self->normal_color[2] = b;
  self->normal_color[3] = a;
}

void
rss_feed_applet_view_set_pressed_color   (RssFeedAppletView *self,
                                          gfloat r,
                                          gfloat g,
                                          gfloat b,
                                          gfloat a)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  self->pressed_color[0] = r;
  self->pressed_color[1] = g;
  self->pressed_color[2] = b;
  self->pressed_color[3] = a;
}

void
rss_feed_applet_view_get_color (RssFeedAppletView *self,
                                gfloat *r,
                                gfloat *g,
                                gfloat *b,
                                gfloat *a)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  if (r != NULL)
    {
      *r = self->normal_color[0];
    }

  if (g != NULL)
    {
      *g = self->normal_color[1];
    }

  if (b != NULL)
    {
      *b = self->normal_color[2];
    }

  if (a != NULL)
    {
      *a = self->normal_color[3];
    }
}

void
rss_feed_applet_view_get_pressed_color (RssFeedAppletView *self,
                                        gfloat *r,
                                        gfloat *g,
                                        gfloat *b,
                                        gfloat *a)
{
  g_return_if_fail (RSS_IS_FEED_APPLET_VIEW (self));

  if (r != NULL)
    {
      *r = self->pressed_color[0];
    }

  if (g != NULL)
    {
      *g = self->pressed_color[1];
    }

  if (b != NULL)
    {
      *b = self->pressed_color[2];
    }

  if (a != NULL)
    {
      *a = self->pressed_color[3];
    }
}
