/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "common.h"
#include "gmozillahistorylist.h"
#include "gmozillacppwrapper.h"

/* GMozillaHistoryList */

static GMozillaHistoryList *singleton_instance = NULL;
static guint ref_count = 0;

int
g_mozilla_history_list_clear_all ()
{
    return g_mozilla_cpp_history_list_remove_all();
}

static gboolean
g_mozilla_history_list_position_ok (GWebHistoryList *self, gint pos)
{
    TRACE_LOG();
    return (G_MOZILLA_HISTORY_LIST(self)->items &&
            pos >= 0 &&
            pos < G_MOZILLA_HISTORY_LIST(self)->num_items);
}

static void
g_mozilla_history_list_delete_item (GMozillaHistoryList *self, gint pos)
{
    g_return_if_fail(G_IS_MOZILLA_HISTORY_LIST(self));

    guint rv;
    TRACE_LOG();

    if (!g_mozilla_history_list_position_ok((GWebHistoryList*)self, pos)) {
        return;
    }
    GtkWebHistoryItem *item = (GtkWebHistoryItem *)g_list_nth_data(self->items, pos);
    if (!item->url)
        return ;
    rv = g_mozilla_cpp_history_list_remove(item->url);
    /* we don't have NS_FAILED, this is NS_FAILED */
    if (rv & 0x80000000) {
        TRACE_LOG("gtk_moz_embed_common_remove_history failed");
        return;
    }

    self->items = g_list_remove(self->items, item);
    EAL_GFREE(item->url);
    EAL_IF_GFREE(item->title);
    EAL_GFREE(item);
}

static GObject*
g_mozilla_history_list_get_items (GMozillaHistoryList *self)
{
    TRACE_LOG();
    // Not in use anymore
    return NULL;
}

static gint
g_mozilla_history_list_get_items_count (GMozillaHistoryList *self)
{
    TRACE_LOG();
    g_return_val_if_fail(G_IS_MOZILLA_HISTORY_LIST(self), 0);
    return self->num_items;
}

static const gchar*
g_mozilla_history_list_get_item_title (GWebHistoryList *self, gint pos)
{
    TRACE_LOG();
    if (G_IS_MOZILLA_HISTORY_LIST(self) &&
        g_mozilla_history_list_position_ok(self, pos)) {
        GtkWebHistoryItem *item = (GtkWebHistoryItem *)g_list_nth_data(
            G_MOZILLA_HISTORY_LIST(self)->items, pos);
        return item->title;
    }
    return NULL;
}

static const gchar*
g_mozilla_history_list_get_item_url (GWebHistoryList *self, gint pos)
{
    TRACE_LOG();
    if (G_IS_MOZILLA_HISTORY_LIST(self) &&
        g_mozilla_history_list_position_ok(self, pos)) {
        GtkWebHistoryItem *item = (GtkWebHistoryItem *)g_list_nth_data(
            G_MOZILLA_HISTORY_LIST(self)->items, pos);
        return item->url;
    }
    return NULL;
}

static const gchar*
g_mozilla_shistory_list_get_item_thumbnail (GWebHistoryList *self, gint pos)
{
    TRACE_LOG();
    return NULL;
}

static GTime
g_mozilla_history_list_get_item_time (GWebHistoryList *self, gint pos)
{
    TRACE_LOG();
    if (G_IS_MOZILLA_HISTORY_LIST(self) &&
        g_mozilla_history_list_position_ok(self, pos)) {
        GtkWebHistoryItem *item = (GtkWebHistoryItem *)g_list_nth_data(
            G_MOZILLA_HISTORY_LIST(self)->items, pos);
        return item->accessed;
    }
    return 0;
}

static void
g_mozilla_history_list_destroy (GMozillaHistoryList *self)
{
    TRACE_LOG();

    // Don't mess with cached array with window history lists
    gint window_history = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(self), "window-history"));

    if (ref_count && !window_history)
        ref_count--;

    if (ref_count == 0 || window_history) {
        g_list_foreach(self->items, free_history_item, NULL);
        EAL_GFREE_FUNC(self->items, g_list_free);
        if (!window_history)
            singleton_instance = NULL;
    }
}

static void
g_mozilla_history_list_interface_init (GWebHistoryListIface *iface)
{
    TRACE_LOG();
    iface->destroy         = (void (*) (GWebHistoryList*))
                             g_mozilla_history_list_destroy;
    iface->delete_item     = (void (*) (GWebHistoryList*, gint))
                             g_mozilla_history_list_delete_item;
    iface->get_items       = (GObject* (*) (GWebHistoryList*))
                             g_mozilla_history_list_get_items;
    iface->get_items_count = (gint (*) (GWebHistoryList*))
                             g_mozilla_history_list_get_items_count;
    iface->get_item_title  = (gchar* (*) (GWebHistoryList*, gint))
                             g_mozilla_history_list_get_item_title;
    iface->get_item_url    = (gchar* (*) (GWebHistoryList*, gint))
                             g_mozilla_history_list_get_item_url;
    iface->get_item_time   = (GTime (*) (GWebHistoryList*, gint))
                             g_mozilla_history_list_get_item_time;
    iface->get_item_thumbnail  = (gchar* (*) (GWebHistoryList*, gint))
                             g_mozilla_shistory_list_get_item_thumbnail;
}

static void
g_mozilla_history_list_instance_init (GTypeInstance *instance, gpointer  g_class)
{
    TRACE_LOG();
    G_MOZILLA_HISTORY_LIST(instance)->items     = NULL;
    G_MOZILLA_HISTORY_LIST(instance)->num_items = 0;
}

GType
g_mozilla_history_list_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
            {
                sizeof (GMozillaHistoryListClass),
                NULL,   /* base_init */
                NULL,   /* base_finalize */
                NULL,   /* class_init */
                NULL,   /* class_finalize */
                NULL,   /* class_data */
                sizeof (GMozillaHistoryList),
                0,      /* n_preallocs */
                g_mozilla_history_list_instance_init    /* instance_init */
            };
        static const GInterfaceInfo iface_info =
            {
                (GInterfaceInitFunc) g_mozilla_history_list_interface_init,
                NULL,   /* interface_finalize */
                NULL    /* interface_data */
            };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       "GMozillaHistoryList",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB_HISTORY_LIST,
                                     &iface_info);
    }
    return type;
}

GObject*
g_mozilla_history_list_new (void)
{
    TRACE_LOG();

    if (!singleton_instance) { 
        singleton_instance = G_MOZILLA_HISTORY_LIST(g_object_new(g_mozilla_history_list_get_type(), NULL));
        singleton_instance->isSHistory = FALSE;
        singleton_instance->num_items = g_mozilla_cpp_get_history_list(
                        &singleton_instance->items,
                        singleton_instance->num_items);
        ref_count = 1;
    } else
        ref_count++;

    return G_OBJECT(singleton_instance);
}

typedef struct _HistoryMessage HistoryMessage;
struct _HistoryMessage
{
    GMozillaEngine *engine;
    guint message;
    gint number;
    GWebEngineEncoding encoding;
};

static gboolean
idle_cb (gpointer user_data)
{
    TRACE_LOG();

    HistoryMessage *msg = (HistoryMessage *)user_data;

    if (singleton_instance)
        singleton_instance->num_items =
            g_mozilla_cpp_get_history_list(&singleton_instance->items,
                                           singleton_instance->num_items);

    g_object_set_data(G_OBJECT(msg->engine), "async", GINT_TO_POINTER(1));
    g_signal_emit_by_name(msg->engine,
                          G_WEBWIDGET_SIGNAL_HISTORY_CHANGED,
                          msg->message, msg->number, msg->encoding);

    g_free(msg);
    return FALSE;
}

void send_history_message(GMozillaEngine* engine, unsigned int message, int number, GWebEngineEncoding encoding)
{
  TRACE_LOG();

  HistoryMessage *msg = g_try_new0(HistoryMessage, 1);
  if (!msg)
      return;

  msg->engine = engine;
  msg->message = message;
  msg->number = number;
  msg->encoding = encoding;

  // Let engine side process to handle history addition before
  // sending the notification forward
  g_idle_add ((GSourceFunc)idle_cb, msg);
}


