/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Oleg Romashin <romaxa@gmail.com>
 *
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *        Egor Starkov <egor.starkov@nokia.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "nsSHistoryListener.h"
#include "nsISHistory.h"
#include "nsIWindowWatcher.h"
#include "nsIDOMWindow.h"
#include "nsStringGlue.h"
#include "nsServiceManagerUtils.h"
#include "nsIWebBrowserChrome.h"
#include "nsIWebNavigation.h"
#include "gmozillahistorylist.h"
#include "MicrobEalUtils.h"

#include <stdio.h>
#include "common.h"

NS_IMPL_ISUPPORTS2(nsSHistoryListener, nsISHistoryListener, nsSupportsWeakReference)

nsSHistoryListener::nsSHistoryListener()
{
  mEngine = nsnull;
}

nsSHistoryListener::~nsSHistoryListener()
{
  mEngine = nsnull;
  nsCOMPtr<nsISHistory> sessionHistory;
  nsresult rv = GetSessionHistory(getter_AddRefs(sessionHistory));
  mDOMWindow = nsnull;
  NS_ENSURE_SUCCESS(rv, );
  sessionHistory->RemoveSHistoryListener(static_cast<nsISHistoryListener*>(this));
}

nsresult
nsSHistoryListener::Init(nsIDOMWindow *aDOMWindow, GMozillaEngine* aEngine)
{
  NS_ENSURE_ARG(aDOMWindow);
  NS_ENSURE_ARG(aEngine);

  mEngine = aEngine;
  mDOMWindow = aDOMWindow;

  nsresult rv;
  nsCOMPtr<nsISHistory> sessionHistory;
  rv = GetSessionHistory(getter_AddRefs(sessionHistory));
  NS_ENSURE_SUCCESS(rv, rv);
  return sessionHistory->AddSHistoryListener(static_cast<nsISHistoryListener*>(this));
}

nsresult
nsSHistoryListener::GetSessionHistory(nsISHistory** aSessionHistory)
{
  nsCOMPtr<nsIWindowWatcher> wwatch =
    do_GetService("@mozilla.org/embedcomp/window-watcher;1");
  NS_ENSURE_TRUE(wwatch, NS_ERROR_NULL_POINTER);

  nsCOMPtr<nsIWebBrowserChrome> chrome;
  wwatch->GetChromeForWindow(mDOMWindow, getter_AddRefs(chrome));
  NS_ENSURE_TRUE(chrome, NS_ERROR_NULL_POINTER);

  nsCOMPtr<nsIWebBrowser> webBrowser;
  chrome->GetWebBrowser(getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_NULL_POINTER);

  nsCOMPtr<nsIWebNavigation> navigation = do_QueryInterface(webBrowser);
  NS_ENSURE_TRUE(navigation, NS_ERROR_NULL_POINTER);

  nsCOMPtr<nsISHistory> sessionHistory;
  nsresult rv = navigation->GetSessionHistory(getter_AddRefs(sessionHistory));
  NS_ENSURE_SUCCESS(rv, rv);
  NS_ENSURE_TRUE (sessionHistory, NS_ERROR_NULL_POINTER);

  *aSessionHistory = sessionHistory;
  NS_ADDREF(*aSessionHistory);
  return rv;
}

GWebEngineEncoding
nsSHistoryListener::GetEncoding()
{
  GMozillaEngine* engine = mEngine ? mEngine : GetGMozEngineForDOMWindow(nsnull);
  NS_ENSURE_TRUE(engine, G_WEBENGINE_ENCODING_AUTOMATIC);

  GtkMozEmbed *embed = GTK_MOZ_EMBED(engine->engine);
  NS_ENSURE_TRUE(embed, G_WEBENGINE_ENCODING_AUTOMATIC);

  return g_mozilla_engine_get_encoding(embed);
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryNewEntry(nsIURI *aNewURI)
{
  NS_ENSURE_ARG(mEngine);
  NS_ENSURE_ARG(aNewURI);

  nsCAutoString newUrl;
  aNewURI->GetSpec(newUrl);
  bool hashUrl = newUrl.RFindChar('#') >= 0;
  int flag = !hashUrl;

  if (mURI)
  {
    nsCAutoString oldUrl;
    mURI->GetSpec(oldUrl);

    if (newUrl.Equals(oldUrl))
      flag = 0;

    if (mEngine->mMinWindowSize.width || mEngine->mMinWindowSize.height) {
      nsCAutoString newPre;
      nsCAutoString oldPre;
      mURI->GetPrePath(oldPre);
      aNewURI->GetPrePath(newPre);
      ULOG("FrameEXP state: minSize:%i,%i, PrePathEq:%i, hashUrl:%i",
            mEngine->mMinWindowSize.width, mEngine->mMinWindowSize.height, oldPre.Equals(newPre), hashUrl);
      if (!(oldPre.Equals(newPre) && hashUrl)) {
        mEngine->mMinWindowSize.width = 0;
        mEngine->mMinWindowSize.height = 0;
      }
    }
  }
  aNewURI->Clone(getter_AddRefs(mURI));

  mEngine->full_update_required = TRUE;

  if (flag) {
    mEngine->mLastVisibleArea.x = mEngine->mLastVisibleArea.y = 0;
    mEngine->mLastTargetArea.x = mEngine->mLastTargetArea.y = 0;
  }

  send_history_message(mEngine, G_WEBENGINE_HISTORY_NEW_ENTRY, flag, GetEncoding());
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::PrepareHistoryNavigation(nsIURI *aGoURI)
{
  aGoURI->Clone(getter_AddRefs(mURI));
  mEngine->mLastVisibleArea.x = mEngine->mLastVisibleArea.y = 0;
  mEngine->mLastTargetArea.x = mEngine->mLastTargetArea.y = 0;
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryGoBack(nsIURI *aBackURI, PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG(mEngine);
  NS_ENSURE_ARG(aBackURI);

  PrepareHistoryNavigation(aBackURI);

  send_history_message(mEngine, G_WEBENGINE_HISTORY_GO_BACK, 0, GetEncoding());
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryGoForward(nsIURI *aForwardURI, PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG(mEngine);
  NS_ENSURE_ARG(aForwardURI);

  PrepareHistoryNavigation(aForwardURI);

  send_history_message(mEngine, G_WEBENGINE_HISTORY_GO_FORWARD, 0, GetEncoding());
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryReload(nsIURI *aReloadURI, PRUint32 aReloadFlags, PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG(mEngine);
  send_history_message(mEngine, G_WEBENGINE_HISTORY_RELOAD, 0, GetEncoding());
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryGotoIndex(PRInt32 aIndex, nsIURI *aGotoURI, PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG(mEngine);

  PrepareHistoryNavigation(aGotoURI);

  send_history_message(mEngine, G_WEBENGINE_HISTORY_GOTO_INDEX, aIndex, GetEncoding());
  return NS_OK;
}

NS_IMETHODIMP
nsSHistoryListener::OnHistoryPurge(PRInt32 aNumEntries, PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG(mEngine);
  send_history_message(mEngine, G_WEBENGINE_HISTORY_PURGE, aNumEntries, GetEncoding());
  return NS_OK;
}
