/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Anton Rogaynis <anton.rogaynis@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

//#define MOZEAL_LOGGING 1

#include "common.h"
#include "gmozillacppwrapper.h"
#include "gmozillaweb.h"
#include "gmozillaengine.h"

#include "MicrobEalDownloadMgr.h"
#include "MicrobEalUtils.h"

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>

#include "gtkmozembed_internal.h"

#include "nsServiceManagerUtils.h"
#include "nsCOMPtr.h"
#include "nsMemory.h"
#include "nsILoginManager.h"
#include "nsILoginInfo.h"
#include "nsEmbedString.h"
#include "nsINavHistoryService.h"
#include "nsToolkitCompsCID.h"
#include "nsIBrowserHistory.h"
#include "nsNetUtil.h"
#include "nsIWebBrowser.h"
#include "nsIBaseWindow.h"
#include "nsIObserverService.h"
#include "nsIPrefBranch.h"
#include "nsIPrefService.h"
#include "nsIDocCharset.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIWebNavigation.h"
#include "nsIHistoryEntry.h"
#include "nsISHistory.h"
#include "nsIClipboardCommands.h"
#include "nsIDOMMimeType.h"
#include "nsIHttpAuthManager.h"

#include "nsIDOMHTMLDocument.h"
#include "nsIDOMHTMLCollection.h"
#include "nsIImageLoadingContent.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMHTMLIFrameElement.h"
#include "nsIDOMHTMLFrameElement.h"

#include "nsIRDFDataSource.h"
#include "nsIRDFService.h"
#include "nsIExtensionManager.h"

#include <nsIWindowWatcher.h>
#include <nsIDOMWindow.h>
#include <nsIDOMDocument.h>
#include <nsIDOMHTMLCanvasElement.h>
#include <nsIDOMDocumentEvent.h>
#include <nsIDOMEventTarget.h>
#include <nsIPrivateDOMEvent.h>
#include <nsIDOMWindowInternal.h>
#include <nsICanvasRenderingContextInternal.h>
#include <nsIDOMCanvasRenderingContext2D.h>
#include <nsIComponentManager.h>
#include <plbase64.h>
#include <nsPIDOMWindow.h>

// certs
#include "nsIDocShell.h"
#include "nsISSLStatus.h"
#include "nsISSLStatusProvider.h"
#include "nsIX509Cert.h"
#include "nsISecureBrowserUI.h"

#include "nsIDOMPlugin.h"
#include "nsIPluginHost.h"

// JS
#include "nsIScriptObjectPrincipal.h"
#include "nsIScriptGlobalObject.h"
#include "nsIScriptContext.h"

#include "nsIObserver.h"

#include "MicrobEalCertificate.h"
#include "nsSHistoryListener.h"
#include "MicrobEalNotificationBox.h"

#include "nsISelectionController.h"
#include "nsIExtensionManager.h"
#include "nsIComponentRegistrar.h"

#include "MicrobEalXshmRenderer.h"
#include "nsIViewManager.h"
#include "nsIView.h"
#include "nsIWeakReferenceUtils.h"
#include "nsGUIEvent.h"
#include "nsIDocument.h"
#include "nsIPresShell.h"
#include "nsIDOMWindowUtils.h"
#include "nsIDOMNSEvent.h"
#include "nsIDOMNSDocument.h"
#include "nsIDOMClientRect.h"
#include "nsIDOMNSElement.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMHTMLObjectElement.h"
#include "nsIDOMHTMLEmbedElement.h"
#include "nsIDOMHTMLLinkElement.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLSelectElement.h"
#include "nsIDOMHTMLOptionElement.h"
#include "nsISelectElement.h"
#include "nsIDOMNSHTMLElement.h"
#include "nsIDOMKeyEvent.h"
#include "nsIWebBrowserFocus.h"
#include "nsITypeAheadFind.h"
#include <nsIImageDocument.h>
#include "imgIContainer.h"
#include "nsIContent.h"
#include "nsISelection2.h"
#include "nsPresContext.h"
#include "nsIDOMRange.h"

#include "nsIDOMWindowCollection.h"
#include "nsWidgetsCID.h"
#include "nsIAppShell.h"

#include "MicrobEalContextMenuInfo.h"

#include "gtkmicrob_context.h"

#define EM_RDF_EXT_DES "http://www.mozilla.org/2004/em-rdf#description"
#define EM_RDF_EXT_CREATOR "http://www.mozilla.org/2004/em-rdf#creator"
#define EM_RDF_EXT_STATE "http://www.mozilla.org/2004/em-rdf#userDisabled"
#define EM_RDF_EXT_APP_DISABLED "http://www.mozilla.org/2004/em-rdf#appDisabled"

#define MAX_NESTED_FRAMES 4
#define SUSPEND_JS_TIMEOUT_0   0
#define SUSPEND_JS_TIMEOUT_1   15000
#define SUSPEND_JS_TIMEOUT_2   30000
#define SUSPEND_JS_TIMEOUT_3   60000

static NS_DEFINE_CID(kAppShellCID, NS_APPSHELL_CID);

/* Clear all stored logins */
void g_mozilla_cpp_clear_all_passwords()
{
  // Get login manager
  nsCOMPtr<nsILoginManager> loginmgr(
    do_GetService(NS_LOGINMANAGER_CONTRACTID));

  if (!loginmgr) {
    return;
  }

  loginmgr->RemoveAllLogins();

  //Clear all hosts for which password saving is disabled.
  nsresult rv;
  PRUnichar **hostnames;
  PRUint32 count;
  
  rv = loginmgr->GetAllDisabledHosts(&count, &hostnames);
  NS_ENSURE_SUCCESS(rv, );
  for (PRUint32 i = 0; i < count; i++) {
     loginmgr->SetLoginSavingEnabled(nsDependentString(hostnames[i]),
                                      PR_TRUE);
  }
  NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY(count, hostnames);
}

/* Remove a login by index */
void g_mozilla_cpp_clear_passwords_by_index(int index)
{
  // Get login manager
  nsCOMPtr<nsILoginManager> loginmgr(
    do_GetService(NS_LOGINMANAGER_CONTRACTID));

  if (!loginmgr) {
    return;
  }

  nsILoginInfo** logins;
  PRUint32 count = 0;

  nsresult rv = loginmgr->GetAllLogins(&count, &logins);
  if (NS_FAILED(rv)) {
    return;
  }

  if (index >= 0 && index < (int)count) {
    loginmgr->RemoveLogin(logins[index]);
  }

  NS_FREE_XPCOM_ISUPPORTS_POINTER_ARRAY(count, logins);
}

/* Clear all authenticated sessions */
void g_mozilla_cpp_clear_authenticated_sessions()
{
  // Fetch the authentication manager:
  nsCOMPtr<nsIHttpAuthManager> authManager =
    do_GetService(NS_HTTPAUTHMANAGER_CONTRACTID);
  if (authManager)
    authManager->ClearAll();
}

/* Get logins matching uri */
int g_mozilla_cpp_get_logins(const char* uri, GList** list)
{
  int ret = 0;

  nsCOMPtr<nsILoginManager> loginmgr(
    do_GetService(NS_LOGINMANAGER_CONTRACTID));

  if (!loginmgr) {
    return ret;
  }

  nsresult rv;
  nsILoginInfo** logins;
  PRUint32 count = 0;
  nsString empty;

  rv = loginmgr->FindLogins(&count, NS_ConvertUTF8toUTF16(uri),
                            empty, empty, &logins);
  NS_ENSURE_SUCCESS(rv, ret);

  nsEmbedString loginHost;
  nsEmbedString loginUser;
  nsEmbedString loginPass;
  nsILoginInfo* curLogin = NULL;


  // Go through logins list
  for (; ret < (int)count; ret++) {
    curLogin = logins[ret];

    if (!curLogin) continue;
    // Append to GList
    if (list) {
      GtkMozLogin* login = g_new0(GtkMozLogin, 1);
      if (!login) {
        rv = NS_ERROR_OUT_OF_MEMORY;
        break;
      }

      rv = curLogin->GetHostname(loginHost);
      if (NS_FAILED(rv)) {
        break;
      }

      rv = curLogin->GetUsername(loginUser);
      if (NS_FAILED(rv)) {
        break;
      }
      rv = curLogin->GetPassword(loginPass);
      if (NS_FAILED(rv)) {
        break;
      }

      // Fill login struct
      login->user = ToNewUTF8String(loginUser);
      login->pass = ToNewUTF8String(loginPass);
      login->host = ToNewUTF8String(loginHost);
      login->index = ret;

      *list = g_list_append(*list, login);
    }
  }

  NS_FREE_XPCOM_ISUPPORTS_POINTER_ARRAY(count, logins);
  return ret;
}

void
free_history_item(gpointer data,
                  gpointer user_data)
{
  GtkWebHistoryItem *item = (GtkWebHistoryItem *)data;
  g_free(item->title);
  g_free(item->url);
  g_free(item);
}

nsresult
get_content_list(GList **aHistoryList,
                 gint *aCounter,
                 nsINavHistoryContainerResultNode *root,
                 gint aCurCount)
{
  NS_ENSURE_ARG_POINTER(aHistoryList);
  *aCounter = 0;
  root->SetContainerOpen(PR_TRUE);

  PRUint32 childCount = 0;
  root->GetChildCount(&childCount);

  if (childCount == 0) {
    root->SetContainerOpen(PR_FALSE);
    return NS_OK;
  }

  if (childCount == (guint)aCurCount && *aHistoryList) {
    // No new items, visit time of a previously visited
    // history item is changed, no way of knowing which
    // item to update so we need to dump our cache here
    g_list_foreach(*aHistoryList, free_history_item, NULL);
    g_list_free(*aHistoryList);
    *aHistoryList = NULL;
    aCurCount = 0;
  }

  // Get history item count delta
  PRInt32 newItems = childCount - aCurCount;

  GtkWebHistoryItem* historyItem = NULL;
  // Add new items to the beginning of the list
  for (PRInt32 i = 0; i < newItems; ++i) {
    nsCOMPtr<nsINavHistoryResultNode> child;
    root->GetChild(i, getter_AddRefs(child));
    if (!child) {
      continue;
    }

    PRUint32 type = 0;
    child->GetType(&type);

    if (type == nsINavHistoryResultNode::RESULT_TYPE_URI) {
      historyItem = g_try_new0(GtkWebHistoryItem, 1);
      if (!historyItem) {
          root->SetContainerOpen(PR_FALSE);
          return NS_ERROR_OUT_OF_MEMORY;
      }
      nsCString string;
      child->GetUri(string);
      historyItem->url = strdup(PromiseFlatCString(string).get());

      child->GetTitle(string);
      historyItem->title = strdup(PromiseFlatCString(string).get());

      PRTime theTime, temp;
      child->GetTime(&theTime);
      LL_DIV(temp, theTime, PR_USEC_PER_SEC);
      LL_L2I(historyItem->accessed, temp);
      *aHistoryList = g_list_insert(*aHistoryList, historyItem, i);
    }
  }

  *aCounter = childCount;
  root->SetContainerOpen(PR_FALSE);
  return NS_OK;
}

gint
g_mozilla_cpp_get_history_list(GList** history_list,
                               int cur_count)
{
  gint count = 0;

  nsCOMPtr<nsINavHistoryService> histSvc =
    do_GetService(NS_NAVHISTORYSERVICE_CONTRACTID);
  if (!histSvc) {
    return count;
  }

  nsCOMPtr<nsINavHistoryQuery> query;
  histSvc->GetNewQuery(getter_AddRefs(query));
  if (!query) {
    return count;
  }

  nsCOMPtr<nsINavHistoryQueryOptions> options;
  histSvc->GetNewQueryOptions(getter_AddRefs(options));
  if (!options) {
    return count;
  }

  nsCOMPtr<nsINavHistoryResult> result;
  histSvc->ExecuteQuery(query, options, getter_AddRefs(result));
  if (!result) {
    return count;
  }

  result->SetSortingMode(nsINavHistoryQueryOptions::SORT_BY_DATE_DESCENDING);

  nsCOMPtr<nsINavHistoryContainerResultNode> rootNode;
  result->GetRoot(getter_AddRefs(rootNode));
  if (!rootNode) {
    return count;
  }

  if (NS_OK != get_content_list(history_list, &count, rootNode, cur_count)) {
    return 0;
  }

  return count;
}

gint
g_mozilla_cpp_history_list_remove_all()
{
  nsresult rv(NS_ERROR_FAILURE);
  nsCOMPtr<nsINavHistoryService> histSvc =
    do_GetService(NS_NAVHISTORYSERVICE_CONTRACTID);

  if (!histSvc) {
    return rv;
  }

  nsCOMPtr<nsIBrowserHistory> browserHistory = do_QueryInterface(histSvc, &rv);
  if (NS_OK == rv) {
    browserHistory->RemoveAllPages();
  }

  return rv;
}

gint
g_mozilla_cpp_shistory_list_remove_all(GMozillaWeb *mozweb)
{
  NS_ENSURE_ARG_POINTER(mozweb);
  nsresult rv(NS_ERROR_FAILURE);
  GMozillaEngine *engine = NULL;
  GtkMozEmbed *embed = NULL;
  nsCOMPtr<nsIWebBrowser> browser;
  nsCOMPtr<nsIWebNavigation> navigation;
  nsCOMPtr<nsISHistory> shistory;
  GSList *c = NULL;
  PRInt32 count = 0;

  // we must clear all session histories
  for (c = (GSList*) mozweb->window_list; c; c = c->next)
  {
    count = 0;
    engine = NULL;
    embed = NULL;
    browser = nsnull;

    engine = G_MOZILLA_ENGINE(c->data);
    embed = GTK_MOZ_EMBED(engine->engine);
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(browser));
    navigation = do_QueryInterface(browser, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = navigation->GetSessionHistory(getter_AddRefs(shistory));
    NS_ENSURE_SUCCESS(rv, rv);
    shistory->GetCount(&count);
    shistory->PurgeHistory(count);
  }

  return rv;
}

gint g_mozilla_cpp_history_list_remove(const gchar* url)
{
  if (!url) {
    return NS_ERROR_INVALID_ARG;
  }

  nsresult rv(NS_ERROR_FAILURE);
  nsCOMPtr<nsINavHistoryService> histSvc =
    do_GetService(NS_NAVHISTORYSERVICE_CONTRACTID);

  if (!histSvc) {
    return rv;
  }

  nsCOMPtr<nsIBrowserHistory> browserHistory = do_QueryInterface(histSvc, &rv);
  if (NS_FAILED(rv)) {
    return rv;
  }

  nsCOMPtr<nsIURI> uri;
  rv = NS_NewURI(getter_AddRefs(uri), url);
  if (NS_OK == rv) {
    browserHistory->RemovePage(uri);
  }
  return rv;
}

int
g_mozilla_reflow_content(GtkMozEmbed *embed)
{
  nsresult rv;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(embed, getter_AddRefs(webBrowser));
  nsCOMPtr<nsIBaseWindow> baseWindow = do_QueryInterface(webBrowser, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = baseWindow->SetPosition(0, 0);
  return rv;
}

int
g_mozilla_engine_download_start(GMozillaTransferItem *item)
{
  nsresult rv = NS_OK;
  NS_ENSURE_ARG_POINTER(item);
  if (item->download_id < 0) {
    PRUint32 id = 0;
    nsCOMPtr<nsIDownload> download;
    rv = MicrobEalDownloadMgr::InitTransferItem(item->server, item->file_name, &id, getter_AddRefs(download));
    if (NS_SUCCEEDED(rv))
      item->download_id = id;
  }
  if (item->download_id >= 0)
    rv = MicrobEalDownloadMgr::StartDownload(item->download_id);
  return rv;
}

int
g_mozilla_engine_download_pause(GMozillaTransferItem *item)
{
  nsresult rv;
  NS_ENSURE_ARG_POINTER(item);
  nsCOMPtr<nsIDownloadManager> dm = do_GetService(G_MOZILLA_DOWNLOAD_MANAGER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = dm->PauseDownload(item->download_id);
  return rv;
}

int
g_mozilla_engine_download_cancel(GMozillaTransferItem *item)
{
  nsresult rv;
  NS_ENSURE_ARG_POINTER(item);
  nsCOMPtr<nsIDownloadManager> dm = do_GetService(G_MOZILLA_DOWNLOAD_MANAGER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = dm->CancelDownload(item->download_id);
  return rv;
}

int
g_mozilla_engine_download_setup(void **listener, void *web_global)
{
  nsresult rv;
  rv = MicrobEalDownloadMgr::SetupDownloadListener(listener, web_global);
  return rv;
}

int
g_mozilla_engine_get_downloads_count(void)
{
  nsresult rv;
  nsCOMPtr<nsIDownloadManager> dm = do_GetService(G_MOZILLA_DOWNLOAD_MANAGER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  PRInt32 activeDownloadCount = 0;
  dm->GetActiveDownloadCount(&activeDownloadCount);
  return activeDownloadCount;
}

GList*
g_mozilla_engine_get_download_items(void *web_global)
{
  nsresult rv;
  NS_ENSURE_TRUE(web_global, 0);

  nsCOMPtr<nsISimpleEnumerator> dirIterator;
  nsCOMPtr<nsIDownloadManager> dm = do_GetService(G_MOZILLA_DOWNLOAD_MANAGER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, nsnull);

  dm->GetActiveDownloads(getter_AddRefs(dirIterator));
  GList *ret_list = nsnull;

  PRBool hasMore = PR_FALSE;
  while (dirIterator->HasMoreElements(&hasMore), hasMore) {
    nsCOMPtr<nsIDownload> download;
    rv = dirIterator->GetNext((nsISupports**)getter_AddRefs(download));
    if (NS_FAILED(rv))
      continue;
    PRUint32 id;
    PRInt64 size, transferred;
    nsCAutoString src_spec, target_spec;
    nsCOMPtr<nsIURI> source;
    nsCOMPtr<nsIURI> target;
    download->GetSource(getter_AddRefs(source));
    download->GetTarget(getter_AddRefs(target));
    source->GetSpec(src_spec);
    target->GetSpec(target_spec);
    download->GetId(&id);
    download->GetSize(&size);
    download->GetAmountTransferred(&transferred);
    GMozillaTransferItem *item = (GMozillaTransferItem*)g_mozilla_transfer_item_new_with_url(src_spec.get(), target_spec.get());
    item->download_id = id;
    item->file_size = size;
    item->downloaded_size = transferred;
    ret_list = g_list_append(ret_list, item);
    g_object_set_data(G_OBJECT(item), "global", web_global);
  }
  ((GMozillaWeb*)web_global)->ongoing_download_list = ret_list;
  return ret_list;
}

int
g_mozilla_engine_save_target(void *engine, const char* aUrl,
                             const char* aDestination, int aSetting, void *aCtx)
{
  ULOG("from:%s, to:%s, setting:%i, ctx:%p\n", aUrl, aDestination, aSetting, aCtx);
  NS_ENSURE_ARG_POINTER(engine);
  int mode = 2;
  g_signal_emit_by_name (G_OBJECT(engine), G_WEBWIDGET_SIGNAL_DOWNLOAD_REQUEST, aUrl, aDestination, "", (gulong)0, mode);
  GMozillaWeb* web = G_MOZILLA_WEB(((GMozillaEngine*)engine)->global);
  if (!web)
      return FALSE;
  GList *list = web->ongoing_download_list;
  if (!list)
      return FALSE;
  GMozillaTransferItem *item = (GMozillaTransferItem *)g_list_nth_data(list, g_list_length(list) - 1);
  if (!item)
      return FALSE;
  nsresult rv = NS_OK;
  nsCOMPtr<nsIDownload> download;
  if (item->download_id < 0) {
    PRUint32 id = 0;
    rv = MicrobEalDownloadMgr::InitTransferItem(item->server, item->file_name, &id, getter_AddRefs(download));
    if (NS_SUCCEEDED(rv))
      item->download_id = id;
  }
  void* webbrowser = aCtx;
  PRBool isDocument = (aSetting < 2);
  nsCOMPtr<nsIWebBrowser> webBrowser;
  if (!webbrowser && isDocument) {
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(G_MOZILLA_ENGINE(engine)->engine), getter_AddRefs(webBrowser));
    webbrowser = webBrowser;
  }
  rv = MicrobEalDownloadMgr::StartDownload(item->download_id, download, isDocument, webbrowser, (aSetting == 1));
  return NS_SUCCEEDED(rv)?TRUE:FALSE;
}

int
g_mozilla_engine_observe(const char* service_id,
                         void *object,
                         const char* topic,
                         gunichar2* data)
{
  nsresult rv;
  if (service_id && *service_id) {
    nsCOMPtr<nsISupports> service = do_GetService(service_id, &rv);
    NS_ENSURE_SUCCESS(rv, FALSE);
    nsCOMPtr<nsIObserver> Observer = do_QueryInterface(service, &rv);
    NS_ENSURE_SUCCESS(rv, FALSE);
    rv = Observer->Observe((nsISupports*)object, topic, (PRUnichar*)data);
  } else {
    //This is the correct?
    nsCOMPtr<nsIObserverService> obsService =
      do_GetService(G_MOZILLA_OBSERVER_SERVICE_CONTRACTID, &rv);
    if (obsService)
      rv = obsService->NotifyObservers((nsISupports*)object, topic, (PRUnichar*)data);
  }
  return NS_FAILED(rv) ? false : true;
}

int
g_mozilla_engine_get_pref_type(const char *name)
{
  g_return_val_if_fail (name != NULL, G_TYPE_INVALID);

  int rv = G_TYPE_INVALID;
  PRInt32 type;

  nsCOMPtr<nsIPrefBranch> pref = do_GetService(NS_PREFSERVICE_CONTRACTID);
  if (pref) {
    pref->GetPrefType(name, &type);
    switch (type) {
    case nsIPrefBranch::PREF_STRING: {
      rv = G_TYPE_STRING;
      break;
    }
    case nsIPrefBranch::PREF_INT: {
      rv = G_TYPE_INT;
      break;
    }
    case nsIPrefBranch::PREF_BOOL: {
      rv = G_TYPE_BOOLEAN;
      break;
    }
    default:
      break;
    }
  }
  return rv;
}

gboolean
g_mozilla_engine_pref_has_user_value(const char *name)
{
  nsresult rv;
  nsCOMPtr<nsIPrefBranch> prefService = do_GetService(NS_PREFSERVICE_CONTRACTID, &rv);
  NS_ENSURE_TRUE(prefService, rv);
  PRBool val = PR_FALSE;
  return NS_SUCCEEDED(prefService->PrefHasUserValue(name, &val)) && val;
}

int
g_mozilla_engine_set_pref(GtkType type, const char *name, void *value)
{
  g_return_val_if_fail (name != NULL, FALSE);
  g_return_val_if_fail (value != NULL, FALSE);

  nsCOMPtr<nsIPrefBranch> pref = do_GetService(NS_PREFSERVICE_CONTRACTID);

  if (pref) {
    nsresult rv = NS_ERROR_FAILURE;
    PRInt32 curType = 0;
    // Make sure that we can set the preference even if the old type doesn't match
    rv = pref->GetPrefType(name, &curType);
    switch (type) {
    case GTK_TYPE_BOOL: {
      if (NS_SUCCEEDED(rv) && curType != nsIPrefBranch::PREF_BOOL)
        pref->ClearUserPref(name);
      /* I doubt this cast pair is correct */
      rv = pref->SetBoolPref(name, !!*(int*)value);
      break;
    }
    case GTK_TYPE_INT: {
      if (NS_SUCCEEDED(rv) && curType != nsIPrefBranch::PREF_INT)
        pref->ClearUserPref(name);
      /* I doubt this cast pair is correct */
      rv = pref->SetIntPref(name, *(int*)value);
      break;
    }
    case GTK_TYPE_STRING: {
      if (NS_SUCCEEDED(rv) && curType != nsIPrefBranch::PREF_STRING)
        pref->ClearUserPref(name);
      g_return_val_if_fail (value, FALSE);
      rv = pref->SetCharPref(name, (gchar*)value);
      break;
    }
    default:
      break;
    }
    return NS_SUCCEEDED(rv);
  }
  return FALSE;
}

int
g_mozilla_engine_get_pref(GtkType type, const char *name, void *value)
{
  g_return_val_if_fail (name != NULL, FALSE);

  nsCOMPtr<nsIPrefBranch> pref = do_GetService(NS_PREFSERVICE_CONTRACTID);

  nsresult rv = NS_ERROR_FAILURE;
  if (pref) {
    switch (type) {
    case GTK_TYPE_BOOL: {
      rv = pref->GetBoolPref(name, (gboolean*)value);
      break;
    }
    case GTK_TYPE_INT: {
      rv = pref->GetIntPref(name, (gint*)value);
      break;
    }
    case GTK_TYPE_STRING: {
      rv = pref->GetCharPref(name, (gchar**)value);
      break;
    }
    default:
      break;
    }
    return NS_SUCCEEDED(rv);
  }
  return FALSE;
}

int
g_mozilla_engine_save_prefs(void)
{
  nsresult rv;
  nsCOMPtr<nsIPrefService> prefService = do_GetService(NS_PREFSERVICE_CONTRACTID, &rv);
  NS_ENSURE_TRUE(prefService, rv);
  return NS_SUCCEEDED(prefService->SavePrefFile (nsnull));
}

char *
g_moz_engine_get_encoding(GtkMozEmbed *embed, int frame_number)
{
  char *encoding = nsnull;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, 0);
  nsCOMPtr<nsIDocCharset> docCharset = do_GetInterface(webBrowser);
  NS_ENSURE_TRUE(docCharset, 0);
  docCharset->GetCharset(&encoding);
  return encoding;
}

int
g_moz_engine_set_encoding(GtkMozEmbed *embed, const char *encoding, int frame_number)
{
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  nsCOMPtr<nsIContentViewer> contentViewer;
  GetContentViewer(webBrowser, getter_AddRefs(contentViewer));
  NS_ENSURE_TRUE (contentViewer, NS_ERROR_FAILURE);
  nsCOMPtr<nsIMarkupDocumentViewer> mDocViewer = do_QueryInterface(contentViewer);
  NS_ENSURE_TRUE (mDocViewer, NS_ERROR_FAILURE);
  return mDocViewer->SetForceCharacterSet(nsDependentCString(encoding));
}

int
g_mozilla_engine_get_shistory_list(GMozillaEngine *mozilla_engine,
                                   GList **history_list,
                                   unsigned int type)
{
  g_return_val_if_fail ((mozilla_engine != NULL), 0);
  NS_ENSURE_TRUE(history_list, 0);
  GtkMozEmbed *embed = GTK_MOZ_EMBED(mozilla_engine->engine);

  PRInt32 curIndex = 0, totalCount = 0, navIndex = 0, maxItems = 0;
  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  nsCOMPtr<nsIWebNavigation> navigation = do_QueryInterface(webBrowser, &rv);
  NS_ENSURE_TRUE (navigation, NS_ERROR_FAILURE);

  nsCOMPtr<nsISHistory> sessionHistory;
  rv = navigation->GetSessionHistory(getter_AddRefs(sessionHistory));
  NS_ENSURE_SUCCESS(rv, rv);

  //Get the current index at session History
  sessionHistory->GetIndex(&curIndex);
  //Gets the number of toplevel documents available in session history.
  sessionHistory->GetCount(&totalCount);

  if (type == GTK_MOZ_COMMON_BACK_SHISTORY) {
    navIndex = curIndex;
    maxItems = curIndex + 1;
  } else if (type == GTK_MOZ_COMMON_FORWARD_SHISTORY) {
    navIndex = curIndex + 1;
    maxItems = totalCount - navIndex;
  }

  if (maxItems <= 0)
    return 0;

  GtkWebHistoryItem *item = NULL;
  
  for (PRInt32 numItems = 0; numItems < maxItems; numItems++) {
    // Get the HistoryEntry at a given index.
    nsCOMPtr<nsIHistoryEntry> curEntry;
    rv = sessionHistory->GetEntryAtIndex((navIndex), PR_FALSE,
                                         getter_AddRefs(curEntry));
    if (NS_FAILED(rv) || (!curEntry))
      continue;

    // Get the URI of the HistoryEntry
    nsCOMPtr<nsIURI> uri;
    rv = curEntry->GetURI(getter_AddRefs(uri));
    if (NS_FAILED(rv) || (!uri))
      continue;

    nsCString uriString;
    rv = uri->GetSpec(uriString);
    if (NS_FAILED(rv) || uriString.IsEmpty())
      continue;

    // Get the title of the HistoryEntry
    PRUnichar* title;
    rv = curEntry->GetTitle (&title);
    if (NS_FAILED(rv) || (!title))
      continue;

    item = g_try_new0(GtkWebHistoryItem, 1);
    if (!item)
      continue;
    
    item->url = strdup(uriString.get());
    item->title = strdup(NS_ConvertUTF16toUTF8(title).get());
    item->accessed = 0;
    *history_list = g_list_insert(*history_list, item, numItems);

    if (type == GTK_MOZ_COMMON_BACK_SHISTORY)
      navIndex--;
    else if (type == GTK_MOZ_COMMON_FORWARD_SHISTORY)
      navIndex++;
  }

  return (int)maxItems;
}

int
g_mozilla_engine_get_shistory_index(GtkMozEmbed *embed)
{
  g_return_val_if_fail ((embed != NULL), -1);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), -1);

  PRInt32 curIndex = -1;
  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, 0);
  nsCOMPtr<nsIWebNavigation> navigation = do_QueryInterface(webBrowser, &rv);
  NS_ENSURE_TRUE (navigation, curIndex);

  nsCOMPtr<nsISHistory> sessionHistory;
  rv = navigation->GetSessionHistory(getter_AddRefs(sessionHistory));
  NS_ENSURE_SUCCESS(rv, curIndex);

  sessionHistory->GetIndex(&curIndex);

  return (int)curIndex;
}

int
g_mozilla_engine_shistory_goto_index(GtkMozEmbed *embed, int index)
{
  g_return_val_if_fail (embed != NULL, NS_ERROR_FAILURE);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), NS_ERROR_FAILURE);

  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  nsCOMPtr<nsIWebNavigation> navigation = do_QueryInterface(webBrowser, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  return navigation->GotoIndex(index);
}

int
g_mozilla_engine_clipboard(GtkMozEmbed *embed, unsigned int action)
{
  g_return_val_if_fail(embed != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, FALSE);

  nsresult rv = NS_OK;
  nsCOMPtr <nsIDOMWindow> window;
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));

  nsCOMPtr<nsIDOMDocument> ddoc;
  window->GetDocument(getter_AddRefs(ddoc));
  NS_ENSURE_TRUE(ddoc, NS_OK);

  nsCOMPtr<nsIDocument> doc;
  doc = do_QueryInterface(ddoc, &rv);
  NS_ENSURE_TRUE(doc, NS_OK);

  nsIPresShell *presShell = doc->GetPrimaryShell();
  if (presShell) {
    nsCOMPtr<nsISelectionController> selCon = do_QueryInterface(presShell);
    if (selCon)
      selCon->SetDisplaySelection(nsISelectionController::SELECTION_ON);
  }

  PRBool canDo = PR_TRUE;
  nsCOMPtr<nsIClipboardCommands> clipboard(do_GetInterface(webBrowser));
  NS_ENSURE_TRUE(clipboard, FALSE);
  NS_ENSURE_TRUE(webBrowser, FALSE);

  nsCOMPtr<nsIWebBrowserFocus> webBrowserFocus(do_QueryInterface(webBrowser));
  if (webBrowserFocus)
    webBrowserFocus->Activate();

  switch (action) {
  case GTK_MOZ_CLIPBOARD_SELECT_ALL: {
    rv = clipboard->SelectAll();
    break;
  }
  case GTK_MOZ_CLIPBOARD_CUT: {
    rv = clipboard->CutSelection();
    break;
  }
  case GTK_MOZ_CLIPBOARD_COPY: {
    rv = clipboard->CopySelection();
    break;
  }
  case GTK_MOZ_CLIPBOARD_PASTE: {
    rv = clipboard->Paste();
    break;
  }
  case GTK_MOZ_CLIPBOARD_CAN_CUT: {
    rv = clipboard->CanCutSelection(&canDo);
    break;
  }
  case GTK_MOZ_CLIPBOARD_CAN_PASTE: {
    rv = clipboard->CanPaste(&canDo);
    break;
  }
  case GTK_MOZ_CLIPBOARD_CAN_COPY: {
    rv = clipboard->CanCopySelection(&canDo);
    break;
  }
  default:
    break;
  }
  NS_ENSURE_SUCCESS(rv, FALSE);
  return canDo?TRUE:FALSE;
}

unsigned int
g_mozilla_engine_get_security_mode(unsigned int sec_state)
{
  const unsigned int wpl_security_bits = nsIWebProgressListener::STATE_IS_SECURE |
                                         nsIWebProgressListener::STATE_IS_BROKEN |
                                         nsIWebProgressListener::STATE_IS_INSECURE |
                                         nsIWebProgressListener::STATE_SECURE_HIGH |
                                         nsIWebProgressListener::STATE_SECURE_MED |
                                         nsIWebProgressListener::STATE_SECURE_LOW;

  /* sec_state is defined as a bitmask that may be extended in the future.
   * We filter out any unknown bits before testing for known values.
   */
  unsigned int sec_mode;
  switch (sec_state & wpl_security_bits) {
  case nsIWebProgressListener::STATE_IS_INSECURE:
    sec_mode = G_WEBENGINE_NO_SECURITY;
    break;
  case nsIWebProgressListener::STATE_IS_BROKEN:
    sec_mode = G_WEBENGINE_NO_SECURITY;
    break;
  case nsIWebProgressListener::STATE_IS_SECURE |
      nsIWebProgressListener::STATE_SECURE_HIGH:
    sec_mode = G_WEBENGINE_HIGH_SECURITY;
    break;
  case nsIWebProgressListener::STATE_IS_SECURE |
      nsIWebProgressListener::STATE_SECURE_MED:
    sec_mode = G_WEBENGINE_MEDIUM_SECURITY;
    break;
  case nsIWebProgressListener::STATE_IS_SECURE |
      nsIWebProgressListener::STATE_SECURE_LOW:
    sec_mode = G_WEBENGINE_LOW_SECURITY;
    break;
  default:
    sec_mode = G_WEBENGINE_UNKNOWN_SECURITY;
    break;
  }
  return sec_mode;
}

static PRBool
g_mozilla_get_node_value(nsIRDFDataSource *eDataSource,
                         nsIRDFResource *itemResource,
                         nsIRDFService *rdfService,
                         const char* nodeType, PRUnichar **retval)
{
  nsCOMPtr<nsIRDFResource> nodeRDFResource;
  rdfService->GetResource(NS_LITERAL_CSTRING(nodeType), getter_AddRefs(nodeRDFResource));
  if (!nodeRDFResource)
    return PR_FALSE;

  nsCOMPtr<nsIRDFNode> itemNode;
  eDataSource->GetTarget(itemResource, nodeRDFResource, PR_TRUE, getter_AddRefs(itemNode));
  if (!itemNode)
    return PR_FALSE;

  nsCOMPtr<nsIRDFLiteral> itemNodeLiteral = do_QueryInterface(itemNode);
  if (!itemNodeLiteral)
    return PR_FALSE;

  PRUnichar *itemType = nsnull;
  itemNodeLiteral->GetValue(&itemType);
  *retval= itemType;

  return PR_TRUE;
}

static PRBool
g_mozilla_get_component_list(PRUint32 *length, GList **pluginArray)
{
  PRUint32 count = 0;

  nsIUpdateItem **extensions = nsnull;

  nsCOMPtr<nsIExtensionManager> eManager(do_GetService(G_MOZILLA_EXTENSIONS_MANAGER_CONTRACTID));
  if (!eManager)
    return PR_FALSE;

  nsCOMPtr<nsIRDFService> rdfService;
  rdfService = do_GetService(G_MOZILLA_RDF_SERVICE_CONTRACTID);
  if (!rdfService)
    return PR_FALSE;

  nsCOMPtr<nsIRDFDataSource> eDataSource;
  eManager->GetDatasource(getter_AddRefs(eDataSource));
  if (!eDataSource)
    return PR_FALSE;

  eManager->GetItemList(nsIUpdateItem::TYPE_EXTENSION, &count,
                        &extensions);

  if (!pluginArray)
    return FALSE;

  NS_ENSURE_TRUE(extensions, PR_FALSE);
  for (PRUint32 i = 0; i < count; i++) {
    nsIUpdateItem *extension = extensions[i];
    if (extension) {
      GtkMicrobPlugin *list_item = g_new0(GtkMicrobPlugin, 1);
      nsAutoString itemName;
      extension->GetName(itemName);
      if (itemName.IsEmpty())
        return PR_FALSE;

      nsAutoString itemId;
      extension->GetId(itemId);
      if (itemId.IsEmpty())
        return PR_FALSE;

      nsCString resourceID(G_MOZILLA_RESOURCES);
      resourceID.Append(NS_ConvertUTF16toUTF8(itemId));
      nsCOMPtr<nsIRDFResource> itemResource;
      rdfService->GetResource(resourceID, getter_AddRefs(itemResource));
      if (!itemResource)
        return PR_FALSE;

      PRUnichar *extAppDisabled = nsnull;
      g_mozilla_get_node_value(eDataSource, itemResource, rdfService,
                               EM_RDF_EXT_APP_DISABLED, &extAppDisabled);
      if ((extAppDisabled) && (nsDependentString(extAppDisabled).Equals(NS_LITERAL_STRING("true"))))
        return PR_FALSE;

      list_item->title = g_strdup((char *)NS_ConvertUTF16toUTF8(itemName).get());
      list_item->path =  g_strdup((char *)NS_ConvertUTF16toUTF8(itemId).get());

      PRUnichar *itemCreator = nsnull;
      g_mozilla_get_node_value(eDataSource, itemResource, rdfService,
                               EM_RDF_EXT_CREATOR, &itemCreator);
      if (itemCreator)
        list_item->creator = g_strdup(NS_ConvertUTF16toUTF8(itemCreator).get());

      PRUnichar *itemDescrption = nsnull;
      g_mozilla_get_node_value(eDataSource, itemResource, rdfService,
                               EM_RDF_EXT_DES, &itemDescrption);
      if (itemDescrption)
        list_item->type = g_strdup(NS_ConvertUTF16toUTF8(itemDescrption).get());

      PRUnichar *itemState = nsnull;
      g_mozilla_get_node_value(eDataSource, itemResource, rdfService,
                               EM_RDF_EXT_STATE, &itemState);
      if ((itemState) && (nsDependentString(itemState).Equals(NS_LITERAL_STRING("true"))))
        list_item->isDisabled = TRUE;
      else
        list_item->isDisabled = FALSE;

      list_item->isPlugin = FALSE;
      *pluginArray = g_list_append(*pluginArray, list_item);
      *length = *length + 1;
    }
  }
  return PR_TRUE;
}

static PRBool
g_mozilla_get_npplugin_list(PRUint32 *aLength, GList **pluginArray)
{
  nsresult rv;
  nsCOMPtr<nsIPluginHost> pluginHost(do_GetService(MOZ_PLUGIN_HOST_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, FALSE);

  pluginHost->ReloadPlugins(PR_FALSE);  //FIXME XXX MEMLEAK
  pluginHost->GetPluginCount(aLength);

  if (!pluginArray)
    return FALSE;

  nsIDOMPlugin **aItems = new nsIDOMPlugin*[*aLength];
  NS_ENSURE_TRUE(aItems, FALSE); // OUT OF Memory

  rv = pluginHost->GetPlugins(*aLength, aItems);
  if (NS_FAILED(rv)) {
    for (PRUint32 i = 0; i < *aLength; i++)
      NS_IF_RELEASE(aItems[i]);
    delete [] aItems;
    return FALSE;
  }

  nsString string;
  for (int plugin_index = 0; plugin_index < (int) *aLength; plugin_index++) {
    GtkMicrobPlugin *list_item = g_new0(GtkMicrobPlugin, 1);
    if (!list_item)
      continue;

    rv = aItems[plugin_index]->GetName(string);
    if (!NS_FAILED(rv))
      list_item->title = g_strdup(NS_ConvertUTF16toUTF8(string).get());

    aItems[plugin_index]->GetFilename(string);
    if (!NS_FAILED(rv))
      list_item->path = g_strdup(NS_ConvertUTF16toUTF8(string).get());

    nsCOMPtr<nsIDOMMimeType> mimeType;
    PRUint32 mime_count = 0;
    rv = aItems[plugin_index]->GetLength(&mime_count);
    if (NS_FAILED(rv))
      continue;

    nsString single_mime;
    string.SetLength(0);
    for (int mime_index = 0; mime_index < (int)mime_count; ++mime_index) {
      rv = aItems[plugin_index]->Item(mime_index, getter_AddRefs(mimeType));
      if (NS_FAILED(rv))
        continue;
      rv = mimeType->GetDescription(single_mime);
      if (!NS_FAILED(rv)) {
        if (mime_index > 0)
          string.AppendLiteral("; ");
        string.Append(single_mime);
      }
    }

    list_item->type = g_strdup(NS_ConvertUTF16toUTF8(string).get());
    list_item->isPlugin = TRUE;

    if (!NS_FAILED(rv))
      *pluginArray = g_list_append(*pluginArray, list_item);
  }
  for (PRUint32 i = 0; i < *aLength; i++)
    NS_IF_RELEASE(aItems[i]);
  delete [] aItems;
  return TRUE;
}

int g_mozilla_plugins_enable(GMozillaWeb *aWeb, gboolean aEnable)
{
  nsresult rv;
  nsCOMPtr<nsIPluginHost> pluginHost(do_GetService(MOZ_PLUGIN_HOST_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  if (aEnable) {
    pluginHost->Init();
    pluginHost->LoadPlugins();
  } else {
    pluginHost->Destroy();
  }
  return NS_OK;
}

int
g_mozilla_get_plugins_list(GList **pluginArray)
{
  PRUint32 aLength = 0;
  g_mozilla_get_npplugin_list(&aLength, pluginArray);
  g_mozilla_get_component_list(&aLength, pluginArray);
  return (int)aLength;
}

int
g_mozilla_set_extension_status(char *extensionID, int status)
{
  g_return_val_if_fail (extensionID != NULL, FALSE);

  nsCOMPtr<nsIExtensionManager> eManager(do_GetService(G_MOZILLA_EXTENSIONS_MANAGER_CONTRACTID));
  if (!eManager)
    return FALSE;

  nsAutoString id;
  id.Append(NS_ConvertUTF8toUTF16(extensionID));

  if (status)
    eManager->EnableItem(id);
  else
    eManager->DisableItem(id);
  PRBool aNeedsRestart = PR_FALSE;
  eManager->Start(&aNeedsRestart);
  if (aNeedsRestart) {
    //g_idle_add(restart_notify, NULL);
  }

  return TRUE;
}


/* Loads all images from all documents (frames and iframes) inside aDoc, recursively */
static int
g_mozilla_load_images(nsIDOMDocument *aDoc)
{
  /* look for img tags */
  nsresult rv;
  nsCOMPtr<nsIDOMHTMLDocument> htmlDoc = do_QueryInterface(aDoc, &rv);
  NS_ENSURE_SUCCESS(rv, FALSE);

  nsCOMPtr<nsIDOMHTMLCollection> imgList;
  rv = htmlDoc->GetImages(getter_AddRefs(imgList));
  NS_ENSURE_SUCCESS(rv, FALSE);

  PRUint32 imgCount;
  rv = imgList->GetLength(&imgCount);
  NS_ENSURE_SUCCESS(rv, FALSE);

  guint32 i;
  for (i = 0; i < imgCount; i++) {
    nsCOMPtr<nsIDOMNode> imgNode;
    rv = imgList->Item(i, getter_AddRefs(imgNode));
    if (NS_FAILED(rv))
      continue;

    nsCOMPtr<nsIImageLoadingContent> imgContent = do_QueryInterface(imgNode, &rv);
    if (NS_FAILED(rv))
      continue;

    imgContent->ForceReload();
  }

  /* Look for iframes */
  nsCOMPtr<nsIDOMNodeList> iframeList;
  rv = aDoc->GetElementsByTagName(NS_LITERAL_STRING("iframe"),
                                  getter_AddRefs(iframeList));
  NS_ENSURE_SUCCESS(rv, FALSE);

  PRUint32 iframeCount;
  rv = iframeList->GetLength(&iframeCount);
  NS_ENSURE_SUCCESS(rv, FALSE);

  /* look for frames */
  nsCOMPtr<nsIDOMNodeList> frameList;
  rv = aDoc->GetElementsByTagName(NS_LITERAL_STRING("frame"),
                                  getter_AddRefs(frameList));
  NS_ENSURE_SUCCESS(rv, FALSE);

  PRUint32 frameCount;
  rv = frameList->GetLength(&frameCount);
  NS_ENSURE_SUCCESS(rv, FALSE);

  if (iframeCount == 0 && frameCount == 0)
    return TRUE;

  for (i = 0; i < iframeCount; i++) {
    nsCOMPtr<nsIDOMNode> iframeNode;
    rv = iframeList->Item(i, getter_AddRefs(iframeNode));
    if (NS_FAILED(rv))
      continue;

    nsCOMPtr<nsIDOMHTMLIFrameElement> iframeElement = do_QueryInterface(iframeNode, &rv);
    if (NS_FAILED(rv))
      continue;

    nsCOMPtr<nsIDOMDocument> iframeDoc;
    rv = iframeElement->GetContentDocument(getter_AddRefs(iframeDoc));
    if (NS_FAILED(rv))
      continue;

    g_mozilla_load_images(iframeDoc);
  }

  for (i = 0; i < frameCount; i++) {
    nsCOMPtr<nsIDOMNode> frameNode;
    rv = frameList->Item(i, getter_AddRefs(frameNode));
    if (NS_FAILED(rv))
      continue;

    nsCOMPtr<nsIDOMHTMLFrameElement> frameElement = do_QueryInterface(frameNode, &rv);
    if (NS_FAILED(rv))
      continue;

    nsCOMPtr<nsIDOMDocument> frameDoc;
    rv = frameElement->GetContentDocument(getter_AddRefs(frameDoc));
    if (NS_FAILED(rv))
      continue;

    g_mozilla_load_images(frameDoc);
  }

  return TRUE;
}

int
g_mozilla_set_image_policy(GMozillaEngine *engine, int policy)
{
  // Need fix for all engines
  g_return_val_if_fail(engine != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(engine->engine), FALSE);

  nsresult rv;
  nsCOMPtr<nsIDOMDocument> domDoc;

  gboolean download_images = TRUE, curr_download_images = TRUE;
  gint moz_level = 1, curr_moz_level = 1;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));
  nsCOMPtr<nsIWebNavigation> navigation = do_QueryInterface(webBrowser, &rv);
  NS_ENSURE_TRUE (navigation, NS_ERROR_FAILURE);

  rv = navigation->GetDocument(getter_AddRefs(domDoc));
  NS_ENSURE_SUCCESS(rv, rv);

  switch (policy) {
  case G_WEBENGINE_POLICY_ALL_IMAGES:
    download_images = TRUE;
    moz_level = 1;
    break;
  case G_WEBENGINE_POLICY_NO_IMAGES:
    download_images = FALSE;
    moz_level = 2;
    break;
  case G_WEBENGINE_POLICY_LOADED_IMAGES:
    download_images = FALSE;
    moz_level = 1;
    break;
  }

  gboolean save = FALSE;

  // Get current prefs
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &curr_download_images);
  g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &curr_moz_level);

  // NOTE: G_MOZILLA_PREF_DOWNLOAD_IMAGES="microb.download_images",
  //       see: microbImageLoadingPolicy.js
  if (g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &download_images))
    save = TRUE;
  if (g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &moz_level))
    save = TRUE;

  if (save)
    g_mozilla_engine_save_prefs();

  // If settings have changed do reloads according to new policy
  if (curr_download_images != download_images || curr_moz_level != moz_level) {
    switch (policy) {
      case G_WEBENGINE_POLICY_ALL_IMAGES:
        rv = g_mozilla_load_images(domDoc);
        break;
     case G_WEBENGINE_POLICY_NO_IMAGES:
       gtk_moz_embed_reload(GTK_MOZ_EMBED(engine->engine), GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
       break;
     case G_WEBENGINE_POLICY_LOADED_IMAGES:
       rv = g_mozilla_load_images(domDoc);
       break;
    }
  }

  return rv;
}

int
g_mozilla_get_server_cert(GtkMozEmbed *embed, void **cert, void* ctx)
{
  g_return_val_if_fail(embed != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, FALSE);

  nsCOMPtr<nsIDocShell> docShell(do_GetInterface((nsISupports*)webBrowser));
  NS_ENSURE_TRUE(docShell, FALSE);

  nsCOMPtr<nsISecureBrowserUI> mSecureUI;
  docShell->GetSecurityUI(getter_AddRefs(mSecureUI));
  NS_ENSURE_TRUE(mSecureUI, FALSE);

  nsCOMPtr<nsISSLStatusProvider> mSecureProvider = do_QueryInterface(mSecureUI);
  NS_ENSURE_TRUE(mSecureProvider, FALSE);

  nsCOMPtr<nsISSLStatus> SSLStatus;
  mSecureProvider->GetSSLStatus(getter_AddRefs(SSLStatus));
  NS_ENSURE_TRUE(SSLStatus, FALSE);

  nsCOMPtr<nsIX509Cert> serverCert;
  SSLStatus->GetServerCert(getter_AddRefs(serverCert));
  NS_ENSURE_TRUE(serverCert, FALSE);

  *cert = serverCert;
  return TRUE;
}

gboolean
g_mozilla_get_page_verifier_name (GtkMozEmbed *embed, gchar **verifier_name)
{
  gpointer serverCert = NULL;
  gboolean status = FALSE;
  nsIX509Cert* cert = NULL;
  nsAutoString organization;
  gboolean ret;

  status = g_mozilla_get_server_cert(GTK_MOZ_EMBED(embed), &serverCert, NULL);

  if (status && serverCert != NULL) {
    cert = static_cast<nsIX509Cert*> (serverCert);

    cert->GetIssuerOrganization(organization);
    // dbus will free the verifier string after a copy is made to client side.
    *verifier_name = (gchar*) NS_strdup(NS_ConvertUTF16toUTF8(organization).get());
    ret = TRUE;
  } else {
    ret = FALSE;
  }

  return ret;
}

int
g_mozilla_load_image(void* node, const char *url)
{
  nsresult rv;
  nsCOMPtr<nsIImageLoadingContent> imgContent = do_QueryInterface(static_cast<nsISupports*>(node), &rv);
  nsCOMPtr<nsIURI> imgUri;
  NS_ENSURE_TRUE(imgContent, FALSE);
  rv = imgContent->GetCurrentURI(getter_AddRefs(imgUri));
  if (NS_FAILED(rv) || !imgUri)
    return FALSE;

  nsCAutoString spec;
  rv = imgUri->GetSpec(spec);
  if (NS_FAILED(rv))
    return FALSE;

  if (spec.Equals(url)) {
    gint show_value = 1, show_cur_value;
    g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &show_cur_value);
    if (show_cur_value != 1) {
      gboolean download_value = TRUE, download_cur_value;
      g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &download_cur_value);

      // Set image loading conf so that we both load the image and show it
      g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &show_value);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &download_value);
      
      imgContent->ForceReload();
      
      // Set image loading conf back to what it was
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &download_cur_value);
      g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &show_cur_value);
    } else
      imgContent->ForceReload();
  }
  return TRUE;
}

int
g_mozilla_evaluate_js(GMozillaEngine* moz_engine, const gchar* script)
{
  NS_ENSURE_ARG_POINTER(moz_engine);
  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(moz_engine->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);

  webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  nsCOMPtr<nsIScriptGlobalObject> sgo = do_QueryInterface(window);
  NS_ENSURE_TRUE(sgo, NS_ERROR_FAILURE);

  nsCOMPtr<nsIScriptContext> ctx = sgo->GetContext();
  NS_ENSURE_TRUE(ctx, NS_ERROR_FAILURE);

  nsCOMPtr<nsIScriptObjectPrincipal> sgoPrincipal = do_QueryInterface(sgo);
  NS_ENSURE_TRUE(sgoPrincipal, NS_ERROR_FAILURE);

  ctx->EvaluateString(NS_ConvertUTF8toUTF16(script), sgo->GetGlobalJSObject(),
                      sgoPrincipal->GetPrincipal(),
                      "mozembed", 0, nsnull, nsnull, nsnull);

  return NS_OK;
}

void
g_mozilla_notification_button_pressed(GtkButton *button, gpointer notification)
{
  if ((!GTK_IS_BUTTON(button)) || (NULL == notification)) {
    return;
  }

  MicrobEalNotificationButton *notif_button = (MicrobEalNotificationButton *) g_object_get_data((GObject *) button, "button-data");
  if (NULL == notif_button) {
    return;
  }

  nsresult rv;
  nsCOMPtr<nsINotificationBox> notifBox(do_GetService(NOTIFICATIONBOX_CONTRACTID, &rv));
  if (NS_SUCCEEDED(rv)) {
    notifBox->ButtonPressed((nsINotification *) notification, notif_button);
  }
}

gboolean g_moz_engine_permit_unload(GtkMozEmbed *embed)
{
    g_return_val_if_fail(embed, TRUE);
    // If something fails during execution of this method, it is better
    // to permit the unload, otherwise this prevents the user to close
    // the window. 
    gboolean okToUnload = TRUE;
    nsCOMPtr<nsIContentViewer> contentViewer;
    nsCOMPtr <nsIWebBrowser> webBrowser;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));
    NS_ENSURE_TRUE(webBrowser, TRUE);
    GetContentViewer(webBrowser, getter_AddRefs(contentViewer));
    NS_ENSURE_TRUE(contentViewer, TRUE);

    // This call invokes the unload confirmation query if such wanted:
    nsresult rv = contentViewer->PermitUnload(PR_TRUE, &okToUnload);
    NS_ENSURE_SUCCESS(rv, TRUE);
    return okToUnload;
}

gint
g_mozilla_get_text_zoom(GtkMozEmbed *embed)
{
  gint zoom_level = 100;
  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));

  nsCOMPtr<nsIContentViewer> contentViewer;
  GetContentViewer(webBrowser, getter_AddRefs(contentViewer));

  nsCOMPtr<nsIMarkupDocumentViewer> markupViewer = do_QueryInterface(contentViewer);
  NS_ENSURE_TRUE(markupViewer, -1);

  float zoomLevelFloat = 0.0;
  rv = markupViewer->GetTextZoom(&zoomLevelFloat);
  NS_ENSURE_SUCCESS(rv, -1);
  zoom_level = (int)round(zoomLevelFloat * 100.0);

  return zoom_level;
}

gboolean
g_mozilla_set_text_zoom(GtkMozEmbed *embed, gint zoom_value)
{
  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), getter_AddRefs(webBrowser));

  nsCOMPtr<nsIContentViewer> contentViewer;
  GetContentViewer(webBrowser, getter_AddRefs(contentViewer));
  NS_ENSURE_TRUE (contentViewer, FALSE);

  nsCOMPtr<nsIMarkupDocumentViewer> markupViewer = do_QueryInterface(contentViewer);
  NS_ENSURE_TRUE(markupViewer, FALSE);

  float zoomLevelFloat = (float) zoom_value / 100.0;
  rv = markupViewer->SetTextZoom(zoomLevelFloat);

  NS_ENSURE_TRUE(rv, FALSE);

  return TRUE;
}

int
g_mozilla_initialize_extensions(void)
{
  nsresult rv(NS_ERROR_FAILURE);
  nsCOMPtr<nsIExtensionManager> em(do_GetService(G_MOZILLA_EXTENSIONS_MANAGER_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);
  PRBool needsRestart = PR_FALSE;
  rv = em->CheckForMismatches(&needsRestart);
  if (!needsRestart) {
    em->Start(&needsRestart);
  }
  nsCOMPtr<nsIComponentRegistrar> cr;

#ifndef DEBUG // by some unknown reason it cause crash
  NS_GetComponentRegistrar(getter_AddRefs(cr));
#endif

  if (cr)
    cr->AutoRegister(nsnull);

  return needsRestart;
}

void
draw_layout(GMozillaEngine *engine, int id, int left, int top, int width, int height, int zoom)
{
  g_return_if_fail(engine);
  g_return_if_fail(GTK_IS_MOZ_EMBED(engine->engine));

  MicrobEalXshmRenderer *mshm = (MicrobEalXshmRenderer*)g_object_get_data(G_OBJECT(engine->global), G_MOZILLA_PAINT_LISTENER);

  if (!mshm)
    return;

  if (!mshm->mShmData.id)
    return;

  GdkRectangle r = { left, top, width, height };
  mshm->PushUpdateRequest(engine, r, id, zoom);
}

static void
DispatchSelectElementEvents(nsIDOMHTMLSelectElement *element, PRBool beforeShow)
{
  nsresult rv;
  nsCOMPtr<nsIDOMEventTarget> target(do_QueryInterface(element));
  if (!target)
    return;

  nsCOMPtr<nsIDOMDocument> ownerDocument;
  element->GetOwnerDocument(getter_AddRefs(ownerDocument));
  nsCOMPtr<nsIDOMDocumentEvent> document(do_QueryInterface(ownerDocument));
  if (!document)
    return;

  nsCOMPtr<nsIDOMEvent> event;
  rv = document->CreateEvent(NS_LITERAL_STRING("Events"),
                             getter_AddRefs(event));
  if (NS_FAILED(rv))
    return;

  nsCOMPtr<nsIPrivateDOMEvent> privateEvent(do_QueryInterface(event));

  if (!privateEvent)
    return;

  rv = privateEvent->SetTrusted(PR_TRUE);
  if (NS_FAILED(rv))
    return;

  PRBool defaultActionEnabled = PR_TRUE;
  if (beforeShow) {
    rv = event->InitEvent(NS_LITERAL_STRING("focus"), PR_TRUE, PR_TRUE);
    if (NS_FAILED(rv))
      return;
    target->DispatchEvent(event, &defaultActionEnabled);
  } else {
    rv = event->InitEvent(NS_LITERAL_STRING("change"), PR_TRUE, PR_TRUE);
    if (NS_FAILED(rv))
       return;

     PRBool defaultActionEnabled = PR_TRUE;
     target->DispatchEvent(event, &defaultActionEnabled);

    rv = event->InitEvent(NS_LITERAL_STRING("click"), PR_TRUE, PR_TRUE);
    if (NS_FAILED(rv))
      return;
    target->DispatchEvent(event, &defaultActionEnabled);
  }
}

void 
g_mozilla_select_widget_item_selected (GMozillaEngine *self, gpointer select, gint* selected_items, gint count)
{
  if (!select) 
    return;

  nsCOMPtr<nsIDOMHTMLSelectElement> element = do_QueryInterface((nsISupports*)select);
  nsCOMPtr<nsISelectElement> select_element = do_QueryInterface(element);
  NS_ENSURE_TRUE(select_element, );

  nsresult rv;
  PRBool mult;
  element->GetMultiple(&mult);

  if (!mult)
  {
    rv = element->SetSelectedIndex(selected_items[0]);
    if (NS_FAILED(rv))
      return;
  }
  else
  {
    //uselect all elements
    element->SetSelectedIndex(-1);
    for (gint i = 0; i < count; ++i)
    {
      rv = select_element->SetOptionsSelectedByIndex(selected_items[i], selected_items[i], PR_TRUE, PR_FALSE, PR_TRUE, PR_TRUE, nsnull);
      if (NS_FAILED(rv))
        return;
    }
  }
  DispatchSelectElementEvents(element, PR_FALSE);
}

nsresult
create_select_widget (GMozillaEngine *self, gpointer dom_node,
                      gpointer original_node)
{
#ifdef HILDON_2_2
   g_return_val_if_fail(self, NS_ERROR_FAILURE);

   if (self->dialog_is_active)
     return NS_OK;

   if (!dom_node)
     return NS_ERROR_FAILURE;

   self->dialog_is_active = true;

   nsresult rv;
   nsCOMPtr<nsIDOMHTMLSelectElement> select;
   nsCOMPtr<nsIDOMNode> aDOMNode = do_QueryInterface((nsISupports*)dom_node, &rv);
   nsCOMPtr<nsIDOMHTMLOptionElement> origEl = do_QueryInterface((nsISupports*)original_node);

   NS_ENSURE_SUCCESS(rv, rv);

   select = do_QueryInterface(aDOMNode, &rv);
   NS_ENSURE_SUCCESS(rv, rv);

   PRBool elemDisabled = PR_FALSE;
   select->GetDisabled(&elemDisabled);

   if (elemDisabled) {
     self->dialog_is_active = false;
     return NS_OK; //not showing selector for disabled list
   }

   DispatchSelectElementEvents(select, PR_TRUE);
   select->Focus();

   nsCOMPtr<nsISelectElement> selectElem;
   selectElem = do_QueryInterface(select, &rv);
   NS_ENSURE_SUCCESS(rv, rv);

   PRBool multiple;
   select->GetMultiple(&multiple);
   PRUint32 length = 0;
   select->GetLength(&length);
   nsCOMPtr<nsIDOMHTMLOptionsCollection> options;
   rv = select->GetOptions(getter_AddRefs(options));
   NS_ENSURE_SUCCESS(rv, rv);

   GSList* list = NULL;
   PRBool sel;
   PRBool all_items_disabled = PR_TRUE;
   gint clicked_item = -1;
   SelectedItem* item = NULL;

   for (int i = 0; i < (int)length; i++) {
       nsCOMPtr<nsIDOMNode> node;
       options->Item(i, getter_AddRefs(node));
       if (!node) continue;
       nsCOMPtr<nsIDOMHTMLOptionElement> el = do_QueryInterface(node);
       if (!el) continue;
       PRBool nodeDisabled;
       selectElem->IsOptionDisabled(i, &nodeDisabled);
       nsString nodeTitle;

       if (!nodeDisabled)
       {
         item = g_new0(SelectedItem, 1);
         all_items_disabled = PR_FALSE;
         el->GetText(nodeTitle);
         item->name = g_strdup(NS_ConvertUTF16toUTF8(nodeTitle).get());
         el->GetSelected(&sel);

         if (sel)
           item->selected = true;
         else
           item->selected = false;

         item->number = i;

         if (origEl == el)
             clicked_item = i;

         list = g_slist_append(list, item);
       }
   }

   if (all_items_disabled) //not showing selector for disabled list
   {
     self->dialog_is_active = false;
     g_slist_free(list);
     return NS_OK;
   }

   if (list) {
       nsString title;
       nsCOMPtr<nsIDOMElement> element = do_QueryInterface(select, &rv);
       if (element)
         element->GetAttribute(NS_LITERAL_STRING("title"), title);
       g_mozilla_create_select_notification(self, list, (gpointer)select,
                                            (gboolean)multiple, title.IsEmpty() ? NULL : NS_ConvertUTF16toUTF8(title).get(), clicked_item);
   } else
       self->dialog_is_active = false;
#endif
   return NS_OK;
}

// Return node element dimensions relative to window scroll position
// nearest == TRUE - will try to search nearest dom element if current is not dom element (ex: nsIDOMText)
int
get_node_box_or_nearest(void *aNode, float *left, float *top, float *width, float *height, int nearest)
{
  NS_ENSURE_ARG_POINTER((aNode && left && top && width && height));
  nsCOMPtr <nsIDOMNode> node = do_QueryInterface((nsISupports*)aNode);
  NS_ENSURE_TRUE(node, NS_ERROR_FAILURE);
  nsCOMPtr <nsIDOMNSElement> domNSEl = do_QueryInterface((nsISupports*)aNode);
  nsresult rv;
  if (!domNSEl && nearest) {
    nsCOMPtr<nsIDOMNode> nearesNode;
    node->GetPreviousSibling(getter_AddRefs(nearesNode));
    domNSEl = do_QueryInterface(nearesNode, &rv);
    if (NS_FAILED(rv)) {
      node->GetNextSibling(getter_AddRefs(nearesNode));
      domNSEl = do_QueryInterface(nearesNode, &rv);
      if (NS_FAILED(rv)) {
        node->GetParentNode(getter_AddRefs(nearesNode));
        domNSEl = do_QueryInterface(nearesNode, &rv);
      }
    }
  }

  NS_ENSURE_TRUE(domNSEl, NS_ERROR_FAILURE);
  nsCOMPtr <nsIDOMClientRect> clRect;
  nsString str;
  node->GetNodeName(str);
  rv = domNSEl->GetBoundingClientRect(getter_AddRefs(clRect));
  NS_ENSURE_TRUE(clRect, NS_ERROR_FAILURE);
  clRect->GetTop(top);
  clRect->GetLeft(left);
  clRect->GetWidth(width);
  clRect->GetHeight(height);
  ULOG("node:'%s': [%g,%g,%g,%g]", NS_ConvertUTF16toUTF8(str).get(), *left, *top, *width, *height);
  return NS_OK;
}
 
static nsresult
CheckFrameOffSet(nsIDOMElement *aDOMElement, PRInt32 aX, PRInt32 aY, PRInt32 scrollX, PRInt32 scrollY, PRInt32 *frameOffsetTop, PRInt32 *frameOffsetLeft, nsIDOMElement **frameElement)
{
  ULOG("aDOMElement:%p, a[%i,%i], scr[%i,%i]", aDOMElement, aX, aY, scrollX, scrollY);
  NS_ENSURE_ARG_POINTER(frameOffsetTop && frameOffsetLeft && frameElement);

  int nestLevel = 0;
  nsCOMPtr<nsIDOMElement> frameEl = do_QueryInterface(aDOMElement);
  NS_ENSURE_TRUE(frameEl, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMHTMLFrameElement> frame;
  while((++nestLevel < MAX_NESTED_FRAMES) && (frame = do_QueryInterface(frameEl)))
  {
    nsCOMPtr<nsIDOMDocument> frameDoc;
    frame->GetContentDocument(getter_AddRefs(frameDoc));
    if (!frameDoc) break;

    nsCOMPtr<nsIDOMNSDocument> nsDoc = do_QueryInterface(frameDoc);
    if (!nsDoc) break;

    nsCOMPtr<nsIDOMNSHTMLElement> nsHtmlElem = do_QueryInterface(frameEl);
    if (!nsHtmlElem) break;

    nsHtmlElem->GetOffsetTop(frameOffsetTop);
    nsHtmlElem->GetOffsetLeft(frameOffsetLeft);

    PRInt32 frameX, frameY;
    frameX = aX - *frameOffsetLeft - scrollX;
    frameY = aY - *frameOffsetTop - scrollY;
    nsDoc->ElementFromPoint(frameX, frameY, getter_AddRefs(frameEl));
  }

  nestLevel = 0;
  nsCOMPtr<nsIDOMHTMLIFrameElement> iframe;
  while((++nestLevel < MAX_NESTED_FRAMES) && (iframe = do_QueryInterface(frameEl)))
  {
    nsCOMPtr<nsIDOMDocument> frameDoc;
    iframe->GetContentDocument(getter_AddRefs(frameDoc));
    if (!frameDoc) break;

    nsCOMPtr<nsIDOMNSDocument> nsDoc = do_QueryInterface(frameDoc);
    if (!nsDoc) break;

    float t = *frameOffsetTop, l = *frameOffsetLeft, w, h;
    get_node_box_or_nearest(frameEl, &l, &t, &w, &h, false);
    *frameOffsetLeft += l;
    *frameOffsetTop += t;
    PRInt32 frameX, frameY;
    frameX = aX - *frameOffsetLeft - scrollX;
    frameY = aY - *frameOffsetTop - scrollY;
    nsDoc->ElementFromPoint(frameX, frameY, getter_AddRefs(frameEl));
  }
  if (frameElement && frameEl)
    NS_ADDREF(*frameElement = frameEl);
  return NS_OK;
}

static inline
int dist_points(nsPoint aPt1, nsPoint aPt2)
{
  return roundf(sqrt((aPt2.x-aPt1.x)*(aPt2.x-aPt1.x)+(aPt2.y-aPt1.y)*(aPt2.y-aPt1.y)));
}

static inline int
calc_dist_from_node_to_point(nsPoint &aPt, int aScrollX, int aScrollY, nsIDOMElement *aElement)
{
  float t=0, l=0, w=0, h=0, b=0, r=0;
  get_node_box_or_nearest(aElement, &l, &t, &w, &h, false);
  l += aScrollX;
  t += aScrollY;
  b=t+h;
  r=l+w;
  int dist = INT_MAX;
  // Check that point has perpendicular to node rect
  if (aPt.x > l && aPt.x < r) {
    // Vertical perpendicular
    if (aPt.y < t)
      dist = t - aPt.y;
    else if (aPt.y > b)
      dist = aPt.y - b;
    else // Point inside rect
      dist = 0;
  }
  else if (aPt.y > t && aPt.y < b) {
    // Horizontal perpendicular
    if (aPt.x < l)
      dist = l - aPt.x;
    else if (aPt.x > r)
      dist = aPt.x - r;
    else // Point inside rect
      dist = 0;
  } else {
    // No perpendicular to rect, cal distance to rect corners
    if (aPt.x < l && aPt.y < t)
      dist = dist_points(aPt, nsPoint(l, t));
    if (aPt.x > r && aPt.y < t)
      dist = dist_points(aPt, nsPoint(r, t));
    if (aPt.x < l && aPt.y > b)
      dist = dist_points(aPt, nsPoint(l, b));
    if (aPt.x > r && aPt.y > b)
      dist = dist_points(aPt, nsPoint(r, b));
  }
  return dist;
}

#ifdef MICROB_API_GET_NODES_FOR_AREA
static int
FindAncElement(nsIContent *aContent, nsIDOMHTMLAnchorElement * *ancEl)
{
  nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aContent);
  nsCOMPtr<nsIDOMNode> parent;
  node->GetParentNode(getter_AddRefs(parent));
  nsCOMPtr<nsIDOMHTMLAnchorElement> anc;
  int parentIndex = 0; // Check 3 parents for anchor
  while (parent && parentIndex < 3) {
    parentIndex++;
    anc = do_QueryInterface(parent);
    if (anc) {
      NS_ADDREF(*ancEl = anc);
      return parentIndex;
    }
    nsCOMPtr<nsIDOMNode> nextParent;
    parent->GetParentNode(getter_AddRefs(nextParent));
    parent = nextParent;
  }
  return parentIndex;
}

static nsresult
select_possible_element_from_list(nsPoint &targetPt, int aScrollX, int aScrollY, nsISupportsArray *aNodes, nsIDOMElement * *aElement)
{
  nsresult rv;
  PRUint32 count;
  rv = aNodes->Count(&count);
  NS_ENSURE_SUCCESS(rv, rv);

  targetPt.x += aScrollX;
  targetPt.y += aScrollY;
  nsCOMPtr<nsIContent> currentFlavor;
  nsCOMPtr<nsIContent> closestElem;
  PRInt32 closestDist = INT_MAX;
  for ( PRUint32 flavorIndex = 0; flavorIndex < count; ++flavorIndex) {
    nsCOMPtr<nsISupports> genericWrapper;
    aNodes->GetElementAt(flavorIndex, getter_AddRefs(genericWrapper));
    currentFlavor = do_QueryInterface(genericWrapper);
    if (currentFlavor) {
      nsCOMPtr<nsIDOMHTMLAnchorElement> ancEl = do_QueryInterface(currentFlavor);
      if (!ancEl) {
        int level = FindAncElement(currentFlavor, getter_AddRefs(ancEl));
        if (ancEl) {
          nsCOMPtr<nsIDOMHTMLImageElement> imgEl = do_QueryInterface(currentFlavor);
          if (level == 1 && imgEl) {
            currentFlavor = do_QueryInterface(imgEl);
          } else
            currentFlavor = do_QueryInterface(ancEl);
        }
      }
      if (ancEl || currentFlavor->IsNodeOfType(nsIContent::eHTML_FORM_CONTROL)) {
        nsCOMPtr<nsIDOMElement> vel = do_QueryInterface(currentFlavor);
        if (!vel) continue;
        int dist = calc_dist_from_node_to_point(targetPt, aScrollX, aScrollY, vel);
        if (dist < closestDist) {
          closestDist = dist;
          closestElem = do_QueryInterface(currentFlavor);
        }
      }
    }
  }

  nsCOMPtr<nsIDOMElement> retEl = do_QueryInterface(closestElem, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  NS_ADDREF(*aElement = retEl);
#if 0
  nsString nodename, nodeid;
  nsCOMPtr<nsIDOMNode> retnode = do_QueryInterface(retEl, &rv);
  retnode->GetNodeName(nodename);
  retEl->GetAttribute(NS_LITERAL_STRING("id"), nodeid);
  printf("Found el:%s, id:%s, distance - %i\n", NS_ConvertUTF16toUTF8(nodename).get(), NS_ConvertUTF16toUTF8(nodeid).get(), closestDist);
#endif

  return NS_OK;
}
#endif

int
get_element_info(GMozillaEngine *self, const int aX, const int aY, int aRadius, GdkRectangle *aBoundingRect)
{
  NS_ENSURE_ARG_POINTER(self);
  nsresult rv;
  nsCOMPtr <nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_TRUE(window, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMWindowUtils> utils(do_GetInterface(window));
  NS_ENSURE_TRUE(utils, NS_ERROR_FAILURE);
  nsCOMPtr <nsIDOMElement> domEl;
  PRInt32 scrollX = 0, scrollY = 0;
  PRInt32 frameOffsetTop=0, frameOffsetLeft=0;
  rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(window, nsnull, nsnull, nsnull, nsnull, &scrollX, &scrollY);
#ifdef MICROB_API_GET_NODES_FOR_AREA
  if (aRadius) {
    // if calcWithHitPoints == TRUE, then array from engine will be shorter
    nsCOMPtr<nsIDOMDocument> ddoc;
    // Get DOM Document
    rv = window->GetDocument(getter_AddRefs(ddoc));
    NS_ENSURE_SUCCESS(rv, rv);
    // Get Document
    nsCOMPtr<nsIDocument> doc = do_QueryInterface(ddoc, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    // Get Primary Shell
    nsIPresShell *presShell = doc->GetPrimaryShell();
    NS_ENSURE_TRUE(presShell, rv);

    nsPoint mainPt(aX-scrollX, aY-scrollY);
    nsRect r(mainPt.x - aRadius, mainPt.y - aRadius, aRadius*2, aRadius*2);
    nsCOMPtr<nsISupportsArray> array;

    nsTArray<nsPoint> points;
    points.AppendElement(mainPt);
    int step = aRadius >= 4 ? aRadius/4 : 1;
    for (int px = r.x; px <= r.XMost(); px+=step)
      for (int py = r.y; py <= r.YMost(); py+=step) {
        if (px == mainPt.x && py == mainPt.y) continue;
        points.AppendElement(nsPoint(px, py));
      }
    rv = presShell->GetNodesForArea(r, getter_AddRefs(array), &points,
                                    PR_TRUE, PR_TRUE);
    if (NS_SUCCEEDED(rv) && array)
      select_possible_element_from_list(mainPt, scrollX, scrollY, array, getter_AddRefs(domEl));
  }
  if (!domEl)
#endif
    rv = utils->ElementFromPoint(aX-scrollX, aY-scrollY, PR_TRUE, PR_TRUE, getter_AddRefs(domEl));

  nsCOMPtr <nsIDOMElement> domFrameEl;
  CheckFrameOffSet(domEl, aX, aY, scrollX, scrollY, &frameOffsetTop, &frameOffsetLeft, getter_AddRefs(domFrameEl));
  if (domFrameEl && domFrameEl != domEl)
    domEl = domFrameEl;

  ULOG("pt[%i,%i], scroll[%i,%i] domEl:%p", aX, aY, scrollX, scrollY, domEl.get());
  nsCOMPtr <nsIDOMNode> dnode = do_QueryInterface(domEl, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  gtk_microb_context_update_from_element(GTK_MICROB_CONTEXT(self->ctx), dnode);
  gtk_microb_context_update_scrollable(GTK_MICROB_CONTEXT(self->ctx));
  self->user_ctx->common_data.ctx_type = ctx_mozembed_to_eal(GTK_MICROB_CONTEXT(self->ctx)->type);

  gtk_microb_context_save_node(GTK_MICROB_CONTEXT(self->ctx), dnode);
  float t, l, w, h;
  rv = get_node_box_or_nearest(dnode, &l, &t, &w, &h, false);
  NS_ENSURE_SUCCESS(rv, rv);
  l += scrollX + frameOffsetLeft;
  t += scrollY + frameOffsetTop;

  rv = send_element_info(self, G_WEB_ELEMENT_INFO_ID_UI_REQUEST,
                         self->user_ctx->common_data.ctx_type,
                         t, l, w, h,
                         gtk_microb_context_get_scrollable_type(GTK_MICROB_CONTEXT(self->ctx)));
  return rv;
}

int
recalculate_element_info(GMozillaEngine *self, gboolean recalculate)
{
  nsresult rv = NS_ERROR_FAILURE;

  NS_ENSURE_TRUE(self->ctx, rv);
  NS_ENSURE_TRUE(self->user_ctx, rv);
  NS_ENSURE_TRUE(self->mouse_ctx, rv);

  gpointer node = gtk_microb_context_get_saved_node(GTK_MICROB_CONTEXT(self->ctx));
  NS_ENSURE_TRUE(node, rv);
  nsCOMPtr<nsIDOMNode> dnode(static_cast<nsIDOMNode*>(node));

  nsCOMPtr<nsIDOMElement> frElem;
  nsCOMPtr<nsIDOMHTMLFrameElement> frame;

  nsCOMPtr<nsIDOMWindow> window;
  rv = GetDOMWindowByNode(dnode, getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIDOMElement> frameEl;
  PRInt32 offsetTop = 0, offsetLeft = 0;
  PRInt32 scrollX = 0, scrollY = 0;

  while (window)
  {
    PRInt32 winScrollX = 0, winScrollY = 0;
    window->GetScrollX(&winScrollX);
    window->GetScrollY(&winScrollY);

    scrollX += winScrollX;
    scrollY += winScrollY;
    nsCOMPtr<nsIDOMWindowInternal> intWin = do_QueryInterface(window, &rv);
    if (intWin)
    {
      intWin->GetFrameElement(getter_AddRefs(frElem));
      if (frElem)
      {
        frame = do_QueryInterface(frElem);
        if (frame)
        {
          nsCOMPtr<nsIDOMNSHTMLElement> nsHtmlElem = do_QueryInterface(frame, &rv);
          if (!nsHtmlElem) break;

          PRInt32 frameOffsetTop, frameOffsetLeft;
          nsHtmlElem->GetOffsetTop(&frameOffsetTop);
          nsHtmlElem->GetOffsetLeft(&frameOffsetLeft);
          offsetTop += frameOffsetTop;
          offsetLeft += frameOffsetLeft;
        }
        else
        {
          nsCOMPtr<nsIDOMHTMLIFrameElement> iframe = do_QueryInterface(frElem);
          if (iframe)
          {
            float fl, ft, fw, fh;
            get_node_box_or_nearest(frElem, &fl, &ft, &fw, &fh, false);
            offsetTop += ft;
            offsetLeft += fl;
          }
        }
      }
    }

    nsIDOMWindow* parentWindow;
    window->GetParent(&parentWindow);
    if (window == parentWindow)
      break;
    window = parentWindow;
  }

  int message_id = G_WEB_ELEMENT_INFO_ID_UPDATE;
  if (recalculate)
  {
    gtk_microb_context_recalculate(GTK_MICROB_CONTEXT(self->ctx), dnode);
    gtk_microb_context_update_scrollable(GTK_MICROB_CONTEXT(self->ctx));
    self->user_ctx->common_data.ctx_type = ctx_mozembed_to_eal(GTK_MICROB_CONTEXT(self->ctx)->type);
    message_id = G_WEB_ELEMENT_INFO_ID_RECALCULATED;
  }

  float t, l, w, h;
  rv = get_node_box_or_nearest(gtk_microb_context_get_saved_node(GTK_MICROB_CONTEXT(self->ctx)),
                                                                 &l, &t, &w, &h, FALSE);
  NS_ENSURE_SUCCESS(rv, rv);
  l += scrollX + offsetLeft;
  t += scrollY + offsetTop;

  ULOG("RESIZE ELEMENT INFO [%g:%g:%g:%g]", l, t, w, h);
  rv = send_element_info(self, message_id, self->mouse_ctx->common_data.ctx_type, t, l, w, h,
                         gtk_microb_context_get_scrollable_type(GTK_MICROB_CONTEXT(self->ctx)));
  return NS_OK;
}

int
get_mouse_over_element_info(GMozillaEngine *self, gpointer node)
{
  g_return_val_if_fail(self, 0);
  g_return_val_if_fail(node, 0);

  nsresult rv;
  nsCOMPtr <nsIDOMNode> eventNode = do_QueryInterface((nsISupports *) node);
  NS_ENSURE_TRUE(eventNode, NS_ERROR_FAILURE);
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  nsCOMPtr <nsIDOMWindow> window;
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_TRUE(window, NS_ERROR_FAILURE);

  PRInt32 scrollX = 0, scrollY = 0;
  rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(window, nsnull, nsnull, nsnull, nsnull, &scrollX, &scrollY);

  float l, t, w, h;
  rv = get_node_box_or_nearest(eventNode, &l, &t, &w, &h, false);
  NS_ENSURE_SUCCESS(rv, rv);

  if (self->doc_index != -1 || self->mouse_ctx->common_data.moz_ctx_type & GTK_MICROB_EAL_CONTEXT_IFRAME) {
    nsCOMPtr<nsIDOMWindow> parWin;
    rv = GetDOMWindowByNode(eventNode, getter_AddRefs(parWin));
    NS_ENSURE_SUCCESS(rv, rv);
    nsCOMPtr<nsIDOMWindowInternal> intWin = do_QueryInterface(parWin, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    nsCOMPtr<nsIDOMElement> frElem;
    rv = intWin->GetFrameElement(getter_AddRefs(frElem));
    NS_ENSURE_SUCCESS(rv, rv);

    float lFrElem, tFrElem, wFrElem, hFrElem;
    rv = get_node_box_or_nearest(frElem, &lFrElem, &tFrElem, &wFrElem, &hFrElem, false);
    NS_ENSURE_SUCCESS(rv, rv);

    float differenceLeft = l;
    float differenceTop = t;
    l += lFrElem;
    t += tFrElem;

    nsSize winSize(0,0);
    rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(intWin, &winSize.width, &winSize.height, nsnull, nsnull, nsnull, nsnull);
    NS_ENSURE_SUCCESS(rv, rv);

    // to element info message clip element size to max what is visible. right side frame
    if (lFrElem + winSize.width < l + w)
      w -= ((l + w) - (lFrElem + winSize.width));
    // bottom
    if (tFrElem + winSize.height < t + h)
      h -= ((t + h) - (tFrElem + winSize.height));
    // left
    if (l < lFrElem) {
      l = lFrElem;
      w -= ABS(differenceLeft);
    }
    // up
    if (t < tFrElem) {
      t = tFrElem;
      h -= ABS(differenceTop);
    }

    nsCOMPtr<nsIDOMWindow> parent;
    parWin->GetParent(getter_AddRefs(parent));
    NS_ENSURE_TRUE(parent, NS_ERROR_FAILURE);
    int nestLevel = 0;

    while (parent != window && nestLevel < MAX_NESTED_FRAMES) {
      parWin = parent;
      NS_ENSURE_TRUE(parWin, NS_ERROR_FAILURE);
      intWin = do_QueryInterface(parWin, &rv);
      NS_ENSURE_SUCCESS(rv, rv);
      nsCOMPtr<nsIDOMElement> frElem;
      rv = intWin->GetFrameElement(getter_AddRefs(frElem));
      NS_ENSURE_SUCCESS(rv, rv);

      rv = get_node_box_or_nearest(frElem, &lFrElem, &tFrElem, &wFrElem, &hFrElem, false);
      NS_ENSURE_SUCCESS(rv, rv);

      l += lFrElem;
      t += tFrElem;
      nestLevel += 1;
      parWin->GetParent(getter_AddRefs(parent));
      NS_ENSURE_TRUE(parent, NS_ERROR_FAILURE);
    }
  }

  l += scrollX;
  t += scrollY;
  rv = send_element_info(self, G_WEB_ELEMENT_INFO_ID_MOUSE_OVER,
                         self->mouse_ctx->common_data.ctx_type, t, l, w, h, -1);

  return rv;
}

int send_element_info(GMozillaEngine *self, int id, int type, float t, float l, float w, float h, int aScrollType)
{
  g_return_val_if_fail(self, 0);
  GMozillaWeb* web = G_MOZILLA_WEB(self->global);

  g_return_val_if_fail(web, NS_ERROR_FAILURE);

  if (g_mozilla_web_check_bus(web)) {
    ElementInfoMessage msg = { (guint32) self, id/*request id*/, type, aScrollType, (int)l, (int)t, (int)w, (int)h, 0 };
    if (!g_web_bus_send_message(web->web_bus, web->ui_bus_name, ELEMENT_INFO_MSG, &msg, sizeof(msg)));
      return NS_ERROR_FAILURE;
  }
  return NS_OK;
}

int
scroll_moz_scrollable_view(GMozillaEngine *self, int x, int y, int *overflowx, int *overflowy)
{
  int prev_type = gtk_microb_context_get_scrollable_type(GTK_MICROB_CONTEXT(self->ctx));
  gtk_microb_context_scroll_scrollable(GTK_MICROB_CONTEXT(self->ctx), x, y, overflowx, overflowy);
  int type = gtk_microb_context_get_scrollable_type(GTK_MICROB_CONTEXT(self->ctx));
  // Send notification about scrollable view type change
  if (type != prev_type)
    send_element_info(self, G_WEB_ELEMENT_INFO_ID_SCROLLVIEW, 0, 0, 0, 0, 0, type);

  return NS_OK;
}

int
g_mozilla_engine_send_activate_event(GMozillaEngine *self, gboolean activate)
{
  nsresult rv;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  if (webBrowser) {
    nsCOMPtr<nsIWebBrowserFocus> webBrowserFocus(do_QueryInterface(webBrowser));
    if (webBrowserFocus) {
      if (activate)
        webBrowserFocus->Activate();
      else
        webBrowserFocus->Deactivate();
    }
  }

  nsCOMPtr<nsIViewManager> viewManager;
  nsCOMPtr<nsIWidget> widget;

  rv = GetSendEventTools(self, getter_AddRefs(widget), getter_AddRefs(viewManager));
  NS_ENSURE_SUCCESS(rv, rv);

  self->check_pending_im_enter_keyvent = FALSE;
  nsGUIEvent event(PR_TRUE, activate?NS_ACTIVATE:NS_DEACTIVATE, widget);

  nsEventStatus status;
  nsIView* rootView;
  viewManager->GetRootView(rootView);
  return viewManager->DispatchEvent(&event, rootView, &status);
}

int
send_event_to_layout(GMozillaEngine *self, int type, int x, int y, int mod, int button, int clickcount, int time)
{
  NS_ENSURE_ARG_POINTER(self);
  ULOG("type:%i, pt[%i,%i], mod:%i, bt:%i, cnt:%i, time:%i\n", type, x, y, mod, button, clickcount, time);
  nsresult rv;
  nsCOMPtr <nsIDOMWindow> window;
  nsCOMPtr <nsIViewManager> viewManager;
  nsCOMPtr <nsIWidget> widget;

  rv = GetSendEventTools(self, getter_AddRefs(widget),
                         getter_AddRefs(viewManager),
                         getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, FALSE);

  nsIView* rootView = nsnull;
  viewManager->GetRootView(rootView);
  NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);

  nsEventStatus status;
  PRBool contextMenuKey = PR_FALSE;
  PRInt32 msg = type;
  PRInt32 aModifiers = 0;
  PRInt32 ebutton = 0;

  switch (button) {
    case 1:
      ebutton = nsMouseEvent::eLeftButton;
      break;
    case 2:
      ebutton = nsMouseEvent::eMiddleButton;
      break;
    case 3:
      ebutton = nsMouseEvent::eRightButton;
      break;
    default:
      break;
  }

  PRBool doClickEvent = PR_FALSE;

  switch (type) {
    case G_WEBENGINE_MOUSE_DOWN:
      msg = NS_MOUSE_BUTTON_DOWN;
      break;
    case G_WEBENGINE_MOUSE_UP:
      msg = NS_MOUSE_BUTTON_UP;
      break;
    case G_WEBENGINE_MOUSE_MOVE:
      msg = NS_MOUSE_MOVE;
      break;
    case G_WEBENGINE_MOUSE_CLICK:
      msg = NS_MOUSE_CLICK;
      doClickEvent = PR_TRUE;
      break;
    case G_WEBENGINE_MOUSE_ENTER:
      msg = NS_MOUSE_ENTER;
      break;
    case G_WEBENGINE_MOUSE_EXIT:
      msg = NS_MOUSE_EXIT;
      break;
    case G_WEBENGINE_MOUSE_DOUBLECLICK:
      msg = NS_MOUSE_DOUBLECLICK;
      break;
    case G_WEBENGINE_MOUSE_LONG_PRESS:
      msg = NS_CONTEXTMENU;
      ebutton = nsMouseEvent::eRightButton;
      break;
    default:
      msg = NS_MOUSE_MOVE;
      break;
  }

  nsMouseEvent event(PR_TRUE, doClickEvent?NS_MOUSE_MOVE:msg, rootView->GetWidget(), nsMouseEvent::eReal, contextMenuKey ? nsMouseEvent::eContextMenuKey : nsMouseEvent::eNormal);

  event.isShift = (aModifiers & nsIDOMNSEvent::SHIFT_MASK) ? PR_TRUE : PR_FALSE;
  event.isControl = (aModifiers & nsIDOMNSEvent::CONTROL_MASK) ? PR_TRUE : PR_FALSE;
  event.isAlt = (aModifiers & nsIDOMNSEvent::ALT_MASK) ? PR_TRUE : PR_FALSE;
  event.isMeta = (aModifiers & nsIDOMNSEvent::META_MASK) ? PR_TRUE : PR_FALSE;

  event.button = ebutton;
  event.widget = widget;

  event.clickCount = 1;

  event.ignoreRootScrollFrame = PR_TRUE;

  PRInt32 scrollX = 0, scrollY = 0;
  rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(window, nsnull, nsnull, nsnull, nsnull, &scrollX, &scrollY);
  event.refPoint.x = x - scrollX;
  event.refPoint.y = y - scrollY;
  ULOG("xy[%i,%i], event.refPoint[%i,%i] bt:%i\n", x, y, event.refPoint.x, event.refPoint.y, event.button);

  event.time = PR_IntervalNow();
  viewManager->DispatchEvent(&event, rootView, &status);
  viewManager->GetRootView(rootView); // Make sure that we are still alive
  NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);
  if (doClickEvent) {
    event.flags &= ~NS_EVENT_FLAG_CANT_CANCEL;
    event.message = NS_MOUSE_BUTTON_DOWN;
    event.time = PR_IntervalNow();
    viewManager->DispatchEvent(&event, rootView, &status);
    viewManager->GetRootView(rootView);
    NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);
    event.message = NS_MOUSE_BUTTON_UP;
    event.time = PR_IntervalNow();
    viewManager->DispatchEvent(&event, rootView, &status);
    self->check_pending_im_enter_keyvent = FALSE;
    viewManager->GetRootView(rootView);
    NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);
    self->user_iteraction_click = TRUE;
  }

  viewManager->GetRootView(rootView);
  NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);

  if (event.button == nsMouseEvent::eRightButton
      && doClickEvent) {
    event.message = NS_CONTEXTMENU;
    event.time = PR_IntervalNow();
    viewManager->DispatchEvent(&event, rootView, &status);
  }
  static int last_cursor = -1;
  int cursor = -1;

  if (type == G_WEBENGINE_MOUSE_DOWN ||
      type == G_WEBENGINE_MOUSE_UP ||
      type == G_WEBENGINE_MOUSE_MOVE ||
      type == G_WEBENGINE_MOUSE_CLICK ||
      type == G_WEBENGINE_MOUSE_DOUBLECLICK) {

    // get the cursor from the main widget
    nsCOMPtr <nsIWebBrowser> webBrowser;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
    NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);

    nsCOMPtr<nsIBaseWindow> mBaseWindow = do_QueryInterface(webBrowser, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    nsCOMPtr<nsIWidget> mainWidget;
    mBaseWindow->GetMainWidget(getter_AddRefs(mainWidget));
    NS_ENSURE_TRUE(mainWidget, NS_ERROR_FAILURE);
    cursor = (int)mainWidget->GetCursor();
    
    // Only send cursor update messages when the cursor has actually changed.
    if (cursor != last_cursor) {
      send_cursor_update_message(self, cursor);
      last_cursor = cursor;
    }
  }

  return NS_OK;
}

gboolean send_fake_key_event_to_layout(GMozillaEngine *self)
{
  nsCOMPtr<nsIViewManager> viewManager;
  nsCOMPtr<nsIWidget> widget;
  nsresult rv;

  rv = GetSendEventTools(self, getter_AddRefs(widget));
  NS_ENSURE_SUCCESS(rv, FALSE);

  GtkWidget *cont = GTK_WIDGET(widget->GetNativeData(NS_NATIVE_WIDGET));
  gboolean retval = FALSE;

  GdkEventKey event;
  event.window = NULL;
  event.send_event = FALSE;
  event.hardware_keycode = 37;
  event.time = 0;
  event.group = 0;
  event.length = 0;
  event.string = NULL;
  event.is_modifier = 0;
  event.keyval = 65507;

  event.state = 0;
  g_signal_emit_by_name(cont, "key_press_event", &event, &retval);
  event.state = 4;
  g_signal_emit_by_name(cont, "key_release_event", &event, &retval);

  self->fake_key_id = 0;

  return FALSE;
}

gboolean
g_mozilla_engine_send_key_event(GMozillaEngine *self,
                                int type, int keyval, int length, int state, int hardware_keycode,
                                int group, guint is_modifier, int time)
{
  ULOG("type:%i, keyv:%i, state:%i, hw:%i, group:%i, time:%i", type, keyval, state, hardware_keycode, group, time);
  nsresult rv;

  nsCOMPtr <nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_TRUE(window, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMWindowUtils> utils(do_GetInterface(window));
  NS_ENSURE_TRUE(utils, NS_ERROR_FAILURE);

  GdkEventKey event;
  event.window = NULL;
  event.keyval = keyval;
  event.send_event = FALSE;
  event.state = state;
  event.hardware_keycode = hardware_keycode;
  event.time = time;
  event.group = group;
  event.length = length;
  event.string = NULL;
  event.is_modifier = is_modifier;

  if (type == G_WEBENGINE_KEY_PRESS) {
    event.type = GDK_KEY_PRESS;
  }
  if (type == G_WEBENGINE_KEY_RELEASE) {
    event.type = GDK_KEY_RELEASE;
  }
  self->check_pending_im_enter_keyvent = TRUE;

  if (register_im_context(self) &&
      (user_ctx_text_input_or_password(self) || self->dom_focused_input)) {
    if (gtk_im_context_filter_keypress((GtkIMContext *)self->gtkIMContext, &event))
      return TRUE;
  }

  if ((self->user_ctx) && (!(self->user_ctx->common_data.ctx_type & G_WEBENGINE_POPUPMENU_EMBED))) {
    // Stop handling special keys
    switch (keyval) {
      case GDK_ISO_Level3_Shift:    // The Fn key
        return TRUE;
    }
  }

  nsCOMPtr<nsIViewManager> viewManager;
  nsCOMPtr<nsIWidget> widget;

  rv = GetSendEventTools(self, getter_AddRefs(widget));
  NS_ENSURE_SUCCESS(rv, FALSE);

  GtkWidget *cont = GTK_WIDGET(widget->GetNativeData(NS_NATIVE_WIDGET));
  if (cont) {
    gboolean retval = FALSE;
    if (type == G_WEBENGINE_KEY_PRESS) {
      g_signal_emit_by_name(cont, "key_press_event", &event, &retval);
      ULOG("KEYPRESS key=%d handled=%d\n", keyval, retval);
      // check if nobody consumed the backspace
      if (keyval == GDK_BackSpace && !retval) {
        // if so, go back in history
        ULOG("GOING BACK or FORWARD\n");
        if (state & GDK_SHIFT_MASK)
          g_mozilla_engine_go_forward(self, 1);
        else
          g_mozilla_engine_go_back(self, 1);
        retval = TRUE;
      }
      return retval;
    }
    else if (type == G_WEBENGINE_KEY_RELEASE)
      g_signal_emit_by_name(cont, "key_release_event", &event, &retval);

    return retval;
  }

  return FALSE;
}

int
g_mozilla_simple_navigation(GMozillaEngine *self, void *event)
{
  NS_ENSURE_ARG(self);
  NS_ENSURE_ARG(event);

  nsresult rv;
  nsCOMPtr <nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_TRUE(window, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMWindowUtils> utils(do_GetInterface(window));
  NS_ENSURE_TRUE(utils, NS_ERROR_FAILURE);
  nsCOMPtr <nsIDOMKeyEvent> keyEvent = do_QueryInterface((nsISupports*)event);
  NS_ENSURE_TRUE(keyEvent, NS_ERROR_FAILURE);
  PRBool result = PR_FALSE;
  PRBool ctrlKey = 0;
  PRUint32 keyCode = 0;
  keyEvent->GetCtrlKey(&ctrlKey);
  keyEvent->GetKeyCode(&keyCode);
  ULOG("is Ctrl:%i, key:%i", ctrlKey, keyCode);
  if (utils && ctrlKey && (keyCode == nsIDOMKeyEvent::DOM_VK_LEFT || keyCode == nsIDOMKeyEvent::DOM_VK_RIGHT)) {
    if (ctx_text_input_or_password(self))
    {
      EAL_IF_GFREE_FUNC(self->fake_key_id, g_source_remove);
      send_fake_key_event_to_layout(self);
    }

    utils->SendKeyEvent(NS_LITERAL_STRING("keypress"), NS_VK_TAB, 0,
                        (keyCode == nsIDOMKeyEvent::DOM_VK_LEFT) ? nsIDOMNSEvent::SHIFT_MASK : 0,
                        PR_FALSE, &result);
  }
  return NS_OK;
}

void
send_cursor_update_message (GMozillaEngine *self, int cursor)
{
    if(self == NULL)
        return;

    GWebCursorType gwebcursor = G_WEB_CURSOR_NONE; 

    // For list of nsCursor values see: source/widget/public/nsIWidget.h in gecko
    switch ((nsCursor)cursor) {
        case eCursor_select:
        case eCursor_vertical_text:
            gwebcursor = G_WEB_CURSOR_TEXT;
            break;
        case eCursor_hyperlink:
            gwebcursor = G_WEB_CURSOR_LINK;
            break;
        case eCursor_crosshair:
        case eCursor_grab:
            gwebcursor = G_WEB_CURSOR_GRAB;
            break;
        case eCursor_move:
        case eCursor_grabbing:
            gwebcursor = G_WEB_CURSOR_GRABBING;
            break;
        default:
            // Anything else will get default cursor
            gwebcursor = G_WEB_CURSOR_DEFAULT;
            break;
    }

    ULOG ("ENG Cursor: %d  ===>>  UI Cursor: %d.\n", cursor, gwebcursor);

    GMozillaWeb* web = G_MOZILLA_WEB(self->global);
    if (g_mozilla_web_check_bus(web)) {
        CursorUpdateMessage msg = { (guint32) self, gwebcursor };
        if (!g_web_bus_send_message(web->web_bus,
                                  web->ui_bus_name,
                                  CURSOR_UPDATE_MSG,
                                  &msg,
                                  sizeof(msg))) {
            ULOG ("Failed to send cursor update message to UI.");
        }
    }
}

void
update_default_window_size(GMozillaEngine *self)
{
  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, );
  webBrowser->GetContentDOMWindow(getter_AddRefs(window));

  nsCOMPtr<nsPIDOMWindow> pidomWindow = do_QueryInterface(window);
  if (!pidomWindow) {
    printf("ERROR!: Failed to get nsPIDOMWindow from dom window!!!\n");
    return;
  }

  nsCOMPtr<nsIDOMWindowInternal> winInt = do_QueryInterface(window);
  if (winInt) {
    PRInt32 scrollX = 0, scrollY = 0;
    PRInt32 width = 0, height = 0;
    PRInt32 scrollMaxX = 0, scrollMaxY = 0;
    MicrobEalContextMenuInfo::GetDomWindowScrollInfo(window, &width, &height, nsnull, nsnull, &scrollX, &scrollY);

    // Important to do resize first, otherwise scrollTo will fail
    int w = self->mMinWindowSize.width ? self->mMinWindowSize.width : self->mWindowSize.width;
    int h = self->mMinWindowSize.height ? self->mMinWindowSize.height : self->mWindowSize.height;

    //check if it is image
    nsCOMPtr<nsIDOMDocument> domDoc;
    window->GetDocument(getter_AddRefs(domDoc));
    nsCOMPtr<nsIImageDocument> imgDoc = do_QueryInterface(domDoc);

    if (w != width || h != height && !imgDoc) {
      ULOG("ResizeTo:[%i,%i]", w, h);
      winInt->ResizeTo(w, h);
    }
    else if (self->recalculate_element)
    {
      self->recalculate_element = FALSE;
      recalculate_element_info(self, FALSE);
    }

    PRInt32 bidiOffset = 0;
    MicrobEalContextMenuInfo::GetDomWindowScrollInfo(window, &width, &height, &scrollMaxX, &scrollMaxY, nsnull, nsnull, &bidiOffset);
    ULOG("Default Window resize [%i,%i], wh[%i,%i], scroll[%i,%i], curScr[%i,%i], scrmax[%i,%i], bidiOff:%i",
         w, h, width, height, self->mWindowSize.x, self->mWindowSize.y, scrollX, scrollY, scrollMaxX, scrollMaxY,
         bidiOffset);

    // Check that we are not doing bad scrolling
    self->mWindowSize.y = MIN(self->mWindowSize.y, scrollMaxY);
    self->mWindowSize.x = MIN(self->mWindowSize.x, scrollMaxX);

    if (self->mWindowSize.x != scrollX || self->mWindowSize.y != scrollY)
      window->ScrollTo(self->mWindowSize.x - bidiOffset, self->mWindowSize.y);
  }
}

gboolean
g_mozilla_handle_key_event (void* event)
{
  PRUint32 keyCode = -1;
  nsCOMPtr<nsIDOMKeyEvent> aEvent = do_QueryInterface((nsISupports*) event);
  if (aEvent) {
      aEvent->GetKeyCode(&keyCode);
      if (keyCode == nsIDOMKeyEvent::DOM_VK_BACK_SPACE)
          return TRUE;
  }

  return FALSE;
}

/** Get the text contents of given node.
  * NOTE: Current implementation only gets text from the first child node.
  * TODO: Get text from all child nodes, their childs etc..
  * @param node A gpointer to nsIDOMNode instance.
  * @return Newly allocated string containing the node text or NULL on failure.
  */
gchar *
g_mozilla_get_node_text (gpointer node)
{
  nsCOMPtr<nsIDOMNode> domNode = do_QueryInterface((nsISupports*)node);
  NS_ENSURE_TRUE (domNode, NULL);

  nsCOMPtr<nsIDOMNode> childNode;
  domNode->GetFirstChild (getter_AddRefs (childNode));
  NS_ENSURE_TRUE (childNode, NULL);

  PRUint16 nodeType;
  childNode->GetNodeType (&nodeType);
  if(nodeType == nsIDOMNode::TEXT_NODE) {
    nsString value;
    childNode->GetNodeValue (value);
    nsCString cValue (NS_ConvertUTF16toUTF8 (value).get());

    return g_strdup (cValue.get ());
  }

  return NULL;
}

static int
send_find_element_info (GMozillaEngine *self, nsISelection2* sel)
{
  nsresult rv = NS_ERROR_FAILURE;
  NS_ENSURE_TRUE(sel, rv);

  nsCOMPtr<nsIDOMClientRect> selRect;
#ifdef MICROB_API_FIND2
  sel->GetCaretRect(getter_AddRefs(selRect));
#endif
  NS_ENSURE_TRUE(selRect, rv);

  float l = 0, t = 0, r = 0, b = 0;
  selRect->GetLeft(&l);
  selRect->GetTop(&t);
  selRect->GetRight(&r);
  selRect->GetBottom(&b);
  int x = (PRInt32)nsPresContext::AppUnitsToFloatCSSPixels(l);
  int y = (PRInt32)nsPresContext::AppUnitsToFloatCSSPixels(t);
  int width = (PRInt32)nsPresContext::AppUnitsToFloatCSSPixels(r - l);
  int height = (PRInt32)nsPresContext::AppUnitsToFloatCSSPixels(b - t);
  send_element_info(self, G_WEB_ELEMENT_INFO_ID_FIND, 0, y, x, width, height, -1);

  return NS_OK;
}

static int
get_selection(nsIDOMWindow* window, nsISelection2** sel)
{
  nsresult rv = NS_ERROR_FAILURE;
  NS_ENSURE_TRUE(*sel, rv);
  NS_ENSURE_TRUE(window, rv);
  
  nsCOMPtr<nsIDOMDocument> ddoc;
  window->GetDocument(getter_AddRefs(ddoc));
  NS_ENSURE_TRUE(ddoc, rv);

  nsCOMPtr<nsIDocument> doc;
  doc = do_QueryInterface(ddoc, &rv);
  NS_ENSURE_TRUE(doc, rv);
  rv = NS_ERROR_FAILURE;

  nsCOMPtr<nsISelectionController> selCon;
  nsIPresShell *presShell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(presShell, rv);
  selCon = do_QueryInterface(presShell);
  NS_ENSURE_TRUE(selCon, rv);

  nsCOMPtr<nsISelection> selection;
  selCon->GetSelection(nsISelectionController::SELECTION_NORMAL, getter_AddRefs(selection));
  NS_ENSURE_TRUE(selection, rv);
  
  nsCOMPtr<nsISelection2> selection2 = do_QueryInterface(selection, &rv);
  NS_ENSURE_TRUE(selection, rv);

  *sel = selection2;
  NS_ADDREF(*sel);
  return NS_OK;
}

guint
g_mozilla_find_text(GMozillaEngine *self, const gchar *string,
                    gboolean reverse, gboolean case_sensitive,
                    gboolean restart, gint target)
{
  unsigned short res = nsITypeAheadFind::FIND_NOTFOUND;
  nsresult rv = NS_ERROR_FAILURE;

  nsCOMPtr<nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, G_WEBENGINE_SEARCH_NOTFOUND);

  nsCOMPtr<nsIDocShell> docShell(do_GetInterface((nsISupports *)webBrowser));
  NS_ENSURE_TRUE(docShell, G_WEBENGINE_SEARCH_NOTFOUND);

  nsITypeAheadFind *taFind = NULL;

  if (!self->find) {
    nsCOMPtr<nsITypeAheadFind> find(do_CreateInstance("@mozilla.org/typeaheadfind;1"));
    NS_ENSURE_TRUE(find, G_WEBENGINE_SEARCH_NOTFOUND);
    rv = find->Init(docShell);
    NS_ENSURE_SUCCESS(rv, G_WEBENGINE_SEARCH_NOTFOUND);
    NS_ADDREF(find);
    self->find = taFind = find;
  } else
    taFind = static_cast<nsITypeAheadFind *>(self->find);

  nsCOMPtr<nsIDOMWindow> window;
  taFind->GetCurrentWindow(getter_AddRefs(window));

  nsCOMPtr<nsISelection2> sel;
  get_selection(window, getter_AddRefs(sel));
#ifdef MICROB_API_FIND2
  if (sel)
      sel->SetDoScroll(PR_FALSE);
#endif

  taFind->SetCaseSensitive(case_sensitive ? PR_TRUE : PR_FALSE);

  gboolean find_again = FALSE;

  if (self->prev_find_str)
      find_again = g_ascii_strcasecmp(self->prev_find_str, string) == 0;

  if (!restart && find_again)
    rv = taFind->FindAgain(reverse ? PR_TRUE : PR_FALSE, PR_FALSE, &res);
  else {
    rv = taFind->Find(NS_ConvertUTF8toUTF16(string), PR_FALSE, &res);
    EAL_GFREE(self->prev_find_str);
    self->prev_find_str = g_strdup(string);
  }

  NS_ENSURE_SUCCESS(rv, G_WEBENGINE_SEARCH_NOTFOUND);

  // Map res to GWebEngineSearchResult enums
  switch (res) {
    case nsITypeAheadFind::FIND_FOUND: {
      send_find_element_info(self, sel);
      return G_WEBENGINE_SEARCH_FOUND;
    }
    case nsITypeAheadFind::FIND_NOTFOUND:
      return G_WEBENGINE_SEARCH_NOTFOUND;
    case nsITypeAheadFind::FIND_WRAPPED: {
      send_find_element_info(self, sel);
      return G_WEBENGINE_SEARCH_WRAPPED;
    }
    default:
      break;
  }

  return G_WEBENGINE_SEARCH_NOTFOUND;
}

/**
 * Find currently focused DOM element and send a notification
 * message to UI via web bus.
 *
 * @param self GMozillaEngine instance
 * @return NS_OK on success or nsresult error value on failure.
 */
int
send_focused_element_message (GMozillaEngine * self, GWebEnginePopupMenuType *aType)
{
  nsresult rv (NS_OK);
  GWebEnginePopupMenuType type = G_WEBENGINE_POPUPMENU_NOTYPE;
  nsCOMPtr<nsIWebBrowserFocus> focus;
  nsCOMPtr<nsIDOMElement> element;

  if (!self)
    return NS_ERROR_FAILURE;

  nsCOMPtr <nsIWebBrowser> browser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(browser));
  if (!browser) {
    // Failed to get browser
    rv = NS_ERROR_FAILURE;
  }

  if (rv == NS_OK) {
    focus = do_GetInterface (browser);
    if (!focus) {
      // Failed to get focus
      rv = NS_ERROR_FAILURE;
    }
  }

  if (rv == NS_OK) {
    rv = focus->GetFocusedElement (getter_AddRefs (element));
    if (element) {
      nsCOMPtr<nsIDOMNode> node = do_QueryInterface (element);
      if (node) {

        unsigned short ns_node_type = 0;
        nsString name;

        node->GetNodeType (&ns_node_type);
        node->GetLocalName(name);

        ULOG_DEBUG_F ("Focused node: %s", NS_ConvertUTF16toUTF8 (name).get());

        if (ns_node_type == nsIDOMNode::ELEMENT_NODE) {
          if (name.LowerCaseEqualsLiteral ("input")) {
            type = G_WEBENGINE_POPUPMENU_INPUT;
          } else if (name.LowerCaseEqualsLiteral ("textarea")) {
            type = G_WEBENGINE_POPUPMENU_INPUT;
          } else if (name.LowerCaseEqualsLiteral ("object")) {
            type = G_WEBENGINE_POPUPMENU_EMBED;
          } else if (name.LowerCaseEqualsLiteral ("embed")) {
            type = G_WEBENGINE_POPUPMENU_EMBED;
          } else if (name.LowerCaseEqualsLiteral ("applet")) {
            type = G_WEBENGINE_POPUPMENU_EMBED;
          } else {
            nsCOMPtr<nsIDOMNSHTMLElement> nsel = do_QueryInterface (element);
            if (nsel) {
              nsString conteditable;
              nsel->GetContentEditable(conteditable);
              if (conteditable.EqualsLiteral("true")) {
                type = G_WEBENGINE_POPUPMENU_INPUT;
              }
            } else
              // TODO: Add other element types as necessary
              type = G_WEBENGINE_POPUPMENU_UNKNOWN_PROTOCOL;
          }
        }
      } else {
        // Failed to get node
        rv = NS_ERROR_FAILURE;
      }
    } else {
      // Failed to get element: there's no focused element
      type = G_WEBENGINE_POPUPMENU_NOTYPE;
      rv = NS_ERROR_FAILURE;
    }
  }

  if (aType)
    *aType = type;

  GMozillaWeb* web = G_MOZILLA_WEB (self->global);
  if (g_mozilla_web_check_bus(web)) {

    int error = (rv != NS_OK);

    ElementInfoMessage msg = { (guint32) self, G_WEB_ELEMENT_INFO_ID_DOM_FOCUS, type, 0, 0, 0, 0, 0, error };
    g_web_bus_send_message (web->web_bus,
                            web->ui_bus_name,
                            ELEMENT_INFO_MSG,
                            &msg,
                            sizeof (msg));
  } else {
    rv = NS_ERROR_FAILURE;
  }

  return rv;
}

static int
g_mozilla_engine_window_set_image_animation(nsIDOMWindow *window, int state)
{
  nsresult rv;
  nsCOMPtr<nsIDOMWindowCollection> frames;
  rv = window->GetFrames(getter_AddRefs(frames));
  NS_ENSURE_SUCCESS(rv, rv);
  PRUint32 frameCount = 0;
  frames->GetLength(&frameCount);
  nsCOMPtr<nsIDOMWindow> currentWindow;
  NS_ENSURE_SUCCESS(rv, rv);
  for (unsigned int i= 0; i < frameCount; i++) {
    frames->Item(i, getter_AddRefs(currentWindow));
    if (currentWindow) {
      nsCOMPtr<nsIDOMWindowUtils> utils(do_GetInterface(currentWindow, &rv));
      utils->SetImageAnimationMode(state);
    }
  }
  nsCOMPtr<nsIDOMWindowUtils> utils(do_GetInterface(window, &rv));
  if (NS_SUCCEEDED(rv))
    rv = utils->SetImageAnimationMode(state);
  
  // Notify all observers about image animation suspension
  nsCOMPtr<nsIObserverService> os = do_GetService("@mozilla.org/observer-service;1");
  if (os) {
    const char* value = (state == imgIContainer::kDontAnimMode) ? "true" : "false";
    rv = os->NotifyObservers(nsnull, "animation-suspension", NS_ConvertUTF8toUTF16(value).get());
  }
  return rv;
}

static int
g_mozilla_engine_window_suspend_timeouts(nsIDOMWindow *window, PRBool active)
{
  nsresult rv;

  nsCOMPtr<nsPIDOMWindow> pwindow(do_QueryInterface(window, &rv));
  NS_ENSURE_SUCCESS(rv, rv);
  if (active)
    pwindow->ResumeTimeouts();
  else if (!pwindow->TimeoutSuspendCount())
      pwindow->SuspendTimeouts();

  nsCOMPtr<nsIDOMWindowCollection> frames;
  rv = window->GetFrames(getter_AddRefs(frames));
  NS_ENSURE_SUCCESS(rv, rv);
  PRUint32 frameCount = 0;
  frames->GetLength(&frameCount);
  nsCOMPtr<nsIDOMWindow> currentWindow;
  NS_ENSURE_SUCCESS(rv, rv);

  for (unsigned int i= 0; i < frameCount; i++) {
    frames->Item(i, getter_AddRefs(currentWindow));
    nsCOMPtr<nsPIDOMWindow> pcwindow(do_QueryInterface(currentWindow));
    if (pcwindow) {
      if (active)
        pcwindow->ResumeTimeouts();
      else if (!pcwindow->TimeoutSuspendCount())
        pcwindow->SuspendTimeouts();
    }
  }
  return rv;
}


int
g_mozilla_web_suspend_native(GMozillaWeb *aWeb, gboolean aSuspend)
{
  g_return_val_if_fail(aWeb, NS_ERROR_FAILURE);

  nsCOMPtr<nsIAppShell> appShell = do_GetService(kAppShellCID);
  NS_ENSURE_TRUE(appShell, NS_ERROR_FAILURE);
  if (aSuspend)
    appShell->SuspendNative();
  else
    appShell->ResumeNative();

  ULOG_INFO("FullSusp:%i, web active:%i->%i", aSuspend, aWeb->web_is_active, !aSuspend);
  aWeb->web_is_active = !aSuspend;
  return NS_OK;
}

static gboolean 
suspend_timeouts_timer_callback(GMozillaEngine *engine)
{
  NS_ENSURE_TRUE(engine, FALSE);
  NS_ENSURE_TRUE(engine->engine, FALSE);
  nsresult rv;

  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, FALSE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, rv);

  g_mozilla_engine_window_suspend_timeouts (window, PR_FALSE);
  engine->timeouts_suspended = TRUE;
  return FALSE;
}

void
suspend_timeouts_timer_cleanup(GMozillaEngine *engine)
{
  NS_ENSURE_TRUE (engine, );
  engine->suspend_timeouts_timer_id = 0;
}

int
g_mozilla_engine_window_set_active(GMozillaEngine *engine, int state)
{
  ULOG("eng:%p, state:%i", engine, state);
  NS_ENSURE_ARG(engine);
  NS_ENSURE_ARG(engine->engine);

  nsresult rv;

  gboolean suspend_images = FALSE;
  gboolean suspend_plugins = FALSE;
  gint suspend_js = FALSE;
  gboolean suspend_native = FALSE;
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SUSPEND_IMAGES, &suspend_images);
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SUSPEND_PLUGINS, &suspend_plugins);
  g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SUSPEND_JAVASCRIPT, &suspend_js);
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SUSPEND_NATIVE, &suspend_native);
  if (!suspend_images && !suspend_plugins && suspend_js == G_TIMEOUTS_SUSPEND_NEVER && !suspend_native)
    return NS_OK;

  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, rv);

  PRBool active = state ? PR_TRUE : PR_FALSE;

  nsCOMPtr<nsPIDOMWindow> pwindow(do_QueryInterface(window));
  nsCOMPtr<nsIDOMEventTarget> target(do_QueryInterface(pwindow->GetChromeEventHandler()));
  if (!target) return NS_OK;

  nsCOMPtr<nsIDOMDocument> ownerDocument;
  rv = window->GetDocument(getter_AddRefs(ownerDocument));
  nsCOMPtr<nsIDOMDocumentEvent> document(do_QueryInterface(ownerDocument));
  if (!document) return NS_OK;

  if (suspend_plugins) {
    nsCOMPtr<nsIDOMEvent> event;
    rv = document->CreateEvent(NS_LITERAL_STRING("Events"), getter_AddRefs(event));
    NS_ENSURE_SUCCESS(rv, rv);

    nsCOMPtr<nsIPrivateDOMEvent> privateEvent(do_QueryInterface(event));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = privateEvent->SetTrusted(PR_TRUE);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = event->InitEvent(active?NS_LITERAL_STRING("pluginsshow"):NS_LITERAL_STRING("pluginshide"), PR_TRUE, PR_TRUE);
    NS_ENSURE_SUCCESS(rv, rv);

    PRBool defaultActionEnabled = PR_TRUE;
    ULOG("Dispatch event visible:%i!!!!!", active);
    target->DispatchEvent(event, &defaultActionEnabled);
  }

  nsCOMPtr<nsIDocShell> docShell(do_GetInterface((nsISupports*)webBrowser));
  NS_ENSURE_TRUE(docShell, FALSE);

  EAL_IF_GFREE_FUNC(engine->suspend_timeouts_timer_id, g_source_remove);
  if (active) {
    engine->is_layout_active = TRUE;
    if (suspend_images)
      g_mozilla_engine_window_set_image_animation(window, imgIContainer::kNormalAnimMode);
    if (engine->timeouts_suspended) {
      g_mozilla_engine_window_suspend_timeouts(window, active);
      engine->timeouts_suspended = FALSE;
    }
    MicrobEalXshmRenderer::ResumePendingUpdates(engine);
#ifdef MICROB_API_SET_ACTIVE_DOCSHELL
    docShell->SetIsActive(PR_TRUE);
#endif
    // Set current doc shell for find
    if (engine->find) {
      nsITypeAheadFind *taFind = static_cast<nsITypeAheadFind *>(engine->find);
      taFind->SetDocShell(docShell);
    }
    EAL_GFREE(engine->prev_find_str);
  }
  else {
    engine->is_layout_active = FALSE;
    if (suspend_images)
      g_mozilla_engine_window_set_image_animation(window, imgIContainer::kDontAnimMode);
    if (!engine->timeouts_suspended) {
      gint timeout = -1;
      switch (suspend_js) {
        case G_TIMEOUTS_SUSPEND_0_S:
          timeout = SUSPEND_JS_TIMEOUT_0;
          break;
        case G_TIMEOUTS_SUSPEND_15_S:
          timeout = SUSPEND_JS_TIMEOUT_1;
          break;
        case G_TIMEOUTS_SUSPEND_30_S:
          timeout = SUSPEND_JS_TIMEOUT_2;
          break;
        case G_TIMEOUTS_SUSPEND_60_S:
          timeout = SUSPEND_JS_TIMEOUT_0;
          break;
        case G_TIMEOUTS_SUSPEND_NEVER:
        default:
          break;
      }
      if (timeout >= 0)
        engine->suspend_timeouts_timer_id = g_timeout_add_full (G_PRIORITY_DEFAULT,
                                                                timeout,
                                                                (GSourceFunc)suspend_timeouts_timer_callback,
                                                                engine,
                                                                (GDestroyNotify)suspend_timeouts_timer_cleanup);
    }
    MicrobEalXshmRenderer::SuspendPendingUpdates(engine);
#ifdef MICROB_API_SET_ACTIVE_DOCSHELL
    docShell->SetIsActive(PR_FALSE);
#endif
  }

  if (!suspend_native)
    return NS_OK;

  GMozillaWeb *web = G_MOZILLA_WEB(engine->global);
  NS_ENSURE_TRUE(web, NS_ERROR_FAILURE);

  gboolean do_suspend_change = FALSE;
  if (web->window_list) {
    gboolean all_suspended = TRUE;
    GList *c = NULL;
    for (c = (GList*) web->window_list; c; c = c->next) {
        GMozillaEngine * eng = (GMozillaEngine*)c->data;
        if (eng->is_layout_active) {
          all_suspended = FALSE;
          break;
        }
    }
    if (!web->web_is_active != all_suspended)
      do_suspend_change = TRUE;
  }
  if (do_suspend_change)
    g_mozilla_web_suspend_native(web, web->web_is_active ? FALSE : TRUE);

  return NS_OK;
}

static inline GtkWidget *
_get_gtk_widget_from_nsIWidget(nsIWidget *aWidget)
{
  GtkWidget *container = NULL;
  GdkWindow *gtkwin = GDK_WINDOW(aWidget->GetNativeData(NS_NATIVE_WINDOW));
  if (gtkwin) {
    gpointer user_data = NULL;
    gdk_window_get_user_data(gtkwin, &user_data);
    NS_ENSURE_TRUE(user_data, nsnull);
    //this should be a MozContainer, and his parent should be gtkmozembed
    container = GTK_WIDGET(user_data);
  } else {
    // if NATIVE_WINDOW not available then try NATIVE_WIDGET
    void * data = aWidget->GetNativeData(NS_NATIVE_WIDGET);
    NS_ENSURE_TRUE(data, nsnull);
    if (data && GTK_IS_WIDGET(data)) {
      container = GTK_WIDGET(data);
      GtkWidget *parent = gtk_widget_get_parent(container);
      if (parent)
        container = parent;
    }
  }
  return container;
}

GtkWidget *
g_mozilla_engine_get_mozcontainer_from_domwindow(void *aDOMWindow)
{
  NS_ENSURE_TRUE(aDOMWindow, nsnull);

  nsresult rv;
  nsCOMPtr<nsIDocument> doc;
  nsCOMPtr<nsIDOMDocument> domDoc;
  nsCOMPtr<nsIViewManager> viewManager;
  nsCOMPtr<nsIWidget> widget;
  nsCOMPtr<nsIDOMWindow> window = do_QueryInterface((nsISupports*)aDOMWindow, &rv);
  NS_ENSURE_TRUE(window, nsnull);
  rv = window->GetDocument(getter_AddRefs(domDoc));
  doc = do_QueryInterface(domDoc, &rv);
  NS_ENSURE_TRUE(doc, nsnull);
  nsIPresShell *shell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(shell, nsnull);
  viewManager = shell->GetViewManager();
  NS_ENSURE_TRUE(viewManager, nsnull);
  rv = viewManager->GetRootWidget(getter_AddRefs(widget));
  NS_ENSURE_TRUE(widget, nsnull);

  return _get_gtk_widget_from_nsIWidget(widget);
}

GtkWidget *
g_mozilla_engine_get_mozcontainer(GMozillaEngine *self)
{
  NS_ENSURE_TRUE(self, nsnull);

  nsresult rv;
  nsCOMPtr<nsIViewManager> viewManager;
  nsCOMPtr<nsIWidget> widget;
  nsCOMPtr<nsIDOMWindow> window;
  rv = GetSendEventTools(self, getter_AddRefs(widget), getter_AddRefs(viewManager), getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, nsnull);
  NS_ENSURE_TRUE(widget, nsnull);

  return _get_gtk_widget_from_nsIWidget(widget);
}

int
g_mozilla_xshm_buffer_set_block(GMozillaEngine *self, int aSetBlocked)
{
  NS_ENSURE_ARG(self);

  MicrobEalXshmRenderer *mshm = (MicrobEalXshmRenderer*)g_object_get_data(G_OBJECT(self->global), G_MOZILLA_PAINT_LISTENER);
  NS_ENSURE_TRUE(mshm, NS_ERROR_FAILURE);

  ULOG("Unblock painting:%p:%i->%i", self, mshm->mShmData.IsBlocked(), aSetBlocked);
  mshm->mShmData.SetBlocked(aSetBlocked);
  return NS_OK;
}

int
g_mozilla_engine_update_fixed_frames(GMozillaEngine *self, gboolean checkOnly)
{
#ifdef MICROB_API_INVALIDATE_FIXED_FRAMES
  nsresult rv;

  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(self->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
  rv = webBrowser->GetContentDOMWindow(getter_AddRefs(window));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIDOMDocument> ddoc;
  // Get DOM Document
  rv = window->GetDocument(getter_AddRefs(ddoc));
  NS_ENSURE_SUCCESS(rv, rv);
  // Get Document
  nsCOMPtr<nsIDocument> doc = do_QueryInterface(ddoc, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  // Get Primary Shell
  nsIPresShell *presShell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(presShell, rv);
  return presShell->InvalidateFixedFrames(checkOnly?PR_TRUE:PR_FALSE);
#else
  return NS_ERROR_NOT_IMPLEMENTED;
#endif
}

int
g_mozilla_engine_destroy_internal(GMozillaEngine *self)
{
  if (self->shistory_listener) {
    nsSHistoryListener* shistory = static_cast<nsSHistoryListener*>(self->shistory_listener);
    NS_IF_RELEASE(shistory);
    self->shistory_listener = NULL;
  }
  if (self->find) {
    nsITypeAheadFind *taFind = static_cast<nsITypeAheadFind *>(self->find);
    NS_IF_RELEASE(taFind);
    self->find = NULL;
  }
  
  // send memory pressure notification to release any unnecessary allocations
  nsCOMPtr<nsIObserverService> os = do_GetService("@mozilla.org/observer-service;1");
  if (os)
    os->NotifyObservers(nsnull, "memory-pressure", NS_LITERAL_STRING("low-memory").get());

  return NS_OK;
}

void
send_frame_scroll_overflow_message (GMozillaEngine *self, int overflowx, int overflowy)
{
    if(self == NULL)
        return;

    ULOG ("Frame scroll overflow (%d,%d)", overflowx, overflowy);

    GMozillaWeb* web = G_MOZILLA_WEB(self->global);
    if (g_mozilla_web_check_bus(web)) {
        FrameScrollOverflowMessage msg = { (guint32) self, (guint32) overflowx, (guint32) overflowy };
        if (!g_web_bus_send_message(web->web_bus,
                                  web->ui_bus_name,
                                  FRAME_SCROLL_OVERFLOW_MSG,
                                  &msg,
                                  sizeof(msg))) {
            ULOG ("Failed to send frame scroll overflow message to UI.");
        }
    }
}

int
g_mozilla_web_destroy_internal(GMozillaWeb *self)
{
  MicrobEalXshmRenderer *mxshm = (MicrobEalXshmRenderer*)g_object_steal_data(G_OBJECT(self), G_MOZILLA_PAINT_LISTENER);
  NS_IF_RELEASE(mxshm);
  MicrobEalCertificate *cert = (MicrobEalCertificate*)g_object_steal_data(G_OBJECT(self), G_MOZILLA_CERTIFICATE_LISTENER);
  NS_IF_RELEASE(cert);
  return NS_OK;
}

void
g_mozilla_web_force_online(void)
{
  nsCOMPtr<nsIIOService> ioService = do_GetService("@mozilla.org/network/io-service;1");
  if (!ioService)
    return;
  ioService->SetOffline(PR_FALSE);
}
