/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "common.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"
#include "gmozillapluginlist.h"
#include "gmozillahistorylist.h"
#include "gtkmozembed.h"
#include "gmozillamarshal.h"
#include "gmozillatransferitem.h"
#include "gmozillatransferitemlist.h"
#include "gmozillapassman.h"
#include <glib/gstdio.h>
#include "gmozillacppwrapper.h"
#include <libosso.h>
#include <malloc.h>
#include <plstr.h>
#include <gwebwidget.h>

#ifdef USE_OPENSSL
#include <openssl/x509.h>
#endif

#ifdef USE_CST
#include <cst.h>
#endif

#include <unistd.h>
#include <string.h>

#include "microb_eal_components.h"

enum
{
    ASK_COOKIE,                   /** < When an ask cookie dialog is required */
    CERTIFICATE_DIALOG,           /** < When a certificate dialog is required */
    CERTIFICATE_PASSWORD_DIALOG,  /** < When a certificate dialog requires a password */
    CERTIFICATE_DETAILS,          /** < When a certificate details dialog is required */
    HISTORY_ADDED,                /** < When a url is added into history */
    SELECT_MATCH_SIGNAL,
    ON_SUBMIT_SIGNAL,
    ICON_SIGNAL,
    MODAL_DIALOG,
    CONNECTIVITY_STATUS,
    ON_SERVICE_NOTIFY,
    LAST_SIGNAL                   /** < Last signal used */
};

guint g_mozilla_web_signals[LAST_SIGNAL] = { 0 };

static GMozillaWeb* gWebGlobal = NULL;

typedef struct _GWebCookieContext GWebCookieContext;
struct _GWebCookieContext
{
  gboolean remember_decision;
  gboolean accept;
};

#ifdef MOZEAL_LOGGING
static gboolean bmozeal_trace = FALSE;
void
pr_mozeal_trace (const char *fmt, ...)
{
    if (bmozeal_trace)
    {
        va_list vargs;
        va_start (vargs, fmt);
        vprintf (fmt, vargs);
        va_end (vargs);
    }
}
#endif

// class stuff
static void g_mozilla_web_class_init (GMozillaWebClass *);
static void g_mozilla_web_set_default_config (GMozillaWeb *self);
static gboolean g_mozilla_create_extensions_folder(void);
void g_mozilla_connectivity_set_global (GMozillaWeb *self);
void g_mozilla_connectivity_handle(GMozillaEngine *self, gint event);
void g_mozilla_connectivity_set_proxy ();
void g_mozilla_connectivity_unset_global(GMozillaWeb *self);

// GWebBus message handler
static void web_bus_message_handler(guint msg_id, gpointer* msg, guint msg_len, GMozillaWeb* web);
static void web_bus_pingpong_handler(guint msg_id, gpointer* msg, guint msg_len, GMozillaWeb* web);


#ifdef UNUSED
static void
g_mozilla_web_show_error (const char *message)
{
    TRACE_LOG();
    /* MTBD | Not used, but may be it is usefull stuff */
    return;
}
#endif /*UNUSED*/

GMozillaWeb* g_mozilla_get_current_web(void)
{
  return gWebGlobal;
}

void
g_web_exit(void)
{
  g_mozilla_engine_observe (NULL, NULL, "quit-application", (gunichar2 *) "s\0h\0u\0t\0d\0o\0w\0n\0\0");
  gtk_moz_embed_pop_startup();
  return;
}

static void
g_mozilla_web_destroy (GMozillaWeb *self)
{
    TRACE_LOG();
    // Checking self pointer
    g_return_if_fail(self);

    if (self->window_list)
    {
        GList *c = NULL;
        ULOG("Client init web without closing old %i engines", g_list_length(self->window_list));
        for (c = (GSList*) self->window_list; c; c = c->next)
        {
            g_web_engine_destroy(c->data);
        }
        g_list_free(self->window_list);
    }

    if (GTK_MOZ_EMBED_COMMON(self->global))
    {
        gtk_widget_destroy(GTK_WIDGET(self->global));
        self->global = NULL;
    }
    if (self->default_cursor)
    {
        gdk_cursor_unref(self->default_cursor);
        self->default_cursor = NULL;
    }

    g_mozilla_connectivity_unset_global(self);

    microb_eal_components_del();

    EAL_IF_GFREE_FUNC(self->web_bus, g_object_unref);
    // Check Password Manager
    EAL_IF_GFREE_FUNC(self->password_manager, g_object_unref);
    EAL_IF_GFREE(self->ui_bus_name);
    gWebGlobal = NULL;

    g_mozilla_web_destroy_internal(self);

    if (self->shutdown_xpcom) {
      g_mozilla_plugins_enable(self, FALSE);
      g_web_bus_unref(self->web_bus);
      self->web_bus = NULL;
      gtk_main_quit();
    }
}

static void
connectivity_status_cb(GMozillaWeb *global, gint code, GMozillaEngine *self)
{
    if (self)
        g_mozilla_connectivity_handle(self, code);
}

static GWebEngine*
g_mozilla_web_new_web_engine_window_with_context (GMozillaWeb *self, gpointer context, gboolean leave_tracks, gboolean without_history)
{
    TRACE_LOG();

    g_return_val_if_fail(self, NULL);

    GMozillaEngine * engine = G_MOZILLA_ENGINE(g_mozilla_engine_new_with_context_and_tracks(context, 
                                               leave_tracks,
                                               without_history,
                                               GPOINTER_TO_UINT(g_object_steal_data(G_OBJECT(self), G_MOZILLA_CHROME_FLAGS))));

    ULOG_INFO("New e:%p, g:%p, ctx:%p", engine, engine?engine->engine:NULL, context);
    if (self->web_bus) {

        g_web_bus_update_channel(self->web_bus);
    }

    g_return_val_if_fail(engine, NULL);

    engine->global = G_OBJECT(self);
    
    self->window_list = g_list_append(self->window_list, engine);

    g_signal_connect(G_OBJECT(self), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, G_CALLBACK(connectivity_status_cb), engine);

    if (!self->is_configured) {
        g_mozilla_web_set_default_config (self);
        g_mozilla_connectivity_set_proxy();
    }

#ifdef MICROB_GTK_MOZ_EMBED
    if (engine->engine)
        GTK_MOZ_EMBED(engine->engine)->common = (GtkObject*)self->global;
#endif

    return G_WEB_ENGINE(engine);
}

static GWebEngine*
g_mozilla_web_new_web_engine_window (GMozillaWeb *self)
{
    TRACE_LOG();

    return g_mozilla_web_new_web_engine_window_with_context (self, NULL, FALSE, FALSE);
}

static GWebTransferItem*
g_mozilla_web_new_web_transfer_item (GMozillaWeb *self, const gchar* url, const gchar* filename)
{
    TRACE_LOG();

    ULOG_DEBUG_F("url:'%s', filename: '%s'", url, filename);

    GWebTransferItem *i = G_WEB_TRANSFER_ITEM(g_mozilla_transfer_item_new_with_url(url, filename));
    self->ongoing_download_list = g_list_append(self->ongoing_download_list, G_OBJECT (i));
    g_object_set_data(G_OBJECT(i), G_MOZILLA_GLOBAL, self);
    return i;
}

static GWebTransferItem*
g_mozilla_web_initialize_web_transfer_item (GMozillaWeb *self, GObject *web_transfer_item)
{
    TRACE_LOG();
    GWebTransferItem *i = G_WEB_TRANSFER_ITEM(g_mozilla_transfer_item_new_with_item(web_transfer_item));
    self->ongoing_download_list = g_list_append(self->ongoing_download_list, G_OBJECT (i));
    g_object_set_data(G_OBJECT(i), G_MOZILLA_GLOBAL, self);
    return i;
}

void
g_mozilla_web_pause_all_transfer_item (GMozillaWeb *self)
{
    TRACE_LOG();

    gint i, max = g_list_length (self->ongoing_download_list);
    for (i = 0; i < max; i++) {
        g_signal_emit_by_name(G_OBJECT (g_list_nth_data (self->ongoing_download_list, i)), G_WEBWIDGET_SIGNAL_ERROR);
    }

    return ;
}

static GWebTransferItemList*
g_mozilla_web_get_web_transfer_item_list (GMozillaWeb *self)
{
    TRACE_LOG();
    return G_WEB_TRANSFER_ITEM_LIST(g_mozilla_transfer_item_list_new(self));
}

GWebStatus
g_mozilla_web_set_home_page (GMozillaWeb *self, const gchar* home_page)
{
    TRACE_LOG();
    if (g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_HOME_PAGE_LOCATION, (gchar*)home_page)) {
        g_mozilla_engine_save_prefs();

        return G_WEB_STATUS_OK;
    }
    return G_WEB_ERR;
}

static gchar*
g_mozilla_web_get_home_page (GMozillaWeb *self)
{
    TRACE_LOG();
    gchar *home_page = NULL;
    if (!g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_HOME_PAGE_LOCATION, &home_page))
        return NULL;
    return home_page;
}

static gchar*
g_mozilla_web_get_default_home_page (GMozillaWeb *self)
{
    TRACE_LOG();

    return g_strdup (BROWSER_DEFAULT_PAGE);
}

GWebStatus
g_mozilla_web_set_default_home_page (GMozillaWeb *self, const gchar* home_page)
{
    TRACE_LOG();

    // NOT SUPPORTED

    return G_WEB_STATUS_OK;
}

static gint
g_mozilla_web_get_maximum_cache_size (GMozillaWeb *self)
{
    TRACE_LOG();
    /* MTBD | Not Used */
    return -1;
}

static gboolean
g_mozilla_web_home_page_available (GMozillaWeb *self)
{
    TRACE_LOG();
    /* MTBD | Not Used */
    return TRUE;
}

static void
g_mozilla_web_set_frame_borders_enabled (GMozillaWeb *self, gboolean enabled)
{
    TRACE_LOG();
    /* TBD | Strange feature, useless also as Frame merging ;) */
    if (g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DISPLAY_FRAME_BORDER, &enabled))
        g_mozilla_engine_save_prefs();
    return;
}

static gboolean
g_mozilla_web_get_frame_borders_enabled (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD | Strange feature, useless also as Frame merging ;) */
    gboolean enabled = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DISPLAY_FRAME_BORDER, &enabled))
        return FALSE;
    return enabled;
}

static GWebStatus
g_mozilla_web_set_location_completion_suffixes (GMozillaWeb *self, const gchar* suffixes)
{
    TRACE_LOG();
    if (g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_WEB_ADDRESS_SUFFIXES, (gchar*)suffixes)) {
        g_mozilla_engine_save_prefs();
        return G_WEB_STATUS_OK;   
    }
    return G_WEB_ERR;
}

static gchar*
g_mozilla_web_get_location_completion_suffixes (GMozillaWeb *self)
{
    TRACE_LOG();
    gchar *web_suffix = NULL;
    if (!g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_WEB_ADDRESS_SUFFIXES, &web_suffix))
        return NULL;
    return web_suffix;
}

static GWebStatus
g_mozilla_web_set_plugins_enabled (GMozillaWeb *self, gboolean enabled)
{
    TRACE_LOG();
    /*IMP | May be pref ?
      if (!gtk_moz_embed_common_set_plugins_enabled(GTK_MOZ_EMBED_COMMON(self->global), enabled))
      return G_WEB_ERR;
      */
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_plugins_enabled (GMozillaWeb *self)
{
    TRACE_LOG();
    /*IMP | May be pref ?
      return gtk_moz_embed_common_get_plugins_enabled(self->global);
      */
    return TRUE;
}

static GWebStatus
g_mozilla_web_set_ssl2_support (GMozillaWeb *self, gboolean enabled)
{
    TRACE_LOG();
    if (g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_ENABLE_SSL2, &enabled)) {
        g_mozilla_engine_save_prefs();
        return G_WEB_STATUS_OK;   
    }
    return G_WEB_ERR;
}

static gboolean
g_mozilla_web_get_ssl2_support (GMozillaWeb *self)
{
    TRACE_LOG();
    gboolean enabled = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_ENABLE_SSL2, &enabled))
        return FALSE;
    return enabled;
}

static GWebStatus
g_mozilla_web_set_js_enabled (GMozillaWeb *self, gboolean enabled)
{
    TRACE_LOG();
    if (g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_ENABLE_JAVA_SCRIPT, &enabled)) {
        g_mozilla_engine_save_prefs();
        return G_WEB_STATUS_OK;   
    }
    return G_WEB_ERR;
}

static gboolean
g_mozilla_web_get_js_enabled (GMozillaWeb *self)
{
    TRACE_LOG();
    gboolean enabled = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_ENABLE_JAVA_SCRIPT, &enabled))
        return FALSE;
    return enabled;
}

#ifdef UNUSED
static void
g_mozilla_web_show_memory_cache_error (const char *message)
{
    TRACE_LOG("%s", message);
    /* MTBD | What this stuff do here??? Delete? */
    return;
}
#endif /*UNUSED*/

static void
g_mozilla_web_set_memory_cache (GMozillaWeb *self, GMemoryCache acceptance)
{
    TRACE_LOG();
    gint size = G_MEMORY_CACHE_AUTOMATIC_SIZE;
    switch (acceptance)
    {
    case G_MEMORY_CACHE_AUTOMATIC:
        size = G_MEMORY_CACHE_AUTOMATIC_SIZE;
        break;
    case G_MEMORY_CACHE_OFF:
        size = G_MEMORY_CACHE_OFF_SIZE;
        break;
    case G_MEMORY_CACHE_SMALL:
        size = G_MEMORY_CACHE_SMALL_SIZE;
        break;
    case G_MEMORY_CACHE_DEFAULT:
        size = G_MEMORY_CACHE_DEFAULT_SIZE;
        break;
    case G_MEMORY_CACHE_MEDIUM:
        size = G_MEMORY_CACHE_MEDIUM_SIZE;
        break;
    case G_MEMORY_CACHE_LARGE:
        size = G_MEMORY_CACHE_LARGE_SIZE;
        break;
    default:
        size = G_MEMORY_CACHE_AUTOMATIC_SIZE;
        break;
    }
    if (g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_MEMORY_CACHE_SIZE, &size)) {
        g_mozilla_engine_save_prefs();
    }
    return;
}

static GMemoryCache
g_mozilla_web_get_memory_cache (GMozillaWeb *self)
{
    TRACE_LOG();
    gint mem_cache = G_MEMORY_CACHE_AUTOMATIC_SIZE;
    if (!g_mozilla_engine_get_pref (G_TYPE_INT, G_MOZILLA_PREF_MEMORY_CACHE_SIZE, &mem_cache))
        return G_MEMORY_CACHE_AUTOMATIC;

    switch (mem_cache)
    {
    case G_MEMORY_CACHE_OFF_SIZE:
        return G_MEMORY_CACHE_OFF;
    case G_MEMORY_CACHE_SMALL_SIZE:
        return G_MEMORY_CACHE_SMALL;
    case G_MEMORY_CACHE_DEFAULT_SIZE:
        return G_MEMORY_CACHE_DEFAULT;
    case G_MEMORY_CACHE_MEDIUM_SIZE:
        return G_MEMORY_CACHE_MEDIUM;
    case G_MEMORY_CACHE_LARGE_SIZE:
        return G_MEMORY_CACHE_LARGE;
    default:
        return G_MEMORY_CACHE_AUTOMATIC;
    }
}

static void
g_mozilla_web_reset_cache_history_length (GMozillaWeb *self)
{
    TRACE_LOG();
    /* MTBD | Not Used */
}

static GWebStatus
g_mozilla_web_set_cache_size (GMozillaWeb *self, gint size)
{
    TRACE_LOG();
    /* MTBD | Not used */
    if (!g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_MEMORY_CACHE_SIZE, &size))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();
    return G_WEB_STATUS_OK;
}

static gint
g_mozilla_web_get_cache_size (GMozillaWeb *self)
{
    TRACE_LOG();
    gint size = G_MEMORY_CACHE_AUTOMATIC_SIZE;
    /* MTBD | Not used */
    if (!g_mozilla_engine_get_pref (G_TYPE_INT, G_MOZILLA_PREF_MEMORY_CACHE_SIZE, &size))
        return G_MEMORY_CACHE_AUTOMATIC_SIZE;
    return size;
}

static GWebStatus
g_mozilla_web_set_browse_offline (GMozillaWeb *self, gboolean offline)
{
    TRACE_LOG();
    if (!g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_OFFLINE_BROWSER, &offline))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_browse_offline (GMozillaWeb *self)
{
    TRACE_LOG();
    gboolean offline = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_OFFLINE_BROWSER, &offline))
        return FALSE;
    return offline;
}

static GWebStatus
g_mozilla_web_set_automatic_redirection_enabled (GMozillaWeb *self, gboolean enabled)
{
    TRACE_LOG();
    gint redirection_limit = enabled ? G_MOZILLA_PREF_REDIRECTION_DEFAULT_LIMIT : 0;
    if (!g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_REDIRECTION_LIMIT, &redirection_limit))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_automatic_redirection_enabled (GMozillaWeb *self)
{
    TRACE_LOG();
    gint redirection_limit = G_MOZILLA_PREF_REDIRECTION_DEFAULT_LIMIT;

    /* FIXME - there is no "browser.accept.redirects */
    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_REDIRECTION_LIMIT, &redirection_limit))
        return TRUE;

    return redirection_limit != 0;
}

static GWebStatus
g_mozilla_web_set_load_images (GMozillaWeb *self, GWebEngineImagePolicy level)
{
    TRACE_LOG();
    GSList *c = NULL;

    for (c = (GSList*) self->window_list; c; c = c->next)
    {
       g_mozilla_set_image_policy(c->data, level);
    }
    
    return G_WEB_STATUS_OK;
}

static GWebEngineImagePolicy
g_mozilla_web_get_load_images (GMozillaWeb *self)
{
    TRACE_LOG();
    gint show_images = 1;
    gboolean download_images = TRUE;
    
    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SHOW_IMAGES, &show_images))
        return G_WEBENGINE_POLICY_ALL_IMAGES;
    
    switch (show_images)
    {
    case 1:
        // NOTE: G_MOZILLA_PREF_DOWNLOAD_IMAGES="microb.download_images", 
        //       see: microbImageLoadingPolicy.js
        if (g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_DOWNLOAD_IMAGES, &download_images))
        {
            if (!download_images)
                return G_WEBENGINE_POLICY_LOADED_IMAGES;
        }
        return G_WEBENGINE_POLICY_ALL_IMAGES;
    case 2:
        return G_WEBENGINE_POLICY_NO_IMAGES;
    default:
        return G_WEBENGINE_POLICY_ALL_IMAGES;
    }
}

static void
g_mozilla_web_disconnect (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD | Not Used */
    return ;
}

static GWebStatus
g_mozilla_web_clear_cache (GMozillaWeb *self)
{
    TRACE_LOG();
    //guint memoryCache = 1;
    if (gtk_moz_embed_common_clear_cache())
        return G_WEB_ERR;

    return G_WEB_STATUS_OK;
}

static GWebPluginList*
g_mozilla_web_get_plugins_list (GMozillaWeb *self)
{
    TRACE_LOG();
    GObject * pl_list = g_mozilla_plugin_list_new();
    if (pl_list)
      return G_WEB_PLUGIN_LIST(pl_list);
    return NULL;
}

static GWebHistoryList*
g_mozilla_web_get_history_list (GMozillaWeb *self)
{
    TRACE_LOG();
    return G_WEB_HISTORY_LIST(g_mozilla_history_list_new ());
}

static GWebStatus
g_mozilla_web_set_new_window_loads (GMozillaWeb *self, gboolean loads)
{
    TRACE_LOG();

    if (!g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_USE_DEFAULT_HOME_PAGE, &loads))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();

    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_new_window_loads (GMozillaWeb *self)
{
    TRACE_LOG();

    gboolean loads;

    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_USE_DEFAULT_HOME_PAGE, &loads))
        return TRUE;

    return loads;
}

static void
g_mozilla_web_set_cookie_acceptance (GMozillaWeb *self, GCookieAcceptance acceptance)
{
    TRACE_LOG();
    gint life_time_policy = 0;
    gint behaviour = 0;
    gboolean warning_before = FALSE;
    switch (acceptance)
    {
    case G_ACCEPT_COOKIE_ALWAYS:
        behaviour = 0;
        life_time_policy = 0;
        break;
    case G_ACCEPT_COOKIE_PROMPT:
        warning_before = TRUE;
        behaviour = 0;
        life_time_policy = 1;
        break;
    case G_ACCEPT_COOKIE_NEVER:
        behaviour = 2;
        life_time_policy = 0;
        break;
    default:
        g_print("You should not be here\n");
        break;
    }

    if (g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_COOKIE_BEHAVIOUR, &behaviour) && 
        g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_COOKIE_WARN_ABOUT_COOKIES, &warning_before) && 
        g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_COOKIE_LIFETIME_POLICY, &life_time_policy)) {
        g_mozilla_engine_save_prefs(); 
    }
}

static GCookieAcceptance
g_mozilla_web_get_cookie_acceptance (GMozillaWeb *self)
{
    TRACE_LOG();
    gint cookie_behaviour;
    gint cookie_lifetime;

    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_COOKIE_BEHAVIOUR, &cookie_behaviour))
       cookie_behaviour = 0;
    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_COOKIE_LIFETIMEPOLICY, &cookie_lifetime))
       cookie_lifetime = 0;

    if (cookie_behaviour == 2)
    {
        return G_ACCEPT_COOKIE_NEVER;
    }
    else if (cookie_behaviour == 0)
    {
        if (cookie_lifetime == 0)
            return G_ACCEPT_COOKIE_ALWAYS;
        else if (cookie_lifetime == 1)
            return G_ACCEPT_COOKIE_PROMPT;
        else
            return G_ACCEPT_COOKIE_PROMPT;
    }

    return G_ACCEPT_COOKIE_PROMPT;
}

#ifdef UNUSED
static GSList*
g_mozilla_web_get_cookies_list (GMozillaWeb *self)
{
    TRACE_LOG();
    GList * list = NULL;
    /* MTBD | Not used, can't find this function in interface */
    /*IMP
      list = gtk_moz_embed_common_get_cookie_list(self->global);
      */
    return (GSList*)list;
}
#endif /*UNUSED*/

static void
g_mozilla_web_set_popup_acceptance (GMozillaWeb *self, GPopupAcceptance acceptance)
{
    TRACE_LOG();
    gboolean blockDuringLoad = FALSE;
    gint popupPolicy = 0;

    switch (acceptance)
    {
    case G_ACCEPT_POPUP_ALWAYS:
        popupPolicy = 1;
        blockDuringLoad = FALSE;
        break;
    case G_ACCEPT_POPUP_PROMPT:
        popupPolicy = 1;
        blockDuringLoad = TRUE;
        break;
    case G_ACCEPT_POPUP_NEVER:
        popupPolicy = 2;
        blockDuringLoad = FALSE;
        break;
    }
    g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_ENABLE_POPUP_WINDOWS, &popupPolicy);
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_OPEN_POPUP_DURING_LOAD, &blockDuringLoad);

    g_mozilla_engine_save_prefs();
}

static GPopupAcceptance
g_mozilla_web_get_popup_acceptance (GMozillaWeb *self)
{   /* MTBD | Not used, can't find this function in interface */
    TRACE_LOG();
    gboolean blockDuringLoad = FALSE;
    gint popupPolicy = 0;

    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_OPEN_POPUP_DURING_LOAD, &blockDuringLoad))
        return G_ACCEPT_POPUP_ALWAYS;

    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_ENABLE_POPUP_WINDOWS, &popupPolicy))
       return G_ACCEPT_POPUP_ALWAYS;

    if (blockDuringLoad && popupPolicy == 1)
       return G_ACCEPT_POPUP_PROMPT;

    if (!blockDuringLoad && popupPolicy == 2)
       return G_ACCEPT_POPUP_NEVER;

    return G_ACCEPT_POPUP_ALWAYS;
}

static GWebStatus
g_mozilla_web_set_language (GMozillaWeb *self, const gchar* language)
{
    ULOG_DEBUG_F("language:%s\n", language);
    gchar *moz_lang_short = NULL;
    gchar *moz_lang_long = NULL;
    gchar *cur_lang = NULL;
    if (g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_LANGUAGE, &cur_lang) && language)
    {
        if (cur_lang[0]==language[0]
            && cur_lang[1]==language[1]) {
            EAL_GFREE(cur_lang);
            return G_WEB_STATUS_OK;
        }
    }
    if (language)
    {
        gint len = strlen(language);
        if (len >=2)
            moz_lang_short = g_strndup(language, 2);
        if (len >=5)
        {
            moz_lang_long = g_strndup(language, 5);
            if (moz_lang_long) moz_lang_long[2]='-';
        }
    }
    gchar *result_lang = moz_lang_short?moz_lang_short:G_MOZILLA_LANG_SHORT_DEFAULT;
    // g_strdup_printf("%s,%s,en-us,en", moz_lang_long?moz_lang_long:"", moz_lang_short?moz_lang_short:"");
    if (!g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_LANGUAGE, result_lang))
        return G_WEB_ERR;
    g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_EAL_LOCALE, moz_lang_long?moz_lang_long:G_MOZILLA_EAL_LOCALE_DEFAULT);
    g_mozilla_engine_save_prefs();
    EAL_GFREE(moz_lang_short);
    EAL_GFREE(moz_lang_long);
    EAL_IF_GFREE(cur_lang);

    return G_WEB_STATUS_OK;
}

static gchar*
g_mozilla_web_get_language (GMozillaWeb *self)
{
    TRACE_LOG();
    gchar *sval = NULL;
    if (!g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_LANGUAGE, &sval))
        sval = NULL;
    return sval;
}

static GWebStatus
g_mozilla_web_suspend (GMozillaWeb *self)
{
    TRACE_LOG();
    if (self->web_is_active)
      g_mozilla_web_suspend_native(self, TRUE);
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_suspended (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD */
    return !self->web_is_active;

}

static void
g_mozilla_web_resume (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD */
    g_mozilla_web_suspend_native(self, FALSE);
}

static void
g_mozilla_web_set_ua_string (GMozillaWeb *self, gchar *ua_string)
{
    TRACE_LOG();
    g_return_if_fail(self && ua_string); 
    gchar * sval = NULL;
    g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_USER_UA_ADD, &sval);
    gchar *temp_str = g_strdup_printf ("%s%s", sval ? sval : MOZILLA_DEFAULT_VENDOR_NAME, ua_string);
    g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_EAL_NAME, temp_str);
    g_mozilla_engine_save_prefs();
    g_free(temp_str);
    g_free(sval);
    sval = NULL;
}

static void
g_mozilla_web_enable_url_loading (GMozillaWeb *self, gboolean enable)
{
    TRACE_LOG();
    /* TBD */
}

static void
g_mozilla_web_free_memory (GMozillaWeb *self, gboolean progressive)
{
    TRACE_LOG();
    /* TBD */
}


static GWebStatus
g_mozilla_web_store_transfers (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD */
    g_mozilla_engine_download_setup(&self->download_listener, self);
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_restore_transfers (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD */
    void *listener = NULL;
    g_mozilla_engine_download_setup(&listener, self);
    self->download_listener = listener;
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_delete_all_cookies (GMozillaWeb *self)
{
    TRACE_LOG();
    GSList *cookiesList = NULL;

    cookiesList = gtk_moz_embed_common_get_cookie_list();

    if (!gtk_moz_embed_common_delete_all_cookies(cookiesList))
        return G_WEB_ERR;

    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_confirm_cookie (GMozillaWeb *self, gpointer context, gboolean accept, GWebCookieAction action)
{
    TRACE_LOG();
    g_return_val_if_fail(context, G_WEB_ERR);

    if (action == G_WEB_COOKIE_REFUSE_ALL_FROM_SERVER || action == G_WEB_COOKIE_NO_ACTION) 
        ((GWebCookieContext*) context)->accept = FALSE;
    else
        ((GWebCookieContext*) context)->accept = TRUE;

    if (action == G_WEB_COOKIE_ACCEPT_ALL_FROM_SERVER || action == G_WEB_COOKIE_REFUSE_ALL_FROM_SERVER)
        ((GWebCookieContext*) context)->remember_decision = TRUE;
    else
        ((GWebCookieContext*) context)->remember_decision = FALSE;

    return G_WEB_STATUS_OK;
}

static GObject*
g_mozilla_web_get_global (GMozillaWeb *self)
{
    TRACE_LOG();
    return G_OBJECT(self);
}

static GWebCertificateMessage
g_mozilla_web_get_certificate_enum_message (GMozillaWeb *self,
                                            GObject *context)
{
    TRACE_LOG();

    g_return_val_if_fail(context, G_WEB_CERT_MESS_UNKNOWN);
    g_return_val_if_fail(self, G_WEB_CERT_MESS_UNKNOWN);

    self->mCertContexts = g_list_append(self->mCertContexts, context);

    return ((GMozillaWebCertContext*)context)->error;
}

static GWebCertificateReaction
g_mozilla_web_perform_crtificate_context_action (GMozillaWeb *self,
                                                 GObject *context,
                                                 GWebCertificateAction action,
                                                 GObject *cert)
{
    TRACE_LOG();

    g_return_val_if_fail(context, G_WEB_CERT_REACTION_NOTHING);

    if (action == G_WEB_CERT_ACTION_ACCEPT) {
        GMozillaWebCertContext* ctx = (GMozillaWebCertContext*) context;
        if (cert && ctx->mozcert != cert) {
            return G_WEB_CERT_REACTION_CLOSE;
        }
    } else if (action == G_WEB_CERT_ACTION_CANCEL) {
        return G_WEB_CERT_REACTION_CLOSE;
    }
    TRACE_LOG();

    return G_WEB_CERT_REACTION_NOTHING;
}

static GWebStatus
g_mozilla_web_get_certificate_context_iter (GMozillaWeb *self,
                                            GObject* context,
                                            GObject **iterator)
{
    TRACE_LOG();
    g_return_val_if_fail(self, G_WEB_ERR);
    g_return_val_if_fail(context, G_WEB_ERR);
    g_return_val_if_fail(iterator, G_WEB_ERR);

    if ((*iterator = (GObject*)g_list_find(self->mCertContexts, context)) != NULL)
        return G_WEB_STATUS_OK;

    return G_WEB_ERR_NULL_POINTER;
}

static GObject*
g_mozilla_web_get_certificate_iter (GMozillaWeb *self, GObject** iterator)
{
    TRACE_LOG();
    g_return_val_if_fail(self, NULL);
    g_return_val_if_fail(iterator, NULL);
    GList * iter = (GList*)*iterator;
    if (iter && iter->data) {
        GMozillaWebCertContext* context = (GMozillaWebCertContext*)iter->data;
        return (GObject*)context->mozcert;
    }

    return NULL;
}

static void
g_mozilla_web_certificate_free (GMozillaWeb *self, GObject* cert)
{
    TRACE_LOG();
    /* TBD */
}

static void
g_mozilla_web_certificate_iter_free (GMozillaWeb *self, GObject* iterator)
{
    TRACE_LOG();
    /* TBD */
}

static void
g_mozilla_web_notify_plugins (GMozillaWeb *self, gint event)
{
    TRACE_LOG();

    gboolean status = FALSE; 

    if (self && (self->disabled_pllist)) {
        if (G_WEB_PLUGINS_LOW_MEMORY == event)
            status = g_mozilla_plugin_disable_plugins(&self->disabled_pllist);
        else if (G_WEB_PLUGINS_NORMAL_MEMORY == event)
            status = g_mozilla_plugin_enable_plugins (&self->disabled_pllist);
    }

    if (status)
      gtk_moz_embed_common_reload_plugins (FALSE);
    if (G_WEB_PLUGINS_LOW_MEMORY != event)
      return;

    g_mozilla_engine_observe (NULL, NULL, G_MOZILLA_MEMORY_PRESSURE, (gunichar2 *) G_MOZILLA_HEAP_MINIMIZE_UNICODE);
    malloc_trim (0);
}

static void
g_mozilla_web_notify_service (GMozillaWeb *self, const gchar *service_name, gpointer *context, const gchar *topic, gchar *data)
{
    ULOG_DEBUG_F("service_name:%s, topic:%s, ctx:%p", service_name, topic, context);

    if (0 == g_ascii_strcasecmp(topic, G_MOZILLA_FOCUS_OUT))
    {
      gdk_keyboard_ungrab (GDK_CURRENT_TIME);
      return;
    }

    if (0 == g_ascii_strcasecmp(topic, G_MOZILLA_ENGINE_CLEAN_AUTH ))
       g_mozilla_web_store_transfers (self);

    glong items_written = 0;
    glong items_read = 0;
    gunichar2 *name = NULL;
    if (data)
        name = g_utf8_to_utf16 (data, strlen (data), &items_read, &items_written, NULL);
    g_mozilla_engine_observe (service_name, context, topic, name);
    EAL_IF_GFREE(name);
}


static GWebStatus
g_mozilla_web_set_user_agent_id (GMozillaWeb *self, gint ua_id)
{
    TRACE_LOG();
    if (!g_mozilla_engine_set_pref(G_TYPE_INT, G_MOZILLA_PREF_IDENTIFY_AS, &ua_id))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();

    return G_WEB_STATUS_OK;
}

static gint
g_mozilla_web_get_user_agent_id (GMozillaWeb *self)
{
    TRACE_LOG();
    gint ua_id=0;
    if (!g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_IDENTIFY_AS, &ua_id))
        return 0;

    return ua_id;
}

static GWebPasswordManager*
g_mozilla_web_get_password_manager (GMozillaWeb *self, const gchar* url, gint index)
{
    TRACE_LOG();
    // Checking self pointer
    if (self) {

        ULOG_DEBUG_F("get_pwd_mgr: '%s', %i, list: %p", url, index, self->logins_list);

        if (self->logins_list && g_list_length(self->logins_list) >= index) {
            // Check Password Manager
            if (self->password_manager) {
                // Free Password Manager
                g_object_unref(self->password_manager);
            }
            // Create new Password Manager instance
            self->password_manager = g_mozilla_password_manager_login_new(url, index, self->logins_list);
            // Return New instance of the Password Manager
            return G_WEB_PASSWORD_MANAGER(self->password_manager);
        }
    }

    return NULL;
}

static gint
g_mozilla_web_username_matchinfo_get_matches(GMozillaWeb *self, GObject* info)
{
    TRACE_LOG();
    gint matches_count = 0;

    if (info) {

        // NOTE: we should be getting matches_count from glist, but due to UI flow, it is safer to get from engine.
        // matches_count = g_list_length((GList*)info);
        matches_count = g_mozilla_cpp_get_logins(NULL, NULL);
    }

    ULOG_DEBUG_F("matches_count %d\n", matches_count);

    return matches_count;
}

static void
g_mozilla_web_username_matchinfo_delete (GMozillaWeb *self,  GObject* info, gint index)
{
    TRACE_LOG();
    if (!info)
        return;

    { // evil hacking to make it works with current UI's flow.
        gint num = g_list_length((GList*)info);

        if (!index && (num > 1)) { // 0

            *((GList*) info) = *(g_list_nth ((GList*)info, index+1));
            // TODO: return value should not be ignored.
            GList* ret = g_list_remove ((GList*) info, g_list_nth_data ((GList*) info, num));
            // Ignoring warning about ignored return value
            (void) ret;

        } else if (num == 1) {
            ((GList *) info)->data = NULL;
        }
        else {
            // TODO: return value should not be ignored.
            GList* ret = g_list_remove ((GList*) info, g_list_nth_data ((GList*) info, index));
            // Ignoring warning about ignored return value
            (void) ret;
        }
    }

    // engine part !!!
    g_mozilla_cpp_clear_passwords_by_index(index);
}

static GWebStatus
g_mozilla_web_username_matchinfo_get_username (GMozillaWeb *self, GObject* info, gint index, gchar **username)
{
    TRACE_LOG();
    g_return_val_if_fail(info, G_WEB_ERR);
    g_return_val_if_fail(username, G_WEB_ERR);

    //FIXME XXX Here probably memory leak
    *username = g_strdup (g_list_nth_data ((GList*)info, index));

    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_store_personal_data (GMozillaWeb *self, const gchar* server,
                                   const gchar* username, const gchar* password)
{
    TRACE_LOG();
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_delete_personal_data (GMozillaWeb *self, const gchar* server,
                                    const gchar* username)
{
    TRACE_LOG();
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_set_pref (GMozillaWeb *self, gchar* name, GValue *value) 
{
    TRACE_LOG();

    gboolean result = FALSE;
    GType type = G_VALUE_TYPE (value);

    switch (type) {
    case G_TYPE_STRING:
        {
            const gchar *data = g_value_get_string (value);
            result = g_mozilla_engine_set_pref((GtkType)type, name, (void*)data);
            break;
        }
    case G_TYPE_INT:
        {
            gint data = g_value_get_int (value);
            result = g_mozilla_engine_set_pref((GtkType)type, name,  &data);
            break;
        }
    case G_TYPE_BOOLEAN:
        {
            gboolean data = g_value_get_boolean (value);
            result = g_mozilla_engine_set_pref((GtkType)type, name, &data);
            break;
        }
    default:
        break;
    }
    if (result != FALSE)
        result = g_mozilla_engine_save_prefs();
    return result;
}

static gboolean
g_mozilla_web_get_pref (GMozillaWeb *self, gchar* name, GValue *value) 
{

    gboolean result = FALSE;
    gpointer data;
    GType type = (GType) g_mozilla_engine_get_pref_type(name);

    result = g_mozilla_engine_get_pref((GtkType)type, name, &data);
    if (!result)
        return FALSE;

    g_value_init(value, type);
    switch (type) {
    case G_TYPE_STRING:
        {
            g_value_set_string (value, (gchar *)data);
            g_free(data);
            break;
        }
    case G_TYPE_INT:
        {
            g_value_set_int (value, (gint)data);
            break;
        }
    case G_TYPE_BOOLEAN:
        {
            g_value_set_boolean (value, (gboolean)data);
            break;
        }
    default:
        break;
    }
    return result;
}

static GWebStatus
g_mozilla_web_clear_all_history (GMozillaWeb *self)
{
    TRACE_LOG();
    if (NS_OK != g_mozilla_cpp_shistory_list_remove_all (self))
        return G_WEB_ERR;
    if (NS_OK != g_mozilla_history_list_clear_all ())
        return G_WEB_ERR;
    return G_WEB_STATUS_OK;
}

static void
g_mozilla_web_clear_personal_data (GMozillaWeb *self)
{
    TRACE_LOG();
    /* TBD */
}

static GWebStatus
g_mozilla_web_clear_passwords (GMozillaWeb *self)
{
    TRACE_LOG();
    g_mozilla_cpp_clear_all_passwords();
    return G_WEB_STATUS_OK;
}

static GWebStatus
g_mozilla_web_clear_authenticated_sessions (GMozillaWeb *self)
{
    TRACE_LOG();
    g_mozilla_cpp_clear_authenticated_sessions();
    return G_WEB_STATUS_OK;
}


static GWebStatus
g_mozilla_web_set_remember_password (GMozillaWeb *self, gboolean loads)
{
    TRACE_LOG();
    if (!g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_REMEMBER_PASSWORDS, &loads))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_remember_password (GMozillaWeb *self)
{
    TRACE_LOG();
    gboolean remember_password = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_REMEMBER_PASSWORDS, &remember_password))
        return FALSE;
    return remember_password;
}

static GWebStatus
g_mozilla_web_set_remember_password_for_forms (GMozillaWeb *self, gboolean loads)
{
    TRACE_LOG();
    /* TBD */
    if (!g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_PASSWORDS_SAVING, &loads))
        return G_WEB_ERR;
    g_mozilla_engine_save_prefs();
    return G_WEB_STATUS_OK;
}

static gboolean
g_mozilla_web_get_remember_password_for_forms (GMozillaWeb *self)
{
    TRACE_LOG();
    gboolean remember_password = FALSE;
    if (!g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_PASSWORDS_SAVING, &remember_password))
        return FALSE;
    return remember_password;
}

static gboolean
g_mozilla_web_get_plugin_enabled (GMozillaWeb *self, gchar *name)
{
    TRACE_LOG();
    gboolean enabled = TRUE;
    enabled = g_mozilla_plugin_get_item_status(name);
    return enabled;
}

static GWebStatus
g_mozilla_web_set_plugin_enabled (GMozillaWeb *self, gchar *name, gboolean enabled)
{
    TRACE_LOG();
    if (g_mozilla_plugin_set_item_status(name, enabled)) {
        gtk_moz_embed_common_reload_plugins (enabled);
        return G_WEB_STATUS_OK;
    }
    return G_WEB_ERR;
}

static GWebStatus
g_mozilla_web_set_oom_cb(GMozillaWeb *self, GFunc callback)
{
    TRACE_LOG();
    /* MTBD | Not used*/
    return G_WEB_STATUS_OK;
}


static GWebStatus
g_mozilla_web_set_plugin_ignore_filepath (GMozillaWeb *self)
{
    TRACE_LOG();
    GWebStatus status = (GWebStatus) 0;
    /* TBD */
    /*IMP
      Needs to implements of reading ignored plugins before Browser starting
      */
    return status;
}

static void
g_mozilla_web_interface_init (GWebIface *iface)
{
    TRACE_LOG();
    // Check Interfce
    g_return_if_fail(iface);
    
    iface->destroy                            = (void (*) (GWeb*))
        g_mozilla_web_destroy;
    iface->new_web_engine_window              = (GWebEngine* (*) (GWeb*))
        g_mozilla_web_new_web_engine_window;
    iface->new_web_engine_window_with_context = (GWebEngine* (*) (GWeb*, gpointer, gboolean, gboolean))
        g_mozilla_web_new_web_engine_window_with_context;
    iface->web_set_pref = (gboolean (*) (GWeb*, const gchar*, GValue*))
        g_mozilla_web_set_pref;
    iface->web_get_pref = (gboolean (*) (GWeb*, const gchar*, GValue*))
        g_mozilla_web_get_pref;
    iface->new_web_transfer_item              = (GWebTransferItem* (*) (GWeb*, const gchar*, const gchar*))
        g_mozilla_web_new_web_transfer_item;
    iface->initialize_web_transfer_item       = (GWebTransferItem* (*) (GWeb*, GObject*))
        g_mozilla_web_initialize_web_transfer_item;
    iface->get_web_transfer_item_list         = (GWebTransferItemList* (*) (GWeb*))
        g_mozilla_web_get_web_transfer_item_list;
    iface->set_home_page                      = (GWebStatus (*) (GWeb*, const gchar*))
        g_mozilla_web_set_home_page;
    iface->get_home_page                      = (gchar* (*) (GWeb*))
        g_mozilla_web_get_home_page;
    iface->set_language                       = (GWebStatus (*) (GWeb*, const gchar*))
        g_mozilla_web_set_language;
    iface->get_language                       = (gchar* (*) (GWeb*))
        g_mozilla_web_get_language;
    iface->reset_cache_history_length         = (void (*) (GWeb*))
        g_mozilla_web_reset_cache_history_length;
    iface->get_maximum_cache_size             = (gint (*) (GWeb*))
        g_mozilla_web_get_maximum_cache_size;
    iface->get_default_home_page              = (gchar* (*) (GWeb*))
        g_mozilla_web_get_default_home_page;
    iface->set_default_home_page              = (GWebStatus (*) (GWeb*self, const gchar*))
        g_mozilla_web_set_default_home_page;
    iface->home_page_set                      = (gboolean (*) (GWeb*))
        g_mozilla_web_home_page_available;
    iface->set_frame_borders_enabled          = (void (*) (GWeb*, gboolean))
        g_mozilla_web_set_frame_borders_enabled;
    iface->get_frame_borders_enabled          = (gboolean (*) (GWeb*))
        g_mozilla_web_get_frame_borders_enabled;
    iface->set_location_completion_suffixes   = (GWebStatus (*) (GWeb*, const gchar*))
        g_mozilla_web_set_location_completion_suffixes;
    iface->get_location_completion_suffixes   = (gchar* (*) (GWeb*))
        g_mozilla_web_get_location_completion_suffixes;
    iface->set_plugins_enabled                = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_plugins_enabled;
    iface->set_ssl2_support                   = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_ssl2_support;
    iface->get_ssl2_support                   = (gboolean (*) (GWeb*))
        g_mozilla_web_get_ssl2_support;
    iface->get_plugins_enabled                = (gboolean (*) (GWeb*))
        g_mozilla_web_get_plugins_enabled;
    iface->set_js_enabled                     = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_js_enabled;
    iface->get_js_enabled                     = (gboolean (*) (GWeb*))
        g_mozilla_web_get_js_enabled;
    iface->set_memory_cache                   = (void (*) (GWeb*, GMemoryCache))
        g_mozilla_web_set_memory_cache;
    iface->get_memory_cache                   = (GMemoryCache (*) (GWeb*))
        g_mozilla_web_get_memory_cache;
    iface->set_cache_size                     = (GWebStatus (*) (GWeb*, gint))
        g_mozilla_web_set_cache_size;
    iface->get_cache_size                     = (gint (*) (GWeb*))
        g_mozilla_web_get_cache_size;
    iface->set_browse_offline                 = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_browse_offline;
    iface->get_browse_offline                 = (gboolean (*) (GWeb*))
        g_mozilla_web_get_browse_offline;
    iface->set_automatic_redirection_enabled  = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_automatic_redirection_enabled;
    iface->get_automatic_redirection_enabled  = (gboolean (*) (GWeb*))
        g_mozilla_web_get_automatic_redirection_enabled;
    iface->set_load_images                    = (GWebStatus (*) (GWeb*, GWebEngineImagePolicy))
        g_mozilla_web_set_load_images;
    iface->get_load_images                    = (GWebEngineImagePolicy (*) (GWeb*))
        g_mozilla_web_get_load_images;
    iface->disconnect                         = (void (*) (GWeb*))
        g_mozilla_web_disconnect;
    iface->clear_cache                        = (GWebStatus (*) (GWeb*))
        g_mozilla_web_clear_cache;
    iface->get_plugins_list                   = (GWebPluginList* (*) (GWeb*))
        g_mozilla_web_get_plugins_list;
    iface->get_history_list                   = (GWebHistoryList* (*) (GWeb*))
        g_mozilla_web_get_history_list;
    iface->set_new_window_loads               = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_new_window_loads;
    iface->get_new_window_loads               = (gboolean (*) (GWeb*))
        g_mozilla_web_get_new_window_loads;
    iface->set_cookie_acceptance              = (void (*) (GWeb*, GCookieAcceptance))
        g_mozilla_web_set_cookie_acceptance;
    iface->get_cookie_acceptance              = (GCookieAcceptance (*) (GWeb*))
        g_mozilla_web_get_cookie_acceptance;
    iface->set_popup_acceptance               = (void (*) (GWeb*, GPopupAcceptance))
        g_mozilla_web_set_popup_acceptance;
    iface->get_popup_acceptance               = (GPopupAcceptance (*) (GWeb*))
        g_mozilla_web_get_popup_acceptance;
    iface->suspend                            = (GWebStatus (*) (GWeb*))
        g_mozilla_web_suspend;
    iface->suspended                          = (gboolean (*) (GWeb*))
        g_mozilla_web_suspended;
    iface->resume                             = (void (*) (GWeb*))
        g_mozilla_web_resume;
    iface->set_ua_string                      = (void (*) (GWeb*, gchar*))
        g_mozilla_web_set_ua_string;
    iface->enable_url_loading                 = (void (*) (GWeb*, gboolean))
        g_mozilla_web_enable_url_loading;
    iface->free_memory                        = (void (*) (GWeb*, gboolean))
        g_mozilla_web_free_memory;
    iface->store_transfers                    = (GWebStatus (*) (GWeb*))
        g_mozilla_web_store_transfers;
    iface->restore_transfers                  = (GWebStatus (*) (GWeb*))
        g_mozilla_web_restore_transfers;
    iface->delete_all_cookies                 = (GWebStatus (*) (GWeb*))
        g_mozilla_web_delete_all_cookies;
    iface->confirm_cookie                     = (GWebStatus (*) (GWeb*, gpointer, gboolean, GWebCookieAction))
        g_mozilla_web_confirm_cookie;
    iface->get_global                         = (GObject* (*) (GWeb*))
        g_mozilla_web_get_global;
    iface->get_certificate_enum_message       = (GWebCertificateMessage (*) (GWeb*, GObject*))
        g_mozilla_web_get_certificate_enum_message;
    iface->perform_crtificate_context_action  = (GWebCertificateReaction (*) (GWeb*,
                                                                              GObject*,
                                                                              GWebCertificateAction,
                                                                              GObject*))
        g_mozilla_web_perform_crtificate_context_action;
    iface->get_certificate_context_iter       = (GWebStatus (*) (GWeb*, GObject*, GObject**))
        g_mozilla_web_get_certificate_context_iter;
    iface->get_certificate_iter               = (GObject* (*) (GWeb*, GObject**))
        g_mozilla_web_get_certificate_iter;
    iface->certificate_free                   = (void (*) (GWeb*, GObject*))
        g_mozilla_web_certificate_free;
    iface->certificate_iter_free              = (void (*) (GWeb*, GObject*))
        g_mozilla_web_certificate_iter_free;
    iface->notify_plugins                     = (void (*) (GWeb*, gint))
        g_mozilla_web_notify_plugins;
    iface->username_matchinfo_get_matches     = (gint (*) (GWeb*, GObject*))
        g_mozilla_web_username_matchinfo_get_matches;
    iface->username_matchinfo_get_username    = (GWebStatus (*) (GWeb*, GObject*, gint, gchar**))
        g_mozilla_web_username_matchinfo_get_username;
    iface->username_matchinfo_delete          = (void (*) (GWeb*, GObject*, gint))
        g_mozilla_web_username_matchinfo_delete;
    iface->store_personal_data 		      = (GWebStatus (*) (GWeb*, const gchar*, const gchar*, const gchar*))
        g_mozilla_web_store_personal_data;
    iface->delete_personal_data 	      = (GWebStatus (*) (GWeb*, const gchar*, const gchar*))
        g_mozilla_web_delete_personal_data;
    iface->get_password_manager 	      =	(GWebPasswordManager* (*) (GWeb*, const gchar*, gint))
        g_mozilla_web_get_password_manager;
    iface->clear_passwords                    = (GWebStatus (*) (GWeb*))
        g_mozilla_web_clear_passwords;
    iface->clear_authenticated_sessions       = (GWebStatus (*) (GWeb*))
        g_mozilla_web_clear_authenticated_sessions;
    iface->clear_all_history                  = (GWebStatus (*) (GWeb*))
        g_mozilla_web_clear_all_history;
    iface->clear_personal_data                = (void (*) (GWeb*))
        g_mozilla_web_clear_personal_data;
    iface->set_remember_password              = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_remember_password;
    iface->get_remember_password	      = (gboolean (*) (GWeb*))
        g_mozilla_web_get_remember_password;
    iface->set_remember_password_for_forms    = (GWebStatus (*) (GWeb*, gboolean))
        g_mozilla_web_set_remember_password_for_forms;
    iface->get_remember_password_for_forms    = (gboolean (*) (GWeb*))
        g_mozilla_web_get_remember_password_for_forms;
    iface->get_plugin_enabled                = (gboolean (*) (GWeb*, gchar*))
        g_mozilla_web_get_plugin_enabled;
    iface->set_plugin_enabled                = (GWebStatus (*) (GWeb*, gchar*, gboolean))
        g_mozilla_web_set_plugin_enabled;
    iface->set_oom_cb                        = (GWebStatus (*) (GWeb*, GFunc))
        g_mozilla_web_set_oom_cb;
    iface->set_plugin_ignore_filepath        = (GWebStatus (*) (GWeb*))
        g_mozilla_web_set_plugin_ignore_filepath;
    iface->set_user_agent_id                  = (GWebStatus (*) (GWeb*, gint))
        g_mozilla_web_set_user_agent_id;
    iface->get_user_agent_id                  = (gint (*) (GWeb*))
        g_mozilla_web_get_user_agent_id;
    iface->notify_service                     = (void (*) (GWeb*, const gchar *, gpointer *, const gchar *, gchar *))
        g_mozilla_web_notify_service;
    iface->create_web_buses                   = (void (*) (GWeb*))
        g_mozilla_web_create_web_buses;
}


static void
g_mozilla_web_instance_init (GTypeInstance *instance, gpointer g_class)
{
    TRACE_LOG();

    // Check Instance
    g_return_if_fail(instance);

    GMozillaWeb* self = G_MOZILLA_WEB(instance);

    self->global = NULL;
    self->disabled_pllist = NULL;
    self->mCertContexts = NULL;
    self->default_cursor = gdk_cursor_new (GDK_LEFT_PTR);
    self->download_listener = NULL;
    self->no_xembed = FALSE;
    self->web_is_active = TRUE;
    self->shutdown_xpcom = TRUE;
#ifdef MOZEAL_LOGGING
    if (g_strrstr(g_getenv("MOZEAL_LOG"), "trace") || g_str_has_prefix(g_getenv("MOZEAL_LOG"), "all")) {
        bmozeal_trace = TRUE;
    }
#endif

    // GWebBus initialization
    self->web_bus = NULL;

    // Init Password Manager
    self->password_manager = NULL;
}

gboolean
g_mozilla_web_check_bus(GMozillaWeb* self)
{
  g_return_val_if_fail(self, FALSE);
  if (!self->web_bus)
    return FALSE;

  g_return_val_if_fail(self->ui_bus_name, FALSE);
  return TRUE;
}

void
g_mozilla_web_destroy_web_bus(GMozillaWeb* self)
{
    EAL_IF_GFREE_FUNC(self->web_bus, g_object_unref);
}

void
g_mozilla_web_create_web_buses(GMozillaWeb* self)
{
    g_return_if_fail(self);
    if (self->web_bus)
        g_mozilla_web_destroy_web_bus(self);

    const char *socket_name = g_getenv(G_WEB_BUS_EAL_NAME);
    self->web_bus = g_web_bus_new(socket_name ? socket_name : "eal");

    const char* ui_bus_name = g_getenv(G_WEB_BUS_UI_NAME);
    if (ui_bus_name) {
        if (self->ui_bus_name)
            g_free(self->ui_bus_name);
        self->ui_bus_name = g_strdup(ui_bus_name);
    }
    else
      self->ui_bus_name = g_strdup("ui");

    g_web_bus_register_callback(self->web_bus, UPDATE_AREA_MSG,
        (GWebBusCallback) web_bus_message_handler, self);
    g_web_bus_register_callback(self->web_bus, VISIBLE_AREA_MSG,
        (GWebBusCallback) web_bus_message_handler, self);
    g_web_bus_register_callback(self->web_bus, SEND_EVENT_MSG,
        (GWebBusCallback) web_bus_message_handler, self);
    g_web_bus_register_callback(self->web_bus, SEND_KEY_EVENT_MSG,
        (GWebBusCallback) web_bus_message_handler, self);
    g_web_bus_register_callback(self->web_bus, ELEMENT_INFO_MSG,
        (GWebBusCallback) web_bus_message_handler, self);
    g_web_bus_register_callback(self->web_bus, PING_MSG,
        (GWebBusCallback) web_bus_pingpong_handler, self);

}


GType
g_mozilla_web_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
        {
            sizeof (GMozillaWebClass),
            NULL,   /* base_init */
            NULL,   /* base_finalize */
            (GClassInitFunc) g_mozilla_web_class_init, /* class_init */
            NULL,   /* class_finalize */
            NULL,   /* class_data */
            sizeof (GMozillaWeb),
            0,      /* n_preallocs */
            g_mozilla_web_instance_init    /* instance_init */
        };
        static const GInterfaceInfo iface_info =
        {
            (GInterfaceInitFunc) g_mozilla_web_interface_init,
            NULL,   /* interface_finalize */
            NULL    /* interface_data */
        };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       G_MOZILLA_WEB_NAME,
                                       &info, (GTypeFlags) 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB,
                                     &iface_info);
    }
    return type;
}

static void
g_mozilla_web_class_init (GMozillaWebClass * klass)
{
    TRACE_LOG ();
    GObjectClass *gobject_class;
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    gobject_class = (GObjectClass *) klass;
    object_class = (GtkObjectClass *) klass;
    widget_class = (GtkWidgetClass *) klass;

    g_mozilla_web_signals[CERTIFICATE_DIALOG] =
        g_signal_new(G_WEBWIDGET_SIGNAL_CERTIFICATE_DIALOG,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, certificate_dialog),
                     NULL, NULL,
                     g_mozeal_marshal_BOOL__POINTER,
                     G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);

    g_mozilla_web_signals[CERTIFICATE_PASSWORD_DIALOG] =
        g_signal_new(G_WEBWIDGET_SIGNAL_CERTIFICATE_PASSWORD_DIALOG,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, certificate_password_dialog),
                     NULL, NULL,
                     g_mozeal_marshal_VOID__STRING_STRING_POINTER,
                     G_TYPE_NONE, 3,
                     G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE,
                     G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE,
                     G_TYPE_POINTER);

    g_mozilla_web_signals[CERTIFICATE_DETAILS] =
        g_signal_new(G_WEBWIDGET_SIGNAL_CERTIFICATE_DETAILS,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, certificate_details),
                     NULL, NULL,
                     g_cclosure_marshal_VOID__STRING,
                     G_TYPE_NONE, 1, G_TYPE_STRING);

    g_mozilla_web_signals[HISTORY_ADDED] =
        g_signal_new(G_WEBWIDGET_SIGNAL_GLOBAL_HISTORY_ITEM_ADDED,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, history_added),
                     NULL, NULL,
                     gtk_marshal_VOID__STRING,
                     G_TYPE_NONE, 1,
                     G_TYPE_STRING | G_SIGNAL_TYPE_STATIC_SCOPE);

    g_mozilla_web_signals[MODAL_DIALOG] =
        g_signal_new(G_WEBWIDGET_SIGNAL_MODAL_DIALOG,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, modal_dialog),
                     NULL, NULL,
                     g_mozeal_marshal_INT__STRING_STRING_INT_INT_INT_INT,
                     G_TYPE_INT,
                     6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT,
                     G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);

    g_mozilla_web_signals[CONNECTIVITY_STATUS] =
        g_signal_new(G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, connectivity_status),
                     NULL, NULL,
                     g_cclosure_marshal_VOID__INT,
                     G_TYPE_NONE, 1, G_TYPE_INT);

    g_mozilla_web_signals[ON_SERVICE_NOTIFY] =
        g_signal_new(G_WEBWIDGET_SIGNAL_ON_SERVICE_NOTIFY,
                     G_OBJECT_CLASS_TYPE (gobject_class),
                     G_SIGNAL_RUN_LAST,
                     G_STRUCT_OFFSET(GMozillaWebClass, on_service_notify),
                     NULL, NULL,
                     g_mozeal_marshal_INT__POINTER_STRING_STRING,
                     G_TYPE_INT, 3, G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_STRING);
}

void
g_mozilla_web_ask_cookie (GMozillaEngine *engine,
                          gboolean *remember,
                          gint *action,
                          gint expire, 
                          const gchar * url,
                          const gchar * name, const gchar * value,
                          const gchar * domain, const gchar * path)
{

    TRACE_LOG();
    g_return_if_fail (engine);

    GWebCookieContext * ctx = (GWebCookieContext*) g_new0(GWebCookieContext*, 1);
    ctx->remember_decision = remember?*remember:FALSE;
    ctx->accept = action?*action:0;

    g_signal_emit_by_name(G_OBJECT(engine), G_WEBWIDGET_SIGNAL_ASK_COOKIE, ctx, 1, url, name, value, domain, path, FALSE, expire);

    if (action)
        *action = !ctx->accept;
    if (remember)
        *remember = ctx->remember_decision;

    g_free(ctx);
}

gboolean
g_mozilla_web_prompt (GMozillaEngine *engine, const gchar * dialog_title,
           const gchar * dialog_msg, gchar ** default_msg,
           const gchar * check_msg, gboolean * check_value)
{

    TRACE_LOG ();
    g_return_val_if_fail (engine, FALSE);

    gchar *response = NULL;
    g_signal_emit_by_name (G_OBJECT (engine), G_WEBWIDGET_SIGNAL_PROMPT,
                           dialog_msg, *default_msg, &response, NULL);

    if (*default_msg && response)
    {
        EAL_GFREE (*default_msg);
        *default_msg = g_strdup_printf ("%s", response);
        return TRUE;
    }
    return FALSE;
}

gint
g_mozilla_web_confirm_ex (GMozillaEngine *engine,
               const char *title, const char *text, guint bt_flags,
               const char *button1, const char *button2, const char *button3,
               const char *check_msg, gboolean *check_val)
{
    TRACE_LOG ();
    g_return_val_if_fail (engine, 1);

    gint ret_val;
    gchar *ntext = (gchar *) text;
    gint cur = 0, i = 0;
    gboolean prev_space = FALSE, isspace = FALSE;
    for (i = 0; i != strlen (text); i++)
    {
        isspace = g_ascii_isspace (text[i]);
        if (!isspace || !prev_space)
            ntext[cur++] = text[i];
        prev_space = isspace;
    }
    ntext[cur] = 0;

    g_signal_emit_by_name (G_OBJECT (engine), G_WEBWIDGET_SIGNAL_CONFIRM_EX,
                           title, ntext, bt_flags, button1, button2, button3,
                           check_msg, (gpointer)check_val,
                           &ret_val, NULL);

    return ret_val;
}

gboolean
g_mozilla_web_prompt_auth (GMozillaEngine * engine, GObject * mozembed,
                const gchar * dialog_title, const gchar * dialog_msg,
                gchar ** user_name, gchar ** user_passwd,
                const gchar * check_msg, gboolean * check_value)
{
    TRACE_LOG ();
    g_return_val_if_fail (engine, FALSE);

    GtkMozEmbed* embed = GTK_MOZ_EMBED(mozembed);
    g_return_val_if_fail (embed, FALSE);

    GList *logins = NULL;

    EAL_IF_GFREE (engine->username);
    EAL_IF_GFREE (engine->password);
    engine->accept = FALSE;

    gint logins_count = g_mozilla_cpp_get_logins (dialog_title, &logins);

    GMozillaWeb *web = G_MOZILLA_WEB(engine->global);

    g_return_val_if_fail (web, FALSE);

    web->logins_list = NULL;
    if (logins_count)
        web->logins_list = logins;

    // FIXME: please, that is an workaround for bug 42663, so remove it as soon as it gets solved in a another way.
    gint authentification_type;

    gchar *temp_current_location = gtk_moz_embed_get_location (embed);

    if (engine->last_location
        && !strcmp (engine->last_location, temp_current_location))
        authentification_type = G_WEBENGINE_AUTHENTICATION_WRONG;
    else
        authentification_type = G_WEBENGINE_AUTHENTICATE;

    EAL_IF_GFREE (engine->last_location);
    engine->last_location = temp_current_location;
    temp_current_location = NULL;

    g_signal_emit_by_name (G_OBJECT (engine),
                           G_WEBWIDGET_SIGNAL_AUTHENTIFICATION_REQUIRED,
                           authentification_type, dialog_title, dialog_msg);

    GList *ptr = g_list_first (logins);
    while (ptr)
    {
        GtkMozLogin *login = (GtkMozLogin *) ptr->data;
        if (login)
        {
            EAL_IF_GFREE (login->user);
            EAL_IF_GFREE (login->pass);
            EAL_IF_GFREE (login->host);
        }
        ptr = ptr->next;
    }

    if (engine->accept)
    {
        if (engine->username && *engine->username)
        {
            if (user_name)
            {
                if (*user_name)
                    g_free (*user_name);
                *user_name = g_strdup (engine->username);
                EAL_GFREE (engine->username);
            }
        }
        if (engine->password)
        {
            if (user_passwd)
            {
                if (*user_passwd)
                    g_free (*user_passwd);
                *user_passwd = g_strdup (engine->password);
                EAL_GFREE (engine->password);
            }
        }
        engine->accept = FALSE;
        gboolean rremember;
        if (!g_mozilla_engine_get_pref
            (G_TYPE_BOOLEAN, G_MOZILLA_PREF_REMEMBER_PASSWORDS, &rremember))
            rremember = FALSE;
        g_return_val_if_fail(check_value, FALSE);
        *check_value = rremember;
        return TRUE;
    }
    return FALSE;
}

gboolean
g_mozilla_web_select (GMozillaEngine * engine, const gchar * dialog_title,
           const gchar * dialog_text, gpointer dialog_options,
           gpointer dialog_choice)
{
    TRACE_LOG ();
    g_return_val_if_fail (engine, FALSE);

    gboolean ret_val;
    g_signal_emit_by_name (G_OBJECT (engine), G_WEBWIDGET_SIGNAL_SELECT,
                           dialog_title, dialog_text, dialog_options,
                           dialog_choice, &ret_val, NULL);
    return ret_val;
}

static gint
gtk_moz_common_username_select_login_cb (GtkMozEmbedCommon * common,
                                         GtkMozEmbed * mozembed,
                                         GList * logins,
                                         GMozillaWeb * selfweb)
{
  TRACE_LOG ();
  gint ret_val = -1;
  GMozillaEngine *self = g_object_get_data (G_OBJECT (mozembed), G_MOZILLA_EAL_PARENT);
  g_return_val_if_fail (G_MOZILLA_ENGINE (self), FALSE);
  g_signal_emit_by_name (G_OBJECT (self), G_WEBWIDGET_SIGNAL_SELECT_MATCH, (const gpointer) logins, &ret_val);

  return ret_val;
};

static void
new_window_orphan_cb (GtkMozEmbedSingle * embed, GtkMozEmbed ** retval, guint chromemask, GMozillaWeb *self)
{
  TRACE_LOG();
  GSList *c = NULL;
  GMozillaEngine *s = NULL;
  g_return_if_fail(self);

  for (c = (GSList*) self->window_list; c; c = c->next)
  {
    s = (GMozillaEngine*) c->data;

    if (s)
        break;
  }

  if (s) {
    g_object_set_data(s->global, G_MOZILLA_CHROME_FLAGS, GUINT_TO_POINTER(chromemask));
    g_signal_emit_by_name(G_OBJECT(s), G_WEBWIDGET_SIGNAL_WINDOW_REQUEST, retval, "", chromemask, FALSE);
    g_object_steal_data(s->global, G_MOZILLA_CHROME_FLAGS);
  }
}

static void
g_mozilla_web_global_connect(GMozillaWeb *self)
{
    TRACE_LOG();
    g_mozilla_connectivity_set_global(self);

    g_signal_connect (G_OBJECT (self->global), G_WEBWIDGET_SIGNAL_SELECT_LOGIN,
                      G_CALLBACK (gtk_moz_common_username_select_login_cb), self);
}


static void
g_mozilla_web_set_default_config (GMozillaWeb *self)
{
    TRACE_LOG();
    gchar *pref_string = NULL;
    gint pref_int = 0;
    gboolean pref_bool = TRUE;
    const gboolean kTRUE = TRUE;
    const gboolean kFALSE = FALSE;

#define SET_DEFAULT(prefname, preftype, storage, default) \
  if (!g_mozilla_engine_get_pref(preftype, prefname, (gpointer)&storage)) \
    g_mozilla_engine_set_pref(preftype, prefname, (gpointer)&default);
#define SET_DEFAULT_BOOL(prefname, default) \
  SET_DEFAULT(prefname, G_TYPE_BOOLEAN, pref_bool, default)
#define SET_DEFAULT_STRING(prefname, default) \
  SET_DEFAULT(prefname, G_TYPE_STRING, pref_string, default) \
  EAL_IF_GFREE(pref_string);
#define SET_DEFAULT_INT(prefname, default) \
  pref_int = default; \
  SET_DEFAULT(prefname, G_TYPE_INT, pref_int, pref_int)

#define SET_DEFAULT_WEAK(prefname, preftype, storage, default) \
  if (!g_mozilla_engine_pref_has_user_value(prefname)) \
    g_mozilla_engine_set_pref(preftype, prefname, (gpointer)&default);

#define SET_DEFAULT_BOOL_WEAK(prefname, default) \
  SET_DEFAULT_WEAK(prefname, G_TYPE_BOOLEAN, pref_bool, default)
#define SET_DEFAULT_STRING_WEAK(prefname, default) \
  SET_DEFAULT_WEAK(prefname, G_TYPE_STRING, pref_string, default) \
  EAL_IF_GFREE(pref_string);
#define SET_DEFAULT_INT_WEAK(prefname, default) \
  pref_int = default; \
  SET_DEFAULT_WEAK(prefname, G_TYPE_INT, pref_int, pref_int)

    // for flash, need a better name :)
    // This change was requested by Leonid.
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_PLUGINS_FLASHBACK, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_PLUGINS_ENABLE_SUSPENDING, kFALSE)

    // enabling xul error pages
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_XUL_ERROR_PAGE, kTRUE)

    // no proxy for tbox - this is only for test purposes, and has to be removed when not needed anymore.
    SET_DEFAULT_STRING(G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, G_MOZILLA_LOCAL_HOST_IP)

    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SSR_ENABLED, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SNAV_IGNORE_TEXT_FIELDS, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SNAV_DISABLE_JS, kFALSE)
    SET_DEFAULT_STRING(G_MOZILLA_PREF_SNAV_BLOCK, G_MOZILLA_SNAV_BLOCK_DEFAULTS)

    SET_DEFAULT_INT_WEAK(G_MOZILLA_PREF_ACCESSIBILITY_TABFOCUS, G_MOZILLA_ACCESSIBILITY_TABFOCUS_DEFAULT)
    // activating snav
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SPATIAL_NAVIGATION, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SNAV_ENABLED, kFALSE)
    
    // block panning mode site
    SET_DEFAULT_STRING(G_MOZILLA_PREF_WIDGETUTILS_BLOCK, G_MOZILLA_WIDGETUTILS_BLOCK_DEFAULT)

    // focus border styling
#if 0
    SET_DEFAULT_INT(G_MOZILLA_PREF_FOCUS_RING_WIDTH, 3)
    SET_DEFAULT_BOOL("browser.display.focus_ring_on_anything", kTRUE)
#endif
#if 0
    SET_DEFAULT_STRING(G_MOZILLA_PREF_FOCUS_TEXT_COLOR, "#9d0202")
    SET_DEFAULT_STRING(G_MOZILLA_PREF_FOCUS_BACKGROUND_COLOR, "#f6ff08")
#endif
#if 0
    if (g_mozilla_engine_get_pref(G_TYPE_STRING, G_MOZILLA_PREF_USER_UA_ADD, &pref_string))
    {
        gchar *temp_str = g_strdup_printf ("%s", sval);
        g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_EAL_NAME, temp_str);
        EAL_IF_GFREE(pref_string);
    }
    else {
        g_mozilla_engine_set_pref(G_TYPE_STRING, G_MOZILLA_PREF_EAL_NAME, MOZILLA_DEFAULT_VENDOR_NAME);
    }

    SET_DEFAULT_STRING(G_MOZILLA_PREF_EAL_VER, VERSION)
    SET_DEFAULT_STRING(G_MOZILLA_PREF_EAL_MILESTONE, EAL_MILESTONE)
#endif

#if 0
    if (config) {
        SET_DEFAULT_STRING(G_MOZILLA_PREF_UI_STRING, config->ua_string)
    }
#endif

    SET_DEFAULT_STRING(G_MOZILLA_PREF_HOME_PAGE_LOCATION, BROWSER_DEFAULT_PAGE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_USE_DEFAULT_HOME_PAGE, kTRUE)
    SET_DEFAULT_STRING(G_MOZILLA_PREF_WEB_ADDRESS_SUFFIXES, G_MOZILLA_WEB_ADDRESS_SUFFIXES)
    SET_DEFAULT_STRING(G_MOZILLA_PREF_GNOMEVFS_SUPPORT, G_MOZILLA_GNOMEVFS_SUPPORT)
    SET_DEFAULT_INT(G_MOZILLA_PREF_MEMORY_CACHE_SIZE, G_MEMORY_CACHE_SMALL_SIZE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_ENABLE_JAVA_SCRIPT, kTRUE)

    const char *shm_env = getenv(G_MOZILLA_MOZ_ENABLE_XSHM);
    if (shm_env && g_str_has_prefix(shm_env, "1")) {
      SET_DEFAULT_STRING(G_MOZILLA_PREF_FORCE_WMODE, G_MOZILLA_PREF_WMODE_OPAQUE)
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_ENABLED, kFALSE)
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_BLOCK_ZOOM_ENABLED, kFALSE)

      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_GFX_FORCE_OFFSCR_IMAGE, (gpointer)&kTRUE);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_FORCE_24BPP, (gpointer)&kTRUE);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_DISABLE, (gpointer)&kTRUE);
      const char *no_xembed_env = getenv(G_MOZILLA_MOZ_DISABLE_XEMBED);
      gboolean not_realize = no_xembed_env && g_str_has_prefix(no_xembed_env, "1");
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_NOT_REALIZED, &not_realize);
      self->no_xembed = not_realize;
      
    } else {
      // activating widgetUtils
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_ENABLED, kTRUE)
      // enable mono mode for PE3
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_MONOMODE, kFALSE)
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_MONOKINETIC, kFALSE)
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_ABSORBMOUSEDOWN, kTRUE)
      SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGETUTILS_PLUGINSUPPORT, kTRUE)

      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_GFX_FORCE_OFFSCR_IMAGE, (gpointer)&kFALSE);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_DISABLE, (gpointer)&kFALSE);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_FORCE_24BPP, (gpointer)&kFALSE);
      g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_WIDGET_NOT_REALIZED, (gpointer)&kFALSE);
      self->no_xembed = FALSE;
    }
#if 0
    SET_DEFAULT_BOOL("plugin.expose_full_path", kTRUE)
#endif
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_LAYOUT_FULL_ZOOM_MODE_FAST, kFALSE)
#if 0
    SET_DEFAULT_BOOL("layout.full.true.zoom.mode", kTRUE)
#else
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_LAYOUT_FULL_TRUE_ZOOM_MODE, (gpointer)&kTRUE);
#endif
    //SET_DEFAULT_BOOL("gtkmozembed.nasty_progress", kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_ENABLE_JAVA, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_ENABLE_PLUGINS, kFALSE)
    SET_DEFAULT_INT(G_MOZILLA_PREF_REDIRECTION_LIMIT, G_MOZILLA_PREF_REDIRECTION_DEFAULT_LIMIT)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_ACCEPT_COOKIES, kTRUE)
    SET_DEFAULT_INT(G_MOZILLA_PREF_ENABLE_POPUP_WINDOWS, 1)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_AUTOMATIC_IMAGE_RESIZING, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_OPEN_POPUP_DURING_LOAD, kFALSE)
#if 0
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SECURE_ENTER, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SECURE_LEAVE, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SECURE_SUBMIT, kFALSE)
#else
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SECURE_ENTER, (gpointer)&kFALSE);
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SECURE_LEAVE, (gpointer)&kFALSE);
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SECURE_SUBMIT, (gpointer)&kFALSE);
#endif
    SET_DEFAULT_INT(G_MOZILLA_PREF_SHOW_IMAGES, G_MOZILLA_SHOW_IMAGES_DEFAULT)
    SET_DEFAULT_INT(G_MOZILLA_PREF_PANNING_SENS, G_MOZILLA_PANNING_SENSITIVITY_DEFAULT)

    gchar *lang = getenv(G_MOZILLA_LC_MESSAGES);
    g_mozilla_web_set_language (self, lang ? lang : getenv(G_MOZILLA_LANG));

#if 0
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_FORCE_AUTOCOMPLETION, kTRUE)
#else
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_FORCE_AUTOCOMPLETION, (gpointer)&kTRUE);
#endif
    SET_DEFAULT_STRING(G_MOZILLA_PREF_SIGNON_FILE, G_MOZILLA_SIGNON_FILE_DEFAULT)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_REMEMBER_PASSWORDS, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_REMEMBER_SIGNONS, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_PASSWORDS_SAVING, kTRUE)
    SET_DEFAULT_INT(G_MOZILLA_PREF_BROWSER_HISTORY_EXPIRE_DAYS, G_MOZILLA_BROWSER_HISTORY_EXPIRE_DAYS_DEFAULT)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_PRESERVE_ZOOM_NAV_CHANGE, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_JSFOCUS_FOR_VISIBLE_AREA, kTRUE)

    SET_DEFAULT_BOOL(G_MOZILLA_PREF_WIDGET_ALLOW_NATIVE_THEME, kTRUE)

    SET_DEFAULT_BOOL(G_MOZILLA_PREF_DM_SHOW_ALERTCOMPLETE, kFALSE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_XUL_DOWNLOAD_MGR_SHOW, kFALSE)

    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SUSPEND_IMAGES, kTRUE)
    SET_DEFAULT_INT(G_MOZILLA_PREF_SUSPEND_JAVASCRIPT, G_TIMEOUTS_SUSPEND_TIME_DEFAULT)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_SUSPEND_PLUGINS, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_EXPAND_FRAMES, kTRUE)
    SET_DEFAULT_BOOL(G_MOZILLA_PREF_HIDE_FIXED_FRAMES, kTRUE)
    g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_HIDE_FIXED_FRAMES, &self->mHideFixedFrames);

    // Permanently disable "Get Add-ons" from Options > Components menu
    g_mozilla_engine_set_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_EXTENSIONS_GETADDONS_SHOWPANE, (gpointer)&kFALSE);

    pref_bool = TRUE;
    if (g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, "gtkmozembed.pop_startup_on_last_window", &pref_bool) && pref_bool)
      self->shutdown_xpcom = FALSE;

    g_mozilla_engine_save_prefs();

    self->is_configured = TRUE;
}

GObject*
g_web_new (void)
{
    TRACE_LOG();
    GObject* instance = NULL;
    instance = G_OBJECT(g_object_new(g_mozilla_web_get_type(), NULL));

    G_MOZILLA_WEB(instance)->is_configured = FALSE;

    G_MOZILLA_WEB(instance)->global = G_OBJECT(gtk_moz_embed_common_new());

    g_mozilla_web_global_connect(G_MOZILLA_WEB(instance));

    g_mozilla_engine_observe(G_MOZILLA_CLIENT_OBSERVER_CONTRACTID, instance, G_MOZILLA_WEB_NAME_DEFAULT, NULL);

    G_MOZILLA_WEB(instance)->window_list = NULL;

    G_MOZILLA_WEB(instance)->ui_bus_name = g_strdup(G_MOZILLA_UI);

    // set the chrome type so it's stored in the object
    g_signal_connect(G_OBJECT(gtk_moz_embed_single_get()), G_MOZILLA_NEW_WINDOW_ORPHAN,
                       GTK_SIGNAL_FUNC(new_window_orphan_cb), instance);

    gWebGlobal = instance;
    G_MOZILLA_WEB(instance)->mHideFixedFrames = TRUE;

    return instance;
}

static gboolean
g_mozilla_add_extension_ini(const char *ext_name, const char *target_path)
{
  const char *profilePath = getenv (G_MOZILLA_MICROB_PROFILE_PATH_NAME_DEFAULT);
  char *extensions_ini = g_strdup_printf("%s/%s", profilePath, G_MOZILLA_EXTENSIONS_FILENAME_DEFAULT);
  char *extensions_ini_w = g_strdup_printf("%s/%s", profilePath, G_MOZILLA_EXTENSIONS_FILENAME_NEW_DEFAULT);
  FILE *f = fopen(extensions_ini, "r+");
  FILE *fw = fopen(extensions_ini_w, "w+");
  gboolean exists = FALSE;
  gboolean added = FALSE;
  int ext_num = -1;
  char buf[200];

  // Checking File Write Handle
  if (f) {
    char *s = NULL;
    gboolean ext_section = FALSE;
    while ((s = fgets( (char*) &buf, 199, f))) {
      if (ext_section) {
        if (g_str_has_prefix(s, "Extension")) {
          if (PL_strcasestr(s, ext_name)) {
            exists = TRUE;
            break;
          }
          ext_num++;
        } else {
          ext_num++;
          char * line = g_strdup_printf(G_MOZILLA_EXTENSION_NAME_PARAM, ext_num, target_path);
          // Check File Write Handle
          if (fw)
            fputs(line, fw);

          g_free(line);
          added = TRUE;
        }
      }
      if (!ext_section && g_str_has_prefix(s, "[ExtensionDirs]"))
        ext_section = TRUE;

    // Check File Write Handle
    if (fw)
      fputs(s, fw);
    }
    if (!added) {
      ext_num++;
      char * line = g_strdup_printf(G_MOZILLA_EXTENSION_NAME_PARAM, ext_num, target_path);
      // Check File Write Handle
      if (fw)
        fputs(line, fw);
      g_free(line);
      added = TRUE;
    }
    // Close File
    fclose(f);
    f = NULL;
  } else {
      // Check File WWrite Handle
      if (fw)
        fputs("[ExtensionDirs]\n", fw);
      ext_num++;
      char * line = g_strdup_printf(G_MOZILLA_EXTENSION_NAME_PARAM, ext_num, target_path);
      // Check File Write Handle
      if (fw)
        fputs(line, fw);

      g_free(line);
  }

  // Check File Write Handle
  if (fw) {
    // Closing File Write Handle
    fclose(fw);
    fw = NULL;
    }

  if (!exists) {
    FILE *f = fopen(extensions_ini_w, "r+");
    FILE *fw = fopen(extensions_ini, "w+");
    // Checking File Handles
    if (f && fw) {
      char *s = NULL;
      while ((s = fgets((char*)&buf, 199, f)))
        fputs(s, fw);
    }
    // Check File Handle
    if (f)
      fclose(f);
    // Check File Write Handle
    if (fw)
      fclose(fw);
  }
  unlink(extensions_ini_w);
  g_free(extensions_ini_w);
  g_free(extensions_ini);

  // TODO: This function by definition should return a boolean value. The return value is
  // not checked by caller in any place, so probably this function could be changed to
  // to return a void or return value handling added to the caller functions?
  return added;
}

static gboolean
g_mozilla_create_extensions_folder(void)
{
    const gchar *name = NULL;
    // the extensions path
    gchar *dirExtensionsPath = getenv (G_MOZILLA_INTERNAL_EXTENSIONS_DIR);
    
    GDir *pluginDir;
    g_unlink(dirExtensionsPath);

    if (-1 == g_access(dirExtensionsPath, R_OK))
    {
        g_mkdir_with_parents(dirExtensionsPath, 0755);
    }

    pluginDir = g_dir_open (EXTENSIONS_DIR, 0, NULL);
    if (!pluginDir)
        return FALSE;
    while ((name = g_dir_read_name (pluginDir)))
    {
        gchar *destinyPath = g_strdup_printf("%s/%s", dirExtensionsPath, name);

        if (-1 == g_access(destinyPath, R_OK))
        {
            gchar *path = g_strdup_printf("%s/%s", EXTENSIONS_DIR, name);
            g_mozilla_add_extension_ini(name, destinyPath);
            if (symlink(path, destinyPath) < 0) {
                ULOG_DEBUG_F("cannot make symbolic link %s", destinyPath);
            }
            g_free(path);
        }
        g_free(destinyPath);
    }
    g_dir_close (pluginDir);
    return TRUE;
}

static gboolean
g_mozilla_check_system_extensions(const gchar *full_path, const gchar *profile_path)
{
    struct stat sb, sb2;
    gboolean tt = FALSE;
    if (stat("/var/lock/browser_extensions.stamp", &sb) != -1) {
        gchar * compreg = g_strdup_printf("%s/%s/%s", full_path, profile_path?profile_path:G_MOZILLA_MICROB, "compreg.dat");
        if (compreg && stat(compreg, &sb2) != -1) {
          if (sb.st_mtime > sb2.st_mtime) {
              gchar * xpti = g_strdup_printf("%s/%s/%s", full_path, profile_path?profile_path:G_MOZILLA_MICROB, "xpti.dat");
              unlink(compreg);
              unlink(xpti);
              g_free(xpti);
              tt = TRUE;
          }
        }
        EAL_IF_GFREE(compreg);
    }
    return tt;
}

void
g_web_set_env (void)
{
    TRACE_LOG();
    gtk_moz_embed_set_path(PREFIX"/lib/microb-engine");
    char *home_path;
    char *full_path;
    char *comp_path = NULL;
    char *profile_path = NULL;

    home_path = (char*)getenv(G_MOZILLA_HOME);
    if (!home_path)
    {
        fprintf(stderr, "Failed to get HOME\n");
        exit(1);
    }
    comp_path = (char*)getenv(G_MOZILLA_COMP_PATH_OVVERRIDE);
    profile_path = (char*)getenv(G_MOZILLA_PROFILE_OVERRIDE);

    gchar *plugins_path = g_strdup_printf("%s/%s/%s", getenv (G_MOZILLA_HOME), comp_path?comp_path:".mozilla", "plugins");
    setenv(G_MOZILLA_INTERNAL_PLUGINS_DIR, plugins_path, 0);
    gchar *extensions_path = g_strdup_printf("%s/%s/%s/%s", getenv (G_MOZILLA_HOME), comp_path?comp_path:".mozilla", profile_path?profile_path:G_MOZILLA_MICROB, "extensions");
    setenv(G_MOZILLA_INTERNAL_EXTENSIONS_DIR, extensions_path, 0);
    setenv(G_MOZILLA_PREF_PLUGINS_DIR , plugins_path, 0);
    setenv(G_MOZILLA_DISABLE_SIG_HANDLER, "1", 0);
    full_path = g_strdup_printf("%s/%s", home_path, comp_path?comp_path:".mozilla");
    setenv(G_MOZILLA_MICROB_PROFILE_PATH, full_path, 0);
    gtk_moz_embed_set_profile_path(full_path, profile_path?profile_path:G_MOZILLA_MICROB);
    gtk_moz_embed_set_comp_path(full_path);

    if (symlink(OSSO_PLUGINS, plugins_path) < 0) {
        ULOG_DEBUG_F("cannot make symbolic link %s", plugins_path);
    }

    g_mozilla_create_extensions_folder();
    g_mozilla_check_system_extensions(full_path, profile_path);

    g_free(plugins_path);
    g_free(extensions_path);
    g_free(full_path);

    microb_eal_components_static_init();
    gtk_moz_embed_push_startup();
    microb_eal_components_init();
    g_mozilla_initialize_extensions();
}

/**
 * Checks if the given engine pointer is to an existing engine.
 * @param web    GMozillaWeb instance.
 * @param engine GMozillaEngine pointer to be validated.
 * @returns TRUE if engine pointer is valid.
 */
static gboolean
validate_engine_pointer (GMozillaWeb* web, GMozillaEngine* engine)
{
    g_return_val_if_fail (web, FALSE);

    if (engine == NULL)
        return FALSE;

    return g_list_find (web->window_list, (gconstpointer)engine) != NULL;
}

static void
web_bus_message_handler(guint msg_id, gpointer* msg, guint msg_len, GMozillaWeb* web)
{
    g_return_if_fail(msg && web);

    switch (msg_id) {
        case UPDATE_AREA_MSG:
        {
            g_return_if_fail(msg_len == sizeof(UpdateAreaMessage));
            UpdateAreaMessage* m = (UpdateAreaMessage*) msg;
            GMozillaEngine* engine = (GMozillaEngine*) m->engine_id;
            g_return_if_fail(validate_engine_pointer (web, engine));
            g_return_if_fail(G_IS_MOZILLA_ENGINE(engine));
            g_mozilla_engine_update_area(engine, m->id, m->left, m->top,
                                        m->width, m->height, m->zoom);
            break;
        }

        case VISIBLE_AREA_MSG:
        {
            g_return_if_fail(msg_len == sizeof(VisibleAreaMessage));
            VisibleAreaMessage* m = (VisibleAreaMessage*) msg;
            GMozillaEngine* engine = (GMozillaEngine*) m->engine_id;
            g_return_if_fail(validate_engine_pointer (web, engine));
            g_return_if_fail(G_IS_MOZILLA_ENGINE(engine));
            g_mozilla_engine_visible_area(engine, m->vx, m->vy, m->vw, m->vh,
                                          m->tx, m->ty, m->tw, m->th,
                                          m->wx, m->wy, m->ww, m->wh);
            break;
        }

        case SEND_EVENT_MSG:
        {
            g_return_if_fail(msg_len == sizeof(SendEventMessage));
            SendEventMessage* m = (SendEventMessage*) msg;
            GMozillaEngine* engine = (GMozillaEngine*) m->engine_id;
            g_return_if_fail(validate_engine_pointer (web, engine));
            g_return_if_fail(G_IS_MOZILLA_ENGINE(engine));
            g_mozilla_engine_send_mouse_event(engine, m->type, m->x, m->y, m->mod,
                                              m->button, m->clickcount, m->time);
            break;
        }

        case SEND_KEY_EVENT_MSG:
        {
            g_return_if_fail(msg_len == sizeof(SendKeyEventMessage));
            SendKeyEventMessage* m = (SendKeyEventMessage*) msg;
            GMozillaEngine* engine = (GMozillaEngine*) m->engine_id;
            g_return_if_fail(validate_engine_pointer (web, engine));
            g_return_if_fail(G_IS_MOZILLA_ENGINE(engine));
            g_mozilla_engine_send_key_event(engine, m->type, m->keyval, m->length, m->state,
                                            m->hardware_keycode, m->group, m->is_modifier, m->time);
            break;
        }

        case ELEMENT_INFO_MSG:
        {
            g_return_if_fail (msg_len == sizeof (ElementInfoMessage));
            ElementInfoMessage * m = (ElementInfoMessage *) msg;
            if (m->requestID) {
                GMozillaEngine * engine = (GMozillaEngine *) m->engine_id;
                g_return_if_fail(validate_engine_pointer (web, engine));
                g_return_if_fail (G_IS_MOZILLA_ENGINE (engine));
                send_focused_element_message (engine, NULL);
            }

            break;
        }
    }
}

static void
web_bus_pingpong_handler(guint msg_id, gpointer* msg, guint msg_len, GMozillaWeb* web)
{
    g_return_if_fail(msg && web && web->web_bus && web->ui_bus_name);

    if (msg_id == PING_MSG) 
        g_web_bus_send_message(web->web_bus, web->ui_bus_name, PONG_MSG, "shutup!", 8);
}

