/* ***** BEGIN LICENSE BLOCK *****
 * vim:set ts=2 sw=2 sts=2 tw=80 et cindent:
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard. Portions created by Christopher Blizzard are Copyright (C) Christopher Blizzard.  All Rights Reserved.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *   Ramiro Estrugo <ramiro@eazel.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <stdio.h>
#include "gtkmicrob_context.h"

#include "MicrobEalContextMenuInfo.h"
#include "nsIDOMNode.h"

// class and instance initialization

#define GTK_MICROB_CTX_GET_PRIVATE(object)(G_TYPE_INSTANCE_GET_PRIVATE ((object), GTK_TYPE_MICROB_CONTEXT, GtkMicroBContextPrivate))

struct _GtkMicroBContextPrivate
{
  MicrobEalContextMenuInfo *info;
  GHashTable *props;
};

static void gtk_microb_context_class_init (GtkMicroBContextClass *klass);
static void gtk_microb_context_init (GtkMicroBContext *ctx);
static void gtk_microb_context_finalize (GObject *object);

static GObjectClass *parent_class = NULL;

// GObject + class-related functions

GType
gtk_microb_context_get_type (void)
{
  static GType microb_context_type = 0;
  if (microb_context_type == 0)
  {
    const GTypeInfo our_info =
    {
      sizeof (GtkMicroBContextClass),
      NULL, /* base_init */
      NULL, /* base_finalize */
      (GClassInitFunc) gtk_microb_context_class_init,
      NULL,
      NULL, /* class_data */
      sizeof (GtkMicroBContext),
      0, /* n_preallocs */
      (GInstanceInitFunc) gtk_microb_context_init,
    };

    microb_context_type = g_type_register_static (G_TYPE_OBJECT,
                                                  "GtkMicroBContext",
                                                  &our_info,
                                                  (GTypeFlags)0);
  }

  return microb_context_type;
}

static void
_microb_context_free_g_value (gpointer value)
{
    g_value_unset ((GValue *) value);
    g_free (value);
}

void gtk_microb_context_init (GtkMicroBContext *ctx)
{
  ctx->priv = G_TYPE_INSTANCE_GET_PRIVATE ((ctx), GTK_TYPE_MICROB_CONTEXT, GtkMicroBContextPrivate);

  ctx->priv->info = NULL;
  ctx->priv->props = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, _microb_context_free_g_value);
  ctx->x = ctx->y = ctx->x_root = ctx->y_root = ctx->button = ctx->type = ctx->modifier = 0;
  ctx->docindex = -1;

  return;
}
GtkMicroBContext *
gtk_microb_context_new ()
{
  return GTK_MICROB_CONTEXT(g_object_new(GTK_TYPE_MICROB_CONTEXT, NULL));
}

// GObject methods
void
gtk_microb_context_finalize (GObject *object)
{
  GtkMicroBContext * ctx = GTK_MICROB_CONTEXT(object);

  g_hash_table_destroy (ctx->priv->props);
  ctx->priv->props = NULL;

  if (ctx->priv->info)
    delete (MicrobEalContextMenuInfo*)ctx->priv->info;
  ctx->priv->info = NULL;

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

void
gtk_microb_context_class_init(GtkMicroBContextClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = G_OBJECT_CLASS(g_type_class_peek_parent (klass));
  object_class->finalize = gtk_microb_context_finalize;

  g_type_class_add_private (object_class, sizeof (GtkMicroBContextPrivate));
}

gboolean
gtk_microb_context_update_from_event (GtkMicroBContext *ctx, gpointer event)
{
  gboolean ret = FALSE;
  g_return_val_if_fail(ctx, ret);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), ret);
  g_return_val_if_fail(event, ret);

  if (!ctx->priv->info)
    ctx->priv->info = new MicrobEalContextMenuInfo(ctx, g_object_get_data(G_OBJECT(ctx), "mozembed"));

  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, ret);

  ret = NS_SUCCEEDED(info->UpdateContextData(static_cast<nsIDOMEvent*>(event)));
  return ret;
}

gboolean
gtk_microb_context_update_from_element (GtkMicroBContext *ctx, gpointer aIDOMNSElement)
{
  gboolean ret = FALSE;
  g_return_val_if_fail(ctx, ret);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), ret);
  g_return_val_if_fail(aIDOMNSElement, ret);

  if (!ctx->priv->info)
    ctx->priv->info = new MicrobEalContextMenuInfo(ctx, g_object_get_data(G_OBJECT(ctx), "mozembed"));

  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, ret);

  ret = NS_SUCCEEDED(info->UpdateContextDataFromElement(static_cast<nsIDOMNSElement*>(aIDOMNSElement)));
  return ret;
}

gboolean
gtk_microb_context_set_prop (GtkMicroBContext *ctx, const char *name, GValue *value)
{
  g_return_val_if_fail(ctx, FALSE);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), FALSE);
  g_hash_table_insert (GTK_MICROB_CONTEXT(ctx)->priv->props, g_strdup (name), value);
  return TRUE;
}

const GValue*
gtk_microb_context_get_prop (GtkMicroBContext *ctx, const char *name)
{
  g_return_val_if_fail(ctx, NULL);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), NULL);
  return (const GValue *) g_hash_table_lookup (GTK_MICROB_CONTEXT(ctx)->priv->props, name);
}

gpointer
gtk_microb_context_get_node (GtkMicroBContext *ctx)
{
  g_return_val_if_fail(ctx, NULL);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), NULL);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  if (!info) return NULL;

  return info->mEventNode;
}

gpointer
gtk_microb_context_get_original_node (GtkMicroBContext *ctx)
{
  g_return_val_if_fail(ctx, NULL);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), NULL);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  if (!info) return NULL;

  return info->mOrigNode;
}

gboolean
gtk_microb_context_get_text (GtkMicroBContext *ctx, gchar **text, gint *position)
{
  g_return_val_if_fail(ctx, FALSE);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), FALSE);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, FALSE);

  nsString textStr;
  PRInt32 pos = 0;
  NS_ENSURE_SUCCESS(info->GetContextText(textStr, !(position), &pos), FALSE);
  if (text) *text = ToNewUTF8String(textStr);
  if (position) *position = pos;
  return TRUE;
}

gboolean
gtk_microb_context_insert_text(GtkMicroBContext *ctx, const gchar *string, gpointer node, gint surr1, gint surr2)
{
  g_return_val_if_fail(ctx, FALSE);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), FALSE);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, FALSE);

  if (!string) {
      info->ScrollToSelectedNode((nsISupports*)(node?node:g_object_get_data(G_OBJECT(ctx), "mozembed")));
      return TRUE;
  }
  g_return_val_if_fail(string, FALSE);

  nsString aString;
  aString.Assign(NS_ConvertUTF8toUTF16(string));
  NS_ENSURE_SUCCESS(info->InsertTextToNode((nsIDOMNode*)node, aString, (PRInt32)surr1, (PRInt32)surr2), FALSE);

  return TRUE;
}

int
gtk_microb_context_get_scrollable_type(GtkMicroBContext *ctx)
{
  g_return_val_if_fail(ctx, 0);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), 0);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, 0);

  return info->GetScrollableType();
}

gboolean
gtk_microb_context_scroll_scrollable(GtkMicroBContext *ctx, int adx, int ady)
{
  g_return_val_if_fail(ctx, NULL);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), NULL);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, NULL);

  NS_ENSURE_SUCCESS(info->ScrollScrollableView(adx, ady), FALSE);
  return TRUE;
}

gboolean
gtk_microb_context_update_scrollable(GtkMicroBContext *ctx)
{
  g_return_val_if_fail(ctx, TRUE);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), TRUE);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, TRUE);

  NS_ENSURE_SUCCESS(info->UpdateScrollableView(), FALSE);
  return TRUE;
}

gboolean
gtk_microb_context_do_handle_event_with_target(GtkMicroBContext *ctx, gpointer node)
{
  g_return_val_if_fail(ctx, FALSE);
  g_return_val_if_fail(GTK_IS_MICROB_CONTEXT(ctx), FALSE);
  MicrobEalContextMenuInfo* info = static_cast<MicrobEalContextMenuInfo*>(ctx->priv->info);
  g_return_val_if_fail(info, FALSE);

  NS_ENSURE_SUCCESS(info->DoHandleEventWithTarget((nsIDOMNode *)node), FALSE);
  return TRUE;
}
