/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "common.h"
#include "gmozillatransferitem.h"
#include "gmozillamarshal.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"
#include "gmozillacppwrapper.h"
#include "gmozillaconnectivity.h"

// class and instance initialization
enum
{
    TRANSFER_ITEM_STARTED_SIGNAL,
    TRANSFER_ITEM_COMPLETED_SIGNAL,
    TRANSFER_ITEM_ABORTED_SIGNAL,
    TRANSFER_ITEM_ERROR_SIGNAL,
    TRANSFER_ITEM_PROGRESS_SIGNAL,
    TRANSFER_ITEM_LAST_SIGNAL
};

static guint transfer_item_signals [TRANSFER_ITEM_LAST_SIGNAL] = { 0 };
static gboolean is_transfer_item_restart = FALSE;

static gpointer last_download_object = NULL;

// progress timeout stuff - network down errors.
#define PROGRESS_TOLERANCY_INTERVAL 60000
static gboolean progress_timeout (GMozillaTransferItem *);
void reset_progress_timeout (GMozillaTransferItem *);

// connecting signals from engine.
static gboolean g_mozilla_transfer_item_connect_signals (GMozillaTransferItem *self);

static void
g_mozilla_transfer_item_destroy (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    GMozillaWeb *global = g_object_get_data(G_OBJECT(self), "global");
    if (global)
        global->ongoing_download_list = g_list_remove(global->ongoing_download_list, self);
    if (self->item)
    {
        EAL_IF_GFREE(self->file_name);
        gtk_object_destroy(GTK_OBJECT(self->item));
        self->item = NULL;
    }
}

static void
g_mozilla_transfer_item_start (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    g_mozilla_engine_download_start(self);
}

static void
g_mozilla_transfer_item_stop (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();

    // reset progress timeout.
    g_mozilla_engine_download_pause(self);
    reset_progress_timeout (self);
}

static GWebStatus
g_mozilla_transfer_item_continue (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();

    // fix for bug 55796
#ifdef USE_CONIC
    if (!g_mozilla_connectivity_is_connected())
        if (g_mozilla_connectivity_connect(self, FALSE))
            return G_WEB_ERR;
#endif

    g_mozilla_engine_download_start(self);

    // init progress timeout.
    init_progress_timeout (self);

    return G_WEB_ERR;
}

static void
g_mozilla_transfer_item_cancel (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();

    if (self) {
        g_mozilla_engine_download_cancel(self);
    }
}

static void
g_mozilla_transfer_item_reload (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    //g_mozilla_engine_download_start(self);
}

static GObject*
g_mozilla_transfer_item_get_web_item (GMozillaTransferItem *self)
{
    return (GObject *) self;
}

static const gchar*
g_mozilla_transfer_item_get_url (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    return self->server;
}

static const gchar*
g_mozilla_transfer_item_get_filename (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    return self->file_name;
}

static glong
g_mozilla_transfer_item_get_total_size (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    return self->file_size;
}

static glong
g_mozilla_transfer_item_get_progress (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    return self->downloaded_size;
}

static void
g_mozilla_transfer_item_interface_init (GWebTransferItemIface *iface)
{
    ULOG_DEBUG_F();
    iface->destroy        = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_destroy;
    iface->start          = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_start;
    iface->stop           = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_stop;
    iface->do_continue    = (GWebStatus (*) (GWebTransferItem*))
        g_mozilla_transfer_item_continue;
    iface->cancel         = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_cancel;
    iface->reload         = (void (*) (GWebTransferItem*))
        g_mozilla_transfer_item_reload;
    iface->get_web_item   = (GObject* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_web_item;
    iface->get_url        = (gchar* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_url;
    iface->get_filename   = (gchar* (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_filename;
    iface->get_total_size = (glong (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_total_size;
    iface->get_progress   = (glong (*) (GWebTransferItem*))
        g_mozilla_transfer_item_get_progress;
}

static void
g_mozilla_transfer_item_instance_init (GTypeInstance *instance, gpointer  g_class)
{
    ULOG_DEBUG_F();
    G_MOZILLA_TRANSFER_ITEM(instance)->item = NULL;
    G_MOZILLA_TRANSFER_ITEM(instance)->data = nsnull;
    G_MOZILLA_TRANSFER_ITEM(instance)->download_id = -1;
}

static void
g_mozilla_transfer_item_class_init (GMozillaTransferItemClass *klass)
{
    ULOG_DEBUG_F();

    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    // set up maemo-eal/ui specific signals
    transfer_item_signals[TRANSFER_ITEM_STARTED_SIGNAL] =
        g_signal_new(G_WEB_TRANSFER_ITEM_SIGNAL_STARTED,
                     G_TYPE_FROM_CLASS(klass),
                     G_SIGNAL_RUN_FIRST,
                     G_STRUCT_OFFSET(GMozillaTransferItemClass, started),
                     NULL,
                     NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0);
    transfer_item_signals[TRANSFER_ITEM_COMPLETED_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_COMPLETED,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, completed),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_ERROR_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_ERROR,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, error),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_ABORTED_SIGNAL] =
        g_signal_new ("aborted",
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, aborted),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);
    transfer_item_signals[TRANSFER_ITEM_PROGRESS_SIGNAL] =
        g_signal_new (G_WEB_TRANSFER_ITEM_SIGNAL_PROGRESS,
                      G_OBJECT_CLASS_TYPE (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GMozillaTransferItemClass, progress),
                      NULL, NULL,
                      g_mozeal_marshal_VOID__ULONG_ULONG_ULONG,
                      G_TYPE_NONE,
                      3,
                      G_TYPE_ULONG, G_TYPE_ULONG, G_TYPE_ULONG);

}

GType
g_mozilla_transfer_item_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
        {
            sizeof (GMozillaTransferItemClass),
            NULL,   /* base_init */
            NULL,   /* base_finalize */
            (GClassInitFunc) g_mozilla_transfer_item_class_init,   /* class_init */
            NULL,   /* class_finalize */
            NULL,   /* class_data */
            sizeof (GMozillaTransferItem),
            0,      /* n_preallocs */
            g_mozilla_transfer_item_instance_init    /* instance_init */
        };
        static const GInterfaceInfo iface_info =
        {
            (GInterfaceInitFunc) g_mozilla_transfer_item_interface_init,
            NULL,   /* interface_finalize */
            NULL    /* interface_data */
        };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       "GMozillaTransferItem",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB_TRANSFER_ITEM,
                                     &iface_info);
    }
    return type;
}

/* a macro for creating a new object of our type */
#define GET_NEW (g_object_new(g_mozilla_transfer_item_get_type(), NULL))

GObject*
g_mozilla_transfer_item_new (void)
{
    ULOG_DEBUG_F();
    GObject* instance = G_OBJECT(GET_NEW);
    g_return_val_if_fail (instance, NULL);
    g_mozilla_transfer_item_set_latest_object (instance);
    g_mozilla_transfer_item_set_restart_flag (FALSE);
    return instance;
}

GObject*
g_mozilla_transfer_item_new_with_item (GObject *item)
{
    ULOG_DEBUG_F();

    GObject* instance = g_mozilla_transfer_item_new ();
    if (!instance) return NULL;

    // signal stuff
    g_mozilla_transfer_item_connect_signals (G_MOZILLA_TRANSFER_ITEM(instance));

    G_MOZILLA_TRANSFER_ITEM(instance)->item = item;
    return instance;
}

GObject*
g_mozilla_transfer_item_new_with_url (const gchar *url,
                                      const gchar *filename)
{
    ULOG_DEBUG_F();

    GObject* instance = g_mozilla_transfer_item_new ();
    if (!instance) return NULL;

    G_MOZILLA_TRANSFER_ITEM (instance)->file_name = filename?g_strdup(filename):NULL;
    G_MOZILLA_TRANSFER_ITEM (instance)->server = url?g_strdup(url):NULL;

    // signal stuff
    g_mozilla_transfer_item_connect_signals (G_MOZILLA_TRANSFER_ITEM(instance));

    return instance;
}

static gboolean
g_mozilla_transfer_item_connect_signals (GMozillaTransferItem *self)
{
    ULOG_DEBUG_F();
    return TRUE;
}

void 
g_mozilla_transfer_item_start_download (gchar **file_name_with_path)
{
    ULOG_DEBUG_F();
    
    GMozillaTransferItem* self = G_MOZILLA_TRANSFER_ITEM(g_mozilla_transfer_item_get_latest_object ());
    g_return_if_fail(self);

    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_STARTED);
    if (self->file_name)
        *file_name_with_path = g_strdup (self->file_name);
    else *file_name_with_path = NULL;

    return ;
}

void 
g_mozilla_transfer_item_error_download (void)
{
    ULOG_DEBUG_F();

    GMozillaTransferItem* self = G_MOZILLA_TRANSFER_ITEM(g_mozilla_transfer_item_get_latest_object ());
    g_return_if_fail(self);

    g_signal_emit_by_name(G_OBJECT (self), "error");

    return ;
}

void
g_mozilla_transfer_item_aborted_download (void)
{
    ULOG_DEBUG_F();

    GObject* self = NULL;
    g_return_if_fail(self);

    g_signal_emit_by_name(self, "aborted");

    return ;
}

void 
g_mozilla_transfer_item_completed_download (void)
{
    ULOG_DEBUG_F();

    GMozillaTransferItem* self = NULL;
    g_return_if_fail(self);

    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_COMPLETED);

    // reset progress timeout.
    reset_progress_timeout (self);

}

void 
g_mozilla_transfer_item_progress_download (glong current, glong total, glong mode)
{
//    ULOG_DEBUG_F();

    GMozillaTransferItem* self = NULL;
    g_return_if_fail(self);

    init_progress_timeout (self);

    // emit signal to get download progress and displays on download list dialog
    g_signal_emit_by_name(G_OBJECT (self), G_WEB_TRANSFER_ITEM_SIGNAL_PROGRESS, current, total, 1);
    return ;
}

gboolean progress_timeout (GMozillaTransferItem *self)
{
//    ULOG_DEBUG_F();

    // reset progress timeout.
    reset_progress_timeout (self);

    // forcing download to get paused
    g_signal_emit_by_name (G_OBJECT (self), "aborted");
    return FALSE;
}

void init_progress_timeout (GMozillaTransferItem *self)
{
//    ULOG_DEBUG_F();

    // reset progress timeout.
    reset_progress_timeout (self);

    if (self)
        self->progress_timer_id = g_timeout_add (PROGRESS_TOLERANCY_INTERVAL, (GSourceFunc)
                                             progress_timeout, (gpointer) self);
}

void reset_progress_timeout (GMozillaTransferItem *self)
{
//    ULOG_DEBUG_F();

    if (self && self->progress_timer_id) 
    {
        g_source_remove(self->progress_timer_id);
        self->progress_timer_id = 0;
    }
}

gboolean g_mozilla_transfer_item_get_restart_flag (void)
{
    ULOG_DEBUG_F();
    return is_transfer_item_restart;
}

void g_mozilla_transfer_item_set_restart_flag (gboolean restart_flag)
{
    ULOG_DEBUG_F();
    is_transfer_item_restart = restart_flag;
}

GObject* g_mozilla_transfer_item_get_latest_object (void)
{
    ULOG_DEBUG_F();
    return last_download_object;
}

void g_mozilla_transfer_item_set_latest_object (GObject* latest_object)
{
    ULOG_DEBUG_F();
    last_download_object = latest_object;
}
