/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Nokia code.
 *
 * Contributor(s):
 *   Brian McGillion <ext-brian.mcgillion@nokia.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#ifndef CPPUNITSIGNALEXCEPTION_H
#define CPPUNITSIGNALEXCEPTION_H
#include <cppunit/extensions/HelperMacros.h>
#include "signalcatcher.h"


namespace CppUnit
{
    /*!
      * \brief  An extension MACRO for the CppUnit framework
      * \param message An Additional user defined message
      * \param expression The line of code that is to be run
      *
      * \details This macro takes a line of code that is expected to throw an exception
      * or emmit a fatal signal. This macro is a wrapper around the SignalCatcher class
      * and its associated TRY macro.  When a fatal signal or unhandled exception are found
      * a stack trace is taken and an warning message is output in the CppUnit log, or xml file
      * if xml hooking is used.
      *
      * Example usage:
      * \code
      * #include CppUnitSignalException.h
      *
      * void bvRegion_test::clipCairoTest()
      * {
      *     bvRect area1(2, 3, 50, 50);
      *     bvRegion region1(area1);
      *     //The following line is expected to throw a signal 11 (SIGSEGV) if NULL is unhandled
      *     CPPUNIT_ASSERT_NO_SIGNAL_OR_EXCEPTION_MESSAGE("No NULL checking performed", region1.ClipCairo(NULL));
      * }
      * \endcode
      */
# define CPPUNIT_ASSERT_NO_SIGNAL_OR_EXCEPTION_MESSAGE( message, expression )                \
    do{                                                                     \
        CPPUNIT_NS::Message cpputMsg_( "Fatal signal/exception emitted and caught" ); \
        cpputMsg_.addDetail( message );                                     \
        TRY                                                                 \
        {                                                                   \
            expression;                                                     \
        }                                                                   \
        CATCH                                                               \
        {                                                                   \
            cpputMsg_.addDetail( "Caught: " + CPPUNIT_EXTRACT_EXCEPTION_TYPE_( e, "std::exception or derived" ) ); \
            cpputMsg_.addDetail( e.getStackTrace() );                      \
            CPPUNIT_NS::Asserter::fail( cpputMsg_, CPPUNIT_SOURCELINE() );  \
        }                                                                   \
        catch ( const std::exception &e )                                   \
        {                                                                   \
            cpputMsg_.addDetail( "Caught: " +  CPPUNIT_EXTRACT_EXCEPTION_TYPE_( e, "std::exception or derived" ) );    \
            cpputMsg_.addDetail( std::string("What(): ") + e.what() );      \
            CPPUNIT_NS::Asserter::fail( cpputMsg_, CPPUNIT_SOURCELINE() );  \
        }                                                                   \
        catch(...)                                                          \
        {                                                                   \
            cpputMsg_.addDetail( "Caught: unknown. exception" );            \
            CPPUNIT_NS::Asserter::fail( cpputMsg_, CPPUNIT_SOURCELINE() );  \
        }                                                                   \
    }while(false);

    /*!
      * \brief  An extension MACRO for the CppUnit framework
      * \param expression The line of code that is to be run
      *
      * \details This macro takes a line of code that is expected to throw an exception
      * or emmit a fatal signal. This macro is a wrapper around the SignalCatcher class
      * and its associated TRY macro.  When a fatal signal or unhandled exception are found
      * a stack trace is taken and an warning message is output in the CppUnit log, or xml file
      * if xml hooking is used.
      *
      * Example usage:
      * \code
      * #include CppUnitSignalException.h
      *
      * void bvRegion_test::clipCairoTest()
      * {
      *     bvRect area1(2, 3, 50, 50);
      *     bvRegion region1(area1);
      *     //The following line is expected to throw a signal 11 (SIGSEGV) if NULL is unhandled
      *     CPPUNIT_ASSERT_NO_SIGNAL_OR_EXCEPTION(region1.ClipCairo(NULL));
      * }
      * \endcode
      */
# define CPPUNIT_ASSERT_NO_SIGNAL_OR_EXCEPTION( expression )                             \
   CPPUNIT_ASSERT_NO_SIGNAL_OR_EXCEPTION_MESSAGE( CPPUNIT_NS::AdditionalMessage(), expression )
}
#endif // CPPUNITSIGNALEXCEPTION_H
