/*
 * This file is part of maemopad
 *
 * Copyright (C) 2005-2009 Nokia Corporation. All rights reserved.
 *
 * This maemo code example is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the root
 * directory.
 *
 */
 
#include "maemopad-window.h"
#include <hildon/hildon-file-chooser-dialog.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

/* send via email: */
#include <libmodest-dbus-client/libmodest-dbus-client.h>

/* send via bt: */
#include <conbtdialogs-dbus.h>
#include <dbus/dbus-glib.h>

/* Gnome VFS for file i/o: */
#include <libgnomevfs/gnome-vfs.h>

#include <glib/gi18n.h>

#include <fullscreenmanager.h>

static void create_textarea (MaemopadWindow *self);
static void create_menu (MaemopadWindow *self);

static void maemopad_window_on_menu_file_new (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_file_open (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_file_save (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_file_saveas (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_close(GtkButton *button, gpointer data);
static void maemopad_window_on_menu_edit_cut (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_edit_copy (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_edit_paste (GtkButton *button, gpointer data);
static void maemopad_window_on_menu_sendvia_email (GtkButton *button, gpointer data);
static gboolean maemopad_window_rpc_sendvia_bluetooth (const gchar *path);
static void maemopad_window_on_menu_sendvia_bt (GtkButton *button, gpointer data);
static void maemopad_window_on_font_set (GtkFontButton *button, gpointer data);
static void maemopad_window_on_menu_fullscreen (GtkButton *button, gpointer data);
static void maemopad_window_on_buffer_modified (GtkButton *button, gpointer data);
static gboolean maemopad_window_on_key_press (GtkWidget *widget, GdkEventKey *event,
                     gpointer data);
static gboolean maemopad_window_on_key_release (GtkWidget *widget, GdkEventKey *event,
                     gpointer data);
                     
static void maemopad_window_read_file_to_buffer (MaemopadWindow* self);
static void maemopad_window_write_buffer_to_file (MaemopadWindow* self);


G_DEFINE_TYPE (MaemopadWindow, maemopad_window, HILDON_TYPE_WINDOW)

static void
maemopad_window_dispose (GObject *object)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (object);
  
  /* Free member data.
   * Note that the child widgets are destroyed automatically.
   */
  g_free (self->file_name);
  
  /* TODO: This crashes: */
  /*
  if (self->font_desc)
    pango_font_description_free (self->font_desc);
  */
  
  G_OBJECT_CLASS (maemopad_window_parent_class)->dispose (object);
}

static void
maemopad_window_finalize (GObject *object)
{
  G_OBJECT_CLASS (maemopad_window_parent_class)->finalize (object);
}

static void
maemopad_window_class_init (MaemopadWindowClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);


  object_class->dispose = maemopad_window_dispose;
  object_class->finalize = maemopad_window_finalize;
}

static void
maemopad_window_init (MaemopadWindow *self)
{
  GtkClipboard *clipboard = NULL;
  
  /* Create a vbox which will contain our text view: */
  GtkWidget *main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (self), main_vbox);
    
  /* Create all necessary parts: */
  create_textarea (self);
  create_menu (self);
    
  /* Put the scrolledwindow in the vbox and show it: */
  gtk_box_pack_start (GTK_BOX (main_vbox), self->scrolledwindow, TRUE, TRUE, 0);
  gtk_widget_show (main_vbox);
    
  /* Set the intial focus on the text view: */
  gtk_widget_grab_focus (GTK_WIDGET(self->textview));
  
  /* Connect key-press signals: */
  g_signal_connect (self, "key_press_event",
    G_CALLBACK (maemopad_window_on_key_press), self);
  g_signal_connect (self, "key_release_event",
    G_CALLBACK (maemopad_window_on_key_release), self);
    
  /* Allow clipboard data to be stored even when the application is closed: */
  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_can_store (clipboard, NULL, 0);

  self->fullscreen_manager = fullscreen_create_manager(self);
}

MaemopadWindow*
maemopad_window_new (osso_context_t *osso)
{
  MaemopadWindow *self = 
    MAEMOPAD_WINDOW (g_object_new (MAEMOPAD_TYPE_WINDOW, NULL));
  
  /* Avoid adding extra code such as this to a _new() function when writing 
   * widgets that should be reusable. This should really be a GObject property.
   */
  self->osso = osso; 
  
  return self;
}


/* Create the text area */
static void create_textarea (MaemopadWindow *self)
{
  /* Scrolled window */
  self->scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(self->scrolledwindow);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(self->scrolledwindow),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
  /* Text view */
  self->textview = hildon_text_view_new ();

  /* Some text view settings */
  gtk_text_view_set_editable (GTK_TEXT_VIEW (self->textview), TRUE);
  gtk_text_view_set_left_margin (GTK_TEXT_VIEW (self->textview), 10);
  gtk_text_view_set_right_margin (GTK_TEXT_VIEW (self->textview), 10);

  /* Get handle */
  self->buffer = hildon_text_view_get_buffer (GTK_TEXT_VIEW (self->textview));

  /* Enable Rich Text Support */
  gtk_text_buffer_register_serialize_tagset (self->buffer, "RTF");
  gtk_text_buffer_register_deserialize_tagset (self->buffer, "RTF");

  /* Put textview under scrolledwindow and show it*/
  gtk_container_add(GTK_CONTAINER(self->scrolledwindow), self->textview);
  gtk_widget_show(self->textview);

  /* Change default font throughout the widget */
  self->font_desc = pango_font_description_from_string ("Monospace Regular 22");
  gtk_widget_modify_font (self->textview, self->font_desc);

  /* Connect signals */
  g_signal_connect (self->buffer, "modified-changed",
    G_CALLBACK(maemopad_window_on_buffer_modified), self);
  g_signal_connect (self->buffer, "changed",
    G_CALLBACK(maemopad_window_on_buffer_modified), self);
}


static GtkWidget* add_menu_item (GtkWidget *main_menu, const gchar* title,
    GCallback clicked_callback, gpointer user_data)
{
  HildonAppMenu *app_menu = HILDON_APP_MENU (main_menu);
    
  /* Create a button, add it, and return it: */
  GtkWidget *button = hildon_button_new_with_text (HILDON_SIZE_AUTO, 
        HILDON_BUTTON_ARRANGEMENT_VERTICAL, title, NULL);
  gtk_widget_show (button);
    
  g_signal_connect_after (button, "clicked", 
  	G_CALLBACK (clicked_callback), user_data);
  		      
  hildon_app_menu_append (app_menu, GTK_BUTTON (button));
    
  return button;
}

/* Create the menu items needed: */
static void create_menu (MaemopadWindow *self)
{
  /* Create needed handles */
  GtkWidget *main_menu = hildon_app_menu_new ();
  
  /* Create the menu items */
  self->new_item = add_menu_item (main_menu, _("New"),
    G_CALLBACK (&maemopad_window_on_menu_file_new), self);
  self->open_item = add_menu_item (main_menu, _("Open"),
    G_CALLBACK (&maemopad_window_on_menu_file_open), self);
  self->save_item = add_menu_item (main_menu, _("Save"),
    G_CALLBACK (&maemopad_window_on_menu_file_save), self);
  self->saveas_item = add_menu_item (main_menu, _("Save As..."),
    G_CALLBACK (&maemopad_window_on_menu_file_saveas), self);
  self->cut_item = add_menu_item (main_menu, _("Cut"),
    G_CALLBACK (&maemopad_window_on_menu_edit_cut), self);
  self->copy_item = add_menu_item (main_menu, _("Copy"),
    G_CALLBACK (&maemopad_window_on_menu_edit_copy), self);
  self->paste_item = add_menu_item (main_menu, _("Paste"),
    G_CALLBACK (&maemopad_window_on_menu_edit_paste), self);
  
  self->sendemail_item = add_menu_item (main_menu, _("Send via Email"),
    G_CALLBACK (&maemopad_window_on_menu_sendvia_email), self);
  self->sendbt_item = add_menu_item (main_menu, _("Send via Bluetooth"),
    G_CALLBACK (&maemopad_window_on_menu_sendvia_bt), self);
  
  self->font_item = gtk_font_button_new ();
  gtk_widget_show (self->font_item);
  g_signal_connect_after (self->font_item, "font-set", 
    G_CALLBACK (maemopad_window_on_font_set), self);
  hildon_app_menu_append (HILDON_APP_MENU (main_menu), GTK_BUTTON (self->font_item));
 
  self->fullscreen_item = add_menu_item (main_menu, _("Full Screen"),
    G_CALLBACK (&maemopad_window_on_menu_fullscreen), self);
  self->close_item = add_menu_item (main_menu, _("Close"),
    G_CALLBACK (&maemopad_window_on_menu_close), self);
    
    
  /* Add menu to HildonWindow */
  hildon_window_set_app_menu (
    HILDON_WINDOW (self),
    HILDON_APP_MENU (main_menu));
  
  /* We need to show menu items */
  gtk_widget_show_all (GTK_WIDGET (main_menu));
}

/* Save changes note */
static gboolean maemopad_window_confirm_save_changes (MaemopadWindow *self)
{
  HildonNote *hn = NULL;
  gint response = FALSE;
  g_assert (self);

  hn = HILDON_NOTE (hildon_note_new_confirmation_add_buttons
                     (GTK_WINDOW (self),
                      _("maemopad_save_changes_made"),
                      _("maemopad_yes"), GTK_RESPONSE_YES,
                      _("maemopad_no"), GTK_RESPONSE_NO,
                      NULL, NULL));
  response = gtk_dialog_run (GTK_DIALOG(hn));
  gtk_widget_destroy (GTK_WIDGET(hn));
  
  return response == GTK_RESPONSE_YES;
}


/* File chooser */
static gchar* maemopad_window_choose_file (MaemopadWindow *self, GtkFileChooserAction action)
{
  GtkWidget *dialog;
  gchar* filename = NULL;

  dialog = hildon_file_chooser_dialog_new (GTK_WINDOW (self), action);
  gtk_widget_show_all (GTK_WIDGET(dialog));
    
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  
  gtk_widget_destroy (dialog);
  return filename;
}

/* new */
static void maemopad_window_on_menu_file_new (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);

  /* save changes note if file is edited */
  if (self->file_edited) {
    gboolean confirmed = maemopad_window_confirm_save_changes (self);
    if (confirmed) {
      if (self->file_name == NULL) {
        self->file_name = maemopad_window_choose_file (self, GTK_FILE_CHOOSER_ACTION_SAVE);
      }
      
      maemopad_window_write_buffer_to_file (self);
    }
  }
  
  /* clear buffer, filename and free buffer text */
  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (self->buffer), "", -1);
  self->file_name = NULL;
  self->file_edited = FALSE;
}

/* open */
static void maemopad_window_on_menu_file_open (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  gchar* filename = NULL;
  g_assert (self);
  
  /* save changes note if file is edited */
  if (self->file_edited) {
    gboolean confirmed = maemopad_window_confirm_save_changes (self);
    if (confirmed) {
      /* check is we had a new file */
      if (self->file_name == NULL)
        self->file_name = maemopad_window_choose_file (self, GTK_FILE_CHOOSER_ACTION_SAVE);
      
      maemopad_window_write_buffer_to_file (self);
    }
  }

  /* open new file */
  filename = maemopad_window_choose_file (self, GTK_FILE_CHOOSER_ACTION_OPEN);

  /* if we got a file name from chooser -> open file */
  if (filename) {
    self->file_name = filename;
    maemopad_window_read_file_to_buffer (self);
    self->file_edited = FALSE;
  }
}

/* save */
static void maemopad_window_on_menu_file_save (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);
  
  /* Check if we have a new file: */
  if (self->file_name) {
    maemopad_window_write_buffer_to_file (self);
  } else { 
    gchar* filename = maemopad_window_choose_file (self, GTK_FILE_CHOOSER_ACTION_SAVE);

    /* If we have a file name from the chooser then save the file: */
    if (filename) {
      self->file_name = filename;
      maemopad_window_write_buffer_to_file (self);
      self->file_edited = FALSE;
    }
  }
}

/* save as... */
static void maemopad_window_on_menu_file_saveas (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  gchar* filename = NULL;
  g_assert (self);

  filename = maemopad_window_choose_file (self, GTK_FILE_CHOOSER_ACTION_SAVE);

  /* If we have a file name from the chooser the save the file */
  if (filename) {
    self->file_name = filename;
    maemopad_window_write_buffer_to_file (self);
    self->file_edited = FALSE;
  }
}

/* Close */
static void maemopad_window_on_menu_close(GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  gtk_widget_hide (GTK_WIDGET(self));
}

/* cut */
static void maemopad_window_on_menu_edit_cut (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  GtkClipboard *clipboard = NULL;
  g_assert (self);

  /* do cut */
  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(self->buffer), clipboard, TRUE);
}

/* copy */
static void maemopad_window_on_menu_edit_copy (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  GtkClipboard *clipboard = NULL;
  g_assert (self);

  /* do copy */
  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_text_buffer_copy_clipboard (GTK_TEXT_BUFFER(self->buffer), clipboard);
}

/* paste */
static void maemopad_window_on_menu_edit_paste (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  GtkClipboard *clipboard = NULL;
  g_assert (self);

  /* do paste */
  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_text_buffer_paste_clipboard (GTK_TEXT_BUFFER (self->buffer), clipboard, NULL, TRUE);
}

/*send via stuff */
static void maemopad_window_on_menu_sendvia_email (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);  
  gboolean result = TRUE;
  GSList *list = NULL;
  g_assert (self);

  /* Attach the saved file (and not the one currently on screen). If the file 
   * has not yet been saved, nothing will be attached */

  if (self->file_name) {
    list = g_slist_append(list, self->file_name);
    result = libmodest_dbus_client_compose_mail(self->osso, /*osso_context_t*/
      NULL, /*to*/
      NULL, /*cc*/
      NULL, /*bcc*/
      NULL, /*body*/
      NULL, /*subject*/
      list /*attachments*/);
  }

  g_slist_free(list);

  if (result == FALSE)
    g_print("Could not send via email\n");
}


static gboolean maemopad_window_rpc_sendvia_bluetooth (const gchar *path)
{
  DBusGProxy *proxy = NULL;
  DBusGConnection *sys_conn = NULL;
  GError *error = NULL;
  gboolean result = TRUE;
  gchar **files = NULL;

  /*sys_conn = osso_get_sys_dbus_connection(ctx);*/
  sys_conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);

  if(sys_conn == NULL)
  {
    return FALSE;
  }

  files = g_new0(gchar*, 2);
  *files = g_strdup(path);
  files[1] = NULL;

  /* Open connection for btdialogs service */
  proxy = dbus_g_proxy_new_for_name(sys_conn,
    CONBTDIALOGS_DBUS_SERVICE,
    CONBTDIALOGS_DBUS_PATH,
    CONBTDIALOGS_DBUS_INTERFACE);

  /* Send send file request to btdialogs service */
  if (!dbus_g_proxy_call(proxy, CONBTDIALOGS_SEND_FILE_REQ,
    &error, G_TYPE_STRV, files,
    G_TYPE_INVALID, G_TYPE_INVALID))
  {
    g_print("Error: %s\n", error->message);
    g_clear_error (&error);
    result = FALSE;
  }

  g_strfreev(files);

  g_object_unref(proxy);
  return result;
}


static void maemopad_window_on_menu_sendvia_bt (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);
  
  g_print ("Send %s via bluetooth \n", self->file_name);
  maemopad_window_rpc_sendvia_bluetooth (self->file_name);
}


/* font */
static void maemopad_window_on_font_set (GtkFontButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  const gchar *font_name = NULL;
  PangoFontDescription* new_font = NULL;
  g_assert (self);

  font_name = gtk_font_button_get_font_name (button);
  if (!font_name)
    return;
        
  new_font = pango_font_description_from_string (font_name);

  /* Change the font used in the GtkTextView */
  if (new_font) {
    if (self->font_desc)
      pango_font_description_free (self->font_desc);
            
     self->font_desc = new_font;
     gtk_widget_modify_font (self->textview, self->font_desc);
  }
}

/* fullscreen */
static void maemopad_window_on_menu_fullscreen (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);

  /* toggle fullscreen on<->off */
  self->fullscreen = !self->fullscreen;

  if (self->fullscreen) {
    fullscreen_enable (self->fullscreen_manager);
  } else {
    fullscreen_disable (self->fullscreen_manager);
  }
}


/* buffer edited */
static void maemopad_window_on_buffer_modified (GtkButton *button, gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);

  /* change state */
  self->file_edited = TRUE;
}


/* Error note */
void maemopad_window_show_error (const gchar* error_message, MaemopadWindow *self)
{
  GtkWidget *dialog = NULL;

  dialog = hildon_note_new_information (GTK_WINDOW (self), error_message);

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


/* read file */
static void maemopad_window_read_file_to_buffer (MaemopadWindow* self)
{
  GnomeVFSResult vfs_result = GNOME_VFS_OK;
  GnomeVFSHandle *handle = NULL;
  GnomeVFSFileSize in_bytes = 0;
  GnomeVFSFileInfo finfo;
  gchar *temp_buffer = NULL;
  g_assert (self);

  /* Try to get file info: */
  vfs_result = gnome_vfs_get_file_info(self->file_name, &finfo, GNOME_VFS_FILE_INFO_DEFAULT);
  if (vfs_result != GNOME_VFS_OK) {
    maemopad_window_show_error (_("maemopad_open_failed"), self);
    return;
  }
    
  /* Try to create a handle to the file: */
  vfs_result = gnome_vfs_open(&handle, self->file_name, GNOME_VFS_OPEN_READ);
  if (vfs_result != GNOME_VFS_OK) {
    maemopad_window_show_error (_("maemopad_open_failed"), self);
    return;
  }

  /* Allocate memory for temp_buffer: */
  temp_buffer = g_malloc(finfo.size + 1);
  memset(temp_buffer, 0, finfo.size + 1);
    
  /* Read from file to buffer: */
  gnome_vfs_read(handle, temp_buffer, finfo.size, &in_bytes);

  /* Set text to screen: */
  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (self->buffer), temp_buffer, -1);

  /* Free temp, close file and return: */
  g_free(temp_buffer);
  gnome_vfs_close(handle);
}

/* write to file */
static void maemopad_window_write_buffer_to_file (MaemopadWindow* self)
{
  GnomeVFSResult vfs_result = GNOME_VFS_OK;
  GnomeVFSHandle *handle = NULL;
  GnomeVFSFileSize out_bytes = 0;
  gchar *temp_buffer = NULL;
  GtkTextIter start, end;
  g_assert (self);

  /* try to create handle to file */
  vfs_result = gnome_vfs_create (&handle, self->file_name, GNOME_VFS_OPEN_WRITE, 0, 0600);
  if (vfs_result != GNOME_VFS_OK) {
    maemopad_window_show_error (_("maemopad_save_failed"), self);
    return;
  }

  /* find start and end of text */
  gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (self->buffer), &start, &end);

  /* copy all text from screen to temp_buffer */
  temp_buffer = gtk_text_buffer_get_slice (GTK_TEXT_BUFFER (self->buffer), &start, &end, TRUE);

  /* write text to file */
  gnome_vfs_write(handle, temp_buffer, strlen (temp_buffer), &out_bytes);

  /* free temp, close file and return */
  g_free(temp_buffer);
  gnome_vfs_close(handle);
}


static gboolean maemopad_window_on_key_press(GtkWidget *widget, GdkEventKey *event,
                     gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);
  (void) widget;
    
  if (event->keyval == HILDON_HARDKEY_FULLSCREEN && !self->key_pressed) {
    self->key_pressed = TRUE;

    /* Toggle fullscreen on and off: */
    self->fullscreen = !self->fullscreen;

    if (self->fullscreen) {
      fullscreen_enable (self->fullscreen_manager);
    } else {
      fullscreen_disable (self->fullscreen_manager);
    }
  }
  
  if (event->keyval == HILDON_HARDKEY_ESC && !self->key_pressed)
    self->key_pressed = TRUE;
    
  return FALSE;
}

static gboolean maemopad_window_on_key_release(GtkWidget *widget, GdkEventKey *event,
                     gpointer data)
{
  MaemopadWindow *self = MAEMOPAD_WINDOW (data);
  g_assert (self);
    
  self->key_pressed = FALSE;
  return FALSE;
}


