#include "modeSwitchWin.h"
#include "nsITimer.h"
#include "nsComponentManagerUtils.h"
#include "nsIDOMEvent.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"
#include "nsIWidget.h"
#include "nsWindowListener.h"

#define BUTTON_WIDTH    64
#define BUTTON_HEIGHT   64

NS_IMPL_ISUPPORTS1(ModeSwitchWin, nsISupports)

ModeSwitchWin::ModeSwitchWin(nsWindowListener& aWindowListener):
  mWindow(nsnull),
  mButton(nsnull),
  mParent(nsnull),
  mHideWinTimer(nsnull),
  mWindowListener(aWindowListener),
  mDOMWindow(nsnull),
  mDOMWindowInternal(nsnull),
  mHostWinFocus(PR_FALSE),
  mModeSwitchWinX(0),
  mModeSwitchWinY(0),
  mZoomLevel(0)
{
  Init();
}

ModeSwitchWin::~ModeSwitchWin()
{
  if(mHideWinTimer)
    mHideWinTimer->Cancel();
  if(mWindow)
    gtk_widget_destroy(GTK_WIDGET(mWindow));
}

nsresult
ModeSwitchWin::Init()
{
  nsresult rv;
  mHideWinTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  return NS_OK;
}

nsresult
ModeSwitchWin::ShowModeSwitchWin(PRInt32 aMode, nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if(mHideWinTimer)
    mHideWinTimer->Cancel();
  UpdateFromEvent(aDOMEvent);
  CreateModeSwitchWin(aMode);
  NS_ENSURE_TRUE(mButton, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mWindow, NS_ERROR_FAILURE);
  UpdatePosition();
  gtk_widget_show_all(GTK_WIDGET(mWindow));

  return NS_OK;
}

nsresult
ModeSwitchWin::HideModeSwitchWin(PRBool aDelay)
{
  if(aDelay) {
    NS_ENSURE_TRUE(mHideWinTimer, NS_ERROR_FAILURE);
    mHideWinTimer->Cancel();
    mHideWinTimer->InitWithFuncCallback(ModeSwitchWin::HideWinCallback,
                                        (void*)this,
                                        2000,
                                        nsITimer::TYPE_ONE_SHOT);
  } else {
    NS_ENSURE_TRUE(mWindow, NS_OK);
    NS_ENSURE_TRUE(mDOMEvent, NS_ERROR_FAILURE);
    if(mHideWinTimer)
      mHideWinTimer->Cancel();
    gtk_widget_hide(GTK_WIDGET(mWindow));
    HelperFunctions::SetCursor(mDOMEvent, eCursor_standard);
    }
  return NS_OK;
}

nsresult
ModeSwitchWin::HostWinFocus(nsIDOMEvent *aDOMEvent)
{
  mHostWinFocus = PR_TRUE;
  return NS_OK;
}

nsresult
ModeSwitchWin::HostWinBlur(nsIDOMEvent *aDOMEvent)
{
  mHostWinFocus = PR_FALSE;
  if(mHideWinTimer)
    mHideWinTimer->Cancel();
  return NS_OK;
}

//private
void
ModeSwitchWin::CreateModeSwitchWin(PRInt32 aMode)
{
  if (!mWindow && !mButton) {
    mWindow = gtk_window_new(GTK_WINDOW_POPUP);
    NS_ENSURE_TRUE(mWindow, );
    gtk_window_set_decorated(GTK_WINDOW(mWindow), FALSE);
    gtk_widget_set_size_request(mWindow, BUTTON_WIDTH, BUTTON_HEIGHT);
    gtk_widget_realize(GTK_WIDGET(mWindow));
    if (mParent) {
      // Bind the popup window to the MozContainer's window
      gdk_window_reparent(mWindow->window, mParent->window, 0, 0);
    }

    mButton = GTK_WIDGET(gtk_tool_button_new(NULL, NULL));
    NS_ENSURE_TRUE(mButton, );
    gtk_signal_connect(GTK_OBJECT (mButton), "clicked", GTK_SIGNAL_FUNC (ModeSwitchWin::OnClick), this);
    gtk_signal_connect(GTK_OBJECT (mWindow), "hide", GTK_SIGNAL_FUNC (ModeSwitchWin::OnHide), this);
    gtk_container_add(GTK_CONTAINER(mWindow), mButton);
  }
  if (aMode == MODE_PANNING) {
    UpdateIcon(PR_FALSE);
    NS_ENSURE_TRUE(mDOMEvent, );
    HelperFunctions::SetCursor(mDOMEvent, eCursor_crosshair);
  } else if (aMode == MODE_HOVER) {
    UpdateIcon(PR_FALSE);
    NS_ENSURE_TRUE(mDOMEvent, );
    HelperFunctions::SetCursor(mDOMEvent, eCursor_crosshair);
  } else {//input mode
    UpdateIcon(PR_TRUE);
    NS_ENSURE_TRUE(mDOMEvent, );
    HelperFunctions::SetCursor(mDOMEvent, eCursor_hyperlink);
  }
}

//private static
void
ModeSwitchWin::HideWinCallback(nsITimer *aTimer, void *closure)
{
  NS_ENSURE_TRUE(closure, );
  ModeSwitchWin *self = (ModeSwitchWin*)closure;
  NS_ENSURE_TRUE(self, );
  self->HideModeSwitchWin();
}

//private static
void
ModeSwitchWin::OnClick(GtkWidget *aWidget, gpointer data)
{
  NS_ENSURE_TRUE(aWidget, );
  NS_ENSURE_TRUE(data, );
  ModeSwitchWin *self = (ModeSwitchWin*)data;
  NS_ENSURE_TRUE(self, );

  if(self->mWindowListener.TouchScreenMode() == MODE_PANNING) {
    self->mWindowListener.SetTouchScreenMode(MODE_DINPUT);
    if(self->mHideWinTimer)
      self->mHideWinTimer->Cancel();
    self->UpdateIcon(PR_TRUE);
    NS_ENSURE_TRUE(self->mDOMEvent, );
    self->mWindowListener.SetCursorVisible(PR_TRUE);
    HelperFunctions::SetCursor(self->mDOMEvent, eCursor_hyperlink);
    return;
    }

  if(self->mWindowListener.TouchScreenMode() == MODE_DINPUT) {
    self->mWindowListener.SetTouchScreenMode(MODE_PANNING);
    self->UpdateIcon(PR_FALSE);
    NS_ENSURE_TRUE(self->mDOMEvent, );
    self->mWindowListener.SetCursorVisible(PR_FALSE);
    HelperFunctions::SetCursor(self->mDOMEvent, eCursor_crosshair);
    self->HideModeSwitchWin(PR_TRUE);
    return;
    }
}

//private static
void
ModeSwitchWin::OnHide(GtkWidget *aWidget, gpointer data)
{
  NS_ENSURE_TRUE(data, );
  ModeSwitchWin *self = (ModeSwitchWin*) data;
  NS_ENSURE_TRUE(self, );
  self->UpdatePosition();

  if (self->mWindowListener.TouchScreenMode() == MODE_HOVER && self->mHostWinFocus) {
    NS_ENSURE_TRUE(self->mWindow, );
    gtk_widget_show(GTK_WIDGET(self->mWindow));
    NS_ENSURE_TRUE(self->mDOMEvent, );
    HelperFunctions::SetCursor(self->mDOMEvent, eCursor_crosshair);
  }
  if (self->mWindowListener.TouchScreenMode() == MODE_DINPUT && self->mHostWinFocus) {
    NS_ENSURE_TRUE(self->mWindow, );
    gtk_widget_show(GTK_WIDGET(self->mWindow));
    NS_ENSURE_TRUE(self->mDOMEvent, );
    HelperFunctions::SetCursor(self->mDOMEvent, eCursor_hyperlink);
  }
  return;
}

nsresult
ModeSwitchWin::UpdatePosition()
{
  NS_ENSURE_TRUE(mDOMEvent, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mDOMWindowInternal, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mWindow, NS_ERROR_FAILURE);
  PRInt32 innerWidth, innerHeight, screenX, screenY, parentX = 0, parentY = 0;
  mDOMWindowInternal->GetInnerWidth(&innerWidth);
  mDOMWindowInternal->GetInnerHeight(&innerHeight);
  mDOMWindowInternal->GetScreenX(&screenX);
  mDOMWindowInternal->GetScreenY(&screenY);
  if (mParent) {
    // if we have succesfully reparented window, adjust coordinates to its coordinate space
    gdk_window_get_origin(mParent->window, &parentX, &parentY);
  }
  PRInt32 zoom = HelperFunctions::GetZoomLevel(mDOMEvent);
  mZoomLevel = zoom != 0 ? zoom : mZoomLevel;
  mModeSwitchWinX = screenX - parentX;
  mModeSwitchWinY = screenY - parentY + innerHeight * mZoomLevel / 100 - BUTTON_HEIGHT;

  gtk_window_move(GTK_WINDOW(mWindow), mModeSwitchWinX, mModeSwitchWinY);
  return NS_OK;
}

nsresult
ModeSwitchWin::UpdateIcon(PRBool aPanningOn)
{
  const gchar *icon = aPanningOn ? "browser_panning_mode_on"
                                 : "browser_panning_mode_off";
  gtk_tool_button_set_icon_name(GTK_TOOL_BUTTON(mButton), icon);
  return NS_OK;
}

nsresult
ModeSwitchWin::UpdateFromEvent(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  mDOMEvent = do_QueryInterface(aDOMEvent);

  if (!mDOMWindow) {
    nsCOMPtr<nsIDOMWindow> tmpDOMWindow;
    HelperFunctions::GetDOMWindowFromEvent(aDOMEvent, getter_AddRefs(tmpDOMWindow));
    NS_ENSURE_TRUE(tmpDOMWindow, NS_ERROR_FAILURE);
    nsCOMPtr<nsIDOMWindow> topDOMWindow;
    tmpDOMWindow->GetTop(getter_AddRefs(topDOMWindow));
    NS_ENSURE_TRUE(topDOMWindow, NS_ERROR_FAILURE);
    NS_IF_ADDREF(mDOMWindow = topDOMWindow);
    NS_ENSURE_TRUE(mDOMWindow, NS_ERROR_FAILURE);
  }
  if (!mDOMWindowInternal) {
    mDOMWindowInternal = do_QueryInterface(mDOMWindow);
    NS_ENSURE_TRUE(mDOMWindowInternal, NS_ERROR_FAILURE);
  }
  if (!mParent) {
    // Get pointer to the MozContainer widget to use its associated GdkWindow as a
    // parent for our popup window.
    nsIWidget* widget = nsnull;
    nsresult rv = HelperFunctions::GetWidgetFromEvent(aDOMEvent, &widget);
    NS_ENSURE_SUCCESS(rv, rv);
    NS_ENSURE_TRUE(widget, NS_ERROR_FAILURE);
    gpointer native_window = widget->GetNativeData(NS_NATIVE_WINDOW);
    NS_ENSURE_TRUE(GDK_IS_WINDOW(native_window), NS_ERROR_FAILURE);
    gdk_window_get_user_data(GDK_WINDOW(native_window), (gpointer*) &mParent);
  }

  return NS_OK;
}
