#include "iScrollBars.h"
#include "nsIWidget.h"
#include "nsIViewManager.h"
#include "gfxContext.h"
#include "nsIDOMWindow.h"
#include "nsIBaseWindow.h"
#include "nsIDocShellTreeItem.h"
#include "nsIPresShell.h"
#include "nsIDocument.h"
#include "nsIWebProgress.h"
#include "nsIWebProgressListener.h"
#include "nsIDOMBarProp.h"
#include "nsWindowListener.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMElement.h"
#include "nsIMarkupDocumentViewer.h"

NS_IMPL_ISUPPORTS1(iScrollBars, IBaseMode)

iScrollBars::iScrollBars(nsWindowListener *aListener)
 : mListener(aListener),
   mHideScrollbars(PR_TRUE),
   mDoPaintScrollbars(PR_FALSE),
   mIndicatorWidth(10)
{
  Init();
}

iScrollBars::~iScrollBars()
{
}

nsresult
iScrollBars::Init()
{
  mHideScrollbars = PR_TRUE;
  HelperFunctions::GetPref(PREF_TYPE_BOOL, "webaddon.widgetutils.hide_scrollbars", &mHideScrollbars);
  if (getenv("MOZ_SCROLL"))
    mHideScrollbars = PR_FALSE;
  mIndicatorWidth = 10;
  HelperFunctions::GetPref(PREF_TYPE_INT, "webaddon.widgetutils.ind_width", &mIndicatorWidth);
  char *style = nsnull;
  if (mScrollIndStyle.IsEmpty()) {
    if (NS_SUCCEEDED(HelperFunctions::GetPref(PREF_TYPE_STRING, "webaddon.widgetutils.ind_style", &style))) {
      mScrollIndStyle.AssignLiteral(style);
      NS_Free(style);
    } else
      mScrollIndStyle.AssignLiteral("z-index:500000;position:fixed;background-color:gray;opacity:0.5;border:1px solid white;-moz-border-radius:10px;");
  }
  return NS_OK;
}

nsresult
iScrollBars::OnStateChange(nsIWebProgress *aWebProgress, PRUint32 aStateFlags)
{
  if (mHideScrollbars && aStateFlags & nsIWebProgressListener::STATE_START)
  {
    // Comment this to get disable scrollbars for all frames
    // if (!(aStateFlags & nsIWebProgressListener::STATE_IS_DOCUMENT)) return NS_OK;

    nsresult rv;
    nsCOMPtr<nsIDOMWindow> win;
    rv = aWebProgress->GetDOMWindow(getter_AddRefs(win));
    NS_ENSURE_SUCCESS(rv, NS_OK);
    nsCOMPtr<nsIDOMBarProp> scrollbars;
    rv = win->GetScrollbars(getter_AddRefs(scrollbars));
    NS_ENSURE_SUCCESS(rv, NS_OK);
    PRBool isVisible = PR_TRUE;
    scrollbars->GetVisible(&isVisible);
    if (isVisible)
      scrollbars->SetVisible(PR_FALSE);
  }
  return NS_OK;
}

nsresult
iScrollBars::MouseDown(nsIDOMEvent *aDOMEvent)
{
  nsCOMPtr<nsIDOMWindow> tmpDOMWindow;
  HelperFunctions::GetDOMWindowFromEvent(aDOMEvent, getter_AddRefs(tmpDOMWindow));
  if (tmpDOMWindow)
    mWindow = tmpDOMWindow;
  CreateScrollIndicators(mWindow);
  MouseMove(aDOMEvent);
  return NS_OK;
}

nsresult
iScrollBars::MouseUp(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(mVElement, NS_OK);
  NS_ENSURE_TRUE(mHElement, NS_OK);
  mVElement->SetAttribute(NS_LITERAL_STRING("style"), NS_LITERAL_STRING("visibility:hidden;"));
  mHElement->SetAttribute(NS_LITERAL_STRING("style"), NS_LITERAL_STRING("visibility:hidden;"));
  return NS_OK;
}

nsresult
iScrollBars::MouseMove(nsIDOMEvent *aDOMEvent)
{
  nsresult rv;
  nsCOMPtr<nsIDOMWindow> window;
  HelperFunctions::GetDOMWindowFromEvent(aDOMEvent, getter_AddRefs(window));
  if (!window)
    window = mWindow;
  if (!mWindow) {
    mWindow = window;
    if (!mVElement && !mHElement)
      CreateScrollIndicators(mWindow);
  }
  NS_ENSURE_TRUE(window, NS_OK);

  CreateScrollIndicators(mWindow);

  nsCOMPtr<nsIDOMWindowInternal> winint = do_QueryInterface(window, &rv);

  nsCOMPtr<nsIMarkupDocumentViewer> mv;
  float zoomLevel = 1.0f;
  HelperFunctions::GetMarkupViewerByWindow(window, getter_AddRefs(mv));
  if (mv)
      mv->GetFullZoom(&zoomLevel);

  PRInt32 aInnerWidth = 0, aInnerHeight = 0;
  PRInt32 aScrollMaxX = 0, aScrollMaxY = 0;
  PRInt32 aPageXOffset = 0, aPageYOffset = 0;

  winint->GetInnerWidth(&aInnerWidth);
  winint->GetInnerHeight(&aInnerHeight);
  winint->GetScrollMaxX(&aScrollMaxX);
  winint->GetScrollMaxY(&aScrollMaxY);
  winint->GetPageXOffset(&aPageXOffset);
  winint->GetPageYOffset(&aPageYOffset);

  PRBool drawHBar = (aScrollMaxX);
  PRBool drawVBar = (aScrollMaxY);

  float tmp;
  PRInt32 scrollbarWidth = mIndicatorWidth / (zoomLevel ? zoomLevel : 1.0f);
  float scrollbarWidthHalf = scrollbarWidth / 2;
  float vscrollbarX = aInnerWidth - scrollbarWidthHalf - 1;
  tmp = aInnerHeight + aScrollMaxY;
  float vscrollbarYDivisor = tmp ? tmp : 1.0f;
  float vscrollbarY = (aPageYOffset * aInnerHeight) / vscrollbarYDivisor + scrollbarWidthHalf;
  float vscrollbarSize = (aInnerHeight < (aInnerHeight + aScrollMaxY) ? (aInnerHeight * aInnerHeight) / vscrollbarYDivisor : 0) - scrollbarWidth * 2;

  float hscrollbarY = aInnerHeight - scrollbarWidthHalf - 1;
  tmp = aInnerWidth + aScrollMaxX;
  float hscrollbarXDivisor = tmp ? tmp : 1.0f;
  float hscrollbarX = (aPageXOffset * aInnerWidth) / hscrollbarXDivisor + scrollbarWidthHalf;
  float hscrollbarSize = (aInnerWidth  < (aInnerWidth  + aScrollMaxX) ? (aInnerWidth  * aInnerWidth)  / hscrollbarXDivisor : 0) - scrollbarWidth * 2;

  nsRect vrect(vscrollbarX - scrollbarWidthHalf, vscrollbarY - scrollbarWidthHalf, scrollbarWidth, vscrollbarSize + scrollbarWidth);
  nsRect hrect(hscrollbarX - scrollbarWidthHalf, hscrollbarY - scrollbarWidthHalf, hscrollbarSize + scrollbarWidth*2, scrollbarWidth);

  vrect.height = vrect.height < mIndicatorWidth ? mIndicatorWidth : vrect.height;
  hrect.width = hrect.width < mIndicatorWidth ? mIndicatorWidth : hrect.width;

/*
  printf("Scrollbars: v[%i,%i,%i,%i], h[%i,%i,%i,%i], inner[%i,%i], offs[%i,%i], scr[%i,%i], indw:%i, scrSize[%g,%g]\n\n",
         vrect.x, vrect.y, vrect.width, vrect.height,
         hrect.x, hrect.y, hrect.width, hrect.height,
         aInnerWidth, aInnerHeight, aScrollMaxX,
         aScrollMaxY, aPageXOffset, aPageYOffset,
         mIndicatorWidth, vscrollbarSize, vscrollbarSize);
*/

  if (drawVBar)
    setMarkerPosition(mVElement, vrect);
  if (drawHBar)
    setMarkerPosition(mHElement, hrect);

  return NS_OK;
}

nsresult
iScrollBars::setMarkerPosition(nsIDOMElement *aDOMElement, nsRect &aRect)
{
  NS_ENSURE_TRUE(aDOMElement, NS_ERROR_FAILURE);
  nsString style;

  style.Append(mScrollIndStyle);
  style.Append(NS_LITERAL_STRING("left:"));
  style.AppendInt(aRect.x-1);
  style.Append(NS_LITERAL_STRING("px;top:"));
  style.AppendInt(aRect.y-1);
  style.Append(NS_LITERAL_STRING("px;"));
  style.Append(NS_LITERAL_STRING("width:"));
  style.AppendInt(aRect.width);
  style.Append(NS_LITERAL_STRING("px;height:"));
  style.AppendInt(aRect.height);
  style.Append(NS_LITERAL_STRING("px;"));
  aDOMElement->SetAttribute(NS_LITERAL_STRING("style"), style);
  return NS_OK;
}

nsresult
iScrollBars::CreateScrollIndicators(nsIDOMWindow *dw)
{
  nsresult rv;
  if (mVElement && mHElement) {
    AttachScrollIndicators(dw, PR_TRUE);
    return NS_OK;
  }
  NS_ENSURE_TRUE(dw, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMDocument> doc;
  rv = dw->GetDocument(getter_AddRefs(doc));
  NS_ENSURE_SUCCESS(rv, NS_OK);
  rv = doc->CreateElementNS(NS_LITERAL_STRING("http://www.w3.org/1999/xhtml"), NS_LITERAL_STRING("div"), getter_AddRefs(mVElement));
  NS_ENSURE_SUCCESS(rv, NS_OK);
  mVElement->SetAttribute(NS_LITERAL_STRING("id"), NS_LITERAL_STRING("vmarker"));

  rv = doc->CreateElementNS(NS_LITERAL_STRING("http://www.w3.org/1999/xhtml"), NS_LITERAL_STRING("div"), getter_AddRefs(mHElement));
  NS_ENSURE_SUCCESS(rv, NS_OK);
  mHElement->SetAttribute(NS_LITERAL_STRING("id"), NS_LITERAL_STRING("hmarker"));
  AttachScrollIndicators(dw, PR_TRUE);

  return rv;
}

nsresult
iScrollBars::AttachScrollIndicators(nsIDOMWindow *dw, PRBool aAttach)
{
  nsresult rv;
  NS_ENSURE_TRUE(dw, NS_OK);
  nsCOMPtr<nsIDOMDocument> doc;
  rv = dw->GetDocument(getter_AddRefs(doc));
  NS_ENSURE_SUCCESS(rv, NS_OK);

  nsCOMPtr<nsIDOMHTMLElement> domhel;
  nsCOMPtr<nsIDOMHTMLDocument> domdoc = do_QueryInterface(doc, &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  rv = domdoc->GetBody(getter_AddRefs(domhel));
  NS_ENSURE_SUCCESS(rv, NS_OK);
  nsCOMPtr<nsIDOMNode> domnode = do_QueryInterface((nsISupports*)domhel.get(), &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  nsCOMPtr<nsIDOMNode> newdomnode;
  nsCOMPtr<nsIDOMNode> newdomnode1 = do_QueryInterface(mVElement, &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);
  nsCOMPtr<nsIDOMNode> newdomnode2 = do_QueryInterface(mHElement, &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);

  if (aAttach) {
    rv |= domnode->AppendChild(newdomnode1, getter_AddRefs(newdomnode));
    rv |= domnode->AppendChild(newdomnode2, getter_AddRefs(newdomnode));
  } else {
    rv |= domnode->RemoveChild(newdomnode1, getter_AddRefs(newdomnode));
    rv |= domnode->RemoveChild(newdomnode2, getter_AddRefs(newdomnode));
  }
  return rv;
}

nsresult
iScrollBars::SetScrollIndicatorsVisibility(nsIDOMWindow *dw, PRBool aVisible)
{
  if (!aVisible) {
    mVElement->SetAttribute(NS_LITERAL_STRING("style"), NS_LITERAL_STRING("visibility:hidden;"));
    mHElement->SetAttribute(NS_LITERAL_STRING("style"), NS_LITERAL_STRING("visibility:hidden;"));
  }
  return NS_OK;
}
