#include "monoMode.h"
#include "nsIDOMEvent.h"
#include "nsITimer.h"
#include "nsComponentManagerUtils.h"
#include "nsITouchInteractListener.h"
#include "nsIWidget.h"
#include "nsIViewManager.h"
#include "nsIDOMNSHTMLElement.h"
#include "nsIDOMEventTarget.h"
#include "nsStringGlue.h"
#include "nsIDOMNode.h"
#include "nsIWeakReference.h"
#include "nsIWeakReferenceUtils.h"
#include "nsGUIEvent.h"
#include "speedmanager.h"
#include "nsIDOMMouseEvent.h"
#include "nsIMarkupDocumentViewer.h"

#define PAN_DECAY_STEP 50.0

PRUint32 MonoMode::sIsPanning;

NS_IMPL_ISUPPORTS1(MonoMode, IBaseMode)

MonoMode::MonoMode() :
  mRedispatchTimer(nsnull),
  mViewManager(nsnull),
  mMouseDown(PR_FALSE),
  mHasDispatchEvent(PR_FALSE),
  mPanningTimer(nsnull),
  mEndPanTimer(nsnull),
  mSpeedX(nsnull),
  mSpeedY(nsnull),
  mKineticPanStep(0),
  mKineticEn(PR_FALSE),
  mMouseDownZoomLevel(1.0)
{
  Init();
}

nsresult
MonoMode::Init()
{
  INIT_MOUSE_DISPATCH();
  sIsPanning = 0;
  memset((void*)&mFirstMousedownEvent, 0, sizeof(mFirstMousedownEvent));
  memset((void*)&mLastMouseEvent, 0, sizeof(mLastMouseEvent));
  memset((void*)&mCurrentMouseEvent, 0, sizeof(mCurrentMouseEvent));
  mMaxPressure = -1;
  nsresult rv;
  mRedispatchTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  mKineticEn = PR_FALSE;
  HelperFunctions::GetPref(PREF_TYPE_BOOL, "webaddon.widgetutils.monoKinetic", &mKineticEn);
  if(mKineticEn) {
    mPanningTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    mEndPanTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    mSpeedX = new SpeedManager;
    NS_ENSURE_TRUE(mSpeedX, NS_ERROR_OUT_OF_MEMORY);
    mSpeedY = new SpeedManager;
    NS_ENSURE_TRUE(mSpeedY, NS_ERROR_OUT_OF_MEMORY);
  }
  return NS_OK;
}

MonoMode::~MonoMode()
{
  HelperFunctions::CancelShowCxtMenu();
  if (mPanningTimer)
    mPanningTimer->Cancel();
  if (mEndPanTimer)
    mEndPanTimer->Cancel();
  if (mRedispatchTimer)
    mRedispatchTimer->Cancel();
}

PRBool
MonoMode::SetMouseDown(PRBool aMouseDown)
{
  mMouseDown = aMouseDown;
  return mMouseDown;
}

PRBool
MonoMode::CanShowCxtMenu()
{
  return (sIsPanning==0);
}

nsIDOMEvent*
MonoMode::GetDOMEvent()
{
  return mDOMEvent;
}

nsresult
MonoMode::OnMono(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  mDOMEvent = do_QueryInterface(aDOMEvent);
  return CALL_MOUSE_DISPATCH(aDOMEvent);
}

nsresult
MonoMode::MouseDown(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if (mHasDispatchEvent) {
    if (HelperFunctions::IsHTMLSelectElement(aDOMEvent)
            || HelperFunctions::IsHTMLOptionElement(aDOMEvent)
            || HelperFunctions::IsHTMLInputElement(aDOMEvent)
            || HelperFunctions::IsHTMLTextAreaElement(aDOMEvent)) {
      aDOMEvent->StopPropagation();
      aDOMEvent->PreventDefault();
    }
    HelperFunctions::OpenSelection(aDOMEvent);
    return NS_OK;
  }

  nsCOMPtr<nsIDOMWindow> DOMWindow;
  HelperFunctions::GetDOMWindowFromEvent(aDOMEvent,getter_AddRefs(DOMWindow));
  nsCOMPtr<nsIMarkupDocumentViewer> markupViewer;
  nsresult rv = HelperFunctions::GetMarkupViewerByWindow(DOMWindow, getter_AddRefs(markupViewer));
  if (markupViewer)
    markupViewer->GetFullZoom(&mMouseDownZoomLevel);

  nsCOMPtr<nsIViewManager> tmpViewManager;
  HelperFunctions::GetViewManagerFromEvent(aDOMEvent, getter_AddRefs(tmpViewManager));
  if (tmpViewManager) // check mouse move out of screen
    NS_IF_ADDREF(mViewManager = tmpViewManager);
  NS_ENSURE_TRUE(mViewManager, NS_ERROR_FAILURE);

  if (!HelperFunctions::IsHTMLOptionElement(aDOMEvent))
    mMouseDown = PR_TRUE;
  EndPan();
  PRBool isXul = HelperFunctions::IsXULNode(aDOMEvent);

  MouseEventArg tmpMousedownEvent;
  memcpy((void*)&tmpMousedownEvent, (void*)&mFirstMousedownEvent, sizeof(MouseEventArg));
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mFirstMousedownEvent);
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mLastMouseEvent);

  PRBool modifier;
  nsCOMPtr<nsIDOMMouseEvent> ev = do_QueryInterface(aDOMEvent);
  if (ev)
    ev->GetShiftKey(&modifier);
  if (isXul) {
    if (modifier)
      HelperFunctions::StartShowCxtMenu(this);
    return NS_OK;
  }
  nsString string;
  if (NS_FAILED(HelperFunctions::IsSelectionAvailable(aDOMEvent, string))) {
    HelperFunctions::RemoveSelection(aDOMEvent);
    ChangeFocus(aDOMEvent);
  }
  HelperFunctions::OpenSelection(aDOMEvent);

  mIsDblClick = HelperFunctions::IsDblClick(&mLastMouseEvent, &tmpMousedownEvent);
  if(!mIsDblClick)
    HelperFunctions::StartShowCxtMenu(this);

  if (!HelperFunctions::IsHTMLSelectElement(aDOMEvent)
          && !HelperFunctions::IsHTMLOptionElement(aDOMEvent)
          && !HelperFunctions::IsHTMLInputElement(aDOMEvent)
          && !HelperFunctions::IsHTMLTextAreaElement(aDOMEvent)
          && !mIsDblClick) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }

  GET_STATIC_INTERACTLISTENER(sListener);
  mMaxPressure = 0.1;
  if (!sListener) return NS_OK;

  nsCOMPtr<nsIWidget> widget;
  HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
  return sListener->OnMouseDownMono(aDOMEvent, widget);
}

nsresult
MonoMode::MouseMove(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mMouseDown, NS_OK);
  NS_ENSURE_FALSE(HelperFunctions::IsXULNode(aDOMEvent), NS_OK);

  if(mIsDblClick) {
    HelperFunctions::OpenSelection(aDOMEvent);
  } else {
    HelperFunctions::CloseSelection(aDOMEvent);
  }

  PRInt32 delta = 0;
  PRInt32 dx = 0;
  PRInt32 dy = 0;

  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mCurrentMouseEvent);
  if (mCurrentMouseEvent.pressure >= 0.05 && mCurrentMouseEvent.pressure <= 1)
    mMaxPressure = PR_MAX(mCurrentMouseEvent.pressure, mMaxPressure);
  PRBool isJittering = HelperFunctions::IsMouseJittering(&mCurrentMouseEvent,
                                                         &mLastMouseEvent,
                                                         &delta,
                                                         &dx,
                                                         &dy, sIsPanning?0:mMaxPressure);

  if (mKineticEn && mSpeedX && mSpeedY) {
    mSpeedX->AddSpeed(delta, dx);
    mSpeedY->AddSpeed(delta, dy);
  }

  // To reduce jitters, return if the mousemove was a small delta
  NS_ENSURE_FALSE(isJittering, NS_OK);
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mLastMouseEvent);
  nsCOMPtr<nsIViewManager> tmpViewManager;
  HelperFunctions::GetViewManagerFromEvent(aDOMEvent, getter_AddRefs(tmpViewManager));
  if(tmpViewManager) // check mouse move out of screen
    NS_IF_ADDREF(mViewManager = tmpViewManager);
  NS_ENSURE_TRUE(mViewManager, NS_ERROR_FAILURE);
  if(!mIsDblClick)
    DoPan(aDOMEvent, mViewManager, dx, dy);

  if (sIsPanning > 0) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }

  GET_STATIC_INTERACTLISTENER(sListener);
  if (!sListener) return NS_OK;

  nsCOMPtr<nsIWidget> widget;
  HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
  return sListener->OnMouseMoveMono(aDOMEvent, widget);
}

nsresult
MonoMode::MouseUp(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if(mHasDispatchEvent) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
    mHasDispatchEvent = PR_FALSE;
    return NS_OK;
  }
  mMouseDown = PR_FALSE;
  if (mMaxPressure > 0.25 && sIsPanning == 1)
    sIsPanning = 0;
  mMaxPressure = 0;
  HelperFunctions::CancelShowCxtMenu();
  PRBool isXul = HelperFunctions::IsXULNode(aDOMEvent);

  GET_STATIC_INTERACTLISTENER(sListener);
  if (sListener) {
    nsCOMPtr<nsIWidget> widget;
    HelperFunctions::GetWidgetFromEvent(aDOMEvent, getter_AddRefs(widget));
    sListener->OnMouseUpMono(aDOMEvent, widget);
  }
  if (isXul)
    return NS_OK;

  if(mKineticEn) {
    HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mCurrentMouseEvent);
    PRInt32 distanceX = mFirstMousedownEvent.screenX - mCurrentMouseEvent.screenX;
    PRInt32 distanceY = mFirstMousedownEvent.screenY - mCurrentMouseEvent.screenY;

    if (PR_ABS(distanceX) < 10 && PR_ABS(distanceY) < 10) {
      EndPan(PR_TRUE);//delay
      return NS_OK;
    }

    if (sIsPanning > 0) {
      //aDOMEvent->StopPropagation();
      //aDOMEvent->PreventDefault();
      nsresult rv;
      if (mPanningTimer) {
        rv = mPanningTimer->InitWithFuncCallback(MonoMode::DoKineticPan,
                                                 (void*) this, 0,
                                                 nsITimer::TYPE_ONE_SHOT);
        if (NS_SUCCEEDED(rv))
          return NS_OK;
      }
      EndPan(PR_TRUE);//delay
    }
  }
  return NS_OK;
}

nsresult
MonoMode::MouseClick(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_OK);
  if (sIsPanning > 0 || mHasDispatchEvent || mIsDblClick) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
    mHasDispatchEvent = PR_FALSE;
  } else if (mRedispatchTimer
          && !HelperFunctions::IsHTMLSelectElement(aDOMEvent)
          && !HelperFunctions::IsXULNode(aDOMEvent)) {
    MonoMode::DispatchMouseEvent(mRedispatchTimer, this);
    //mRedispatchTimer->InitWithFuncCallback(MonoMode::DispatchMouseEvent,
    //                                       (void*) this,
    //                                       0,
    //                                       nsITimer::TYPE_ONE_SHOT);
  }
  return NS_OK;
}

nsresult
MonoMode::EndPan(PRBool aDelay)
{
  if (!aDelay) {
    sIsPanning = 0;
    if (mPanningTimer)
      mPanningTimer->Cancel();
    if (mEndPanTimer)
      mEndPanTimer->Cancel();

    if (mSpeedX)
      mSpeedX->ClearSpeed();
    if (mSpeedY)
      mSpeedY->ClearSpeed();
    mKineticPanStep = 0;
  } else {
    if (mEndPanTimer)
      mEndPanTimer->InitWithFuncCallback(MonoMode::EndPanDelayCallBack,
                                         (void*) this, 500,
                                         nsITimer::TYPE_ONE_SHOT);
  }

  return NS_OK;
}
nsresult
MonoMode::DoPan(nsIDOMEvent *aDOMEvent,
                nsIViewManager* aViewManager,
                PRInt32 aDx,
                PRInt32 aDy)
{
  // when kinetic pan, aDOMEvent is nsnull
  //NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(aViewManager, NS_ERROR_FAILURE);

  HelperFunctions::ScrollWindow(aDOMEvent, aViewManager, -aDx, -aDy);
  sIsPanning++;

  return NS_OK;
}

nsresult
MonoMode::ChangeFocus(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if (mFocusElement) {
  mFocusElement->Blur();
  }
  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  aDOMEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_TRUE(eventTarget, NS_ERROR_FAILURE);

  //HelperFunctions::ShowElementTree(aDOMEvent);

  mFocusElement = do_QueryInterface(eventTarget);
  if (mFocusElement) {
    mFocusElement->Focus();
  }
  return NS_OK;
}

SpeedManager*
MonoMode::GetSpeedMngX()
{
  return mSpeedX;
}

SpeedManager*
MonoMode::GetSpeedMngY()
{
  return mSpeedY;
}

PRInt32
MonoMode::GetKineticPanStep()
{
  // ++ for kinetic slow down
  return mKineticPanStep++;
}

//static
void
MonoMode::DoKineticPan(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  MonoMode *self = (MonoMode*)closure;
  NS_ENSURE_TRUE(self, );

  const double kDecayFactor = 0.95;
  const PRInt32 kCutOff = 2;
  const PRInt32 kSpeedLimit = 100;

  SpeedManager *spdmngX = self->GetSpeedMngX();
  SpeedManager *spdmngY = self->GetSpeedMngY();
  PRInt32 step = self->GetKineticPanStep();
  if(!spdmngX || !spdmngY || !self->mViewManager) {
    self->EndPan(PR_TRUE);
    return;
  }
  PRInt32 speedX = spdmngX->GetSpeed();
  PRInt32 speedY = spdmngY->GetSpeed();

  if (PR_ABS(speedX) > kSpeedLimit)
    speedX = speedX > 0 ? kSpeedLimit : -kSpeedLimit;

  if (PR_ABS(speedY) > kSpeedLimit)
    speedY = speedY > 0 ? kSpeedLimit : -kSpeedLimit;

  // We want these numbers to be whole and moving in the direction of zero.
  if (speedX < 0)
    speedX = (PRInt32)ceil(speedX);
  else
    speedX = (PRInt32)floor(speedX);

  if (speedY < 0)
    speedY = (PRInt32)ceil(speedY);
  else
    speedY = (PRInt32)floor(speedY);

  self->DoPan(nsnull, self->mViewManager, speedX, speedY);

  // slow down.
  speedX = PRInt32(speedX*(kDecayFactor - step/PAN_DECAY_STEP));
  speedY = PRInt32(speedY*(kDecayFactor - step/PAN_DECAY_STEP));

  spdmngX->SetSpeed(speedX);
  spdmngY->SetSpeed(speedY);

  // see if we should continue
  if (PR_ABS(speedX) > kCutOff || PR_ABS(speedY) > kCutOff)
    timer->InitWithFuncCallback(DoKineticPan,
                                (void*)self,
                                step*10,
                                nsITimer::TYPE_ONE_SHOT);
  else
    self->EndPan(PR_TRUE);//delay endpan
}

//static
void
MonoMode::EndPanDelayCallBack(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  MonoMode *self = (MonoMode*)closure;
  NS_ENSURE_TRUE(self, );
  self->EndPan();
}

nsresult
MonoMode::ModifierActionEvent(PRInt32 aModifier)
{
  nsIView* rootView = nsnull;
  NS_ENSURE_TRUE(mViewManager, NS_ERROR_FAILURE);
  mViewManager->GetRootView(rootView);
  NS_ENSURE_TRUE(rootView, NS_ERROR_FAILURE);
  nsEventStatus status;
  nsMouseEvent event(PR_TRUE, NS_CONTEXTMENU, rootView->GetWidget(), nsMouseEvent::eReal);
  event.refPoint.x = mFirstMousedownEvent.clientX;
  event.refPoint.y = mFirstMousedownEvent.clientY;
  event.button = nsMouseEvent::eRightButton;
  event.time = PR_IntervalNow();
  mViewManager->DispatchEvent(&event, &status);
  return NS_OK;
}

//static
void
MonoMode::DispatchMouseEvent(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  MonoMode *self = (MonoMode*)closure;
  NS_ENSURE_TRUE(self, );

  nsIView* rootView = nsnull;
  NS_ENSURE_TRUE(self->mViewManager, );
  self->mViewManager->GetRootView(rootView);
  NS_ENSURE_TRUE(rootView, );
  nsEventStatus status;
  nsMouseEvent event(PR_TRUE, NS_MOUSE_MOVE, rootView->GetWidget(), nsMouseEvent::eReal);
  event.refPoint.x = self->mFirstMousedownEvent.clientX * self->GetZoomLevel();
  event.refPoint.y = self->mFirstMousedownEvent.clientY * self->GetZoomLevel();
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
  event.message = NS_MOUSE_BUTTON_DOWN;
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
  event.message = NS_MOUSE_BUTTON_UP;
  event.time = PR_IntervalNow();
  self->mViewManager->DispatchEvent(&event, &status);
  self->mHasDispatchEvent = PR_TRUE;
}
