#include "hoverToolTip.h"
#include "nsIDOMEvent.h"
#include "nsITimer.h"
#include "nsComponentManagerUtils.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMNode.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLImageElement.h"

NS_IMPL_ISUPPORTS1(HoverToolTip, nsISupports)

HoverToolTip::HoverToolTip():
  mToolTipWin(nsnull),
  mToolTipLabel(nsnull),
  mShowTooTipTimer(nsnull),
  mShowToolTip(PR_FALSE)
{
  Init();
}

HoverToolTip::~HoverToolTip()
{
  HideToolTip();
  if(mToolTipWin)
    gtk_widget_destroy(mToolTipWin);
}

nsresult
HoverToolTip::Init()
{
  memset((void*)&mCurrentMouseArg, 0, sizeof(mCurrentMouseArg));
  memset((void*)&mLastMouseArg, 0, sizeof(mLastMouseArg));
  nsresult rv;
  mShowTooTipTimer = do_CreateInstance(NS_TIMER_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  return NS_OK;
}

nsresult
HoverToolTip::ShowToolTip(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  if(NS_FAILED(GetHref(aDOMEvent, mHerf))) {
    HideToolTip();
    return NS_OK;
  }
  HelperFunctions::UpdateMouseEventArg(aDOMEvent, &mCurrentMouseArg);

  if(!mToolTipWin && !mToolTipLabel)
    CreateToolTip();
  NS_ENSURE_TRUE(mToolTipWin, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(mToolTipLabel, NS_ERROR_FAILURE);

  if(!mShowToolTip) { // first show
    mShowToolTip = PR_TRUE;
    if (mShowTooTipTimer)
      mShowTooTipTimer->InitWithFuncCallback(HoverToolTip::ShowToolTipCallback,
                                             (void*)this,
                                             800,
                                             nsITimer::TYPE_ONE_SHOT);
  } else { // move ToolTip according to mouse position
    MoveToolTipWin();
  }
  return NS_OK;
}

nsresult
HoverToolTip::HideToolTip()
{
  mShowToolTip = PR_FALSE;

  NS_ENSURE_TRUE(mToolTipWin, NS_OK);
  gtk_widget_hide(mToolTipWin);

  NS_ENSURE_TRUE(mShowTooTipTimer, NS_OK);
  mShowTooTipTimer->Cancel();

  return NS_OK;
}

nsresult
HoverToolTip::CreateToolTip()
{
  mToolTipWin = gtk_window_new(GTK_WINDOW_POPUP);
  NS_ENSURE_TRUE(mToolTipWin, NS_ERROR_FAILURE);
  gtk_widget_set_app_paintable(mToolTipWin, TRUE);
  gtk_window_set_resizable(GTK_WINDOW(mToolTipWin), TRUE);
  // needed to get colors + fonts etc correctly
  gtk_widget_set_name(mToolTipWin, "gtk-tooltips");
  // realize the widget
  gtk_widget_realize(mToolTipWin);
  g_signal_connect(G_OBJECT(mToolTipWin),
                   "expose_event",
                   G_CALLBACK(HoverToolTip::PaintToolTip),
                   NULL);
  g_signal_connect(G_OBJECT(mToolTipWin),
                   "configure_event",
                   G_CALLBACK(TooltipWindowConfigured),
                   this);
  // set up the label for the tooltip
  mToolTipLabel = gtk_label_new("");
  NS_ENSURE_TRUE(mToolTipLabel, NS_ERROR_FAILURE);
  gtk_label_set_ellipsize(GTK_LABEL(mToolTipLabel), PANGO_ELLIPSIZE_END);
  gtk_container_add(GTK_CONTAINER(mToolTipWin), mToolTipLabel);
  gtk_container_set_border_width(GTK_CONTAINER(mToolTipWin), 3);

  return NS_OK;
}

nsresult
HoverToolTip::GetHref(nsIDOMEvent *aDOMEvent, nsString &aHerf)
{
  NS_ENSURE_TRUE(aDOMEvent, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  aDOMEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_TRUE(eventTarget, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget);
  nsCOMPtr<nsIDOMNode> parentNode;

  nsCOMPtr<nsIDOMHTMLImageElement> image = do_QueryInterface(eventNode);
  if (image) {
    image->GetTitle(aHerf);
    if (!aHerf.IsEmpty())
      return NS_OK;
    image->GetAlt(aHerf);
    if (!aHerf.IsEmpty())
      return NS_OK;
  }

  while (eventNode) {
    nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(eventNode);
    if (anchor) {
      anchor->GetTitle(aHerf);
      if (!aHerf.IsEmpty())
        return NS_OK;
      anchor->GetHref(aHerf);
      return IsAURI(aHerf);
    }
    eventNode->GetParentNode(getter_AddRefs(parentNode));
    if(eventNode == parentNode)
      break;
    eventNode = parentNode;
  }

  eventNode = do_QueryInterface(eventTarget);
  image = do_QueryInterface(eventNode);
  if (image) {
    image->GetSrc(aHerf);
    gchar *nameStart = g_strrstr(NS_ConvertUTF16toUTF8(aHerf).get(), "/");
    if (NS_SUCCEEDED(FixUpFileName(nameStart))) {
      aHerf.AssignLiteral(nameStart);
      return NS_OK;
    }
  }
  return NS_ERROR_FAILURE;
}

nsresult
HoverToolTip::IsAURI(const nsString &aHerf)
{
  const gchar *href = g_ascii_strdown(NS_ConvertUTF16toUTF8(aHerf).get(), -1);
  gboolean prefix = g_str_has_prefix(href, "ftp") || g_str_has_prefix(href, "http");
  NS_ENSURE_TRUE(prefix, NS_ERROR_FAILURE);
  return NS_OK;
}

nsresult
HoverToolTip::FixUpFileName(gchar* &aFileName)
{
  if (aFileName && strlen(aFileName) > 5) { // /x.xxx
    aFileName++;
    const gchar *pattern[] = { ".png", ".jpg", ".gif", "" };
    PRInt32 index = 0;
    gchar *nameEnd = nsnull;
    while (strlen(pattern[index]) > 0) {
      nameEnd = g_strrstr(g_ascii_strdown(aFileName, -1), pattern[index]);
      if (nameEnd) {
        nameEnd += strlen(pattern[index]);
        break;
      }
      index++;
    }
    NS_ENSURE_TRUE(nameEnd, NS_ERROR_FAILURE);
    aFileName[strlen(aFileName) - strlen(nameEnd)] = '\0';
    return NS_OK;
  }
  return NS_ERROR_FAILURE;
}

nsresult
HoverToolTip::MoveToolTipWin()
{
  memcpy((void*)&mLastMouseArg, (void*)&mCurrentMouseArg, sizeof(mCurrentMouseArg));
  NS_ENSURE_TRUE(mToolTipWin, NS_ERROR_FAILURE);
  const gint kMouseOffsetX = 20;
  const gint kMouseOffsetY = 20;
  gint toolTipWidth, toolTipHeight;
  gtk_window_get_size(GTK_WINDOW(mToolTipWin), &toolTipWidth, &toolTipHeight);

  GdkScreen *screen = gtk_window_get_screen(GTK_WINDOW(mToolTipWin));
  NS_ENSURE_TRUE(screen, NS_ERROR_FAILURE);
  gint screenWidth = gdk_screen_get_width(screen);
  gint screenHeight = gdk_screen_get_height(screen);
  if (mCurrentMouseArg.screenX + kMouseOffsetX + toolTipWidth < screenWidth &&
          mCurrentMouseArg.screenY - kMouseOffsetY - kMouseOffsetY > 0 )
    // show tooltip window on the up-right
    gtk_window_move(GTK_WINDOW(mToolTipWin),
                      mCurrentMouseArg.screenX + kMouseOffsetX,
                      mCurrentMouseArg.screenY - kMouseOffsetY - kMouseOffsetY);
  else if (mCurrentMouseArg.screenX + kMouseOffsetX + toolTipWidth < screenWidth)
    // show tooltip window on the right
    gtk_window_move(GTK_WINDOW(mToolTipWin),
                  mCurrentMouseArg.screenX + kMouseOffsetX,
                  mCurrentMouseArg.screenY);
  else { // show tooltip window on the left
    gtk_window_move(GTK_WINDOW(mToolTipWin),
                      mCurrentMouseArg.screenX - kMouseOffsetX - toolTipWidth,
                      mCurrentMouseArg.screenY);
  }
  return NS_OK;
}

//static
void
HoverToolTip::ShowToolTipCallback(nsITimer *timer, void *closure)
{
  NS_ENSURE_TRUE(timer, );
  NS_ENSURE_TRUE(closure, );
  HoverToolTip *self = (HoverToolTip*)closure;
  NS_ENSURE_TRUE(self, );
  NS_ENSURE_TRUE(self->mToolTipWin, );
  NS_ENSURE_TRUE(self->mToolTipLabel, );

  const gint kMaxWidth = 25;
  const gchar *href = NS_ConvertUTF16toUTF8(self->mHerf).get();
  NS_ENSURE_TRUE(href, );

  gint hreflen = strlen(href) > kMaxWidth ? kMaxWidth : strlen(href);
  gtk_label_set_width_chars(GTK_LABEL(self->mToolTipLabel), hreflen);
  gtk_label_set_text(GTK_LABEL(self->mToolTipLabel), NS_ConvertUTF16toUTF8(self->mHerf).get());
  gtk_window_resize(GTK_WINDOW(self->mToolTipWin), 1, 1); // auto adjust the window
  gtk_widget_show_all(GTK_WIDGET(self->mToolTipWin));

  return;
}

//static
gint
HoverToolTip::PaintToolTip(GtkWidget *aWindow)
{
  NS_ENSURE_TRUE(aWindow, FALSE);
  // draw tooltip style border around the text
  gtk_paint_flat_box(aWindow->style,
                     aWindow->window,
                     GTK_STATE_NORMAL,
                     GTK_SHADOW_OUT,
                     NULL,
                     aWindow,
                     "tooltip",
                     0, 0,
                     aWindow->allocation.width,
                     aWindow->allocation.height);
  return FALSE;
}

//static
gboolean
HoverToolTip::TooltipWindowConfigured(GtkWidget *widget,
                                      GdkEventConfigure *event,
                                      gpointer user_data)
{
  // Tooltip window has now calculated its size -> move the window accordingly
  HoverToolTip* self = (HoverToolTip*) user_data;
  NS_ENSURE_TRUE(self, FALSE);
  self->MoveToolTipWin();
  return FALSE;
}
