/*
  Copyright (C) 2008, 2009 Jiri Olsa <olsajiri@gmail.com>

  This file is part of the latrace.

  The latrace is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  The latrace is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the latrace (file COPYING).  If not, see 
  <http://www.gnu.org/licenses/>.
*/


%{

#define YYERROR_VERBOSE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"

int yylex (void);
void yyerror(const char *m);

static struct lt_config_shared *scfg;
static int struct_alive = 0;

#define ERROR1(str) \
do { \
	yyerror(str); \
	exit(1); \
} while(0)

#define ERROR2(str, ...) \
do { \
        char ebuf[1024]; \
        sprintf(ebuf, str, __VA_ARGS__); \
	yyerror(ebuf); \
	exit(1); \
} while(0)

static struct lt_list_head* get_list_head()
{
	struct lt_list_head *h;

 	if (NULL == (h = (struct lt_list_head*) malloc(sizeof(*h))))
		ERROR1("failed to allocate list head");

	lt_init_list_head(h);
	return h;
}

%}

%token NAME FILENAME STRUCT TYPEDEF INCLUDE END 

%union
{
	char *s;
	struct lt_arg *arg;
	struct lt_list_head *head;
}

%type <s>    NAME
%type <s>    FILENAME
%type <head> STRUCT_DEF
%type <head> ARGS
%type <arg>  DEF

%%
entry: 
entry struct_def 
| 
entry func_def
|
entry type_def
|
entry include_def
|
entry END
{
	if (lt_args_buf_close(scfg))
		return 0;
}
|
/* left blank intentionally */

/* struct definitions */
struct_def:
STRUCT NAME '{' STRUCT_DEF '}' ';'
{
	switch(lt_args_add_struct(scfg, $2, $4)) {
	case -1:
		ERROR2("failed to add struct %s\n", $2);
	case 1:
		ERROR2("struct limit reached(%d) - %s\n", LT_ARGS_DEF_STRUCT_NUM, $2);
	};

	/* force cration of the new list head */
	struct_alive = 0;
}

STRUCT_DEF:
STRUCT_DEF DEF ';'
{
	struct lt_arg *def     = $2;
	struct lt_list_head *h = $1;

	if (!struct_alive++)
		h = get_list_head();

	lt_list_add_tail(&def->args_list, h);
	$$ = h;
}
| /* left blank intentionally,
     XXX this could be done like the args_def, but user needs to be 
     able to create an empty structure, so thats why we play 
     with the global struct_alive thingie... 
     there could be better way probably */
{
}

type_def:
TYPEDEF NAME NAME ';'
{
	switch(lt_args_add_typedef(scfg, $2, $3)) {
	case -1: 
		ERROR2("unknown typedef - %s %s\n", $2, $3);
	case  1: 
		ERROR2("typedef alrady defined - %s %s\n", $2, $3);
	case  2: 
		ERROR2("typedef limit reached(%d) - %s %s\n", LT_ARGS_DEF_TYPEDEF_NUM, $2, $3);
	};
}

/* function definitions */
func_def:
DEF '(' ARGS ')' ';'
{
	struct lt_arg *arg = $1;

	if (lt_args_add_sym(scfg, arg, $3))
		ERROR2("failed to add symbol %s\n", arg->name);

	/* force cration of the new list head */
	$3 = NULL;
}

ARGS:
ARGS ',' DEF
{
	struct lt_arg *def     = $3;
	struct lt_list_head *h = $1;

	lt_list_add_tail(&def->args_list, h);
	$$ = h;
}
| DEF
{
	struct lt_list_head *h;
	struct lt_arg *def = $1;

	h = get_list_head();
	lt_list_add_tail(&def->args_list, h);
	$$ = h;
}
| NAME
{
	$$ = get_list_head();
}
| /* left intentionaly blank */
{
	$$ = get_list_head();
}

DEF:
NAME NAME
{
	struct lt_arg *arg;

	if (NULL == (arg = lt_args_getarg(scfg, $1, $2, 0, 1)))
		ERROR2("unknown argument type - %s\n", $1);

	$$ = arg;
}
|
NAME '*' NAME
{
	struct lt_arg *arg;
	if (NULL == (arg = lt_args_getarg(scfg, $1, $3, 1, 1)))
		ERROR2("unknown argument type - %s\n", $1);

	$$ = arg;
}
|
STRUCT NAME NAME
{
	struct lt_arg *arg;
	if (NULL == (arg = lt_args_getarg(scfg, $2, $3, 0, 1)))
		ERROR2("unknown argument type - %s\n", $2);

	$$ = arg;
}
|
STRUCT NAME '*' NAME
{
	struct lt_arg *arg;
	if (NULL == (arg = lt_args_getarg(scfg, $2, $4, 1, 1)))
		ERROR2("unknown argument type - %s\n", $2);

	$$ = arg;
}

include_def: INCLUDE '"' FILENAME '"'
{
	if (lt_args_buf_open(scfg, $3))
		ERROR1("failed to process include");
}

%%

int lt_args_parse_init(struct lt_config_shared *cfg)
{
	scfg = cfg;
	return 0;
}
