/* GStreamer
 *
 * Copyright (C) 2001-2002 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *               2006 Edgard Lima <edgard.lima@indt.org.br>
 *               2008 Nokia Corporation <multimedia@maemo.org>
 *
 * gstv4l2camsrc.c: Video4Linux2 source element
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * SECTION:element-v4l2camsrc
 *
 * <refsect2>
 * v4l2camsrc can be used to capture video from v4l2 devices, like webcams and tv cards.
 * <title>Example launch lines</title>
 * <para>
 * <programlisting>
 * gst-launch v4l2camsrc ! xvimagesink
 * </programlisting>
 * This pipeline shows the video captured from /dev/video0 tv card and for
 * webcams.
 * </para>
 * <para>
 * <programlisting>
 * gst-launch-0.10 v4l2camsrc ! jpegdec ! xvimagesink
 * </programlisting>
 * This pipeline shows the video captured from a webcam that delivers jpeg
 * images.
 * </para>
 * </refsect2>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "v4l2camsrc_calls.h"
#include "gstv4l2camcolorbalance.h"
#include "gstv4l2camvidorient.h"
#include "gstv4l2camphotoiface.h"
#include "driver_wrapper.h"

static const GstElementDetails gst_v4l2camsrc_details =
GST_ELEMENT_DETAILS ("V4L2 Image Capture Source",
    "Source/Image",
    "Captures frames from a video4linux2 device",
    "Ronald Bultje <rbultje@ronald.bitfreak.net>,"
    " Edgard Lima <edgard.lima@indt.org.br>");

GST_DEBUG_CATEGORY (v4l2camsrc_debug);
#define GST_CAT_DEFAULT v4l2camsrc_debug

#define DEFAULT_PROP_DEVICE       "/dev/video0"
#define DEFAULT_PROP_DEVICE_NAME  NULL
#define DEFAULT_PROP_DRIVER_NAME  NULL
#define DEFAULT_PROP_DEVICE_FD    -1

#define DEFAULT_PROP_ALWAYS_COPY        FALSE

/* Custom GstMessage name that will be sent to GstBus when application has
   requested the captured raw bayer images to be published */
#define GST_V4L2CAMSRC_PUBLISH_RAW "raw-image"

/* Message that is sent when capturing process is about to start */
#define GST_V4L2CAMSRC_CAPTURE_START "photo-capture-start"
#define GST_V4L2CAMSRC_CAPTURE_END "photo-capture-end"

enum
{
  PROP_0,
  PROP_DEVICE,
  PROP_DEVICE_NAME,
  PROP_DEVICE_FD,
  PROP_QUEUE_SIZE,
  PROP_ALWAYS_COPY,
  PROP_CAPTURE_MODE,
  PROP_DRIVER_NAME
};

/* Only support YUYV output */
static const guint32 gst_v4l2camsrc_formats[] = {
  V4L2_PIX_FMT_YUYV,
};

GST_IMPLEMENT_V4L2CAMSRC_COLOR_BALANCE_METHODS (GstV4l2CamSrc, gst_v4l2cam);
GST_IMPLEMENT_V4L2CAMSRC_PHOTO_METHODS (GstV4l2CamSrc, gst_v4l2cam);
GST_IMPLEMENT_V4L2CAMSRC_VIDORIENT_METHODS (GstV4l2CamSrc, gst_v4l2cam);

static gboolean
gst_v4l2camsrc_iface_supported (GstImplementsInterface * iface, GType iface_type)
{
  GstV4l2CamSrc * v4l2camsrc = GST_V4L2CAMSRC (iface);

  g_assert (iface_type == GST_TYPE_PHOTOGRAPHY ||
      iface_type == GST_TYPE_COLOR_BALANCE ||
      iface_type == GST_TYPE_VIDEO_ORIENTATION);

  /* We implement GstPhotography in NULL state as well */
  if (iface_type == GST_TYPE_PHOTOGRAPHY)
    return TRUE;
  else if (v4l2camsrc->video_fd == -1)
    return FALSE;

  return TRUE;
}

static void
gst_v4l2camsrc_interface_init (GstImplementsInterfaceClass * klass)
{
  /*
   * default virtual functions 
   */
  klass->supported = gst_v4l2camsrc_iface_supported;
}

void
gst_v4l2camsrc_init_interfaces (GType type)
{
  static const GInterfaceInfo v4l2iface_info = {
    (GInterfaceInitFunc) gst_v4l2camsrc_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo v4l2camsrc_photo_info = {
      (GInterfaceInitFunc) gst_v4l2cam_photo_interface_init,
       NULL,
       NULL,
  };
  static const GInterfaceInfo v4l2camsrc_colorbalance_info = {
    (GInterfaceInitFunc) gst_v4l2cam_color_balance_interface_init,
    NULL,
    NULL,
  };
  static const GInterfaceInfo v4l2camsrc_videoorientation_info = {
    (GInterfaceInitFunc) gst_v4l2cam_video_orientation_interface_init,
    NULL,
    NULL,
  };

  g_type_add_interface_static (type,
      GST_TYPE_IMPLEMENTS_INTERFACE, &v4l2iface_info);
  g_type_add_interface_static (type,
      GST_TYPE_PHOTOGRAPHY, &v4l2camsrc_photo_info);
  g_type_add_interface_static (type,
      GST_TYPE_COLOR_BALANCE, &v4l2camsrc_colorbalance_info);
  g_type_add_interface_static (type,
      GST_TYPE_VIDEO_ORIENTATION, &v4l2camsrc_videoorientation_info);
}


GType
gst_v4l2camsrc_capture_mode_get_type (void)
{
  static GType gst_v4l2camsrc_capture_mode_type = 0;
  static GEnumValue gst_v4l2camsrc_capture_modes[] = {
    {GST_V4L2CAMSRC_CAPTURE_MODE_VIEWFINDER, "Viewfinder mode", "viewfinder"},
    {GST_V4L2CAMSRC_CAPTURE_MODE_STILL, "Still image capture mode", "still"},
    {GST_V4L2CAMSRC_CAPTURE_MODE_VIDEO, "Video capturing mode", "video"},
    {0, NULL, NULL},
  };

  if (G_UNLIKELY (!gst_v4l2camsrc_capture_mode_type)) {
    gst_v4l2camsrc_capture_mode_type =
        g_enum_register_static ("GstV4L2CamSrcCaptureMode",
                                gst_v4l2camsrc_capture_modes);
  }
  return gst_v4l2camsrc_capture_mode_type;
}


GST_BOILERPLATE_FULL (GstV4l2CamSrc, gst_v4l2camsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, gst_v4l2camsrc_init_interfaces);

static void gst_v4l2camsrc_dispose (GObject * object);

/* element methods */
static GstStateChangeReturn
gst_v4l2camsrc_change_state (GstElement * element, GstStateChange transition);

/* basesrc methods */
static gboolean gst_v4l2camsrc_start (GstBaseSrc * src);

static gboolean gst_v4l2camsrc_unlock (GstBaseSrc * src);

static gboolean gst_v4l2camsrc_unlock_stop (GstBaseSrc * src);

static gboolean gst_v4l2camsrc_stop (GstBaseSrc * src);

static gboolean gst_v4l2camsrc_set_caps (GstBaseSrc * src, GstCaps * caps);

static GstCaps *gst_v4l2camsrc_get_caps (GstBaseSrc * src);

static gboolean gst_v4l2camsrc_query (GstBaseSrc * bsrc, GstQuery * query);

static GstFlowReturn gst_v4l2camsrc_create (GstPushSrc * src, GstBuffer ** out);

static void gst_v4l2camsrc_fixate (GstBaseSrc * basesrc, GstCaps * caps);

static gboolean gst_v4l2camsrc_negotiate (GstBaseSrc * basesrc);

static void gst_v4l2camsrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_v4l2camsrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static GstStaticPadTemplate gst_v4l2camsrc_src_template =
    GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-yuv, "
        "format = (fourcc) UYVY, "
        "framerate = (fraction) [ 0, 100 ], "
        "width = (int) [ 1, MAX ], "
        "height = (int) [ 1, MAX ];"
        "video/x-raw-yuv, "
        "format = (fourcc) YUY2, "
        "framerate = (fraction) [ 0, 100 ], "
        "width = (int) [ 1, MAX ], "
        "height = (int) [ 1, MAX ]"));

/*
 */
static void
gst_v4l2camsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (gstelement_class, &gst_v4l2camsrc_details);

  gst_element_class_add_pad_template (gstelement_class,
       gst_static_pad_template_get (&gst_v4l2camsrc_src_template));
}

/*
 */
static void
gst_v4l2camsrc_class_init (GstV4l2CamSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *element_class;
  GstBaseSrcClass *basesrc_class;
  GstPushSrcClass *pushsrc_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);
  basesrc_class = GST_BASE_SRC_CLASS (klass);
  pushsrc_class = GST_PUSH_SRC_CLASS (klass);

  gobject_class->dispose = gst_v4l2camsrc_dispose;
  gobject_class->set_property = gst_v4l2camsrc_set_property;
  gobject_class->get_property = gst_v4l2camsrc_get_property;

  element_class->change_state = gst_v4l2camsrc_change_state;

  g_object_class_install_property (gobject_class, PROP_DEVICE,
      g_param_spec_string ("device", "Device", "Device location",
          DEFAULT_PROP_DEVICE, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, PROP_DEVICE_NAME,
      g_param_spec_string ("device-name", "Device name",
          "Name of the device", DEFAULT_PROP_DEVICE_NAME, G_PARAM_READABLE));
  g_object_class_install_property (gobject_class, PROP_DEVICE_FD,
      g_param_spec_int ("device-fd", "File descriptor",
          "File descriptor of the device", -1, G_MAXINT, DEFAULT_PROP_DEVICE_FD,
          G_PARAM_READABLE));
  g_object_class_install_property (gobject_class, PROP_QUEUE_SIZE,
      g_param_spec_uint ("queue-size", "Queue size",
          "Number of buffers to be enqueud in the driver",
          GST_V4L2CAMSRC_MIN_BUFFERS, GST_V4L2CAMSRC_MAX_BUFFERS,
          GST_V4L2CAMSRC_DEFAULT_BUFFERS, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, PROP_ALWAYS_COPY,
      g_param_spec_boolean ("always-copy", "Always Copy",
          "If the buffer will or not be used directly from mmap",
          DEFAULT_PROP_ALWAYS_COPY, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, PROP_CAPTURE_MODE,
       g_param_spec_enum ("capture-mode", "Capturing mode",
           "Defines what kind of capturing mode to be used",
           GST_TYPE_V4L2CAMSRC_CAPTURE_MODE,
           GST_V4L2CAMSRC_CAPTURE_MODE_VIEWFINDER,
           G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, PROP_DRIVER_NAME,
      g_param_spec_string ("driver-name", "Camera Device driver name",
          "Name of the camera device driver to be loaded dynamically",
          DEFAULT_PROP_DRIVER_NAME, G_PARAM_READWRITE));

  basesrc_class->get_caps = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_get_caps);
  basesrc_class->set_caps = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_set_caps);
  basesrc_class->start = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_start);
  basesrc_class->unlock = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_unlock);
  basesrc_class->unlock_stop = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_unlock_stop);
  basesrc_class->stop = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_stop);
  basesrc_class->query = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_query);
  basesrc_class->fixate = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_fixate);
  basesrc_class->negotiate = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_negotiate);

  pushsrc_class->create = GST_DEBUG_FUNCPTR (gst_v4l2camsrc_create);
}

/*
 */
static void
gst_v4l2camsrc_init (GstV4l2CamSrc * v4l2camsrc, GstV4l2CamSrcClass * klass)
{
  /* Initialize device-specific wrapper layer */
  v4l2camsrc->driver = gst_v4l2camsrc_driver_wrapper_load (v4l2camsrc);

  /* number of buffers requested */
  v4l2camsrc->num_buffers = GST_V4L2CAMSRC_DEFAULT_BUFFERS;
  v4l2camsrc->always_copy = DEFAULT_PROP_ALWAYS_COPY;

  v4l2camsrc->state_lock = g_mutex_new ();

  v4l2camsrc->formats = NULL;

  v4l2camsrc->is_capturing = FALSE;

  gst_base_src_set_format (GST_BASE_SRC (v4l2camsrc), GST_FORMAT_TIME);
  gst_base_src_set_live (GST_BASE_SRC (v4l2camsrc), TRUE);

  v4l2camsrc->fps_d = 0;
  v4l2camsrc->fps_n = 0;

  v4l2camsrc->videodev = g_strdup (DEFAULT_PROP_DEVICE);
  v4l2camsrc->video_fd = DEFAULT_PROP_DEVICE_FD;
  v4l2camsrc->poll = gst_poll_new (TRUE);
  v4l2camsrc->driver_name = NULL;
  v4l2camsrc->buffer = NULL;
  v4l2camsrc->colors = NULL;
  v4l2camsrc->crop_supported = FALSE;
  v4l2camsrc->max_zoom_factor = 1.0;

  /* Photo interface */
  v4l2camsrc->photoconf.zoom = 1.0;
  v4l2camsrc->photoconf.ev_compensation = 0.0;
  v4l2camsrc->photoconf.exposure = 0;   /* 0 = auto */
  v4l2camsrc->photoconf.aperture = 0;   /* 0 = auto */
  v4l2camsrc->photoconf.iso_speed = 0;  /* 0 = auto */
  v4l2camsrc->photoconf.wb_mode = GST_PHOTOGRAPHY_WB_MODE_AUTO;
  v4l2camsrc->photoconf.scene_mode = GST_PHOTOGRAPHY_SCENE_MODE_MANUAL;
  v4l2camsrc->photoconf.flash_mode = GST_PHOTOGRAPHY_FLASH_MODE_AUTO;
  v4l2camsrc->photoconf.tone_mode = GST_PHOTOGRAPHY_COLOUR_TONE_MODE_NORMAL;

  v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;
  v4l2camsrc->shake_risk = GST_PHOTOGRAPHY_SHAKE_RISK_LOW;
  v4l2camsrc->capture_mode = GST_V4L2CAMSRC_CAPTURE_MODE_VIEWFINDER;

  v4l2camsrc->capture_w = 0;
  v4l2camsrc->capture_h = 0;
  v4l2camsrc->capture_fps_n = 0;
  v4l2camsrc->capture_fps_d = 0;

  v4l2camsrc->requested_af_mode = AF_NONE_REQUESTED;
}

/*
 */
static void
gst_v4l2camsrc_dispose (GObject * object)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (object);

  if (v4l2camsrc->formats) {
    gst_v4l2camsrc_clear_format_list (v4l2camsrc);
  }

  if (v4l2camsrc->vf_caps) {
    gst_caps_unref (v4l2camsrc->vf_caps);
  }
  if (v4l2camsrc->probed_caps) {
    gst_caps_unref (v4l2camsrc->probed_caps);
    v4l2camsrc->probed_caps = NULL;
  }

  if (v4l2camsrc->videodev) {
    g_free (v4l2camsrc->videodev);
    v4l2camsrc->videodev = NULL;
  }

  if (v4l2camsrc->poll) {
    gst_poll_free (v4l2camsrc->poll);
  }

  if (v4l2camsrc->state_lock) {
    g_mutex_free (v4l2camsrc->state_lock);
    v4l2camsrc->state_lock = NULL;
  }

  if (v4l2camsrc->driver) {
    gst_v4l2camsrc_driver_wrapper_unload (v4l2camsrc->driver);
    v4l2camsrc->driver = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

/*
 */
static void
gst_v4l2camsrc_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (object);

  switch (prop_id) {
    case PROP_DEVICE:
      g_free (v4l2camsrc->videodev);
      v4l2camsrc->videodev = g_value_dup_string (value);
      break;
    case PROP_QUEUE_SIZE:
      v4l2camsrc->num_buffers = g_value_get_uint (value);
      break;
    case PROP_ALWAYS_COPY:
      v4l2camsrc->always_copy = g_value_get_boolean (value);
      break;
    case PROP_CAPTURE_MODE:
      v4l2camsrc->capture_mode = g_value_get_enum (value);

      if (v4l2camsrc->driver) {
        if (v4l2camsrc->capture_mode == GST_V4L2CAMSRC_CAPTURE_MODE_VIDEO &&
            GST_V4L2CAMSRC_IS_ACTIVE (v4l2camsrc))
        {
          /* Turn on privacy light */
          v4l2camsrc->driver->set_privacy_light (v4l2camsrc->driver, TRUE);
        }
        else if (v4l2camsrc->capture_mode ==
            GST_V4L2CAMSRC_CAPTURE_MODE_VIEWFINDER)
        {
          /* Turn off privacy light */
          v4l2camsrc->driver->set_privacy_light (v4l2camsrc->driver, FALSE);
        }
        v4l2camsrc->driver->set_capture_mode (v4l2camsrc->driver,
                                              v4l2camsrc->capture_mode);
      }
      break;
    case PROP_DRIVER_NAME:
      if (!GST_V4L2CAMSRC_IS_ACTIVE (v4l2camsrc)) {
        gchar *tmp = g_value_dup_string (value);

        if (g_strcmp0 (v4l2camsrc->driver_name, tmp)) {
          gst_v4l2camsrc_driver_wrapper_unload (v4l2camsrc->driver);
          g_free (v4l2camsrc->driver_name);
          v4l2camsrc->driver_name = tmp;
          v4l2camsrc->driver = gst_v4l2camsrc_driver_wrapper_load (v4l2camsrc);
          GST_DEBUG_OBJECT (v4l2camsrc, "new camera driver name: %s",
                            v4l2camsrc->driver_name);
        }
        else {
          g_free (tmp);
        }
      }
      else {
        GST_WARNING ("Can't change driver name when device is active");
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

/*
 */
static void
gst_v4l2camsrc_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (object);

  switch (prop_id) {
    case PROP_DEVICE:
      g_value_set_string (value, v4l2camsrc->videodev);
      break;
    case PROP_DEVICE_NAME:
    {
      const guchar *new = NULL;

      if (GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc)) {
        new = v4l2camsrc->vcap.card;
      } else if (gst_v4l2camsrc_open (v4l2camsrc)) {
        new = v4l2camsrc->vcap.card;
        gst_v4l2camsrc_close (v4l2camsrc);
        gst_v4l2camsrc_empty_lists (v4l2camsrc);
      }
      g_value_set_string (value, (gchar *) new);
      break;
    }
    case PROP_DEVICE_FD:
    {
      if (GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc))
        g_value_set_int (value, v4l2camsrc->video_fd);
      else
        g_value_set_int (value, DEFAULT_PROP_DEVICE_FD);
      break;
    }
    case PROP_QUEUE_SIZE:
      g_value_set_uint (value, v4l2camsrc->num_buffers);
      break;
    case PROP_ALWAYS_COPY:
      g_value_set_boolean (value, v4l2camsrc->always_copy);
      break;
    case PROP_CAPTURE_MODE:
      g_value_set_enum (value, v4l2camsrc->capture_mode);
      break;
    case PROP_DRIVER_NAME:
      g_value_set_string (value, v4l2camsrc->driver_name);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

/* this function is a bit of a last resort */
static void
gst_v4l2camsrc_fixate (GstBaseSrc * basesrc, GstCaps * caps)
{
  GstStructure *structure;

  gint i;

  GST_DEBUG_OBJECT (basesrc, "fixating caps %" GST_PTR_FORMAT, caps);

  for (i = 0; i < gst_caps_get_size (caps); ++i) {
    const GValue *v;

    structure = gst_caps_get_structure (caps, i);

    /* FIXME such sizes? we usually fixate to something in the 320x200
     * range... */
    /* We are fixating to greatest possble size (limited to
       GST_V4L2CAMSRC_MAX_SIZE) and the maximum framerate resolution for
       that size */
    gst_structure_fixate_field_nearest_int (structure, "width",
        GST_V4L2CAMSRC_MAX_SIZE);
    gst_structure_fixate_field_nearest_int (structure, "height",
        GST_V4L2CAMSRC_MAX_SIZE);
    gst_structure_fixate_field_nearest_fraction (structure, "framerate",
        G_MAXINT, 1);

    v = gst_structure_get_value (structure, "format");
    if (v && G_VALUE_TYPE (v) != GST_TYPE_FOURCC) {
      guint32 fourcc;

      g_return_if_fail (G_VALUE_TYPE (v) == GST_TYPE_LIST);

      fourcc = gst_value_get_fourcc (gst_value_list_get_value (v, 0));
      gst_structure_set (structure, "format", GST_TYPE_FOURCC, fourcc, NULL);
    }
  }

  GST_DEBUG_OBJECT (basesrc, "fixated caps %" GST_PTR_FORMAT, caps);
}

/*
 */
static gboolean
gst_v4l2camsrc_negotiate (GstBaseSrc * basesrc)
{
  GstCaps *thiscaps;

  GstCaps *caps = NULL;

  GstCaps *peercaps = NULL;

  gboolean result = FALSE;

  /* first see what is possible on our source pad */
  thiscaps = gst_pad_get_caps (GST_BASE_SRC_PAD (basesrc));
  GST_DEBUG_OBJECT (basesrc, "caps of src: %" GST_PTR_FORMAT, thiscaps);
  /* nothing or anything is allowed, we're done */
  if (thiscaps == NULL || gst_caps_is_any (thiscaps))
    goto no_nego_needed;

  /* get the peer caps */
  peercaps = gst_pad_peer_get_caps (GST_BASE_SRC_PAD (basesrc));
  GST_DEBUG_OBJECT (basesrc, "caps of peer: %" GST_PTR_FORMAT, peercaps);
  if (peercaps && !gst_caps_is_any (peercaps)) {
    GstCaps *icaps = NULL;
    int i;

    /* Prefer the first caps we are compatible with that the peer proposed */
    for (i = 0; i < gst_caps_get_size (peercaps); i++) {
      /* get intersection */
      GstCaps *ipcaps = gst_caps_copy_nth (peercaps, i);

      GST_DEBUG_OBJECT (basesrc, "peer: %" GST_PTR_FORMAT, ipcaps);

      icaps = gst_caps_intersect (thiscaps, ipcaps);
      gst_caps_unref (ipcaps);

      if (!gst_caps_is_empty (icaps))
        break;

      gst_caps_unref (icaps);
      icaps = NULL;
    }

    GST_DEBUG_OBJECT (basesrc, "intersect: %" GST_PTR_FORMAT, icaps);
    if (icaps) {
      /* If there are multiple intersections pick the one with the smallest
       * resolution strictly bigger then the first peer caps */
      if (gst_caps_get_size (icaps) > 1) {
        GstStructure *s = gst_caps_get_structure (peercaps, 0);

        int best = 0;

        int twidth, theight;

        int width = G_MAXINT, height = G_MAXINT;

        if (gst_structure_get_int (s, "width", &twidth)
            && gst_structure_get_int (s, "height", &theight)) {

          /* Walk the structure backwards to get the first entry of the
           * smallest resolution bigger (or equal to) the preferred resolution)
           */
          for (i = gst_caps_get_size (icaps) - 1; i >= 0; i--) {
            GstStructure *is = gst_caps_get_structure (icaps, i);

            int w, h;

            if (gst_structure_get_int (is, "width", &w)
                && gst_structure_get_int (is, "height", &h)) {
              if (w >= twidth && w <= width && h >= theight && h <= height) {
                width = w;
                height = h;
                best = i;
              }
            }
          }
        }

        caps = gst_caps_copy_nth (icaps, best);
        gst_caps_unref (icaps);
      } else {
        caps = icaps;
      }
    }
    gst_caps_unref (thiscaps);
    gst_caps_unref (peercaps);
  } else {
    /* no peer or peer have ANY caps, work with our own caps then */
    caps = thiscaps;
  }
  if (caps) {
    caps = gst_caps_make_writable (caps);
    gst_caps_truncate (caps);

    /* now fixate */
    if (!gst_caps_is_empty (caps)) {
      gst_pad_fixate_caps (GST_BASE_SRC_PAD (basesrc), caps);
      GST_DEBUG_OBJECT (basesrc, "fixated to: %" GST_PTR_FORMAT, caps);

      if (gst_caps_is_any (caps)) {
        /* hmm, still anything, so element can do anything and
         * nego is not needed */
        result = TRUE;
      } else if (gst_caps_is_fixed (caps)) {
        /* yay, fixed caps, use those then */
        result = gst_pad_set_caps (GST_BASE_SRC_PAD (basesrc), caps);
        GST_DEBUG_OBJECT (basesrc, "Set caps returned: %d", result);
      }
    }
    gst_caps_unref (caps);
  }
  return result;

no_nego_needed:
  {
    GST_DEBUG_OBJECT (basesrc, "no negotiation needed");
    if (thiscaps)
      gst_caps_unref (thiscaps);
    return TRUE;
  }
}


/*
 */
static struct v4l2_fmtdesc *
gst_v4l2camsrc_get_format_from_fourcc (GstV4l2CamSrc * v4l2camsrc, guint32 fourcc)
{
  struct v4l2_fmtdesc *fmt;

  GSList *walk;

  if (fourcc == 0)
    return NULL;

  walk = v4l2camsrc->formats;
  while (walk) {
    fmt = (struct v4l2_fmtdesc *) walk->data;
    GST_INFO_OBJECT (v4l2camsrc, "trying fourcc %" GST_FOURCC_FORMAT,
      GST_FOURCC_ARGS (fmt->pixelformat));
    if (fmt->pixelformat == fourcc)
      return fmt;
    walk = g_slist_next (walk);
  }

  GST_WARNING_OBJECT (v4l2camsrc, "can't get format from fourcc %" GST_FOURCC_FORMAT,
    GST_FOURCC_ARGS (fourcc));

  return NULL;
}


static GstStructure *
gst_v4l2camsrc_v4l2fourcc_to_structure (guint32 fourcc)
{
  GstStructure *structure = NULL;

  switch (fourcc) {
    case V4L2_PIX_FMT_YUYV:
    case V4L2_PIX_FMT_UYVY:
    {
      guint32 fcc = 0;

      switch (fourcc) {
        case V4L2_PIX_FMT_YUYV:
          fcc = GST_MAKE_FOURCC ('Y', 'U', 'Y', '2');
          break;
        case V4L2_PIX_FMT_UYVY:
          fcc = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
          break;
        default:
          g_assert_not_reached ();
          break;
      }
      structure = gst_structure_new ("video/x-raw-yuv",
                                     "format", GST_TYPE_FOURCC, fcc, NULL);
      break;
    }
#ifdef V4L2_PIX_FMT_SGRBG10
    case V4L2_PIX_FMT_SGRBG10:
      GST_DEBUG ("Ignoring bayer formats");
      break;
#endif
    default:
      GST_DEBUG ("Unknown fourcc 0x%08x %" GST_FOURCC_FORMAT,
                 fourcc, GST_FOURCC_ARGS (fourcc));
      break;
  }

  return structure;
}

/*
 */
static GstCaps *
gst_v4l2camsrc_get_caps (GstBaseSrc * src)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);
  GstStructure *template;
  GstCaps *ret;
  GSList *walk;

  if (!GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc)) {
    /* FIXME: copy? */
    return
        gst_caps_copy (gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD
            (v4l2camsrc)));
  }

  if (v4l2camsrc->probed_caps)
    return gst_caps_ref (v4l2camsrc->probed_caps);

  if (!v4l2camsrc->formats)
    gst_v4l2camsrc_fill_format_list (v4l2camsrc);

  ret = gst_caps_new_empty ();

  for (walk = v4l2camsrc->formats; walk; walk = walk->next) {
    struct v4l2_fmtdesc *format;

    format = (struct v4l2_fmtdesc *) walk->data;

    template = gst_v4l2camsrc_v4l2fourcc_to_structure (format->pixelformat);

    if (template) {
      GstCaps *tmp;

      tmp = gst_v4l2camsrc_probe_caps_for_format (v4l2camsrc,
          format->pixelformat, template);
      if (tmp)
        gst_caps_append (ret, tmp);

      gst_structure_free (template);
    } else {
      GST_DEBUG_OBJECT (v4l2camsrc, "unknown format %u", format->pixelformat);
    }
  }

  v4l2camsrc->probed_caps = gst_caps_ref (ret);

  GST_INFO_OBJECT (v4l2camsrc, "probed caps: %" GST_PTR_FORMAT, ret);

  return ret;
}

/* collect data for the given caps
 * @caps: given input caps
 * @format: location for the v4l format
 * @w/@h: location for width and height
 * @fps_n/@fps_d: location for framerate
 * @size: location for expected size of the frame or 0 if unknown
 * @capture: location for capture mode
 */
static gboolean
gst_v4l2camsrc_get_caps_info (GstV4l2CamSrc * v4l2camsrc, GstCaps * caps,
    struct v4l2_fmtdesc **format, gint * w, gint * h, guint * fps_n,
    guint * fps_d, guint * size)
{
  GstStructure *structure;
  const GValue *framerate;
  guint32 fourcc;
  guint outsize;

  /* default unknown values */
  fourcc = 0;
  outsize = 0;

  structure = gst_caps_get_structure (caps, 0);

  if (!gst_structure_get_int (structure, "width", w))
    return FALSE;

  if (!gst_structure_get_int (structure, "height", h))
    return FALSE;

  framerate = gst_structure_get_value (structure, "framerate");
  if (!framerate)
    return FALSE;

  *fps_n = gst_value_get_fraction_numerator (framerate);
  *fps_d = gst_value_get_fraction_denominator (framerate);

  gst_structure_get_fourcc (structure, "format", &fourcc);

  switch (fourcc) {
    case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
      fourcc = V4L2_PIX_FMT_YUYV;
      break;
    default:
      fourcc = V4L2_PIX_FMT_UYVY;
      break;
  }
  outsize = (GST_ROUND_UP_2 (*w) * 2) * *h;

  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE) {
    v4l2camsrc->driver->set_capture_fmt (v4l2camsrc->driver, fourcc, outsize,
                                         &fourcc, &outsize);
    v4l2camsrc->driver->set_capture_res (v4l2camsrc->driver, *w, *h);
  }

  *format = gst_v4l2camsrc_get_format_from_fourcc (v4l2camsrc, fourcc);
  if(!*format)
    return FALSE;

  *size = outsize;

  return TRUE;
}

/*
 */
static gboolean
gst_v4l2camsrc_init_from_caps (GstV4l2CamSrc *v4l2camsrc, GstCaps * caps)
{
  gint w = 0, h = 0;
  struct v4l2_fmtdesc *format = NULL;
  guint fps_n, fps_d;
  guint size;
  gboolean init_device = TRUE;

  /* make sure we stop capturing and dealloc buffers */
  if (GST_V4L2CAMSRC_IS_ACTIVE (v4l2camsrc)) {
    /* Tell the driver to stop processing during the resolution change */
    v4l2camsrc->driver->stop (v4l2camsrc->driver);
    /* both will throw an element-error on failure */
    if (!gst_v4l2camsrc_capture_stop (v4l2camsrc))
      return FALSE;
    if (!gst_v4l2camsrc_capture_deinit (v4l2camsrc))
      return FALSE;
  }

  /* we want our own v4l2 type of fourcc codes */
  if (!gst_v4l2camsrc_get_caps_info (v4l2camsrc, caps, &format, &w, &h, &fps_n,
       &fps_d, &size))
  {
    GST_DEBUG_OBJECT (v4l2camsrc,
        "can't get capture format from caps %" GST_PTR_FORMAT, caps);
    return FALSE;
  }

  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE) {
    GST_DEBUG_OBJECT (v4l2camsrc, "capture flag found, calling start_capture");
    v4l2camsrc->driver->start_capture (v4l2camsrc->driver, &init_device);
  }

  v4l2camsrc->current_w = w;
  v4l2camsrc->current_h = h;

  if (init_device) {
    GST_DEBUG_OBJECT (v4l2camsrc, "trying to set_capture %dx%d at %d/%d fps, "
        "format %s", w, h, fps_n, fps_d, format->description);

    if (!gst_v4l2camsrc_set_capture (v4l2camsrc, format->pixelformat, w, h,
         fps_n, fps_d))
      /* error already posted */
      return FALSE;

    /* Only start the driver when not in HQ capture mode, since in HQ mode */
    /* we have already called start_capture() above */
    if (v4l2camsrc->photo_capture_phase != GST_V4L2PHOTO_CAPTURE) {
      if (!v4l2camsrc->driver->start (v4l2camsrc->driver))
      {
        GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
                           ("Failed to start driver module"), (NULL));

        return FALSE;
      }
      if (!v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
              &v4l2camsrc->photoconf, FALSE))
      {
        GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
                           ("Failed to configure driver module"), (NULL));

        return FALSE;
      }
    }

    /* now store the expected output size */
    v4l2camsrc->frame_byte_size = size;

    if (!gst_v4l2camsrc_capture_init (v4l2camsrc, caps))
      return FALSE;

    if (!gst_v4l2camsrc_capture_start (v4l2camsrc))
      return FALSE;
  }

  return TRUE;
}

/*
 */
static gboolean
gst_v4l2camsrc_set_caps (GstBaseSrc * src, GstCaps * caps)
{
  GstV4l2CamSrc *v4l2camsrc;

  v4l2camsrc = GST_V4L2CAMSRC (src);

  GST_DEBUG_OBJECT (v4l2camsrc, "set_caps called");

  /* if we're not open, punt -- we'll get setcaps'd later via negotiate */
  if (!GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc))
    return FALSE;

  return gst_v4l2camsrc_init_from_caps (v4l2camsrc, caps);
}

/*
 */
static gboolean
gst_v4l2camsrc_query (GstBaseSrc * bsrc, GstQuery * query)
{
  GstV4l2CamSrc *src;

  gboolean res = FALSE;

  src = GST_V4L2CAMSRC (bsrc);

  switch (GST_QUERY_TYPE (query)) {
    case GST_QUERY_LATENCY:{
      GstClockTime min_latency, max_latency;

      /* device must be open */
      if (!GST_V4L2CAMSRC_IS_OPEN (src)) {
        GST_WARNING_OBJECT (src,
            "Can't give latency since device isn't open !");
        goto done;
      }

      /* we must have a framerate */
      if (src->fps_n <= 0 || src->fps_d <= 0) {
        GST_WARNING_OBJECT (src,
            "Can't give latency since framerate isn't fixated !");
        goto done;
      }

      /* min latency is the time to capture one frame */
      min_latency =
          gst_util_uint64_scale_int (GST_SECOND, src->fps_d, src->fps_n);

      /* max latency is total duration of the frame buffer */
      max_latency = src->num_buffers * min_latency;

      GST_DEBUG_OBJECT (bsrc,
          "report latency min %" GST_TIME_FORMAT " max %" GST_TIME_FORMAT,
          GST_TIME_ARGS (min_latency), GST_TIME_ARGS (max_latency));

      /* we are always live, the min latency is 1 frame and the max latency is
       * the complete buffer of frames. */
      gst_query_set_latency (query, TRUE, min_latency, max_latency);

      res = TRUE;
      break;
    }
    default:
      res = GST_BASE_SRC_CLASS (parent_class)->query (bsrc, query);
      break;
  }

done:

  return res;
}

/* start and stop are not symmetric -- start will open the device, but not start
 * capture. it's setcaps that will start capture, which is called via basesrc's
 * negotiate method. stop will both stop capture and close the device.
 */
static gboolean
gst_v4l2camsrc_start (GstBaseSrc * src)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);

  v4l2camsrc->offset = 0;
  return TRUE;
}

/*
 */
static gboolean
gst_v4l2camsrc_stop (GstBaseSrc * src)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);

  /* tell driver that we are stopping the capturing */
  v4l2camsrc->driver->stop (v4l2camsrc->driver);

  if (GST_V4L2CAMSRC_IS_ACTIVE (v4l2camsrc)
      && !gst_v4l2camsrc_capture_stop (v4l2camsrc))
    return FALSE;

  if (v4l2camsrc->buffer != NULL) {
    if (!gst_v4l2camsrc_capture_deinit (v4l2camsrc))
      return FALSE;
  }

  v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;
  v4l2camsrc->fps_d = 0;
  v4l2camsrc->fps_n = 0;

  return TRUE;
}

static gboolean
gst_v4l2camsrc_unlock (GstBaseSrc * src)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);

  GST_LOG_OBJECT (v4l2camsrc, "Flushing");
  gst_poll_set_flushing (v4l2camsrc->poll, TRUE);

  return TRUE;
}

static gboolean
gst_v4l2camsrc_unlock_stop (GstBaseSrc * src)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);

  GST_LOG_OBJECT (v4l2camsrc, "No longer flushing");
  gst_poll_set_flushing (v4l2camsrc->poll, FALSE);

  return TRUE;
}

/*
 * This will be called with STATE_LOCK held.
 */
static void
gst_v4l2camsrc_check_autofocus (GstV4l2CamSrc *v4l2camsrc)
{
  /* Check autofocusing status */
  GstFocusStatus status;
  guint32 focused_windows;
  guint8 focus_rows;
  guint8 focus_columns;

  status = v4l2camsrc->driver->get_focus_status (v4l2camsrc->driver,
      &focused_windows, &focus_rows, &focus_columns);

  GST_DEBUG_OBJECT (v4l2camsrc, "autofocus status: %d", status);


  /* FIXME: autofocusing can be continuous. What to do in that case? */
  if (status == GST_PHOTOGRAPHY_FOCUS_STATUS_SUCCESS ||
      status == GST_PHOTOGRAPHY_FOCUS_STATUS_FAIL)
  {
    GstStructure *s;
    GstMessage *msg;

    GST_DEBUG_OBJECT (v4l2camsrc, "autofocusing ended");

    /* Send custom GstMessage "autofocus-done" */
    s = gst_structure_new (GST_PHOTOGRAPHY_AUTOFOCUS_DONE,
                           "status", G_TYPE_INT, status,
                           NULL);

    /* If autofocus succeeded, send the bitmask that defines focused
     * windows too */
    if (status == GST_PHOTOGRAPHY_FOCUS_STATUS_SUCCESS) {
      gst_structure_set (s,
                         "focus_window_mask", G_TYPE_INT, focused_windows,
                         "focus_window_rows", G_TYPE_INT, focus_rows,
                         "focus_window_columns", G_TYPE_INT, focus_columns,
                         NULL);

      GST_DEBUG_OBJECT (v4l2camsrc, "focus rows: %lld", focus_rows);
      GST_DEBUG_OBJECT (v4l2camsrc, "focus columns: %lld", focus_columns);
      GST_DEBUG_OBJECT (v4l2camsrc, "focus mask: %lld", focused_windows);
    }

    msg = gst_message_new_element (GST_OBJECT (v4l2camsrc), s);

    if (gst_element_post_message (GST_ELEMENT (v4l2camsrc), msg) == FALSE) {
      GST_WARNING ("This element has no bus, therefore no message sent!");
    }

    /* Set capture phase back to "VIEWFINDER" */
    v4l2camsrc->driver->set_autofocus (v4l2camsrc->driver, FALSE);
    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;
  }
}


/*
 */
static gboolean
gst_v4l2camsrc_send_makernote (GstV4l2CamSrc *v4l2camsrc)
{
  guint8 *makernote_data;
  guint makernote_len;
  gboolean ret;

  ret = v4l2camsrc->driver->get_makernote (v4l2camsrc->driver, &makernote_data,
                                           &makernote_len);

  GST_DEBUG_OBJECT (v4l2camsrc, "makernote query: %d",  ret);

  if (ret && makernote_data && makernote_len > 0) {
    GstTagList *maker_note;
    GstBuffer *mnbuf;
    GstEvent *event;

    mnbuf = gst_buffer_new ();
    GST_BUFFER_DATA (mnbuf) = makernote_data;
    GST_BUFFER_SIZE (mnbuf) = makernote_len;

    GST_DEBUG_OBJECT (v4l2camsrc, "got makernote data: %d bytes",
                      makernote_len);

    maker_note = gst_tag_list_new ();
    gst_tag_list_add (maker_note, GST_TAG_MERGE_REPLACE,
                      "exif-maker-note", mnbuf, NULL);

    event = gst_event_new_tag (gst_tag_list_copy (maker_note));
    gst_pad_push_event (GST_BASE_SRC_PAD (v4l2camsrc), event);

    GST_DEBUG_OBJECT (v4l2camsrc, "makernote event sent");

    gst_tag_list_free (maker_note);
  }

  return ret;
}


/*
 */
static gboolean
gst_v4l2camsrc_send_postproc_params (GstV4l2CamSrc *v4l2camsrc)
{
  GstStructure *pp_struct;
  gboolean ret;

  ret = v4l2camsrc->driver->get_postproc_params (v4l2camsrc->driver,
                                                 &pp_struct);
  if (ret) {
    GstEvent *stats;

    stats = gst_event_new_custom (GST_EVENT_CUSTOM_DOWNSTREAM,
                                  gst_structure_copy (pp_struct));

    gst_pad_push_event (GST_BASE_SRC_PAD (v4l2camsrc), stats);

    GST_DEBUG_OBJECT (v4l2camsrc, "postproc params event sent");

    gst_structure_free (pp_struct);
  }

  return ret;
}


/*
 */
static gboolean
gst_v4l2camsrc_send_image_tags (GstV4l2CamSrc *v4l2camsrc)
{
  GstV4l2CapturedImageParams *params;
  guint zoom, iso, wbalance;
  GstEvent *tagevent;
  GstTagList *tlist;

  tlist = gst_tag_list_new ();
  zoom = v4l2camsrc->photoconf.zoom * 100;
  iso = v4l2camsrc->photoconf.iso_speed;
  wbalance =
    (v4l2camsrc->photoconf.wb_mode == GST_PHOTOGRAPHY_WB_MODE_AUTO) ? 0 : 1;

  /* Add all possible tags here */
  gst_tag_list_add (tlist, GST_TAG_MERGE_REPLACE,
      "capture-custom-rendered", 0,         /* G_TYPE_UINT: 0 = normal */
      "capture-digital-zoom", zoom, 100,    /* GST_TYPE_FRACTION */
      "capture-iso-speed-ratings", iso,     /* G_TYPE_INT */
      "capture-white-balance", wbalance,    /* G_TYPE_UINT */
      "capture-exposure-program", 0,        /* G_TYPE_UINT: 0 = not defined */
      "capture-color-space", 1,             /* G_TYPE_UINT: 1 = sRGB */
      "image-xresolution", 300, 1,    /* GST_TYPE_FRACTION */ 
      "image-yresolution", 300, 1,    /* GST_TYPE_FRACTION */ 
      NULL);

  if (v4l2camsrc->driver->get_hq_image_params (v4l2camsrc->driver, &params)) {
    guint aperture = params->aperture * 100;
    guint shutter = params->shutter_speed * 100;
    guint exposure = params->exposure_time / 1000;
    guint fnumber = params->fnumber * 100;
    guint flen = params->focal_length * 100;

    /* ISO speed 0 means "automatic" and in this case we take the value from
     * daemon's image information structure */
    if (iso == 0) {
      iso = params->iso_speed;
    }

    gst_tag_list_add (tlist, GST_TAG_MERGE_REPLACE,
        "capture-aperture", aperture, 100,        /* GST_TYPE_FRACTION */
        "capture-iso-speed-ratings", iso,         /* G_TYPE_INT */
        "capture-contrast", params->contrast,     /* G_TYPE_INT */
        "capture-saturation", params->saturation, /* G_TYPE_INT */
        "capture-shutter-speed", shutter, 100,    /* GST_TYPE_FRACTION */
        "capture-exposure-time", exposure, 1000,   /* GST_TYPE_FRACTION */
        "capture-exposure-mode", params->exposure_mode, /* G_TYPE_UINT */
        "capture-flash", params->flash,                 /* G_TYPE_UINT */
        "capture-fnumber", fnumber, 100,          /* GST_TYPE_FRACTION */
        "capture-focal-len", flen, 100,           /* GST_TYPE_FRACTION */
        "capture-gain", params->gain,                   /* G_TYPE_UINT */
        "capture-light-source", params->light_source,   /* G_TYPE_UINT */
        "capture-scene-capture-type", params->scene,    /* G_TYPE_UINT */
        NULL);

    g_free (params);
  }

  tagevent = gst_event_new_tag (gst_tag_list_copy (tlist));
  gst_pad_push_event (GST_BASE_SRC_PAD (v4l2camsrc), tagevent);
  GST_DEBUG_OBJECT (v4l2camsrc, "image tags sent");
  gst_tag_list_free (tlist);

  return TRUE;
}


/*
 */
static void
gst_v4l2camsrc_send_raw_frame (GstV4l2CamSrc *v4l2camsrc, GstBuffer *buf)
{
  GstStructure *msg_s = NULL, *raw_s = NULL;
  GstMessage *msg;
  GstBuffer *raw_buffer = NULL;
  GstCaps *raw_caps = NULL;
  gint raw_size = 0, raw_width = 0, raw_height = 0;

  raw_caps = v4l2camsrc->driver->get_raw_caps (v4l2camsrc->driver);

  if (!raw_caps) {
    GST_WARNING_OBJECT (v4l2camsrc, "could not get raw caps");
    return;
  }

  /* Copy buffer since driver->process() may modify the buffer in-place */
  GST_INFO_OBJECT (v4l2camsrc, "copying raw image data for message");
  raw_buffer = gst_buffer_copy (buf);

  /* calculate raw_size from raw_caps */    
  raw_s = gst_caps_get_structure (raw_caps, 0);
  gst_structure_get_int (raw_s, "width", &raw_width);
  gst_structure_get_int (raw_s, "height", &raw_height);
  raw_size = raw_width * raw_height * 2;

  GST_DEBUG_OBJECT (v4l2camsrc, "raw size: %d, raw caps %" GST_PTR_FORMAT,
		    raw_size, raw_caps);

  gst_buffer_set_caps (raw_buffer, raw_caps);
  GST_BUFFER_SIZE (raw_buffer) = raw_size;

  /* Send custom GstMessage "raw-image" */
  msg_s = gst_structure_new (GST_V4L2CAMSRC_PUBLISH_RAW,
			     "buffer", GST_TYPE_BUFFER, raw_buffer,
			     NULL);

  msg = gst_message_new_element (GST_OBJECT (v4l2camsrc), msg_s);

  if (gst_element_post_message (GST_ELEMENT (v4l2camsrc), msg) == FALSE) {
    GST_WARNING ("This element has no bus, therefore no message sent!");
  }

  gst_buffer_unref (raw_buffer);
  gst_caps_unref (raw_caps);
}

/*
 */
static void
gst_v4l2camsrc_set_capture_caps (GstV4l2CamSrc *v4l2camsrc)
{
  GstCaps *newcaps = NULL;
  gchar *str = NULL;

  v4l2camsrc->vf_caps =
      gst_pad_get_negotiated_caps (GST_BASE_SRC_PAD (v4l2camsrc));

  /* Tell the driver what resolution we want to capture */
  v4l2camsrc->driver->set_capture_res (v4l2camsrc->driver,
                                       v4l2camsrc->capture_w,
                                       v4l2camsrc->capture_h);

  /* Now ask the caps of the captured image, so that we can set */
  /* it to source pad */
  newcaps = v4l2camsrc->driver->get_capture_caps (v4l2camsrc->driver);

  if (newcaps == NULL) {
    newcaps = gst_caps_copy (v4l2camsrc->vf_caps);

    if (v4l2camsrc->capture_w && v4l2camsrc->capture_h) {
      gst_caps_set_simple (newcaps,
                           "width", G_TYPE_INT, v4l2camsrc->capture_w,
                           "height", G_TYPE_INT, v4l2camsrc->capture_h,
                           NULL);
    }
  }

  if (v4l2camsrc->capture_fps_n && v4l2camsrc->capture_fps_d) {
    GST_DEBUG_OBJECT (v4l2camsrc, "SET FRAMERATE: %d/%d",
                      v4l2camsrc->capture_fps_n, v4l2camsrc->capture_fps_d);

    gst_caps_set_simple (newcaps,
                         "framerate", GST_TYPE_FRACTION,
                         v4l2camsrc->capture_fps_n, v4l2camsrc->capture_fps_d,
                         NULL);
  }

  str = gst_caps_to_string (newcaps);
  GST_DEBUG_OBJECT (v4l2camsrc, "trying to set new caps for capture: %s", str);
  g_free (str);

  /* Notify application that we are ready now. This must be called before */
  /* the set_caps() call below, since application needs to know the new caps */
  /* so that it can adjust the caps filter to accept new format before */
  /* srcpad caps are actually changed */
  gst_v4l2camsrc_photo_ready_for_capture (v4l2camsrc, newcaps);

  /* This causes caps nego and switches resolution to hi-res mode */
  if (gst_caps_is_equal (v4l2camsrc->vf_caps, newcaps)) {
    GST_DEBUG_OBJECT (v4l2camsrc, "HQ INIT DIRECTLY");
    gst_v4l2camsrc_init_from_caps (v4l2camsrc, newcaps);
  }
  else {
    GST_DEBUG_OBJECT (v4l2camsrc, "HQ INIT VIA SET_CAPS");
    gst_pad_set_caps (GST_BASE_SRC_PAD (v4l2camsrc), newcaps);
  }
  gst_caps_unref (newcaps);
}

/*
 */
static void
gst_v4l2camsrc_capture_cb (gpointer user_data)
{
  GstV4l2CamSrc *v4l2camsrc;
  GstStructure *s;
  GstMessage *msg;

  v4l2camsrc = (GstV4l2CamSrc *) user_data;
  GST_DEBUG_OBJECT (v4l2camsrc, "capture callback");

  /* Send custom GstMessage "photo-capture-start" */
  s = gst_structure_new (GST_V4L2CAMSRC_CAPTURE_START, NULL);
  msg = gst_message_new_element (GST_OBJECT (v4l2camsrc), s);

  if (gst_element_post_message (GST_ELEMENT (v4l2camsrc), msg) == FALSE) {
    GST_WARNING ("This element has no bus, therefore no message sent!");
  }
}

/*
*/
static void
gst_v4l2camsrc_capture_done_cb (gpointer user_data){
  GstV4l2CamSrc *v4l2camsrc;
  GstStructure *s;
  GstMessage *msg;

  v4l2camsrc = (GstV4l2CamSrc *) user_data;
  GST_DEBUG_OBJECT (v4l2camsrc, "capture done callback");

  /* Send custom GstMessage "photo-capture-end" */
  s = gst_structure_new (GST_V4L2CAMSRC_CAPTURE_END, NULL);
  msg = gst_message_new_element (GST_OBJECT (v4l2camsrc), s);

  if (gst_element_post_message (GST_ELEMENT (v4l2camsrc), msg) == FALSE) {
    GST_WARNING ("This element has no bus, therefore no message sent!");
  }

}

/*
 */
static GstFlowReturn
gst_v4l2camsrc_create (GstPushSrc * src, GstBuffer ** buf)
{
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (src);
  GstFlowReturn ret = GST_FLOW_OK;
  gboolean process_exclusive;
  guint32 tmp_num_buffers = 2;
  GstBuffer *temp;
  guint count = 0;
  guint size;

start_over:

  g_mutex_lock (v4l2camsrc->state_lock);

  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE_START) {
    gst_poll_set_flushing (v4l2camsrc->poll, FALSE);
    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_CAPTURE;

    /* Store the current number of buffers and set it to 1 for capture */
    /* FIXME: What if there is no need for re-initialization? */
    tmp_num_buffers = v4l2camsrc->num_buffers;
    v4l2camsrc->num_buffers = 1;

    if (v4l2camsrc->pool->num_live_buffers) {
      GST_DEBUG_OBJECT (v4l2camsrc, "Flushing old buffers before starting HQ capture");
      gst_pad_push_event (GST_BASE_SRC_PAD (v4l2camsrc), gst_event_new_flush_start ());
      gst_pad_push_event (GST_BASE_SRC_PAD (v4l2camsrc), gst_event_new_flush_stop ());
    }

    gst_v4l2camsrc_set_capture_caps (v4l2camsrc);

    /* If driver wants to handle the image capture by itself, we call
       its capturing function */
    if (v4l2camsrc->driver->capture) {
      gboolean cret;

      cret = v4l2camsrc->driver->capture (v4l2camsrc->driver, buf,
                                          gst_v4l2camsrc_capture_cb,
                                          gst_v4l2camsrc_capture_done_cb,
                                          v4l2camsrc);
      GST_DEBUG_OBJECT (v4l2camsrc, "capture function returned: %d", cret);
      if (cret) {
        goto skip_grab_frame;
      } else {
        goto hq_capture_failed;
      }
    }
    else {
      /* Notify that capture is about to happen */
      gst_v4l2camsrc_capture_cb (v4l2camsrc);
    }
  }
  else if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE_DONE) {

    /* Capture may have left it closed before leaving the create function */
    if (!GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc)) {
      gst_v4l2camsrc_open (v4l2camsrc);
      v4l2camsrc->driver->update_fd (v4l2camsrc->driver, v4l2camsrc->video_fd);
    }

    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;

    GST_DEBUG_OBJECT (v4l2camsrc, "capture done. switching to viewfinder");

    /* Set the normal viewfinder resolution back */
    if (v4l2camsrc->vf_caps) {
      GstCaps *hqcaps;

      GST_DEBUG_OBJECT (v4l2camsrc, "set VF caps");
      hqcaps = gst_pad_get_negotiated_caps (GST_BASE_SRC_PAD (v4l2camsrc));

      if (gst_caps_is_equal (v4l2camsrc->vf_caps, hqcaps)) {
        GST_DEBUG_OBJECT (v4l2camsrc, "VF INIT DIRECTLY");
        gst_v4l2camsrc_init_from_caps (v4l2camsrc, v4l2camsrc->vf_caps);
      }
      else {
        GST_DEBUG_OBJECT (v4l2camsrc, "VF INIT VIA SET_CAPS");
        gst_pad_set_caps (GST_BASE_SRC_PAD (v4l2camsrc), v4l2camsrc->vf_caps);
      }
      gst_caps_unref (hqcaps);
      gst_caps_unref (v4l2camsrc->vf_caps);
      v4l2camsrc->vf_caps = NULL;
    }
    GST_DEBUG_OBJECT (v4l2camsrc, "viewfinder running");
  }

  if (v4l2camsrc->requested_af_mode != AF_NONE_REQUESTED) {

    if (v4l2camsrc->requested_af_mode == AF_ON_REQUESTED) {
      gboolean ret;

      ret = v4l2camsrc->driver->set_autofocus (v4l2camsrc->driver, TRUE);

      if (ret)
        v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_AUTOFOCUS;
    }
    else {
      v4l2camsrc->driver->set_autofocus (v4l2camsrc->driver, FALSE);
      v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;
    }

    v4l2camsrc->requested_af_mode = AF_NONE_REQUESTED;
  }

  g_mutex_unlock (v4l2camsrc->state_lock);

again:

  ret = gst_v4l2camsrc_grab_frame (v4l2camsrc, &temp);

  if (ret != GST_FLOW_OK) {
    /* _prepare_for_capture() may have interrupted frame grabbing. */
    if (ret == GST_FLOW_WRONG_STATE &&
        v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE_START)
    {
      ret = GST_FLOW_OK;
      goto start_over;
    } else {
      goto leave;
    }
  }

  if (v4l2camsrc->frame_byte_size > 0) {
    size = GST_BUFFER_SIZE (temp);

    /* if size does not match what we expected, try again */
    if (size != v4l2camsrc->frame_byte_size) {
      GST_ELEMENT_WARNING (v4l2camsrc, RESOURCE, READ,
                           ("Got unexpected frame size of %u instead of %u.",
                            size, v4l2camsrc->frame_byte_size), (NULL));
      gst_buffer_unref (temp);
      if (count++ > 50)
        goto size_error;

      goto again;
    }
  }

  *buf = temp;

  g_mutex_lock (v4l2camsrc->state_lock);

skip_grab_frame:
  /* Post-capture phase */

  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE) {
    GstCaps *src_caps;

    /* If the capture library didn't provide a capture function, then we need
       to send the "capture done" message explicitly here */
    if (v4l2camsrc->driver->capture == NULL) {
      gst_v4l2camsrc_capture_done_cb (v4l2camsrc);
    }

    /* FIXME: Check return value */
    v4l2camsrc->driver->stop_capture (v4l2camsrc->driver);

    GST_DEBUG_OBJECT (v4l2camsrc, "=== POST_CAPTURE PHASE ===");

    src_caps = gst_pad_get_negotiated_caps (GST_BASE_SRC_PAD (v4l2camsrc));
    gst_buffer_set_caps (*buf, src_caps);
    gst_caps_unref (src_caps);

    if (NULL != g_getenv ("CAMSRC_PUBLISH_RAW")) {
      gst_v4l2camsrc_send_raw_frame (v4l2camsrc, *buf);
    }

    /* Stop streaming and deallocate buffers */
    /* FIXME: Not need to do this when:       */
    /*          1) device is not initialized  */
    /*          2) resolution doesn't change  */
    if (!gst_v4l2camsrc_capture_stop (v4l2camsrc))
      goto device_init_failed;
    if (!gst_v4l2camsrc_capture_deinit (v4l2camsrc))
      goto device_init_failed;

    /* Restore the original number of buffers after capture is done */
    v4l2camsrc->num_buffers = tmp_num_buffers;

    process_exclusive =
        v4l2camsrc->driver->process_excl_access (v4l2camsrc->driver);

    if (process_exclusive) {
      GST_DEBUG_OBJECT (v4l2camsrc, "Driver module requires exclusive access");
      gst_v4l2camsrc_close (v4l2camsrc);
      v4l2camsrc->driver->update_fd (v4l2camsrc->driver, -1);
    }

    if (v4l2camsrc->driver->process (v4l2camsrc->driver, buf,
                                     v4l2camsrc->capture_w,
                                     v4l2camsrc->capture_h) == FALSE)
    {
      goto process_failed;
    }

    /* Trick: check if makernote & post-proc params can be asked from */
    /* driver without opening the device first. This makes HQ image pushing */
    /* a bit quickier, since opening the device can be postponed */
    if (!gst_v4l2camsrc_send_makernote (v4l2camsrc) && process_exclusive) {
      gst_v4l2camsrc_open (v4l2camsrc);
      v4l2camsrc->driver->update_fd (v4l2camsrc->driver, v4l2camsrc->video_fd);
      gst_v4l2camsrc_send_makernote (v4l2camsrc);
      gst_v4l2camsrc_send_postproc_params (v4l2camsrc);
    }
    else if (!gst_v4l2camsrc_send_postproc_params (v4l2camsrc) &&
      process_exclusive)
    {
      gst_v4l2camsrc_open (v4l2camsrc);
      v4l2camsrc->driver->update_fd (v4l2camsrc->driver, v4l2camsrc->video_fd);
      gst_v4l2camsrc_send_postproc_params (v4l2camsrc);
    }

    gst_v4l2camsrc_send_image_tags (v4l2camsrc);
    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_CAPTURE_DONE;
  }

  else if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_AUTOFOCUS) {
    GstPhotoShakeRisk shake_risk;
    gboolean retval = FALSE;

    /* this function sets the capture phase to "VIEWFINDER" when
    autofocus is done / fails */
    gst_v4l2camsrc_check_autofocus (v4l2camsrc);

    /* FIXME: Two consecutive GSTATUS calls */
    retval = v4l2camsrc->driver->get_shake_risk (v4l2camsrc->driver,
        &shake_risk);

    if (retval && v4l2camsrc->shake_risk != shake_risk) {
      GstStructure *s;
      GstMessage *msg;

      /* Send custom GstMessage telling the changed shake risk level */
      s = gst_structure_new (GST_PHOTOGRAPHY_SHAKE_RISK,
                             "shake_risk", G_TYPE_INT, shake_risk, NULL);
      msg = gst_message_new_element (GST_OBJECT (v4l2camsrc), s);

      if (gst_element_post_message (GST_ELEMENT (v4l2camsrc), msg) == FALSE) {
        GST_WARNING ("This element has no bus, therefore no message sent!");
      }
      GST_DEBUG_OBJECT (v4l2camsrc, "Shake indicator message sent, risk = %d",
          shake_risk);
    }
    v4l2camsrc->shake_risk = shake_risk;
  }

done:
  g_mutex_unlock (v4l2camsrc->state_lock);

leave:
  return ret;

  /* ERRORS */
size_error:
  GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
                     ("Error read()ing %d bytes on device '%s'.",
                      v4l2camsrc->frame_byte_size, v4l2camsrc->videodev),
                      (NULL));
  ret = GST_FLOW_ERROR;
  goto done;

device_init_failed:
  GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
                     ("Cannot configure device"), (NULL));
  ret = GST_FLOW_ERROR;
  goto done;

hq_capture_failed:
  GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
                     ("Error during HQ capture"), (NULL));
  ret = GST_FLOW_ERROR;
  goto done;

process_failed:
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
                       ("Error during pre-process operation"), (NULL));
  ret = GST_FLOW_ERROR;
  goto done;
}


static GstStateChangeReturn
gst_v4l2camsrc_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
  GstV4l2CamSrc *v4l2camsrc = GST_V4L2CAMSRC (element);

  switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
      /* Open the device */
      if (!gst_v4l2camsrc_open (v4l2camsrc))
        return GST_STATE_CHANGE_FAILURE;

      /* Initialize driver module */
      if (!v4l2camsrc->driver->init (v4l2camsrc->driver, v4l2camsrc->video_fd))
      {
        gst_v4l2camsrc_close (v4l2camsrc);
        gst_v4l2camsrc_empty_lists (v4l2camsrc);
        return GST_STATE_CHANGE_FAILURE;
      }
      break;
    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {
    case GST_STATE_CHANGE_READY_TO_NULL:
      if (GST_V4L2CAMSRC_IS_OPEN (v4l2camsrc)) {
        v4l2camsrc->driver->deinit (v4l2camsrc->driver);
        gst_v4l2camsrc_close (v4l2camsrc);
        gst_v4l2camsrc_empty_lists (v4l2camsrc);
      }
      break;
    default:
      break;
  }

  return ret;
}

