/* GStreamer
 *
 * Copyright (C) 2008 Nokia Corporation <multimedia@maemo.org>
 *
 * gstv4l2photoiface.c: Photo interface implementation for v4l2camsrc
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <gst/gst.h>

#include "gstv4l2camphotoiface.h"
#include "gstv4l2camsrc.h"
#include "v4l2camsrc_calls.h"

GST_DEBUG_CATEGORY_STATIC (v4l2camphoto_debug);
#define GST_CAT_DEFAULT v4l2camphoto_debug


/*
 *
 */
void
gst_v4l2_photo_interface_init (GstPhotographyInterface * iface)
{
    GST_DEBUG_CATEGORY_INIT (v4l2camphoto_debug, "v4l2camphoto", 0,
                             "V4L2Cam Photo interface debugging");
}

/*
 *
 */
GstPhotoCaps
gst_v4l2_photo_get_capabilities (GstV4l2CamSrc *v4l2camsrc)
{
  return v4l2camsrc->driver->get_capabilities (v4l2camsrc->driver);
}



/*
 *
 */
gboolean
gst_v4l2_photo_prepare_for_capture (GstV4l2CamSrc *v4l2camsrc,
                                    GstPhotoCapturePrepared func,
                                    GstCaps *desired_caps,
                                    gpointer user_data)
{
  gboolean use_vf_caps = TRUE;

  v4l2camsrc->prep_func = func;
  v4l2camsrc->prep_udata = user_data;

  if (desired_caps) {
    GstStructure *cstr;

    cstr = gst_caps_get_structure (desired_caps, 0);

    if (gst_structure_get_int (cstr, "width",  &v4l2camsrc->capture_w) &&
        gst_structure_get_int (cstr, "height", &v4l2camsrc->capture_h) &&
        gst_structure_get_fraction (cstr, "framerate",
                                    &v4l2camsrc->capture_fps_n,
                                    &v4l2camsrc->capture_fps_d))
    {
      use_vf_caps = FALSE;
    }
  }

  if (use_vf_caps) {
    v4l2camsrc->capture_w = v4l2camsrc->current_w;
    v4l2camsrc->capture_h = v4l2camsrc->current_h;
    v4l2camsrc->capture_fps_n = v4l2camsrc->fps_n;
    v4l2camsrc->capture_fps_d = v4l2camsrc->fps_d;
  }

  GST_DEBUG_OBJECT (v4l2camsrc, "starting preparations for capture");

  GST_V4L2CAM_STATE_LOCK (v4l2camsrc);

  /* If autofocus is still running, stop it first */
  /* FIXME: focus to infinity? */
  if  (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_AUTOFOCUS) {
    gst_v4l2_photo_set_autofocus (v4l2camsrc, FALSE);
  }

  v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_CAPTURE;
  GST_V4L2CAM_STATE_UNLOCK (v4l2camsrc);

  return TRUE;
}


/*
 *
 */
void
gst_v4l2_photo_ready_for_capture (GstV4l2CamSrc *v4l2camsrc, GstCaps *newcaps)
{
  GST_DEBUG_OBJECT (v4l2camsrc, "prepare for capture is complete");

  if (v4l2camsrc->prep_func) {
    v4l2camsrc->prep_func (v4l2camsrc->prep_udata, newcaps);
    v4l2camsrc->prep_func = NULL;
    v4l2camsrc->prep_udata = NULL;
  }
  GST_DEBUG_OBJECT (v4l2camsrc, "callback returned");
}


/*
 *
 */
static char *
create_debug_string (const char *base_str, GType type, gint value)
{
  GTypeClass *t = g_type_class_ref (type);
  GEnumValue *val = g_enum_get_value (G_ENUM_CLASS (t), value);
  gchar *ret = g_strconcat (base_str, val->value_nick, NULL);
  g_type_class_unref (t);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_ev_compensation (GstV4l2CamSrc *v4l2camsrc, gfloat ev_comp)
{
  gboolean ret = TRUE;

  v4l2camsrc->photoconf.ev_compensation = ev_comp;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
}
  GST_DEBUG_OBJECT (v4l2camsrc, "set EV: %.2f, ret = %d", ev_comp, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_ev_compensation (GstV4l2CamSrc *v4l2camsrc, gfloat *ev_comp)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *ev_comp = v4l2camsrc->photoconf.ev_compensation;
  GST_DEBUG_OBJECT (v4l2camsrc, "got ev compensation: %.2f, ret = %d",
                    *ev_comp, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_iso_speed (GstV4l2CamSrc *v4l2camsrc, guint iso_speed)
{
  gboolean ret = TRUE;

  v4l2camsrc->photoconf.iso_speed = iso_speed;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  return v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
}
  GST_DEBUG_OBJECT (v4l2camsrc, "set ISO: %d, ret = %d", iso_speed, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_iso_speed (GstV4l2CamSrc *v4l2camsrc, guint *iso_speed)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *iso_speed = v4l2camsrc->photoconf.iso_speed;
  GST_DEBUG_OBJECT (v4l2camsrc, "got iso speed: %d, ret = %d", *iso_speed, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_aperture (GstV4l2CamSrc *v4l2camsrc, guint aperture)
{
  gboolean ret = TRUE;

  v4l2camsrc->photoconf.aperture = aperture;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
}
  GST_DEBUG_OBJECT (v4l2camsrc, "set aperture: %d, ret = %d", aperture, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_aperture (GstV4l2CamSrc *v4l2camsrc, guint *aperture)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *aperture = v4l2camsrc->photoconf.aperture;
  GST_DEBUG_OBJECT (v4l2camsrc, "got aperture: %d, ret = %d", *aperture, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_exposure (GstV4l2CamSrc *v4l2camsrc, guint32 exposure)
{
  gboolean ret = TRUE;

  v4l2camsrc->photoconf.exposure = exposure;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
}
  GST_DEBUG_OBJECT (v4l2camsrc, "set exposure: %d, ret = %d", exposure, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_exposure (GstV4l2CamSrc *v4l2camsrc, guint32 *exposure)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *exposure = v4l2camsrc->photoconf.exposure;
  GST_DEBUG_OBJECT (v4l2camsrc, "got exposure: %d, ret = %d", *exposure, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_wb_mode (GstV4l2CamSrc *v4l2camsrc, GstWhiteBalanceMode mode)
{
  gboolean ret = TRUE;
  gchar *dstr;

  v4l2camsrc->photoconf.wb_mode = mode;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                              &v4l2camsrc->photoconf,
                                              FALSE);
  }
  dstr = create_debug_string ("AWB:", GST_TYPE_WHITE_BALANCE_MODE, mode);
  GST_DEBUG_OBJECT (v4l2camsrc, "set %s, ret = %d", dstr, ret);
  g_free (dstr);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_wb_mode (GstV4l2CamSrc *v4l2camsrc, GstWhiteBalanceMode *mode)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *mode = v4l2camsrc->photoconf.wb_mode;
  GST_DEBUG_OBJECT (v4l2camsrc, "got AWB mode:%d, ret = %d", *mode, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_tone_mode (GstV4l2CamSrc *v4l2camsrc, GstColourToneMode mode)
{
  gboolean ret = TRUE;
  gchar *dstr;

  v4l2camsrc->photoconf.tone_mode = mode;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
  }
  dstr = create_debug_string ("tone:", GST_TYPE_COLOUR_TONE_MODE, mode);
  GST_DEBUG_OBJECT (v4l2camsrc, "set %s, ret = %d", dstr, ret);
  g_free (dstr);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_tone_mode (GstV4l2CamSrc *v4l2camsrc, GstColourToneMode *mode)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *mode = v4l2camsrc->photoconf.tone_mode;
  GST_DEBUG_OBJECT (v4l2camsrc, "got tone mode: %d, ret = %d", *mode, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_scene_mode (GstV4l2CamSrc *v4l2camsrc, GstSceneMode mode)
{
  gboolean ret = FALSE;
  gchar *dstr;

  v4l2camsrc->photoconf.scene_mode = mode;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                            &v4l2camsrc->photoconf,
                                            TRUE);
  if (ret) {
    /* Read the changed parameters to local settings cache */
    v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                       &v4l2camsrc->photoconf);
  }
  }
  dstr = create_debug_string ("scene:", GST_TYPE_SCENE_MODE, mode);
  GST_DEBUG_OBJECT (v4l2camsrc, "set %s, ret = %d", dstr, ret);
  g_free (dstr);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_scene_mode (GstV4l2CamSrc *v4l2camsrc, GstSceneMode *mode)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *mode = v4l2camsrc->photoconf.scene_mode;
  GST_DEBUG_OBJECT (v4l2camsrc, "got scene mode: %d, ret = %d", *mode, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_flash_mode (GstV4l2CamSrc *v4l2camsrc, GstFlashMode mode)
{
  gboolean ret = TRUE;
  gchar *dstr;

  v4l2camsrc->photoconf.flash_mode = mode;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                             &v4l2camsrc->photoconf,
                                             FALSE);
}
  dstr = create_debug_string ("flash:", GST_TYPE_FLASH_MODE, mode);
  GST_DEBUG_OBJECT (v4l2camsrc, "set %s, ret = %d", dstr, ret);
  g_free (dstr);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_flash_mode (GstV4l2CamSrc *v4l2camsrc, GstFlashMode *mode)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
  ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver,
                                           &v4l2camsrc->photoconf);
  }
  *mode = v4l2camsrc->photoconf.flash_mode;
  GST_DEBUG_OBJECT (v4l2camsrc, "got flash mode: %d, ret = %d", *mode, ret);

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_zoom (GstV4l2CamSrc *v4l2camsrc, gfloat zoom)
{
  gboolean ret = TRUE;

  v4l2camsrc->photoconf.zoom = zoom;

  GST_DEBUG_OBJECT (v4l2camsrc, "setting zoom to %.2f", zoom);

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    /* If we are using a driver that handles the HQ capture by itself, */
    /* then we need to forward the zoom value to it */
    if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE &&
        v4l2camsrc->driver->capture != NULL)
    {
      GST_DEBUG_OBJECT (v4l2camsrc, "driver zoom");
      ret = v4l2camsrc->driver->set_zoom (v4l2camsrc->driver, zoom);
    }
    else {
      GST_DEBUG_OBJECT (v4l2camsrc, "v4l2camsrc zoom");
      ret = gst_v4l2camsrc_update_cropping (v4l2camsrc, v4l2camsrc->current_w,
                                            v4l2camsrc->current_h, zoom);
    }
  }
  else {
    /* FIXME: Check if the zoom is within known limits */
  }
  GST_DEBUG_OBJECT (v4l2camsrc, "setting zoom %s", ret ? "success" : "failed");

  return ret;
}


/*
 *
 */
gboolean
gst_v4l2_photo_get_zoom (GstV4l2CamSrc *v4l2camsrc, gfloat *zoom)
{
  *zoom = v4l2camsrc->photoconf.zoom;

  GST_DEBUG_OBJECT (v4l2camsrc, "current zoom = %.2f", *zoom);
  return TRUE;
}


/*
 *
 */
void
gst_v4l2_photo_set_autofocus (GstV4l2CamSrc *v4l2camsrc, gboolean on)
{
  gboolean ret;

  GST_DEBUG_OBJECT (v4l2camsrc, "setting autofocus %s", on ? "ON" : "OFF");

  GST_V4L2CAM_STATE_LOCK (v4l2camsrc);
  ret = v4l2camsrc->driver->set_autofocus (v4l2camsrc->driver, on);

  if (ret && on) {
    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_AUTOFOCUS;
  }
  else {
    v4l2camsrc->photo_capture_phase = GST_V4L2PHOTO_VIEWFINDER;
  }

  GST_V4L2CAM_STATE_UNLOCK (v4l2camsrc);
}


/*
 *
 */
gboolean
gst_v4l2_photo_set_config (GstV4l2CamSrc *v4l2camsrc, GstPhotoSettings * config)
{
  gboolean ret = TRUE;

  if (GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    gboolean scene_override =
        config->scene_mode != GST_PHOTOGRAPHY_SCENE_MODE_MANUAL ? TRUE : FALSE;

    ret = v4l2camsrc->driver->write_settings (v4l2camsrc->driver,
                                              config, scene_override);

    if (ret && scene_override) {
      ret = v4l2camsrc->driver->read_settings (v4l2camsrc->driver, config);
    }
  }
  GST_DEBUG_OBJECT (v4l2camsrc, "set config, ret = %d", ret);

  if (ret) {
    memcpy (&v4l2camsrc->photoconf, config, sizeof (GstPhotoSettings));
  }

  return ret;

}


/*
 *
 */
gboolean
gst_v4l2_photo_get_config (GstV4l2CamSrc *v4l2camsrc, GstPhotoSettings * config)
{
  memcpy (config, &v4l2camsrc->photoconf, sizeof (GstPhotoSettings));

  GST_DEBUG_OBJECT (v4l2camsrc, "got config");
  return TRUE;
}
