/* GStreamer
 *
 * Copyright (C) 2009 Nokia Corporation <multimedia@maemo.org>
 *
 * gstv4l2camdriver.h: V4L2 camera driver abstraction API
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __GST_V4L2CAMSRC_DRV_H__
#define __GST_V4L2CAMSRC_DRV_H__

#include <gst/gst.h>
#include <gst/interfaces/photography.h>

G_BEGIN_DECLS

/**
 * GstCamCaptureMode:
 * @GGST_V4L2CAMDRIVER_CAPTURE_MODE_NONE: No capturing mode defined.
 * @GST_V4L2CAMDRIVER_CAPTURE_MODE_STILL: Still image capturing mode.
 * @GST_V4L2CAMDRIVER_CAPTURE_MODE_VIDEO: Video capturing mode.
 *
 * Image capturing mode.
 */
typedef enum
{
  GST_V4L2CAMDRIVER_CAPTURE_MODE_NONE = 0,
  GST_V4L2CAMDRIVER_CAPTURE_MODE_STILL,
  GST_V4L2CAMDRIVER_CAPTURE_MODE_VIDEO
} GstCamCaptureMode;


typedef struct _GstV4l2CamDriver GstV4l2CamDriver;
typedef struct _GstV4l2CapturedImageParams GstV4l2CapturedImageParams;


/**
 * GstV4l2CamDriver:
 * @exposure_time: image exposure time in microseconds
 * @exposure_mode: image exposure mode:
 *    0 - auto exposure
 *    1 - manual exposure
 *    2 - auto bracket
 * @shutter_speed: used shutter speed when image was captured in APEX units
 * @saturation: saturation value, between -100 ... 100
 * @contrast: contrast value, between -100 ... 100
 * @brightness: brightness value, between -100 ... 100
 * @aperture: aperture value in APEX units
 * @gain: gain value
 * @light_source: light source value
 * @fnumber: F-Number value
 * @focal_length: focal length in millimeters
 * @flash: information about flash usage during capture according to EXIF spec
 * @scene: scene capture type:
 *    0 - Standard
 *    1 - Landscape
 *    2 - Portrait
 *    3 - Night scene
 *
 * This structure contains information about captured image parameters.
 */
struct _GstV4l2CapturedImageParams
{
  guint32   exposure_time;
  guint     exposure_mode;
  guint32   shutter_speed;
  gint      saturation;
  gint      contrast;
  gint      brightness;
  gfloat    aperture;
  guint     gain;
  guint     light_source;
  gfloat    fnumber;
  gfloat    focal_length;
  guint     flash;
  guint     scene;
};

typedef void (*CaptureStarting) (gpointer user_data);
typedef void (*CaptureEnding) (gpointer user_data);

/**
 * GstV4l2CamDriver:
 * @driver_priv: Pointer where driver can store its private data structure.
 * @device_mutex: GMutex used when accessing the camera device.
 * @module: GModule that implements the driver API.
 * @init: Initialize the driver. V4l2camsrc calls this immediately after
 *     the camera device has been opened and registers the device file
 *     descriptor to the driver.
 * @update_fd: Function to tell the driver that video device file handle
 *     has been changed. This may happen when device needs to be closed for
 *     example during the @process call.
 * @deinit: Deinitialize the driver. V4l2camsrc tells the driver to stop
 *     using the camera device, because after this call it is not valid anymore.
 * @get_capabilities: Get the #GstPhotoCaps capabilities to find out what
 *     functionality the driver provides.
 * @set_capture_fmt: Tell the driver what format we want to produce, and
 *     driver tells us what format we should capture. This mechanism exists
 *     because the driver may want to capture some exotic RAW format from
 *     the sensor, process it and finally convert to desired format
 *     (in @preprocess function).
 * @set_capture_res: Tell the driver what resolution we want to capture.
 * @get_raw_caps: Get the caps for raw image format coming from camera sensor.
 *     NOTE: Driver must be informed about desired format by using the
 *     previous @set_capture_fmt function before this function can be called.
 * @get_capture_caps: Get the caps for processed HQ image. These caps will
 *     be attached to the HQ-image GstBuffer that is pushed to pipeline.
 * @start: Called by v4l2camsrc at startup (before STREAMON). Driver should
 *     start processing the frames now.
 * @stop: Called by v4l2camsrc when stopping (before STREAMOFF). Driver
 *     must stop processing the frames.
 * @set_autofocus: Start or stop autofocusing.
 * @set_flash: Turn on / off the flash.
 * @get_flash_intensity: Ask the suggested flash intensity value.
 * @set_flash_intensity: Set flash intensity value.
 * @set_flash_duration: Set the flash timeout value.
 * @set_privacy_light: Turn on / off privacy light.
 * @get_focus_status: V4l2camsrc uses this to query autofocusing status.
 * @get_shake_risk: V4l2camsrc uses this to query the risk for image shaking.
 * @start_capture: Called by v4l2camsrc when the HQ-image capture process
 *     starts.
 * @stop_capture: Called by v4l2camsrc when the HQ-image capture has finished.
 * @capture: If driver implements this vmethod, v4l2camsrc's built-in HQ image
 *     capture mechamism is skipped and this function is used to capture
 *     the HQ image.
 * @process_excl_access: This function can be used to check whether the
 *     @process call needs exclusive access to the video device, a.k.a. video
 *     device should be released (closed) before calling @process function.
 *     Some hardware may require this. If the device is closed and re-opened,
 *     driver need to be informed about this by using @update_fd call.
 * @process: Perform some image improvements for the given buffer. This
 *     vmethod is called after @stop_capture vmethod.
 * @write_settings: Called when some settings in v4l2camsrc's #GstPhotoSettings
 *     structure has changed. Driver needs to forward the settings to device.
 * @read_settings: V4l2camsrc tells the driver to retrieve settings from
 *     device and store them into #GstPhotoSettings structure.
 * @get_makernote; Retrieve MakerNote data chunk from camera device.
 * @get_postproc_params: Retrieve post-processing parameters from device.
 * @get_hq_image_params: Retrieve capture parameters for the HQ image.
 *
 * V4l2camsrc calls driver wrapper's init function, which is responsible
 * for filling up these function pointers. All pointers must be present,
 * and wrapper provides dummy implementation for those which
 * are not implemented by the driver code.
 */
struct _GstV4l2CamDriver
{
  void *driver_priv;
  GMutex *device_mutex;
  GModule *module;

  gboolean
  (*init)              (GstV4l2CamDriver *driver, gint fd);

  gboolean
  (*update_fd)         (GstV4l2CamDriver *driver, gint fd);

  gboolean
  (*deinit)            (GstV4l2CamDriver *driver);

  GstPhotoCaps
  (*get_capabilities)  (GstV4l2CamDriver *driver);

  gboolean
  (*set_capture_fmt)   (GstV4l2CamDriver *driver, guint32 output_fourcc,
                        guint output_outsize, guint32 *internal_fourcc,
                        guint *internal_outsize);

  gboolean
  (*set_capture_res)   (GstV4l2CamDriver *driver,
                        guint width, guint height);

  gboolean
  (*set_capture_mode)  (GstV4l2CamDriver *driver,
                        GstCamCaptureMode mode);

  GstCaps *
  (*get_raw_caps)      (GstV4l2CamDriver *driver);

  GstCaps *
  (*get_capture_caps)  (GstV4l2CamDriver *driver);

  gboolean
  (*start)             (GstV4l2CamDriver *driver);

  gboolean
  (*stop)              (GstV4l2CamDriver *driver);

  gboolean
  (*set_autofocus)     (GstV4l2CamDriver *driver, gboolean on_off);

  gboolean
  (*set_flash)         (GstV4l2CamDriver *driver, gboolean on_off);

  gboolean
  (*get_flash_intensity) (GstV4l2CamDriver *driver, guint32 *intensity);

  gboolean
  (*set_flash_intensity) (GstV4l2CamDriver *driver, guint32 intensity);

  gboolean
  (*set_flash_duration) (GstV4l2CamDriver *driver, guint32 duration);

  gboolean
  (*set_privacy_light) (GstV4l2CamDriver *driver, gboolean on_off);

  GstFocusStatus
  (*get_focus_status)  (GstV4l2CamDriver *driver,
                        guint32 *focused_windows_bitmask,
                        guint8 *focus_rows,
                        guint8 *focus_columns);

  gboolean
  (*get_shake_risk)    (GstV4l2CamDriver *driver, GstPhotoShakeRisk *risk);

  gboolean
  (*start_capture)     (GstV4l2CamDriver *driver, gboolean *auto_config_device);

  gboolean
  (*stop_capture)      (GstV4l2CamDriver *driver);

  gboolean
  (*capture)           (GstV4l2CamDriver *driver, GstBuffer **buf,
                        CaptureStarting func_1, CaptureEnding func_2,
                        gpointer user_data);

  gboolean
  (*process_excl_access) (GstV4l2CamDriver *driver);

  gboolean
  (*process)           (GstV4l2CamDriver *driver, GstBuffer **buf,
                        guint image_w, guint image_h);

  gboolean
  (*write_settings)    (GstV4l2CamDriver *driver,
                        GstPhotoSettings *photoconf,
                        gboolean scene_mode_override);

  gboolean
  (*read_settings)     (GstV4l2CamDriver *driver, GstPhotoSettings *photoconf);

  gboolean
  (*get_makernote)     (GstV4l2CamDriver *driver,
                        guint8 **maker_note,
                        guint *maker_note_size);

  gboolean
  (*get_postproc_params) (GstV4l2CamDriver *driver,
                          GstStructure **params_struct);

  gboolean
  (*get_hq_image_params) (GstV4l2CamDriver *driver,
                          GstV4l2CapturedImageParams **params_struct);

  gboolean
  (*set_zoom)          (GstV4l2CamDriver *driver, gfloat zoomfactor);
};

G_END_DECLS

#endif /* __GST_V4L2CAMSRC_DRV_H__ */
