/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 * Copyright (C) 2008 Free Software Foundation, Inc.
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Based on backtrace code from BFD.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <elf.h>
#include <bfd.h>
#include <libiberty.h>
#include <sys/stat.h>

#include "list.h"
#include "options.h"
#include "resolve.h"

static asymbol **slurp_symtab(bfd *abfd, long *symcount)
{
	static asymbol **symbols;
	unsigned int size;

	if ((bfd_get_file_flags (abfd) & HAS_SYMS) == 0) {
		fprintf(stderr, "error: no symbols in %s\n", bfd_get_filename(abfd));
		return NULL;
	}

	*symcount = bfd_read_minisymbols(abfd, FALSE, (void *) &symbols, &size);
	if (*symcount == 0)
		*symcount = bfd_read_minisymbols(abfd, TRUE , (void *) &symbols, &size);

	if (*symcount < 0) {
		fprintf(stderr, "%s: %s\n", bfd_get_filename(abfd), bfd_errmsg(bfd_get_error()));
		return NULL;
	}

	return symbols;
}

static int translate_addresses(bfd *abfd,
	asymbol **syms,
	t_address_info *addr_info,
	t_address address) {

	static const char *filename, *functionname;
	static unsigned int line;

	asection *section;
	bfd_vma pc, vma;
	bfd_size_type size;

	char addr_hex[LINE_MAX];
	sprintf(addr_hex, "%#lx", address);

	pc = bfd_scan_vma(addr_hex, NULL, 16);

	/* initialize variables */
	strcpy(addr_info->filename, "<undefined>");
	strcpy(addr_info->function, "<undefined>");
	addr_info->line = 0;

	for (section = abfd->sections; section != NULL; section = section->next) {

		if ((bfd_get_section_flags(abfd, section) & SEC_ALLOC) == 0)
			continue;

		vma = bfd_get_section_vma(abfd, section);
		if (pc < vma)
			continue;

		size = bfd_get_section_size(section);
		if (pc >= vma + size)
			continue;

		if (bfd_find_nearest_line(abfd, section, syms, pc - vma,
			&filename, &functionname, &line)) {
			
			if (functionname) {
				char *alloc;

				/* FIXME: workaround to remove unwanted prefix from some
				 * function names.
				 */
				if (functionname[0] == 'I' && functionname[1] == 'A' &&
					functionname[2] == '_' && functionname[3] == '_')
					functionname = functionname + 4;

				alloc = cplus_demangle(functionname, DMGL_ANSI | DMGL_PARAMS);
				if (alloc != NULL) { /* C++ demangled method */
					strncpy(addr_info->function, alloc, LINE_MAX);
					addr_info->function[LINE_MAX-1] = '\0';
					free(alloc);
				} else strcpy(addr_info->function, functionname);
			}

			if (filename)
				strcpy(addr_info->filename, filename);

			addr_info->line = line;

			return 0;
		}
	}

	return 0;
}

static bfd *open_file(const char *target,
	const char *file_name,
	int try) {

	char *places[]={".","./.debug", "/usr/lib/debug", NULL};
	char *ptr = places[try];
	char *file_dbg_name=NULL;
	bfd *abfd, *ret;

	abfd = bfd_openr (file_name, target);

	if (abfd == NULL) {
		fprintf(stderr, "%s: %s\n", file_name, bfd_errmsg(bfd_get_error()));
		return NULL;
	}

	if (!bfd_check_format (abfd, bfd_object)) {
		fprintf(stderr, "error: file %s not in executable format\n", file_name);
		return NULL;
	}

	if (ptr == NULL)
		return abfd;

	file_dbg_name = bfd_follow_gnu_debuglink (abfd, ptr);
	bfd_close(abfd);

	if (file_dbg_name != NULL)
		file_name = file_dbg_name;

	ret = open_file(target, file_name, try + 1);

	free(file_dbg_name);

	return ret;
}

t_map *find_map_entry_from_address(t_maplist *maplist,
	t_address address) {

	t_map *map;

	map = maplist->first->next; /* skip sentinel */

	while (map)
	{
		if ((address >= map->dyn_addr_init) &&
			(address < map->dyn_addr_end))
			return map;

		map = map->next;
	}

	return NULL; /* address not found on map entries */
}

int resolve_address(t_map *map,
	t_address address,
	t_address_info *addr_info) {

	bfd *abfd;
	asymbol **syms;
	long symcount;
	int ret;

	if (!map->is_absolute) /* address is not absolute */
		address -= map->dyn_addr_init;

	abfd = open_file(TARGET, map->pathname, 0);
	if (!abfd)
		return -EINVAL;

	syms = slurp_symtab(abfd, &symcount);
	if (!syms)
		return -EINVAL;

	if ((ret = translate_addresses(abfd, syms, addr_info, address)))
		return ret;

	free(syms);

	bfd_close(abfd);

	return 0;
}

int is_absolute (char *pathname) {
	FILE *file;
	Elf32_Ehdr elf_header;
	Elf32_Phdr *program_header;
	size_t ret;
	int i, is_absolute = 1;

	if (!(file = fopen(pathname, "r"))) {
		fprintf(stderr, "error: could not open file %s\n", pathname);
		return -ENOENT;
	}

	/* read ELF header */
	ret = fread(&elf_header, sizeof(elf_header), 1, file);
	if (ret != 1) {
		fprintf(stderr, "error: invalid ELF header from %s\n", pathname);
		return -EINVAL;
	}

	program_header = (Elf32_Phdr *) malloc(sizeof(Elf32_Phdr)*elf_header.e_phnum);
	if (!program_header) {
		perror("malloc");
		return -ENOMEM;
	}

	fseek(file, elf_header.e_phoff, SEEK_SET);

	/* read program header table */
	ret = fread(program_header, sizeof(Elf32_Phdr), elf_header.e_phnum, file);
	if (ret != elf_header.e_phnum) {
		fprintf(stderr, "error: could not read program header table from %s\n", pathname);
		return -EINVAL;
	}

	for (i = 0; i < elf_header.e_phnum; i++) {
		if ((program_header[i].p_type == PT_LOAD) &&
			(!program_header[i].p_offset)) {
			if (!program_header[i].p_vaddr)
				is_absolute = 0;
			break;
		}
	}

	free(program_header);
	fclose(file);

	return is_absolute;
}
