/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <config.h>

#include "options.h"

struct t_arguments arguments;

/* print usage information */
static int print_usage (FILE *stream, int exit_code)
{
	fprintf(stream, "\nfunctracer-postproc version %s.\n", PACKAGE_VERSION);
	fprintf(stream, "Supports functracer trace header version %s "
		"or greater.\n\n", FT_VERSION);
	fprintf(stream, "Usage: %s [options] <inputfile>\n", PACKAGE_NAME);
	fprintf(stream,
		"   -a   --all            Enable all options (-c, -l and -r).\n"
		"   -b   --blocks         Sort by blocks count.\n"
		"   -B   --blocklimit N   Show only if more equal than N blocks (default 1, use with -b or -s).\n"
		"   -c   --compress       Join resource allocations with same backtrace (use with -f or -l).\n"
		"   -f   --freed          List freed resource allocations.\n"
		"   -h   --help           Display this usage information.\n"
		"   -l   --leak           List non-freed resource allocations (possible leaks).\n");
	fprintf(stream,
		"   -r   --resolve        Enable symbol name resolution.\n"
		"   -s   --size           Sort by total size.\n"
		"   -v   --verbose        Print verbose messages.\n"
		"   -V   --version        Print program version.\n");

	exit(exit_code);
}

/* process options from command line parameters */
int process_options (int argc, char *argv[])
{
	int next_option;
	extern char *optarg;
	extern int optind;

	/* a string listing valid short option letters. */
	const char* short_options = "abB:cfhlrsvV";

	/* an array describing valid long options. */
	const struct option long_options[] = {
		{	"all",	0, NULL,	'a' },
		{	"blocks",	0, NULL,	'b' },
		{	"blocklimit",	1, NULL,	'B' },
		{	"compress",	0, NULL,	'c' },
		{	"freed",	0,	NULL,	'f' },
		{	"help",	0,	NULL,	'h' },
		{	"leak",	0,	NULL,	'l' },
		{	"resolve",	0,	NULL,	'r' },
		{	"size",	0, NULL,	's' },
		{	"verbose",	0, NULL,	'v' },
		{	"version",	0,	NULL,	'V' },
		{	NULL,	0,	NULL,	0 }
	};

	/* set arguments values to 0 */
	memset(&arguments, 0, sizeof(t_arguments));

	while (1)	{
		next_option = getopt_long (argc, argv, short_options, long_options, NULL);

		if (next_option == -1) break;

		switch (next_option) {
		case 'a': /* -a or --all */
			arguments.compress = 1;
			arguments.leak = 1;
			arguments.resolve = 1;
			break;
		case 'b': /* -b or --blocks */
			arguments.compress = 1;
			arguments.blocks = 1;
			break;
		case 'B': /* -B or --blocklimit */
			arguments.compress = 1;
			arguments.blocklimit = atoi(optarg);
			if (arguments.blocklimit == 0)
				arguments.blocklimit = -1;
			break;
		case 'c': /* -c or --compress */
			arguments.compress = 1;
			break;
		case 'h': /* -h or --help */
			print_usage (stdout, 0);

		case 'v': /* -v or --verbose */
			/* FIXME: verbose level */
			arguments.verbose++;
			break;

		case 'f': /* -f or -freed */
			arguments.freed = 1;
			break;

		case 'l': /* -l or --leak */
			arguments.leak = 1;
			break;

		case 'r': /* -r or --resolve */
			arguments.resolve = 1;
			break;

		case 's': /* -s or --size */
			arguments.compress = 1;
			arguments.size = 1;
			break;
		case 'V': /* -V or --version */
			fprintf(stdout, "%s\n", PACKAGE_STRING);
			exit(0);
			break;

		default:
			print_usage(stdout, -EINVAL);
		}
	}

	argc -= optind;
	if (arguments.leak && arguments.freed) {
		fprintf(stderr, "%s: conflicting options specified: -f and "
			"-l\n", PACKAGE_NAME);
		return -EINVAL;
	}
	if (arguments.blocklimit < 0) {
		fprintf(stderr, "%s: option -B with wrong argument [N > 0]\n",
			PACKAGE_NAME);
		print_usage(stdout, -EINVAL);
	}
	if (argc < 1) {
		fprintf(stderr, "%s: too few arguments\n", PACKAGE_NAME);
		print_usage(stdout, -EINVAL);
	}
	if (argv[optind])
		/* copy inputfile (if any) to arguments.trace_file */
		strcpy(arguments.trace_file, argv[optind]);
	else {
		fprintf(stderr, "error: no input file\n");
		return -EINVAL;
	}
	if (strcmp(argv[0], "functracer-postproc-reduce") == 0) {
		if (arguments.freed) {
			fprintf(stderr, "%s: conflicting option specified: "
				"-f\n", PACKAGE_NAME);
			return -EINVAL;
		}
		arguments.leak = 1;
		arguments.compress = 1;
	} else if (strcmp(argv[0], "functracer-postproc-resolve") == 0)
		arguments.resolve = 1;

	if (arguments.blocks || arguments.size) {
		arguments.compress = 1;
		if (!arguments.blocklimit)
			arguments.blocklimit = 1;
	}
	return 0;
}
