/* common.c  -  Common functions */

/* Written 1993 by Werner Almesberger */

/* FAT32, VFAT, Atari format support, and various fixes additions May 1998
 * by Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de> */

/* A simple block allocator to speed up and cut down the huge memory
 * overhead of dosfsck allocs (with full 8GB file system, cuts 120MB
 * memory usage to 80MB) 2008-02-13 by Eero Tamminen
 * <eero.tamminen@nokia.com> */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include <signal.h>
#include <unistd.h>
#include "io.h"

#include "common.h"


typedef struct _link {
    void *data;
    struct _link *next;
} LINK;

/* pointers to current larger allocation block used for small allocations */
static void *block_head = NULL;
static void *block_end  = NULL;

#define BLOCK_ALLOC_SIZE (4096-8)  /* assume this is one page with libc overhead */
#define MIN_ALLOC_FROM_BLOCK  127  /* use block alloc for anything smaller */


void die(char *msg,...)
{
    va_list args;

    va_start(args,msg);
    vfprintf(stderr,msg,args);
    va_end(args);
    fprintf(stderr,"\n");
    exit(1);
}


void pdie(char *msg,...)
{
    va_list args;

    va_start(args,msg);
    vfprintf(stderr,msg,args);
    va_end(args);
    fprintf(stderr,":%s\n",strerror(errno));
    exit(1);
}


void *alloc(size_t size)
{
    void *this;

    if ((this = malloc(size))) return this;
    if (errno == 0) {
	errno = ENOMEM;
    }
    pdie("malloc");
    return NULL; /* for GCC */
}


/* should be called only from qalloc().
 * get small allocs from a larger block and add those larger
 * blocks to linked list of qalloc()
 */
static void *qalloc_block(void **root, size_t size)
{
    void *ret;
    if (size > MIN_ALLOC_FROM_BLOCK) {
	 return NULL;
    }
    /* dosfsck doesn't use doubles, so it's safe to align just to 4 */
    size = (size+3) & ~3;
    if (!block_head || block_head + size > block_end) {
        LINK *link = alloc(sizeof(LINK));
        link->next = *root;
        *root = link;

        block_head = alloc(BLOCK_ALLOC_SIZE);
        block_end = block_head + BLOCK_ALLOC_SIZE;
        link->data = block_head;
    }
    ret = block_head;
    block_head += size;
    return ret;
}

static void qfree_blocks(void)
{
    block_head = block_end = NULL;
}


void *qalloc(void **root, size_t size)
{
    LINK *link;
    void *block;

    block = qalloc_block(root, size);
    if (block) {
        return block;
    } else {
        link = alloc(sizeof(LINK));
        link->next = *root;
        *root = link;
        return link->data = alloc(size);
    }   
}


void qfree(void **root)
{
    LINK *this;

    while (*root) {
	this = (LINK *) *root;
	*root = this->next;
	free(this->data);
	free(this);
    }
    qfree_blocks();
}


int min(int a,int b)
{
    return a < b ? a : b;
}


char get_key(char *valid,char *prompt)
{
    int ch,okay;

    while (1) {
	if (prompt) printf("%s ",prompt);
	fflush(stdout);
	while (ch = getchar(), ch == ' ' || ch == '\t');
	if (ch == EOF) exit(1);
	if (!strchr(valid,okay = ch)) okay = 0;
	while (ch = getchar(), ch != '\n' && ch != EOF);
	if (ch == EOF) exit(1);
	if (okay) return okay;
	printf("Invalid input.\n");
    }
}


static void time_limit_handler(int signo)
{
    const int errors = fs_close(0);

    printf("dosfsck: check time limit reached, file system errors are %sdetected\n", errors ? "" : "not ");
    exit(errors ? 1 : 0);
} /* time_limit_handler */

int set_time_limit(int theLimit)
{
    /* Check parameters */
    if (theLimit <= 0)
	return 0;

    if (SIG_ERR == signal(SIGALRM, time_limit_handler))
        return 0;

    /* Set the alarm for pointed time in seconds */
    alarm( (unsigned)theLimit );

    return 1;
} /* set_time_limit */


/* Local Variables: */
/* tab-width: 8     */
/* End:             */
