/*
 * @file crash-reporter-ui-utils.c
 *  
 * This file contains most of the UI operations for uploading Rich cores.
 * It uses Crash Reporter Library for uploading
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gconf/gconf-client.h>
#include <hildon/hildon.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-pannable-area.h>
#include <unistd.h>
#include <glib.h>
#include <stdlib.h> 
#include <string.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <glib/gstdio.h>
#include <errno.h>

/*
 * for appending to LZO archive
 */
#define RICH_CORE_TMP_NOTEFILE	"/tmp/rich-core-note.txt"
#define LZOP_PROG		"/usr/bin/lzop"

#include "crash-reporter-settings-file.h"
#include "crash-reporter-handler.h"
#include "crash-reporter-ui-utils.h"
#include "crash-reporter-ui-main.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

/*It stores the list of files to be uploaded*/
extern GSList *files_list;

/* Crash Reporter UI Application context */
extern osso_context_t *context;

/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void creporter_ui_set_connec_callback_notifiers();

/**
  When Crash Reporter UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files shud be appended
  @param TRUE if success else FALSE
*/
static gboolean creporter_ui_collect_all_corefile_names_at_location(const gchar *location);

static void usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer);
static void usecase_reuse_button_clicked (GtkWidget *button, GtkTextBuffer *buffer);
static void usecase_save_note_button_clicked (GtkWidget *button, GtkTextBuffer *buffer);
static void usecase_other_button_clicked (GtkWidget *button, void *arg);
static void usecase_options_button_clicked (GtkWidget *button, void *arg);
static GtkWidget *creporter_ui_usecase_dialog(gchar *text);
static void creporter_ui_process_response(gboolean send_all_files_mode, gint response);
static gchar *compose_crashed_note(gchar *filename);
static GtkWidget *creporter_ui_select_for_sending_dialog(gint num);

/*
 * counts and stats
 */
static gint tried_count = 0, succeeded_count = 0;

static gint creporter_ui_upload_files();
void creporter_ui_show_progress(char *str, double percent);

static gboolean update_progressbar (gpointer dialog);

/*
 * pointer to usecase/options/send/delete dialog,
 * so that we can call response() on it from button handlers
 */
static GtkDialog *dialog = NULL;
static GtkDialog *send_progress_dialog = NULL;
static GtkWidget* progressbar = NULL;
static GtkWidget* progresslabel = NULL;
static double progress_percent = 0.0;

GtkWidget *reuse_button = NULL;

#define PROGRESS_DLG_VISIBLE	(progressbar && progresslabel)

/*
 * declrs and defs for selected upload/delete stuff
 */
#define MAX_SENDING_SHOW (300)
#define MAX_CORE_NAMELEN_FITS_TO_BUTTON (40)
#define PANNING_THRESHOLD 30 
#define MARGIN_HALF 4 
#define MARGIN_DEFAULT 8

gint num_of_select_send_buttons = 0;
creporterSettings *gsettings = NULL;
static gchar * create_server_label_text(gint num_of_checked, gint sum_size, gchar * addr);
static void select_all_button_clicked (GtkWidget *button, void *arg);
GtkWidget *glabel_server;
GtkWidget *selection_checkall_button;
GtkWidget *selection_uncheckall_button;
GtkWidget *selection_send_button;
GtkWidget *selection_delete_button;
gint sel_num_of_checked = 0;
gint sel_sum_size = 0;

/*
 * declrs and defs for stored re-usable use case notes.
 */
#define USECASE_NOTES_STORAGE	"/home/user/.crash_reporter_usecasenotes"
#define USECASE_NOTES_MAX_SAVED	(50)

static void usecase_notes_ringbuf_fromfile(void);
static void usecase_notes_ringbuf_tofile(void);
static void usecase_notes_add_one(gchar *);
static GSList *usecase_notes_list = NULL;
static GSList *usecase_notes_curr = NULL;

static void 
creporter_ui_process_response(gboolean send_all_files_mode, gint response)
{
    struct send_sel_data *sel;
    GSList *temp;

    switch(response) {
    case GTK_RESPONSE_CANCEL:
	osso_log(LOG_DEBUG, "User cancelled");
	creporter_ui_exit();
	break;
	
    case GTK_RESPONSE_OK:
	/* Create an internet connection and wait for gconf connec callback*/
	if (!creporter_connect_iap()) {
	    creporter_ui_display_note(CONF_CONNECTION_ERROR, NULL);
	    osso_log(LOG_DEBUG, "[%s]: could not connect", __FUNCTION__);
	    creporter_ui_exit();
	}
	break;
    /* GTK_RESPONSE_APPLY is the response to delete single Core file, 
     *  or "Delete All" during creporter startup 
     */	
    case GTK_RESPONSE_APPLY:
	if(files_list) {
	    osso_log(LOG_DEBUG, "User selected Delete");
	    if(send_all_files_mode) {
		for (temp = files_list; temp ; temp = g_slist_next(temp)){
		    sel = CORE_SELECT_CTX(temp);
		    if (num_of_select_send_buttons) {
			if(sel->button && !sel->sel_state) {
			    /* button but not selected for action */
			    continue;
			}
			if(!sel->button) {
			    /* no button for this, but there are buttons,
			       means there was too many entries, but this was not selected */
			    continue;
			}
		    }
		    if(g_remove(CORE_FILE_NAME(temp)) != 0) {
			osso_log(LOG_DEBUG,"[%s] Could not remove the file [%s] errno=%d", 
				 __FUNCTION__, CORE_FILE_NAME(temp), errno);
		    }
		    g_free(temp->data);
		}
		g_slist_free(files_list);
		files_list = NULL;
	    } else {
		/*
		 * should check here for selection data for completeness,
		 * but in current code single core deletion never can have
		 * selection of cores, so here this one is not checked
		 */ 
		void *p;
		if(g_remove(CORE_FILE_NAME(files_list)) != 0)
		    osso_log(LOG_DEBUG,"[%s] Could not remove the core file [%s], errno=%d",
			     __FUNCTION__, CORE_FILE_NAME(files_list), errno);
		p = files_list->data;
		files_list = g_slist_remove(files_list, files_list->data);
		g_free(p);
		p = NULL;
	    }
	}
	creporter_ui_exit();
	break;
    default:
	creporter_ui_exit();
	break;
    }
}

/**
  This function asks and takes the user confirmation
  and establishes the iap connection

  @param gboolean TRUE when need to send all files
		  FALSE when only one file is to be sent
  @param void              
*/
void creporter_ui_ask_user_conf(gboolean send_all_files_mode)
{
    gint response = 0;

    /* Display user confirmation to send rich cores */
    if(!send_all_files_mode) {
	response = creporter_ui_display_note(CONF_USER_CONFIRMATION1, NULL);
    }
    else {
	response = creporter_ui_display_note(CONF_USER_CONFIRMATION2, NULL);
    }
    creporter_ui_process_response(send_all_files_mode, response);
}

/* Callback for other buttons */
static void
send_progress_button_clicked (GtkWidget *button, void *arg)
{
    gint response = (gint)arg;

    if (send_progress_dialog) {
	gtk_dialog_response(send_progress_dialog, response);
    }
}

static GtkWidget *
creporter_ui_send_progress_dialog ()
{
    static GtkWidget *window;
    GtkWidget *cancel_button;
    
    window = gtk_dialog_new ();

    gtk_window_set_title (GTK_WINDOW (window), _("Uploading crash report"));
    //    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_widget_set_size_request(window, 600, 150);
    gtk_window_set_modal (GTK_WINDOW (window), FALSE);

    progresslabel = gtk_label_new("initial note");
    gtk_label_set_line_wrap(GTK_LABEL(progresslabel), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), GTK_WIDGET(progresslabel), 
                        TRUE, TRUE, 0);

    /*------------*/
    /* Add progress bar */
    progressbar = gtk_progress_bar_new();
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), 
			progressbar, TRUE, TRUE, 0);
    g_timeout_add(1000, update_progressbar, NULL);

    /*------------*/
    /* Add a "Stop" button to the bottom of the dialog */
    cancel_button = gtk_dialog_add_button(GTK_DIALOG(window),
					  dgettext("hildon-libs",
						   "wdgt_bd_stop"),
					  GTK_RESPONSE_CANCEL);
					  
    g_signal_connect (G_OBJECT (cancel_button), "clicked",
		      G_CALLBACK (send_progress_button_clicked),
		      (void*)GTK_RESPONSE_CANCEL); 
    gtk_widget_set_no_show_all(cancel_button, FALSE);

    gtk_widget_show_all (window);

    return window;
}

/**
  This function is called to upload all the files that stored in files_list
  This will call the creporter_libs APIs to upload the each file

  @param void
  @return void
  */
static gint 
creporter_ui_upload_files(crash_upload_context ** pctx)
{
    GSList *temp = NULL;
    gint rv = -1;
    gint response;
    GtkWidget *wid = NULL;
    gint cur, tot;
    gchar * size_note;
    struct stat file_info;
    gchar *basename = NULL;
    GThread* send_thread = NULL;
    creporterSettings *settings = NULL;
    crash_upload_context * ctx = NULL;
    struct send_sel_data *sel;

    settings = creporter_get_settings();
    if (!settings) {
	osso_log(LOG_DEBUG, "[%s]: Unable to read settings", __FUNCTION__);
	return CREPORTER_NO_SETTINGS_ERROR;
    } else {
	osso_log(LOG_DEBUG, "[%s]: Settings: '%s'-'%s'-'%s'", 
		 __FUNCTION__, settings->server_addr,
		 settings->user_name, settings->passwd);
    }

    ctx = g_new0(crash_upload_context, 1);
    ctx->settings = settings;
    /*
     * if selection was made, total number is by selected set,
     * otherwise it is length of list
     */
    if (num_of_select_send_buttons) {
	tot = sel_num_of_checked;
    } else {
	tot = g_slist_length(files_list);
    }

    tried_count = succeeded_count = 0;
    if (!send_progress_dialog) {
	wid = creporter_ui_send_progress_dialog();
	send_progress_dialog = GTK_DIALOG(wid);
	gtk_window_set_title(GTK_WINDOW(send_progress_dialog), _("Sending crash report"));
    }
    /*
     * cur:counts cores we send (for display)
     */
    for(cur = 0, temp = files_list; temp; temp = g_slist_next(temp)) {

	/*
	 * Skip processing if there is no file (means, it was just sent
	 * and deleted on previous loop round)
	 */
	if(access(CORE_FILE_NAME(temp), R_OK) < 0) {
	    osso_log(LOG_DEBUG, "[%s]: File %s not found, skip", __FUNCTION__, CORE_FILE_NAME(temp));
	    continue;
	}

	/*
	 * selected upload support: if we have selections state,
	 * check it and skip if entry was not selected
	 */
	sel = CORE_SELECT_CTX(temp);
	if (num_of_select_send_buttons) {
	    if(sel->button && !sel->sel_state) {
		/* button but not selected for action */
		continue;
	    }
	    if(!sel->button) {
		/* no button for this, but there are buttons,
		   means there was too many entries, but this was not selected */
		continue;
	    }
	}
	strncpy(ctx->filename, CORE_FILE_NAME(temp), sizeof(ctx->filename) - 1);
	cur += 1;

	if (send_thread) {
	    osso_log(LOG_DEBUG, "[%s] previous upload thread still active, wait...", __FUNCTION__);
	    rv = (gint)g_thread_join(send_thread);
	    osso_log(LOG_DEBUG, "[%s] join on previous upload thread returned %d", __FUNCTION__, rv);
	}
	send_thread = g_thread_create((GThreadFunc)thread_creporter_upload,
				      ctx, TRUE, NULL);

	tried_count += 1;
	if (PROGRESS_DLG_VISIBLE) {
	    if(stat(CORE_FILE_NAME(temp), &file_info) < 0) {
		osso_log(LOG_DEBUG, "[%s]: can  not stat %s, skip", __FUNCTION__, CORE_FILE_NAME(temp));
		continue;
	    }
	    basename = g_filename_display_basename(CORE_FILE_NAME(temp));
	    if (tot > 1) {
		gchar * sending_note = g_strdup_printf(_("Sending crash report %d / %d"), cur, tot);
		gtk_window_set_title (GTK_WINDOW (send_progress_dialog), sending_note);
		g_free(sending_note);
	    } 
	    gtk_label_set_text(GTK_LABEL(progresslabel), basename);
	    size_note = g_strdup_printf(_("%ld kB"), file_info.st_size / 1024);
	    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progressbar), size_note);
	    progress_percent = 0.0;
	    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar), progress_percent);
	    g_free(basename);
	    g_free(size_note);
	    response = gtk_dialog_run(send_progress_dialog);
	} else {
	    /*
	     * dialog was hidden, response remains CLOSE and we go waiting 
	     * on thread again. Just to be sure, and for clarity,
	     * set it CLOSE once more here
	     */
	    response = GTK_RESPONSE_CLOSE;
	}
	if (response == GTK_RESPONSE_CANCEL) {
	    /*  cancel or dialog delete */
	    gtk_widget_destroy(GTK_WIDGET(wid));
	    send_progress_dialog = NULL;
	    progressbar = NULL;
	    progresslabel = NULL;
	    rv = CREPORTER_CANCEL;
	    break;
	}
	else if ((response == GTK_RESPONSE_CLOSE) ||
	    (response == GTK_RESPONSE_DELETE_EVENT)) {
	    /*  hide. continue sending in background */
	    gtk_widget_hide(GTK_WIDGET(send_progress_dialog));
	    while (gtk_events_pending ()) {
		gtk_main_iteration();
	    }
	    progressbar = NULL;
	    progresslabel = NULL;
	    rv = (gint)g_thread_join(send_thread);
	    if (rv == CREPORTER_SUCCESS) {
		succeeded_count += 1;
	    }
	    send_thread = NULL;
	    continue;
	}
	/* returns with status OK if progressbar indicates "complete" by special sign.
	* Otherwise we can not get out of dialog_run easily 
	*/
	else if (response == GTK_RESPONSE_OK) {
	    rv = (gint)g_thread_join(send_thread);
	    send_thread = NULL;
	    if (rv == CREPORTER_SUCCESS) {
		succeeded_count += 1;
	    }
	}
	else {
	    osso_log(LOG_DEBUG, "[%s] unexpected ret val %d from progress dialog run", __FUNCTION__, response);
	}
    }
    if (send_progress_dialog) {
	gtk_widget_destroy(GTK_WIDGET(wid));
	send_progress_dialog = NULL;
    }
    progressbar = NULL;
    progresslabel = NULL;
    g_slist_free(files_list);
    files_list = NULL;
    *pctx = ctx;
    return rv;
}

/**
  This is callback called when there are any changes to the variable
  CREPORTER_GCONF_CONNECTION_STATUS_DIR ie, to get the connection status

  @param the GConfClient that is notifying 
  @param cnxn_id, connection ID from gconf_client_notify_add()
  @param a GConfEntry entry
  @param user data
  @retuen void 
  */
void static creporter_ui_connect_status_cb (GConfClient *gconf_client_connec_dir,
				guint cnxn_id, GConfEntry *entry,
				gpointer user_data)
{
	creporterConnectionState connec_status = -1;
	gint result = -1;
	crash_upload_context * ctx = NULL;

	creporter_libs_gconf_get_connc_status(&connec_status);

	if(connec_status == CONN_CONNECTED){
		/* Succesfully able to connect */
		result = creporter_ui_upload_files(&ctx);
	
		if(result == CREPORTER_SUCCESS) {
			/*upload success */
			creporter_ui_display_note(CONF_FILE_SENT, ctx);
			creporter_ui_exit();	
		} else if (result == CREPORTER_CANCEL) {
			osso_log(LOG_DEBUG, "[%s]: upload cancelled by user", 
				 __FUNCTION__);
			creporter_ui_exit();
		} else if ((result == -1) && (tried_count == 0)) {
			osso_log(LOG_DEBUG, "[%s]: no files were found to upload", 
				 __FUNCTION__);
			/*
			 * it was not really CURL error because we did
			 * not reach curl phase, but this shows handy generic
			 * error msg when no context given
			 */
			creporter_ui_display_note(CONF_CURL_ERROR, NULL);
			creporter_ui_exit();
		} else {
		        /* Upload operation failed */
		    if (dialog) {
			osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CURL_ERROR note", 
				 __FUNCTION__);
		    } else {
			osso_log(LOG_DEBUG,"[%s] Upload operation failed, Error=%d", __FUNCTION__, result);
			creporter_ui_display_note(CONF_CURL_ERROR, ctx);
			creporter_ui_exit();
		    }
		}
	} else if(connec_status == CONN_DISCONNECTED){
	    if (dialog) {
		osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CONNECTION_ERROR note", 
			 __FUNCTION__);
	    } else {
		osso_log(LOG_DEBUG, "[%s] Couldnot create connection, connec_status error is %d", __FUNCTION__, connec_status);
		creporter_ui_display_note(CONF_CONNECTION_ERROR, NULL);
		creporter_ui_exit();
	    }
	}
}

/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void creporter_ui_set_connec_callback_notifiers()
{
	GConfClient *gconf_client_connec_dir = NULL;
	gpointer user_data = NULL; 

	/* set callback notifier for gconf variable
	 * CREPORTER_GCONF_CONNECTION_STATUS*/	
	gconf_client_connec_dir = gconf_client_get_default();

	gconf_client_add_dir(gconf_client_connec_dir, 
			CREPORTER_GCONF_CONNECTION_STATUS_DIR,
			GCONF_CLIENT_PRELOAD_NONE, NULL	);

	gconf_client_notify_add(gconf_client_connec_dir, 
			CREPORTER_GCONF_CONNECTION_STATUS, 
			creporter_ui_connect_status_cb, 
			user_data, NULL, NULL);
}

static gchar *
compose_crashed_note (gchar *filename)
{
    gchar *text = NULL;
    gchar *text2 = NULL;
    gchar *sig_num = NULL;
    gchar *exe_name = NULL;
    gchar *process_id = NULL;
    gint size;

    creporter_get_rcore_fileinfo(filename, &size, &sig_num, &exe_name, &process_id);
    if (sig_num && exe_name && process_id) {
	text = g_strdup_printf(_("'%s' PID %s died to signal %s"), exe_name, process_id, sig_num);
    } else {
	text = g_strdup_printf(_("A crash file %s was generated"), filename);
    }
    text2 = g_strdup_printf(_("%s. Send %d kB crash report for analysis?"), 
			    text, size/1024); 

    g_free(sig_num);
    g_free(exe_name);
    g_free(process_id);
    g_free(text);

    return text2;
}

/**
  This function is used to display all the confirmation notes in the appl.

  @param the type confirmation note to be displayed
  @return the response from the user
  */
gint 
creporter_ui_display_note (creporterConfirmationType conf_type, crash_upload_context *ctx)
{
    GtkWidget *note = NULL;
    gchar *text = NULL;
    gint response = 0;
    creporterSettings *sett = NULL;
    gint num;
    gchar *basename = NULL;

    if (dialog) {
	osso_log(LOG_DEBUG, "[%s]: dialog already active, skip and return", 
		 __FUNCTION__);
	if (ctx) {
	    g_free(ctx);
	}
	return response;
    }
    num_of_select_send_buttons = 0;
    switch(conf_type) {
    case CONF_USER_CONFIRMATION1:
	if(!files_list) {
	    osso_log(LOG_DEBUG, "[%s]: Files list is NULL",__FUNCTION__);
	    if (ctx) {
		g_free(ctx);
	    }
	    return GTK_RESPONSE_CANCEL;
	}
	usecase_notes_ringbuf_fromfile();
	text = compose_crashed_note(CORE_FILE_NAME(files_list));
	note = creporter_ui_usecase_dialog(text);
	break;
    case CONF_USER_CONFIRMATION2:
	sett = creporter_get_settings();
	num = g_slist_length(files_list);
	if (num > 0) {
	    note = creporter_ui_select_for_sending_dialog(num);
	} else {
	    text = g_strdup(_("This system has no stored crash reports."));
	    note = hildon_note_new_information(NULL, text);
	}
	break;
    case CONF_CONNECTION_ERROR:
	text = g_strdup(_("Crash Reporter failed to connect"));
	note = hildon_note_new_information(NULL, text);
	break;
    case CONF_CURL_ERROR:
	if (ctx) {
	    basename = g_filename_display_basename(ctx->filename);
	    if (ctx->server_response[0]) {
		text = g_strdup_printf(_("\
Unexpected response from crash reports server, \
file %s, send status %d, \
http status %d.\nServer response was:\n%s"), 
				       basename, ctx->curl_status, 
				       ctx->http_resp_code, ctx->server_response);
	    } else {
		text = g_strdup_printf(_("\
Crash report upload did not succeed, \
file %s, problem connecting to server, send status %d"),
				       basename, ctx->curl_status);
	    }
	    g_free(basename);
	} else {
	    text = g_strdup(_("Problem with the crash report upload."));
	}
	note = hildon_note_new_information(NULL, text);
	break;
    case CONF_FILE_SENT:
	if (tried_count == 1) {
	    if(succeeded_count == 1) {
		if (ctx) {
		    gchar *basename = g_filename_display_basename(ctx->filename);
		    text = g_strdup_printf(_("The crash report %s was successfully uploaded"), basename);
		    g_free(basename);
		} else {
		    text = g_strdup(_("The crash report was successfully uploaded"));
		}
	    } else {
		/* 
		 * this should not happen .. 
		 */
		text = g_strdup(_("The crash report sending failed"));
	    }
	} else {
	    text = g_strdup_printf(_("Crash reports upload result: %d files attempted, %d succeeded"),
				   tried_count, succeeded_count);
	}
	note = hildon_note_new_information(NULL, text);
	break;
    case NO_MORE_DUPS:
	basename = g_filename_display_basename(ctx->filename);
	text = g_strdup_printf(_("Crash Reporter detected %d duplicates for the %s crash report, the following duplicates will be silently deleted."),
				   MAX_SIMILAR_RCORES, basename);
	note = hildon_note_new_information(NULL, text);
	break;
    default:
	/*
	 * note being NULL protects us from using/showing it below
	 */
	break;
    }
        
    if (note) {
	dialog = GTK_DIALOG(note);
	/*
	 * This loop makes repeated dialog via RESPONSE_YES return value, 
	 * used for example from "send all" dialog when Select/Deselect ALL us used
	 */
	do {
	    response = gtk_dialog_run(dialog);
	} while (response == GTK_RESPONSE_YES);

	dialog = NULL;
	gtk_widget_destroy(GTK_WIDGET(note));
	note = NULL;
    }
    if (text) {
	g_free(text);
    }
    if (ctx) {
	g_free(ctx);
    }
    reuse_button = NULL;
    return response;
}

/* 
 * Store re-usable use case notes to a file.
 * NULL-terminated strings are stored in file one after other,
 * NULL-char included.
 */
static void
usecase_notes_ringbuf_tofile ()
{
    gchar *buf = NULL;
    gsize len, thislen;
    GSList *temp;
    gint cnt;

    if (usecase_notes_list) {
	len = cnt = 0;
	for (temp = usecase_notes_list; temp ; temp = g_slist_next(temp)){
	    thislen = strlen((gchar*)temp->data);
	    buf = g_realloc(buf, len + thislen + 1);
	    strcpy(buf + len, (gchar*)temp->data);
	    len += thislen + 1;
	    /*
	     * we limit endless growth of list by saving max N entries.
	     * Because we only add at head, this discards older entries.
	     */
	    if( ++cnt >= USECASE_NOTES_MAX_SAVED ) {
		break;
	    }
	}
	g_file_set_contents(USECASE_NOTES_STORAGE, buf,
			    len, NULL);
    }
}

/* 
 * Read re-usable use case notes from a file.
 * NULL-terminated strings are in file one after other,
 * NULL-char included.
 */
static void
usecase_notes_ringbuf_fromfile ()
{
    gchar *buf = NULL;
    gsize len;
    gchar *p, *end;
    gboolean rv = g_file_get_contents(USECASE_NOTES_STORAGE, &buf,
				      &len, NULL);
    if (rv) {
	p = buf;
	end = p + len;
	while (p < end) {
	    usecase_notes_list = g_slist_prepend(usecase_notes_list, g_strdup(p));
	    usecase_notes_curr = usecase_notes_list;
	    while (p < end && *p) {
		p++;
	    }
	    p++;
	}
	g_free(buf);
	if (reuse_button && usecase_notes_list && usecase_notes_curr) {
	    gtk_widget_set_sensitive(reuse_button, TRUE);
	}
    }
}

/*
 * insert text to list of re-usable usecase notes
 * Note. this will store argument "text" 
 * directly in list without making a copy, so
 * caller of this should not free it after call.
 */
static void 
usecase_notes_add_one(gchar *text)
{
    GSList *temp;

    if (strlen(text) == 0) {
	return;
    }
    for (temp = usecase_notes_list; temp ; temp = g_slist_next(temp)){
	/*
	 * only if such entry not on list already.
	 */
	if( !g_strcmp0(text, (gchar*)temp->data)) {
	    g_free(text);
	    return;
	}
    }
    usecase_notes_list = g_slist_prepend(usecase_notes_list, text);
    usecase_notes_curr = usecase_notes_list;
    usecase_notes_ringbuf_tofile();
    if (usecase_notes_list && usecase_notes_curr) {
	gtk_widget_set_sensitive(reuse_button, TRUE);
    }
}

/*
 * re-use one of previously used/stored use case comments
 */
static void
usecase_reuse_button_clicked (GtkWidget *button, GtkTextBuffer *buffer)
{
    if (usecase_notes_list && usecase_notes_curr) {
	gtk_text_buffer_set_text(buffer, (gchar *)usecase_notes_curr->data, -1);
	if(usecase_notes_curr->next) {
	    usecase_notes_curr = usecase_notes_curr->next;
	} else {
	    usecase_notes_curr = usecase_notes_list;
	}
    }
}

/* 
 * "add current note into ring-buffer of saved notes" button 
 */
static void
usecase_save_note_button_clicked (GtkWidget *button, GtkTextBuffer *buffer)
{
    GtkTextIter start;
    GtkTextIter end;
    gchar *text;
    
    /* Obtain iters for the start and end of points of the buffer */
    gtk_text_buffer_get_start_iter (buffer, &start);
    gtk_text_buffer_get_end_iter (buffer, &end);

    /* Get the entire buffer text. */
    text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

    /*
     * this eats text and keeps it so we are not leaking it here
     */
    usecase_notes_add_one(text);
}

/* Callback for Send button */
static void
usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer)
{
    GtkTextIter start;
    GtkTextIter end;
    gchar *text;
    gchar *cmd;
    FILE *fp;
    gint len, cmd_len;
    int res;

    /* Obtain iters for the start and end of points of the buffer */
    gtk_text_buffer_get_start_iter (buffer, &start);
    gtk_text_buffer_get_end_iter (buffer, &end);

    /* Get the entire buffer text. */
    text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
    len = strlen(text);

    if (len > 0) {
	fp = fopen(RICH_CORE_TMP_NOTEFILE, "w");
	if(fp){
	    if (fwrite(text, sizeof(char), len, fp) < len) {
		osso_log(LOG_ERR, "[%s]: error writing rich-core-note to file %s", 
			 __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	    }
	    fclose(fp);
	    cmd_len = sizeof(LZOP_PROG) + sizeof(RICH_CORE_TMP_NOTEFILE) + strlen(CORE_FILE_NAME(files_list)) + 20;
	    cmd = g_malloc(cmd_len);
	    if (cmd) {
		sprintf(cmd, "%s -c %s >> %s", LZOP_PROG, RICH_CORE_TMP_NOTEFILE, CORE_FILE_NAME(files_list));
		res = system(cmd);
		g_free(cmd);
	    }
	    unlink(RICH_CORE_TMP_NOTEFILE);
	} else {
	    osso_log(LOG_ERR,"[%s]: error opening file %s", __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	}
	/*
	 * add it also to list of saved notes for re-use.
	 * Note. this will keep text used in list so no need to free it
	 */
	usecase_notes_add_one(text);
    }

    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, GTK_RESPONSE_OK);
    }
}

/* Callback for options button */
static void
usecase_options_button_clicked (GtkWidget *button, void *arg)
{
    gint res = 0;
    GtkWidget *top_window = (GtkWidget*)arg;
    
    if (context && top_window) {
	res = osso_cp_plugin_execute(context, "libcpcrashreporter.so", top_window, TRUE);
    } else {
	osso_log(LOG_DEBUG, "[%s]: no osso context or topwindow?",__FUNCTION__);
    }
}

/* Callback for other buttons */
static void
usecase_other_button_clicked (GtkWidget *button, void *arg)
{
    gint response = (gint)arg;
    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, response);
    }
}

/**
  Create elements of "describe what you were doing" and 
  send/cancel/del/options dialog
  */
static GtkWidget *
creporter_ui_usecase_dialog (gchar *text)
{
    static GtkWidget *window;
    GtkWidget *scrolled_window;
    GtkWidget *label_note;
    GtkWidget *label_server;
    GtkWidget *save_note_button;
    GtkWidget *send_button;
    GtkWidget *delete_button;
    GtkWidget *options_button;
    GtkWidget *text_view;
    GtkTextBuffer *buffer;
    GtkWidget *frame;
    creporterSettings *sett = NULL;
    
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    window = gtk_dialog_new ();
    /***
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                        (GtkSignalFunc) destroy, NULL);
    **/
    gtk_window_set_title (GTK_WINDOW (window), _("Crash Reporter"));
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_widget_set_size_request(window, 700, 360);
    label_note = gtk_label_new(text);
    gtk_label_set_line_wrap(GTK_LABEL(label_note), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), label_note, 
                        FALSE, FALSE, 0);

    /* create a new scrolled window. */
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);

    /* the policy is one of GTK_POLICY AUTOMATIC, or GTK_POLICY_ALWAYS.
     * GTK_POLICY_AUTOMATIC will automatically decide whether you need
     * scrollbars, whereas GTK_POLICY_ALWAYS will always leave the scrollbars
     * there.  The first one is the horizontal scrollbar, the second, 
     * the vertical. */
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    frame = gtk_frame_new(_("Please describe what you were doing:"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), frame, 
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);

    /* Create a multiline text widget. */
    text_view = gtk_text_view_new ();
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_WORD);
    /* Obtaining the buffer associated with the widget. */
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));

    /* pack the text widget into the scrolled window */
    gtk_scrolled_window_add_with_viewport (
                   GTK_SCROLLED_WINDOW (scrolled_window), text_view);

    /*------------*/
    sett = creporter_get_settings();
    if (sett) {
	gchar *srv_txt;
	srv_txt = g_strdup_printf(_("Receiving server: %s"), sett->server_addr); 
	label_server = gtk_label_new(srv_txt);
	gtk_label_set_line_wrap(GTK_LABEL(label_server), TRUE);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), label_server, 
			    FALSE, FALSE, 0);
    }
    /*------------*/
    /* "Re-use note" button */
    reuse_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Previous note"),
					GTK_RESPONSE_YES);
    g_signal_connect (G_OBJECT (reuse_button), "clicked",
		      G_CALLBACK (usecase_reuse_button_clicked),
		      buffer);
    if (!(usecase_notes_list && usecase_notes_curr)) {
	gtk_widget_set_sensitive(reuse_button, FALSE);
    }
    
    /*------------*/
    /* "Save note as template" button */
    save_note_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Keep this note"),
					     GTK_RESPONSE_YES);
    g_signal_connect (G_OBJECT (save_note_button), "clicked",
		      G_CALLBACK (usecase_save_note_button_clicked),
		      buffer);
    
    /*------------*/

    /* FIXME: Now that the buttons are created with gtk_dialog_add_button,
       connecting signals to buttons should perhaps be replaced by
       local gtk_dialog_run + switch/case setup? */

    /* Add a "Send" button to the bottom of the dialog */
    send_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Send"),
					GTK_RESPONSE_NONE);
    g_signal_connect (G_OBJECT (send_button), "clicked",
		      G_CALLBACK (usecase_send_button_clicked),
		      buffer);
    
    /*
     * excluding default grab (and CAN_DEFAULT above) to 
     * clarify the state with default action...
     * problem we have currently (Crash Reporter 1.20) is that
     * when pressing Enter while in multiline edit field, this warning comes:
     *
     GLIB WARNING ** GLib-GObject - gsignal.c:1019: unable to lookup signal "activate" of unloaded type `GtkEntry'
     *
     * currently no idea why is that, at least excluding those 2 things here does not help..
     */
    /*------------*/
    /* Add a "Options" button to the bottom of the dialog */
    options_button = gtk_button_new_with_label(_("Options"));
    
    hildon_gtk_widget_set_theme_size(options_button, 
				     HILDON_SIZE_FINGER_HEIGHT);

    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (window)->action_area),
		       options_button, TRUE, TRUE, 0);

    g_signal_connect (G_OBJECT (options_button), "clicked",
		      G_CALLBACK(usecase_options_button_clicked),
		      window);
    
    /*------------*/
    /* Add a "delete" button to the bottom of the dialog */
    delete_button = gtk_dialog_add_button(GTK_DIALOG(window), 
					  _("Delete"), GTK_RESPONSE_APPLY);
    g_signal_connect (G_OBJECT (delete_button), "clicked",
		      G_CALLBACK(usecase_other_button_clicked),
		      (void*)GTK_RESPONSE_APPLY);
    
    gtk_widget_show_all (window);

    return window;
}


static gchar *
create_server_label_text (gint num_of_checked, gint sum_size, 
			  gchar * addr)
{
    /*
     * original sum_size is in kilo-bytes. for large values show MB or GB
     */
    gfloat shown_size = sum_size;
    gchar magn = 'k';
    static gchar *labelstr = NULL;

    if (shown_size > 1024) {
	magn = 'M';
	shown_size /= 1024;
    }
    if (shown_size > 1024) {
	magn = 'G';
	shown_size /= 1024;
    }
    if (labelstr) {
	g_free(labelstr);
    }
    labelstr = g_strdup_printf(_("%d selected, %.2f %cB\nServer: %s"), 
			       num_of_checked, shown_size, magn, addr); 
    return labelstr;
}

/**
  This function is called as callback when one coredump "send enabled" button
  gets toggled
  
  @param b is pointer to button widget
  @param data is pointer to selection context

  */
static gboolean 
select_sending_toggle_one_callback (HildonCheckButton *b, gpointer data) {
    struct send_sel_data * sel = (struct send_sel_data *)data;
    gchar new_state;

    if(!sel) {
	return FALSE;
    }
    new_state = (gchar)hildon_check_button_get_active(b);
    if(new_state && !sel->sel_state) {
	sel_num_of_checked += 1;
	sel_sum_size += sel->size;
    } else if (!new_state && sel->sel_state) {
	sel_num_of_checked -= 1;
	sel_sum_size -= sel->size;
    }
    sel->sel_state = new_state;

    if (gsettings && glabel_server) {
	gtk_label_set_label(GTK_LABEL(glabel_server), 
			    create_server_label_text(sel_num_of_checked, sel_sum_size, 
						     gsettings->server_addr));
    }

    if (sel_num_of_checked == 0) {
	gtk_widget_set_sensitive(selection_send_button, FALSE);
	gtk_widget_set_sensitive(selection_delete_button, FALSE);
	gtk_widget_set_sensitive(selection_uncheckall_button, FALSE);
	gtk_widget_set_sensitive(selection_checkall_button, TRUE);
    } else {
	gtk_widget_set_sensitive(selection_uncheckall_button, TRUE);
	gtk_widget_set_sensitive(selection_send_button, TRUE);
	gtk_widget_set_sensitive(selection_delete_button, TRUE);
	if (sel_num_of_checked == num_of_select_send_buttons) {
	    gtk_widget_set_sensitive(selection_checkall_button, FALSE);
	} else {
	    gtk_widget_set_sensitive(selection_checkall_button, TRUE);
	}
    }
    return TRUE;
}

/* Callback for Select or Deselect ALL button */
static void
select_all_button_clicked (GtkWidget *button, void *arg)
{
    GSList *temp;
    struct send_sel_data *sel;
    gboolean new_state = (gboolean)arg;

    for (temp = files_list; temp ; temp = g_slist_next(temp)) {
	sel = CORE_SELECT_CTX(temp);
	if ((sel->sel_state != new_state) && sel->button) {
	    hildon_check_button_set_active(HILDON_CHECK_BUTTON(sel->button), new_state);
	}
    }
}

/**
  Create "this system has N core dumps, select for sending" dialog
  */
static GtkWidget *
creporter_ui_select_for_sending_dialog (gint num)
{
    static GtkWidget *window;
    GtkWidget * vbox2;
    GtkBox * vbox;
    GtkWidget * pan;
    GSList *temp = NULL;
    gchar * s = NULL;
    struct stat file_info;
    gchar *title = NULL;
    struct send_sel_data *sel;
    gint len;
 
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    window = gtk_dialog_new ();
    if (!window) {
	return NULL;
    }
    title = g_strdup_printf(_("Crash Reporter is building list of reports ..."));
    gtk_window_set_title (GTK_WINDOW (window), title);
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    //    gtk_widget_set_size_request(window, 700, 380);

    gtk_widget_show_now(window);

    vbox = GTK_BOX((GTK_DIALOG(window))->vbox);

    pan = hildon_pannable_area_new ();
    g_object_set (G_OBJECT (pan), "initial-hint", TRUE, 
		  "mov-mode", HILDON_MOVEMENT_MODE_VERT,
		  "panning_threshold" ,PANNING_THRESHOLD,
		  NULL);    
 
    gtk_widget_set_size_request (pan, -1, 300); 
    vbox2 = gtk_vbox_new (FALSE, MARGIN_HALF);      

    for (temp = files_list, num_of_select_send_buttons = 0 ; 
	 temp && (num_of_select_send_buttons < MAX_SENDING_SHOW); 
	 temp = g_slist_next(temp)) {

	bzero(&file_info, sizeof(file_info));
	if(stat(CORE_FILE_NAME(temp), &file_info) == 0) {
	    gchar *basename;
	    gchar *basename2;
	    gchar *psrc;
	    gchar *pdest;
	    gchar *p;

	    sel = CORE_SELECT_CTX(temp);
	    sel->size = file_info.st_size/1024;
	    sel->button = hildon_check_button_new(HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	    hildon_check_button_set_active(HILDON_CHECK_BUTTON(sel->button), FALSE);

	    basename = g_filename_display_basename(CORE_FILE_NAME(temp));
	    if ((p = g_strrstr(basename, ".rcore.lzo")) != NULL) {
		*p = 0;
	    }
	    if (strlen(basename) > MAX_CORE_NAMELEN_FITS_TO_BUTTON + 2) {
		p = basename + MAX_CORE_NAMELEN_FITS_TO_BUTTON;
		*p = *(p+1) = *(p+2) = '.';
		*(p+3) = 0;
	    }
	    /*
	     * underscores will not show unless doubled, this is GTK button label feature.
	     * I tried to overwork this with gtk_button_set_use_underline(FALSE)
	     * but it does not help, buttons will show no undersocres.
	     * Therefore we will tediously make a longer copy if there is any '_'
	     */
	    len = strlen(basename);
	    if (g_strstr_len(basename, len, "_")) {
		basename2 = g_malloc(2*len + 1);
		for (psrc = basename, pdest = basename2 ; *psrc; psrc++, pdest++) {
		    *pdest = *psrc;
		    if (*pdest == '_') {
			*(pdest + 1) = '_';
			pdest++;
		    }
		}
		*pdest = 0;
		g_free(basename);
		basename = basename2;
	    }
	    s = g_strdup_printf(_("%s\n%d kB\t%s"), 
				basename, sel->size, ctime(&file_info.st_mtime));
	    g_free(basename);
	    gtk_button_set_label (GTK_BUTTON (sel->button), s);
	    /*	    gtk_button_set_use_underline(GTK_BUTTON(sel->button), FALSE);*/
	    hildon_helper_set_logical_font(sel->button, "SmallSystemFont");

	    gtk_box_pack_start (GTK_BOX (vbox2), sel->button, FALSE, FALSE, 0);
	    g_signal_connect (sel->button, "toggled",
			      G_CALLBACK (select_sending_toggle_one_callback), sel);
	    num_of_select_send_buttons += 1;
	}
    }

     hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (pan),
					     GTK_WIDGET (vbox2));
     gtk_box_pack_start (vbox, pan, FALSE, FALSE, MARGIN_DEFAULT);

    /*------------*/
    gsettings = creporter_get_settings();
    if (gsettings) {
	glabel_server = gtk_label_new(create_server_label_text(0, 0, 
							       gsettings->server_addr));
	//	hildon_helper_set_logical_font(glabel_server, "SmallSystemFont");
	gtk_label_set_line_wrap(GTK_LABEL(glabel_server), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), glabel_server, 
			    FALSE, FALSE, 0);
    }
    /*------------*/
    /* Add a "Select all" button */
    selection_checkall_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Select all"),
						      GTK_RESPONSE_YES);
    g_signal_connect (G_OBJECT (selection_checkall_button), "clicked",
		      G_CALLBACK (select_all_button_clicked), (void *)TRUE);
    
    /* Add a "Deselect all" button. Note: same handler, different arg for those two */
    selection_uncheckall_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Deselect all"),
							GTK_RESPONSE_YES);
    g_signal_connect (G_OBJECT (selection_uncheckall_button), "clicked",
		      G_CALLBACK (select_all_button_clicked), FALSE);
    
    /* Add a "Send selected" button */
    selection_send_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Send selected"),
						  GTK_RESPONSE_OK);
    
    /* Add "Delete selected" button */
    selection_delete_button = gtk_dialog_add_button(GTK_DIALOG(window), 
						    _("Delete selected"), GTK_RESPONSE_APPLY);

    gtk_widget_set_sensitive(selection_uncheckall_button, FALSE);
    gtk_widget_set_sensitive(selection_send_button, FALSE);
    gtk_widget_set_sensitive(selection_delete_button, FALSE);
    title = g_strdup_printf(_("Crash Reporter has %u stored %s%s"),
			    num, (num>1)? _("reports") : _("report"),
			    (num > MAX_SENDING_SHOW)? _(" (not all shown)") : "");

    gtk_window_set_title (GTK_WINDOW (window), title);
    gtk_widget_show_all (window);
    return window;
}

static gboolean
update_progressbar (gpointer dialog)
{
    /* 
     * If no progressbar, stop executing and unregister this function
     */
    if ( !(progressbar && send_progress_dialog) ) {
	return 0;
    }
    /*
     * if not visible don't update
     */
    if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(progressbar))) {
	return 1;
    }

    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar), progress_percent);

    return 1;
}

void creporter_ui_show_progress(char *str, double percent)
{
    if(str == NULL) {
	/* str=NULL is end sign */
	if (send_progress_dialog && PROGRESS_DLG_VISIBLE) {
	    gtk_dialog_response(send_progress_dialog, GTK_RESPONSE_OK);
	} 
    } else if (percent >= 0.0 && percent <= 1.0) {
	progress_percent = percent;
    }
}

/**
  Create elements of "describe use case" and 
  send/cancel/del/options dialog
  */

/**
  When Crash Reporter UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files should be appended
  @param TRUE if success else FALSE
*/
static gboolean creporter_ui_collect_all_corefile_names_at_location(const gchar *location)
{
	GDir* dir;
	const gchar* fil;
	gchar * new_elem = NULL;
	
	g_return_val_if_fail(location != NULL, FALSE);
	
	dir = g_dir_open(location, 0, NULL);
	if (dir != NULL) {
	    while ((fil = g_dir_read_name(dir))) {
		if(core_is_valid(fil)) {
		    /*
		     * location has '/' prepended, so no need to put
		     * it in between here
		     */
		    
		    new_elem = g_malloc0(sizeof(struct send_sel_data) + strlen(location) + strlen(fil) + 1);
		    strcpy(new_elem + sizeof(struct send_sel_data), location);
		    strcat(new_elem + sizeof(struct send_sel_data), fil);

		    files_list = g_slist_prepend(files_list, new_elem);
		}
	    }
	    g_dir_close(dir);
	}

	return TRUE;
}


/**
  The function will collect the names of Rich cores present in both the MMCs
  and append them to the files_list for uploading

  @param void
  @void gboolean
*/
gboolean creporter_ui_collect_all_corefile_names()
{
	gboolean rv=TRUE;
	int idx;

	for (idx = 0 ; idx < MAX_CORE_DIRS ; idx++) {
	    if (core_location_registry[idx].mountpoint[0] == 0) {
		break;
	    }
	    if(!creporter_ui_collect_all_corefile_names_at_location(core_location_registry[idx].dir)) {
		osso_log(LOG_DEBUG, "[%s]: failed to append cores from %s", 
			 __FUNCTION__, core_location_registry[idx].dir);
		rv = FALSE;
	    }
	}

	return rv;
}


/**
  This function is preforming delayed exit of the Crash Reporter UI application

  @param void
  @return void
  */
static gboolean creporter_ui_delayed_exit(void * context)
{
    if (gtk_main_level() > 0) {
	gtk_main_quit();
    }

    osso_deinitialize(context);
    osso_log(LOG_DEBUG, "Crash Reporter UI exits");
    exit(0);
}

/**
  This function is called to exit the Crash Reporter UI application.
  As it is often used from callbacks (like RPC) that work better
  when returned, we initiate a delayed exit handler here and return.

  @param void
  @return void
  */
void creporter_ui_exit()
{
    static gint exit_tmo_id = 0;

    if (exit_tmo_id == 0) {
	/*
	 * only do it once.
	 */
	exit_tmo_id = g_timeout_add(100, creporter_ui_delayed_exit, NULL);
    }
}
