/**
 * @file crash-reporter-settings-file.c
 *
 * This file contains the implementation for Crash Reporter settings
 * file functions. 
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <sys/stat.h>

#include "crash-reporter-settings-file.h"

static GKeyFile *creporter_load_settings_key_file()
{
	gchar filename[256];
	GError *err = NULL;

	GKeyFile *key = g_key_file_new();
	if (key == NULL) {
		return NULL;
	}

	/*
	 * try system location 
	 */
	strcpy(filename, CREPORTER_SYSTEM_SETTINGS_LOCATION);
	strncat(filename, SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, &err) == FALSE) {
	    /*
	     * no config file in system location, give up
	     */
	    if (err) {
		osso_log(LOG_DEBUG, "Failed loading key from %s error=%s", filename, err->message);
		g_error_free(err);
		err = NULL;
	    }
	    g_key_file_free(key);
	    osso_log(LOG_ERR, "Error reading server settings\n");
	    return NULL;
	}

	osso_log(LOG_DEBUG, "settings from '%s'", filename);
	return key;
}

static GKeyFile *creporter_load_privacy_settings_key_file()
{
    gchar filename[256];
    GError *err = NULL;

    GKeyFile *key = g_key_file_new();
    if (key == NULL) {
	return NULL;
    }

    /*
     * it should be like this:
     *
     * strncpy(filename, getenv("HOME"), sizeof(filename)-1);
     * 
     * but for daemon there seem to be no HOME,
     * so we are forced to use hard-defined value here... not nice but what can we do
     */
    strcpy(filename, PRIVACY_SETTINGS_LOCATION);
    strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);

    /*
     * it is normal case to fail the first one, means user
     * has not modified the config and there is only system config,
     * so do not ask for error status and don't log anything about this.
     */
    if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, NULL) == FALSE) {
	/*
	 * no privacy config file under HOME, try system location
	 */
	osso_log(LOG_DEBUG, "Failed loading key from %s", filename);
	strcpy(filename, CREPORTER_SYSTEM_SETTINGS_LOCATION);
	strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	if (g_key_file_load_from_file(key, filename, G_KEY_FILE_NONE, &err) == FALSE) {
	    /*
	     * This is more serious: no system config either, give up
	     */
	    if (err) {
		osso_log(LOG_ERR, "Failed loading key from %s error=%s", filename, err->message);
		g_error_free(err);
		err = NULL;
	    }
	    g_key_file_free(key);
	    osso_log(LOG_ERR, "Error reading privacy settings\n");
	    return NULL;
	}
    }

    osso_log(LOG_DEBUG, "settings from '%s'", filename);
    return key;
}

/**
  This function reads the Server & Credentials settings file values.

  @param none
  @return the creporterSettings structure pointer
*/
static creporterSettings *creporter_read_settings()
{
	GKeyFile *key = NULL;
	GError *err = NULL;
	creporterSettings *details = NULL;
	gchar *saddr = NULL;

	key = creporter_load_settings_key_file();
	if (!key) {
	    return NULL;
	}

	details = g_new0(creporterSettings, 1);

	if ((details->user_name = g_key_file_get_value(key, SERVER_SECTION,
						       USER_NAME, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", USER_NAME, err->message);
		g_error_free(err);
		err = NULL;
	}

	if ((details->passwd = g_key_file_get_value(key, SERVER_SECTION, PWD, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", PWD, err->message);
		g_error_free(err);
		err = NULL;
	}

	if ((saddr = g_key_file_get_value(key, SERVER_SECTION,
					  SERVER_ADDR, &err)) == NULL) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", SERVER_ADDR, err->message);
		g_error_free(err);
		err = NULL;
	}
	/*
	 * if server_addr does not end with slash '/', we append one,
	 * making us tolerate settings files with or without
	 */
	if (g_str_has_suffix(saddr, "/")) {
	    details->server_addr = saddr;
	} else {
	    details->server_addr = g_strdup_printf("%s/", saddr);
	    g_free(saddr);
	}
	saddr = NULL;

	details->use_ssl = g_key_file_get_boolean(key, SERVER_SECTION, USE_SSL, &err); 
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", USE_SSL, err->message);
		g_error_free(err);
		err = NULL;
	}

	g_key_file_free(key);

	/* to be freed by calling function */
	return details;
}

/**
  This function reads the Privacy settings file values.

  @param none
  @return the creporterSettings structure pointer
*/
privacySettings *creporter_read_privacy_settings()
{
	GKeyFile *key = NULL;
	GError *err = NULL;
	privacySettings *details = NULL;

	key = creporter_load_privacy_settings_key_file();
	if (!key) {
	    return NULL;
	}

	details = g_new0(privacySettings, 1);

	details->dumping_enabled = g_key_file_get_boolean(key, SETTINGS_SECTION,
							  DUMPING_ENABLED, &err);
	if(err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", DUMPING_ENABLED, err->message);
		g_error_free(err);
		err = NULL;
		details->dumping_enabled = TRUE;
	}

	details->sending_enabled = g_key_file_get_boolean(key, SETTINGS_SECTION,
							  SENDING_ENABLED, &err);
	if(err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", SENDING_ENABLED, err->message);
		g_error_free(err);
		err = NULL;
		details->sending_enabled = TRUE;
	}

	details->avoid_dups = g_key_file_get_boolean(key, SETTINGS_SECTION,
							  AVOID_DUPS, &err);
	if(err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", AVOID_DUPS, err->message);
		g_error_free(err);
		err = NULL;
		details->avoid_dups = TRUE;
	}

	details->lifelog_enabled = g_key_file_get_boolean(key, SETTINGS_SECTION,
							  LIFELOG, &err);
	if(err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", LIFELOG, err->message);
		g_error_free(err);
		err = NULL;
		/*
		 * NOTE!! While most of settingsd default to ON when not in file,
		 * lifelog gets default FALSE if not specified in config file!!
		 */
		details->lifelog_enabled = FALSE;
	}

	details->include_core = g_key_file_get_boolean(key, PRIVACY_SECTION, 
						       INCLUDE_CORE, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_CORE, err->message);
		g_error_free(err);
		err = NULL;
		details->include_core = TRUE;
	}
	details->include_syslog = g_key_file_get_boolean(key, PRIVACY_SECTION, 
							 INCLUDE_SYSLOG, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_SYSLOG, err->message);
		g_error_free(err);
		err = NULL;
		details->include_syslog = TRUE;
	}
	details->include_pkglist = g_key_file_get_boolean(key, PRIVACY_SECTION, 
							  INCLUDE_PKGLIST, &err);
	if (err) {
		osso_log(LOG_ERR, "Error reading %s:%s\n", INCLUDE_PKGLIST, err->message);
		g_error_free(err);
		err = NULL;
		details->include_pkglist = TRUE;
	}

	g_key_file_free(key);

	/* to be freed by calling function */
	return details;
}

/**
  This function writes the privacy settings file.

  @param sett privacySettings structure pointer
  @return void
*/
void creporter_write_privacy_settings(privacySettings * sett)
{
	GKeyFile *key = NULL;
	FILE *file = NULL;
	gchar filename[256];

	key = creporter_load_privacy_settings_key_file();
	if (!key) {
	    return;
	}

	g_key_file_set_boolean(key, SETTINGS_SECTION, DUMPING_ENABLED, sett->dumping_enabled);
	g_key_file_set_boolean(key, SETTINGS_SECTION, SENDING_ENABLED, sett->sending_enabled);
	g_key_file_set_boolean(key, SETTINGS_SECTION, AVOID_DUPS, sett->avoid_dups);
	g_key_file_set_boolean(key, SETTINGS_SECTION, LIFELOG, sett->lifelog_enabled);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_CORE, sett->include_core);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_SYSLOG, sett->include_syslog);
	g_key_file_set_boolean(key, PRIVACY_SECTION,  INCLUDE_PKGLIST, sett->include_pkglist);

	strncpy(filename, getenv("HOME"), sizeof(filename)-1);
	strncat(filename, PRIVACY_SETTINGS_FILE, sizeof(filename)-strlen(filename)-1);
	file = fopen (filename, "w");
	if (file) {
	    gsize length = 0;
	    char *data = g_key_file_to_data (key, &length, NULL);
	    if (length > 0)
		fwrite (data, length, 1, file);
	    g_free (data);
	    fclose (file);
	} else {
	    osso_log(LOG_DEBUG, "[%s]: FAILED: fopen(%s), can not save settings", __FUNCTION__, filename);
	}
	
	g_key_file_free(key);

	/* to be freed by calling function */
}

#ifdef freeing_not_used_read_once_use_manytimes_never_free
/**
  this function frees the creporterSettings structure

  @param creporterSettings pointer that is to be freed
  @return void
*/
void creporter_free_settings(creporterSettings * settings)
{
	if (settings == NULL)
		return;

	g_free(settings->user_name);
	g_free(settings->passwd);
	g_free(settings->server_addr);
	g_free(settings);
	return;
}
#endif /* freeing_not_used_read_once_use_manytimes_never_free */

/**
  this function frees the privacySettings structure

  @param privacySettings pointer that is to be freed
  @return void
*/
void creporter_free_privacy_settings(privacySettings * settings)
{
	g_free(settings);
	return;
}

creporterSettings * 
creporter_get_settings()
{
    static creporterSettings * settings = NULL;
    if(!settings) {
	settings = creporter_read_settings();
    }
    return settings;
}
