/*
 * @file crash-reporter-handler.c
 *
 * This file contains the function defs
 * for  crash-reporter-handler.c.
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <errno.h>

#include "crash-reporter-common.h"

#include <stdlib.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>

gboolean core_is_valid (gchar *filename);
struct core_location_reg * create_core_location_registry(void);
int is_mounted(const char * path);
void mmc_create_dir(int idx);
void creporter_get_rcore_fileinfo (gchar *filename, gint *fsize, 
				   gchar **sig_num, gchar **exe_name, 
				   gchar **process_id);
gboolean run_system_cmd (const gchar * cmd);

extern struct core_location_reg * core_location_registry;

static const char 
mountpoint_env_names[NUM_ENV_MOUNTPOINTS][MAX_MOUNTPOINT_NAMELEN] =
    {
	"MMC_MOUNTPOINT",
	"INTERNAL_MMC_MOUNTPOINT"
    };

static const char 
mountpoint_static[NUM_STATIC_MOUNTPOINTS][MAX_MOUNTPOINT_NAMELEN] =
    {
	"/media/mmc1",
	"/home/user/MyDocs",
	"/media/mmc2"
    };

const char core_dumps_suffix[] = "/core-dumps/";

/*
 */
struct core_location_reg *
create_core_location_registry(void)
{
    struct core_location_reg * reg;
    const char *name;
    char *str;
    int i, found = 0;

    reg = g_malloc(MAX_CORE_DIRS * sizeof(struct core_location_reg));
    bzero(reg, MAX_CORE_DIRS * sizeof(struct core_location_reg));
    /*
     * get mount points from environment
     */
    for(i = 0; i < NUM_ENV_MOUNTPOINTS; i++) {
	name = mountpoint_env_names[i];
	str = getenv(name);
	if (str) {
	    strncpy(reg[found].mountpoint, str, sizeof(reg[found].mountpoint)-1);
	    found += 1;
	}
    }
    if (!found) {
	/*
	 * nothing in environment, use static values as fallback
	 */
	for(i = 0; i < NUM_STATIC_MOUNTPOINTS; i++) {
	    name = mountpoint_static[i];
	    strncpy(reg[found].mountpoint, name, sizeof(reg[found].mountpoint)-1);
	    found += 1;
	}
    }

    for ( i=0 ; i<found ; i++) {
	strncpy(reg[i].dir, reg[i].mountpoint, sizeof(reg[i].mountpoint));
	strncat(reg[i].dir, core_dumps_suffix, sizeof(reg[i].dir) - sizeof(reg[i].mountpoint));
	osso_log(LOG_DEBUG, "registry:%s", reg[i].dir);
    }
    return reg;
}


/**
  This function checks if the file generated can be accepted for upload
  it will reject if the file has no suitable suffix,
  starts with "." or is a crash_reporter_ui Rich core

  @param filename
  @param TRUE if it can be accepted else FALSE
*/
gboolean 
core_is_valid (gchar *filename)
{
    gchar *basename = NULL;
    gboolean rv = FALSE;
	
    g_return_val_if_fail(filename != NULL, FALSE);	
    basename = g_filename_display_basename(filename);

    if(g_str_has_prefix(basename, ".")) {
	/*Ignore the partially created cores,
	 * these have the prefix '.'*/
	rv = FALSE;
    } else if(g_strrstr(filename, CRASH_REPORTER_UI)) {
	/* Ignore crash_reporter_ui cores*/
	rv = FALSE;
    } else if(g_str_has_suffix(basename, ".rcore") ||
	      g_str_has_suffix(basename, ".rcore.lzo")) {
	rv = TRUE;
    }

    g_free(basename);
    return rv;
}

/*
 * Check if path is part of rootfs or is there other fs mounted
 * return value 0: part of rootfs
 * return value 1: not part of rootfs
 */
int 
is_mounted(const char * path)
{

#ifdef RUNS_IN_SDK_HOST
    return 1;
#else
   struct stat st;
    dev_t path_dev, root_dev;

    bzero(&st, sizeof(st));
    if(stat(path, &st)) {
	return 0;
    }
    path_dev = st.st_dev;
    
    bzero(&st, sizeof(st));
    if(stat(ROOTFS_LOCATION, &st)) {
	return 0;
    }
    root_dev = st.st_dev;

    return (path_dev != root_dev);
#endif
}

/*
 * Check if directory exists, create if does not.
 */
void
mmc_create_dir(int idx)
{
    GDir *dir = NULL;
    if (core_location_registry == NULL) {
	osso_log(LOG_ERR, "[%s] No core locations directory!?",__FUNCTION__);
	return;
    }
    if (is_mounted(core_location_registry[idx].mountpoint)) {
	dir = g_dir_open(core_location_registry[idx].dir, 0, NULL);
	if (dir == NULL) {
	    if(g_mkdir(core_location_registry[idx].dir, FILE_PERMISSION)) {
		osso_log(LOG_ERR, "[%s] Can not create directory %s, err=%s",
			 __FUNCTION__, core_location_registry[idx].dir, strerror(errno));
	    } else {
		osso_log(LOG_INFO, "[%s] Created directory %s",__FUNCTION__, core_location_registry[idx].dir);
	    }
	} else {
	    g_dir_close(dir);
	}
    }
}

/**
  this function parses process name, signum, process_id 
  from the Rich core file name

  @param filename of the Rich core
  @return void
*/
void
creporter_get_rcore_fileinfo (gchar *filename, gint *fsize, 
			      gchar **sig_num, gchar **exe_name, 
			      gchar **process_id)
{
	gchar *temp=NULL;
	gchar *p_signum=NULL;
	gchar *p_after_signum=NULL;
	gchar *p_after_name=NULL;
	gchar *p_dot=NULL;
	gchar *p_pid=NULL;
	gchar *core_name = NULL;
	gint size;
	struct stat file_info;

	if (!(filename && fsize && sig_num && exe_name && process_id)) {
	    return;
	}

	bzero(&file_info, sizeof(file_info));
	stat(filename, &file_info);
	size = file_info.st_size;

	core_name = g_filename_display_basename(filename);
	if(core_name == NULL) {
		return;
	}

	osso_log(LOG_DEBUG, "fileinfo:%s size %lu\n", 
		 filename, (unsigned long)file_info.st_size);
	/*
	 * The name is something like: 
	 * osso_pdfviewer-11-3447.rcore.lzo
	 * or:
	 * osso_pdfviewer-MMMM-11-3447.rcore.lzo
	 *
	 * where MMMM is two last octests of device WLAN MAC.
	 * Parsing from start until dash or digits does not work
	 * because both of these may appear as part of executable name;
	 * Instead, we walk backwards from start of ".rcore":
	 * digits: becomes PID
	 * skip one (via --temp): this is dash '-'
	 * digits: becomes SIGNUM
	 * skip one (via --temp): this is dash '-'
	 * Optional:
	 * 4*hex-digit:  would become partial-MAC, but we do not extract this
	 * Finally, "temp" points to dash character after executable name
	 */
	temp = g_strrstr(core_name, ".rcore");
	if(temp == NULL) {
	    return;
	}

	p_dot = temp;
	while(temp >= core_name && g_ascii_isdigit(*--temp));
	p_after_signum = temp;
	p_pid = temp + 1;
	while(temp >= core_name && g_ascii_isdigit(*--temp));
	p_signum = temp + 1;

	/*
	 * as MAC fields were added later, we need to 
	 * handle them optionally, only if substring backwards from 
	 * temp is now "-XXXX-" where X is hexdigit
	 */
	if ((temp > (core_name + 6)) && 
	    (*(temp) == '-') &&
	    (*(temp - 5) == '-') &&
	    g_ascii_isxdigit(*(temp-1)) &&
	    g_ascii_isxdigit(*(temp-2)) &&
	    g_ascii_isxdigit(*(temp-3)) &&
	    g_ascii_isxdigit(*(temp-4))) {

	    p_after_name = temp - 5;
	} else {
	    p_after_name = temp;
	}

	*exe_name = g_strndup(core_name, p_after_name - core_name);
	*sig_num = g_strndup(p_signum, p_after_signum - p_signum);
	*process_id = g_strndup(p_pid, p_dot - p_pid);
	*fsize = size;

	g_free(core_name);
	return;
}

/**
  This function is used to run system command.
  Example use: check if the process-by-name is running,
               or kill process by name (then retval not used)

  @param void
  @return gboolean - TRUE if the process is running, else FALSE
  */
gboolean 
run_system_cmd (const gchar * cmd)
{
    int rv = system(cmd);

    if( rv == 0) {
	return TRUE;
    } else {
	return FALSE;
    }
}

