/**
 * @file crash-reporter-daemon-monitor.c
 *  
 * This file contains the monitoring functions for Crash Reporter daemon
 * and will invoke UI in case they have to be uploaded to server.
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/inotify.h>

#include "crash-reporter-daemon-defs.h"
#include "crash-reporter-daemon-monitor.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

#define MAX_RCORE_PROC_NAME_SZ (64)
#define MAX_COUNTED_APPS (256)
#define MAX_PREV_CORENAME_SZ (256)
struct rcores_counting {
    int signal;
    int size, lower, upper;
    int count;
    char name[MAX_RCORE_PROC_NAME_SZ];
};

struct rcores_counting rcores_counter[MAX_COUNTED_APPS];
char prev_corename[MAX_PREV_CORENAME_SZ];

int inotify_fd = -1;
extern struct core_location_reg * core_location_registry;

extern gboolean sending_enabled;
extern gboolean avoid_dups;

extern void lifelog_add_event(char *event, char *datastr);

/*
 * check for repeating dups, and also register & count this pattern,
 * pattern being: name/signal/size
 *
 * return value:
 * 0 = can send, either avoid_dups not active, or not a dup
 * 1 = skip sending this, avoid_dups is active and this is a dup
 */
static int 
check_and_add_rcore_pattern (char * filename)
{
    int size, ret = 0;
    short sig;
    gchar *sig_num = NULL;
    gchar *exe_name = NULL;
    gchar *process_id = NULL;
    struct rcores_counting *r;
    osso_return_t retval;

    if (!filename) {
	return 1;
    }
    creporter_get_rcore_fileinfo(filename, &size, &sig_num, &exe_name, &process_id);
    if (!(size && sig_num && exe_name && process_id)) {
	return 0;
    }
    sig = atoi(sig_num);
    for (r=&rcores_counter[0]; r<&rcores_counter[MAX_COUNTED_APPS]; r++) {
	if (r->signal == 0) {
	    /*
	     * not found, add new entry
	     */
	    r->signal = sig;
	    r->size = size;
	    r->lower = (size * 95)/100;
	    r->upper = (size * 105)/100;
	    r->count = 1;
	    strncpy(r->name, exe_name, MAX_RCORE_PROC_NAME_SZ - 1);
	    ret = 0;
	    osso_log(LOG_DEBUG,"new dups_count entry:name=%s sig=%d low=%d up=%d", 
		     exe_name, sig, r->lower, r->upper);
	    break;
	}
	if (!strncmp(r->name, exe_name, MAX_RCORE_PROC_NAME_SZ) &&
	    (r->signal == sig) &&
	    (size >= r->lower) &&
	    (size <= r->upper)) {
	    /*
	     * this matches existing entry, increment count and check limit
	     */
	    r->count += 1;
	    osso_log(LOG_DEBUG,"[%s] name=%s size=%d matches dups entry, count now %d (sig=%d low=%d up=%d)", 
		     __FUNCTION__, exe_name, size, r->count, sig, r->lower, r->upper);
	    if (r->count == MAX_SIMILAR_RCORES) {
		retval = osso_rpc_run_with_defaults(context, CREPORTER_SERVICE, 
						    CREPORTER_METHOD_DUPS_NOTE, NULL, 
						    DBUS_TYPE_STRING, filename, DBUS_TYPE_INVALID);

	    }
	    if (r->count >= MAX_SIMILAR_RCORES) {
		ret =  1;
	    } else {
		ret =  0;
	    }
	    break;
	}
    }
    /*
     * debug check: did the table get full?
     */
    if (r >= &rcores_counter[MAX_COUNTED_APPS]) {
	osso_log(LOG_DEBUG, "[%s]: dup counting table overflow", __FUNCTION__);
    }
    return ret;
}


/**
  This function will invoke CREPORTER_UI through RPC with 
  method name crash_reporter_send_file

  @param the filename of the file that was created
  @return FALSE 
  */
static void 
send_rich_core(gchar * filename)
{
	osso_return_t retval;
	
	if(filename == NULL) {
		osso_log(LOG_INFO, "[%s]: NULL param passed", __FUNCTION__);
		return;
	}
	
 	if(!core_is_valid(filename)) {
		osso_log(LOG_INFO,"%s is ignored", filename);
		return;
	}
		
	if(avoid_dups && check_and_add_rcore_pattern(filename)) {
	    osso_log(LOG_INFO, "duplicates limit of %d reached for %s, delete", 
		     MAX_SIMILAR_RCORES, filename);
	    unlink(filename);
	    return;
	}
	if (!sending_enabled) {
	    return;
	}
	/*
	 * if same as previous, user added comment and same-name
	 * lzo file was re-created. Skip remaining part, this is done already.
	 */
	if(!strncmp(prev_corename, filename, MAX_PREV_CORENAME_SZ)) {
	    return;
	}
	strncpy(prev_corename, filename, MAX_PREV_CORENAME_SZ - 1);
	lifelog_add_event("COREDUMP", filename);

	osso_log(LOG_INFO, "invoke ui for %s", filename);
	retval = osso_rpc_run_with_defaults(context, CREPORTER_SERVICE, 
					    CREPORTER_METHOD_SEND_FILE, NULL, 
					    DBUS_TYPE_STRING, filename, DBUS_TYPE_INVALID);

	return;
}

/**
  This function will invoke CREPORTER_UI through RPC
  with method_name crash_reporter_send_all_files

  */
static void 
send_all_rich_cores()
{
	osso_return_t retval;
	
	retval = osso_rpc_run_with_defaults(context, CREPORTER_SERVICE, 
					    CREPORTER_METHOD_SEND_ALL, NULL, 
					    DBUS_TYPE_INVALID);

}


/**
  */
/* size of the event structure, not counting name */
#define EVENT_SIZE  (sizeof (struct inotify_event))

/* reasonable guess as to size of 1024 events */
#define BUF_LEN        (1024 * (EVENT_SIZE + 16))

#define FNAMESZ 256
char buf[BUF_LEN];

static void 
thread_monitor_rcore_locations()
{
    int len, i = 0, idx;
    char * dirname;
    struct inotify_event *event;
    char filename[FNAMESZ];

    while (1) {
	len = read (inotify_fd, buf, BUF_LEN);
	if (len < 0) {
	    if (errno == EINTR) {
                /* need to reissue system call */
		sleep(1);
	    }
	    else {
		    perror ("read");
		sleep(1);
	    }
	} else if (!len) {
	    /* BUF_LEN too small? */
	    perror ("read2");
	    sleep(1);
	} else {

	    for (i = 0; i < len; i += EVENT_SIZE + event->len) {

		event = (struct inotify_event *) &buf[i];

		dirname = NULL;
		for (idx = 0 ; idx < MAX_CORE_DIRS ; idx++) {
		    if (core_location_registry[idx].mountpoint[0] == 0) {
			break;
		    }
		    if (event->wd == core_location_registry[idx].nhandle) {
			dirname = core_location_registry[idx].dir;
			break;
		    }
		}

		if (event->mask & IN_UNMOUNT) {
		    if ((idx < MAX_CORE_DIRS) && dirname) {
			osso_log(LOG_DEBUG, "[%s] directory %s was unmounted", 
				 __FUNCTION__, dirname);
		    }
		    core_location_registry[idx].nhandle = 0;
		    continue;
		}

		if (event->len == 0) {
		    break;
		}

		if (dirname) {
		    strncpy(filename, dirname, FNAMESZ);
		    strncat(filename, event->name, FNAMESZ-strlen(filename));
		    send_rich_core((gpointer) filename);
		} else {
		    osso_log(LOG_DEBUG, "[%s] No dirname for wd=%d\n", 
			     __FUNCTION__, event->wd);
		}
	    }
	
	}
    }
}

/**
  This function checks for the presence of core files at a location

  @param gchar - location
  @param gboolean - TRUE if Cores are present else FALSE
*/
static gboolean 
check_for_cores_at_location(const gchar *location)
{
	gboolean rv = FALSE;
	gint count = 0;
	GDir* dir;
	const gchar* fil;

	if(!location) {
	    osso_log(LOG_DEBUG, "[%s] NULL param passed", __FUNCTION__);
	    return FALSE;
	}
	dir = g_dir_open(location, 0, NULL);
	if (dir != NULL) {
	    while ((fil = g_dir_read_name(dir))) {
		if(core_is_valid(fil)) {
		    count++;
		}
	    }
	    g_dir_close(dir);
	    if (count > 0 ) {
		rv = TRUE;
	    }
	}
	return rv;
}

/**
  This function sets the monitors for directories in MMC1 and MMC2
  The monitor_event() will be called when any changes
  are done to the monitored directories

  @param idx to paths registry
  @return TRUE if success else FALSE
*/

gboolean set_directory_monitor(int idx)
{
	int wd = inotify_add_watch (inotify_fd,
				core_location_registry[idx].dir,
				IN_CLOSE_WRITE);

	if (wd < 0) {
	    perror ("inotify_add_watch");
	    return FALSE;
	} else {
	    core_location_registry[idx].nhandle = wd;
	    return TRUE;
	}
}

/**
  This function checks for the presence of core files in 
  'core-dumps' directories in both the MMCs. If present, it will 
  send the cores by calling send_all_rich_cores()

  @param void
*/
void init_inotify_and_check_for_cores()
{
    int idx;
    for (idx = 0 ; sending_enabled && (idx < MAX_CORE_DIRS) ; idx++) {
	if (core_location_registry[idx].mountpoint[0] == 0) {
	    break;
	}
	if( check_for_cores_at_location(core_location_registry[idx].dir)) {
	    send_all_rich_cores();
	    break;
	}
    }

    inotify_fd = inotify_init();
    if (inotify_fd < 0) {
	perror ("inotify_init");
    }

    g_thread_create_full((GThreadFunc)thread_monitor_rcore_locations,
             NULL,
             128*1024,
             FALSE,
             0,
             G_THREAD_PRIORITY_NORMAL,
             NULL);
}
