/*
 * @file crash-reporter-settings-dialog.c
 *
 * This file contains the implementation of
 * Crash Reporter settings dialog.
 * 
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <errno.h>
#include <stdlib.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gtk/gtkcombobox.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-helper.h>
#include <hildon/hildon-pannable-area.h>
#include <gconf/gconf-client.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <string.h>

#include "crash-reporter-settings-dialog.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

static void creporter_cp_show_privacy_disclaimer(gpointer parent);

static void
log_settings_event(privacySettings *p, char *action)
{
    osso_log(LOG_DEBUG, "Settings %s:dump=%d send=%d avoid_dups=%d include:core=%d log=%d plist=%d",
	     action,
	     p->dumping_enabled,
	     p->sending_enabled,
	     p->avoid_dups,
	     p->include_core,
	     p->include_syslog,
	     p->include_pkglist);
}

/**
  This function is used to re-start the Crash Reporter Daemon.
  */
static 
gboolean creporter_cp_execute_restart()
{
	gchar *argv[3];
        gint pid = 0;
        gint status = 0;

	osso_log(LOG_DEBUG, "[%s]:", 
		 __FUNCTION__);
	pid = fork();
        if (pid == -1) {
                return FALSE;
        }
        /* FIXME: While this will start/stop the daemon, we need to eventually change it:
           - The initscript will eventually be removed once we migrate 100% to Upstart
           - Meanwhile, this confuses Upstart a bit */
	
	if (pid == 0) {
		argv[0] = "sh";
		argv[1] = "restart";
		argv[2] = 0;
		if (execve(SCRIPT, argv, NULL) == -1) {
			osso_log(LOG_DEBUG, "[%s]: Error in execve", __FUNCTION__);
		}
		exit(127);
	}
	do{
		if (waitpid(pid, &status, 0) == -1) {
			break;
		}
	}while (1);
	return TRUE;
}

/** 
  This function will remove a 'core-dumps' directory with contents

  @param gchar * location of the directory
  @return void
*/ 
static void 
creporter_cp_remove_coredumping_dir(gchar *location)
{
    /*
     * the original idea of walking through directory, deleting files,
     * then directory itself, does not work if directory contains other directories,
     * thus needing recursive walk.
     * Lets do it using system(rm -fr) instead, leaving to "rm" recursiveness and
     * all hassle with walks and files/dirs.
     * This makes our code here 20 times smaller and simpler.
     */
    int rv;
    gchar *cmd = g_strdup_printf("rm -fr %s",location);
    osso_log(LOG_DEBUG, "[%s]: running cmd=[%s]", 
	     __FUNCTION__, cmd);
    rv = system(cmd);
    if (rv == -1) {
	osso_log(LOG_ERR, "[%s]: error=-1 while removing core-dumps directory", 
		 __FUNCTION__);
    }
    g_free(cmd);
}


/**
  This function is called as callback when "saving enabled" button
  gets toggled
  
  @param w is pointer to button widget
  @param data is pointer to _creporterData structure

  */
static gboolean 
dumping_enabled_toggle_callback(GtkWidget *w, gpointer data) {
    creporterData *ndata = (creporterData*)data;

    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))) {
	gtk_widget_set_sensitive(ndata->caption_avoid_dups, TRUE);
	gtk_widget_set_sensitive(ndata->caption_what_to_include, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_core, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_syslog, TRUE);
	gtk_widget_set_sensitive(ndata->caption_include_pkglist, TRUE);
    } else {
	gtk_widget_set_sensitive(ndata->caption_avoid_dups, FALSE);
	gtk_widget_set_sensitive(ndata->caption_what_to_include, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_core, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_syslog, FALSE);
	gtk_widget_set_sensitive(ndata->caption_include_pkglist, FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ndata->checkbox_sending_enabled), FALSE);
    }
    return TRUE;
}

/**
  This function is called as callback when "sending enabled" button
  gets toggled
  
  @param w is pointer to button widget
  @param data is pointer to _creporterData structure

  */
static gboolean 
sending_enabled_toggle_callback(GtkWidget *w, gpointer data) {
    creporterData *ndata = (creporterData*)data;
    /*
     * if sending is enabled, force coredumping also ON
     */
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ndata->checkbox_dumping_enabled), TRUE);
    }
    return TRUE;
}

/**
  This function is called for creation of the dialog widgets.
  
  @param data is the pointer to the creporterData structure
  @param settings_dialog widget pointer
  @param enable_sending flag to indicate the creporter service status
  @param context is the osso_context_t pointer for the cp application
  
  @return TRUE on success, else FALSE
  */
static gboolean 
creporter_create_dialog_widgets(creporterData *data,
				GtkWidget *settings_dialog,
				privacySettings *privsett,
				osso_context_t *context)
{
    GtkWidget *vbox = NULL;
    GtkSizeGroup *size_group = NULL;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(settings_dialog != NULL, FALSE);
	
    vbox = (GTK_DIALOG(settings_dialog))->vbox;
    size_group = 
	gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	
    /*-----------*/
    data->checkbox_dumping_enabled = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_dumping_enabled), 
				 privsett->dumping_enabled);

    data->caption_dumping_enabled =
	hildon_caption_new(size_group, _("Save crash reports"),
			   data->checkbox_dumping_enabled, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_dumping_enabled, 
		       FALSE,FALSE, 0);

    g_signal_connect (data->checkbox_dumping_enabled, "toggled",
		      G_CALLBACK (dumping_enabled_toggle_callback), data);

    /*-----------*/
    data->checkbox_sending_enabled = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_sending_enabled), 
				 privsett->sending_enabled);

    data->caption_sending_enabled =
	hildon_caption_new(size_group, _("Upload crash reports"),
			   data->checkbox_sending_enabled, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_sending_enabled, 
		       FALSE,FALSE, 0);

    g_signal_connect (data->checkbox_sending_enabled, "toggled",
		      G_CALLBACK (sending_enabled_toggle_callback), data);

    /*-----------*/
    data->checkbox_avoid_dups = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_avoid_dups), 
				 privsett->avoid_dups);

    data->caption_avoid_dups =
	hildon_caption_new(size_group, _("Auto-delete duplicates"),
			   data->checkbox_avoid_dups, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_avoid_dups, 
		       FALSE,FALSE, 0);

    /*-----------*/
    data->caption_what_to_include = gtk_label_new(_("Include into future crash reports:"));
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_what_to_include,
		       FALSE,FALSE, 4);

    /*-----------*/
    data->checkbox_include_core = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_core), privsett->include_core);

    data->caption_include_core =
	hildon_caption_new(size_group, _("Core dump"),
			   data->checkbox_include_core, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_core, 
		       FALSE,FALSE, 0);

    /*-----------*/
    data->checkbox_include_syslog = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_syslog), privsett->include_syslog);

    data->caption_include_syslog =
	hildon_caption_new(size_group, _("System log (if exists)"),
			   data->checkbox_include_syslog, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_syslog, 
		       FALSE,FALSE, 0);

    /*-----------*/
    data->checkbox_include_pkglist = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->checkbox_include_pkglist), privsett->include_pkglist);

    data->caption_include_pkglist =
	hildon_caption_new(size_group, _("List of installed packages"),
			   data->checkbox_include_pkglist, NULL, 
			   HILDON_CAPTION_OPTIONAL);
    gtk_box_pack_start(GTK_BOX(vbox), data->caption_include_pkglist, 
		       FALSE,FALSE, 0);

    /*-----------*/
    dumping_enabled_toggle_callback(data->checkbox_dumping_enabled, data);
    sending_enabled_toggle_callback(data->checkbox_sending_enabled, data);
    return TRUE;
}


static void creporter_cp_show_privacy_disclaimer(gpointer parent)
{
    static GtkWidget *window;
    GtkWidget *pannable_area;
    GtkWidget *label;
    GtkWidget *frame;
    gint response;
    gchar *stmt = NULL;
    gchar *p;
    struct stat st;
    FILE *fp = NULL;
    
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    
    window = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(window),
		   	 _("Crash Reporter Privacy Note"));
    gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(parent));
    gtk_window_set_modal(GTK_WINDOW(window), TRUE);
    gtk_window_set_destroy_with_parent(GTK_WINDOW(window), TRUE);
    gtk_widget_set_size_request(window, -1, 300);
    /* create a new pannable area. */
    pannable_area = hildon_pannable_area_new();

    frame = gtk_frame_new(NULL);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), frame, 
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (frame), pannable_area);

    if( !stat(PRIVACY_STATEMENT_FILE, &st) 
	&& (stmt = g_malloc(st.st_size + 1))
	&& (fp = fopen(PRIVACY_STATEMENT_FILE, "r"))
	&& (fread(stmt, sizeof(char), st.st_size, fp) == st.st_size)) {
	stmt[st.st_size] = 0;
	/* replace newlines with ' ' because we use wrap option */
        for (p = stmt; *p ; p++) {
	    if (*p == '\n') {
		*p = ' ';
	    }
	}
    } else {
	osso_log(LOG_ERR,"[%s]: error(s) reading file [%s]\n", __FUNCTION__, PRIVACY_STATEMENT_FILE);
	stmt = g_malloc(80);
	if (stmt) {
	    strcpy(stmt, _("[can not open privacy statement file!]"));
	}
    }
    if (fp) {
	fclose(fp);
    }
    /* Create a text label widget */
    label = gtk_label_new (stmt);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    /* pack the text widget into the panned area */
    hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					   label);
    gtk_widget_show_all (window);
    response = gtk_dialog_run(GTK_DIALOG(window));
    gtk_widget_destroy(GTK_WIDGET(window));
    g_free(stmt);
}

/**
  The function creates the Crash Reporter cp applet dialog

  @param window pointer
  @param osso_context_t pointer 

  @return void
*/
void 
creporter_cp_dialog_create(gpointer window, osso_context_t *context)
{
    GtkWidget *settings_dialog = NULL;
    GtkWidget *privacy_stmt = NULL;
    GtkWidget *manage_stmt = NULL;
    creporterData *dialog_data = NULL;
    gboolean validate = FALSE;
    gboolean ret = FALSE;
    gboolean dumping_enabled = FALSE;
    gboolean sending_enabled = FALSE;
    gboolean avoid_dups = FALSE;
    gboolean prev_dumping_enabled = FALSE;
    gboolean prev_sending_enabled = FALSE;
    gboolean prev_avoid_dups = FALSE;
    gint resp = -1;
    privacySettings* privsettings = NULL;
    int idx;
    osso_return_t retval;

    g_return_if_fail(context != NULL);
    g_return_if_fail(window != NULL);

    privsettings = creporter_read_privacy_settings();
    if (privsettings) {
	log_settings_event(privsettings, "loaded");
    } else {
	gchar *text = NULL;
	GtkWidget *note = NULL;
	text = g_strdup(_("Error loading configuration. Is Crash Reporter configuration package installed?"));
	note = hildon_note_new_information(NULL, text);
	gtk_dialog_run(GTK_DIALOG(note));
	gtk_widget_destroy(GTK_WIDGET(note));	    
	g_free(text);
	return;
    }

    dialog_data = (creporterData *) g_new0(creporterData, 1);
	
    prev_dumping_enabled = privsettings->dumping_enabled;
    prev_sending_enabled = privsettings->sending_enabled;
    prev_avoid_dups      = privsettings->avoid_dups;

    settings_dialog = gtk_dialog_new();
    if (!settings_dialog) {
      osso_log(LOG_DEBUG,"Failure creating settings dialog\n");
      creporter_free_privacy_settings(privsettings);
      return;
    }

    gtk_window_set_title (GTK_WINDOW(settings_dialog),
			  _("Crash Reporter Settings"));
    gtk_window_set_transient_for(GTK_WINDOW(settings_dialog),
				 GTK_WINDOW(window));
    gtk_window_set_modal(GTK_WINDOW(settings_dialog), TRUE);

    gtk_dialog_add_button(GTK_DIALOG(settings_dialog),
			  dgettext("hildon-libs",
				   "wdgt_bd_save"),
			  CP_RESPONSE_OK);

    if (!run_system_cmd("pidof " CRASH_REPORTER_UI)) {
	manage_stmt = gtk_dialog_add_button(GTK_DIALOG(settings_dialog),
			      _("Reports"),
			      CP_RESPONSE_SEND_ALL);
    }

    privacy_stmt = gtk_dialog_add_button(GTK_DIALOG(settings_dialog),
					  _("Privacy\nstatement"),
					  CP_RESPONSE_DISCLAIMER);
    hildon_helper_set_logical_font(privacy_stmt, "SmallSystemFont");
    gtk_dialog_add_button(GTK_DIALOG(settings_dialog),
			  _("Cancel"), GTK_RESPONSE_CANCEL);

    gtk_widget_set_size_request(GTK_WIDGET(settings_dialog),
				-1, -1);
    gtk_dialog_set_has_separator(GTK_DIALOG(settings_dialog), 
				 TRUE);
    ret = creporter_create_dialog_widgets(dialog_data,
					  settings_dialog,
					  privsettings,
					  context);
    if(!ret) {
	osso_log(LOG_DEBUG,"Failure creating dialog widgets\n");
	gtk_widget_destroy(settings_dialog);
	creporter_free_privacy_settings(privsettings);
	return;
    }
	
    gtk_widget_show_all(settings_dialog);
    while(validate == FALSE) {
	
	resp = gtk_dialog_run(GTK_DIALOG(settings_dialog));
	switch (resp) {
	case CP_RESPONSE_OK:
		{ 
		gboolean i_core,i_syslog,i_pkglist;
		dumping_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_dumping_enabled));
		sending_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_sending_enabled));
		avoid_dups = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_avoid_dups));

		i_core = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_core));
		i_syslog = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_syslog));
		i_pkglist = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_data->checkbox_include_pkglist));
		if (i_core != privsettings->include_core || 
		    i_syslog != privsettings->include_syslog || 
		    i_pkglist != privsettings->include_pkglist ||
		    prev_sending_enabled != sending_enabled ||
		    prev_dumping_enabled != dumping_enabled ||
		    prev_avoid_dups != avoid_dups) {
		    privsettings->dumping_enabled = dumping_enabled;
		    privsettings->sending_enabled = sending_enabled;
		    privsettings->avoid_dups = avoid_dups;
		    privsettings->include_core    = i_core;
		    privsettings->include_syslog  = i_syslog;
		    privsettings->include_pkglist = i_pkglist;
		    creporter_write_privacy_settings(privsettings);
		    log_settings_event(privsettings, "saved");
		}
		if (prev_dumping_enabled && !dumping_enabled) {
		    /* delete coredump directories */
		    osso_log(LOG_DEBUG, "[%s] dumping enabled 1->0, delete core dumping directories", __FUNCTION__);
		    /* if this is true then also sending_enabled must be FALSE
		     * (forced by UI) 
		     */
		    for (idx = 0 ; idx < MAX_CORE_DIRS ; idx++) {
			if (core_location_registry[idx].mountpoint[0] == 0) {
			    break;
			}
			creporter_cp_remove_coredumping_dir(core_location_registry[idx].dir);
		    }
		}
		if((prev_dumping_enabled != dumping_enabled) ||
		   (prev_sending_enabled != sending_enabled) ||
		   (prev_avoid_dups != avoid_dups)) {
		    /* 
		     * Re-start creporter_daemon when relevant settings
		     * that affect its operation have changed;
		     * daemon will re-create core-dumps directories if dumping active
		     */
		    creporter_cp_execute_restart();
		}
		validate = TRUE;
		break;
	}
	    
        case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_CANCEL:
	    validate = TRUE;
	    break;
	case CP_RESPONSE_DISCLAIMER:
	    creporter_cp_show_privacy_disclaimer(window);
	    break;
	case CP_RESPONSE_SEND_ALL:
	    retval = osso_rpc_run_with_defaults(context, CREPORTER_SERVICE, 
						CREPORTER_METHOD_SEND_ALL, NULL, 
						DBUS_TYPE_INVALID);

	    break;
	}
    }

    gtk_widget_destroy(settings_dialog);
    creporter_free_privacy_settings(privsettings);
    g_free(dialog_data);
    return;
}
