/**
 * @file crash-reporter-ui-main.c
 *  
 * This file contains the main function for Crash Reporter UI.
 * This application is invoked by the Crash Reporter Daemon in two modes -
 * 1. Send all files (Will send all the Rich cores to Crash Reporter Server
 * 2. Send a file (Will send only a newly generated Rich core
 * Based on the above two modes, it will launch appropriate
 * dialog to the user
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <gconf/gconf-client.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>

#include "crash-reporter-ui-main.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-handler.h"
#include "crash-reporter-ui-utils.h"

/* It stores the list of files to be uploaded*/
GSList *files_list = NULL;

/* Crash Reporter UI Application context */
osso_context_t *context = NULL;

/* creporter_ui_status is the application mode
 *  which can take the values -
 * NOT_ALREADY_UP, SEND_FILE, SEND_ALL_FILES */
static creporterUIMode creporter_ui_status = NOT_ALREADY_UP;

struct core_location_reg * core_location_registry = NULL;

/**
  This function is a callback to detect when the device is about to
  shut down. When that occurs, it's time for the Crash Reporter UI to go down.
  
  @param hw_state contains the monitored hardware state changes
  @param data not used in this case

*/
static void hw_state_event_cb(osso_hw_state_t *hw_state, gpointer data)
{
        if (hw_state->shutdown_ind == TRUE) {
                osso_log(LOG_DEBUG, "[%s]: Shutting down Crash Reporter UI\n",
			 __FUNCTION__);
                creporter_ui_exit();
                return;
        }
}

static void
crash_reporter_send_all()
{
    creporter_ui_collect_all_corefile_names();
    creporter_ui_status = SEND_ALL_FILES;
    creporter_ui_set_connec_callback_notifiers();
    creporter_ui_ask_user_conf(TRUE);
}

/**
 This is a callback for the RPC method when requested on the 
 registered interface

 @param The interface that the method is called on
 @param The method which can be -
 	CREPORTER_METHOD_SEND_ALL-Sends all the RichCores present in 'core-dumps' dirs
	CREPORTER_METHOD_SEND_FILE-Sends only one RichCore whose uri is passed as argument
 @param A GArray of osso_rpc_t structures
 @param An application specific pointer
 @param The return value of the method

 @return OSSO_OK if success else OSSO_ERROR
*/
static osso_return_t 
creporter_ui_rpc_callback_handler(const gchar * interface, const gchar * method, 
			      GArray * arguments, gpointer method_name, osso_rpc_t *retval)
{
    gchar *filename = NULL;
    GSList *temp = NULL;
	
    if ((method == NULL) || (interface == NULL)) {
	osso_log(LOG_DEBUG, "Invalid input params received \n");
	return OSSO_ERROR;
    }

    if(!strcmp(method,CREPORTER_METHOD_SEND_ALL)) {
	crash_reporter_send_all();
	return OSSO_OK;
    } else if(!strcmp(method,CREPORTER_METHOD_SEND_FILE)) {
	if((g_array_index(arguments, osso_rpc_t, 0).value.s) == NULL) {
	    osso_log(LOG_DEBUG, "NULL param passed to %s\n",
		     CRASH_REPORTER_UI);
	    return OSSO_ERROR;
	}
	/*
	 * do not add if same name already on the list,
	 * this happens when user note gets added, then lzo file
	 * gets overwritten with same name
	 */
	for(temp = files_list; temp ; temp = g_slist_next(temp)) {
	    if( !strcmp((gchar *)temp->data, g_array_index(arguments, osso_rpc_t, 0).value.s)) {
		osso_log(LOG_DEBUG, "[%s]: skip add %s (exists)", 
			 __FUNCTION__, (gchar *)temp->data);
		return OSSO_OK;
	    }
	}
	/* Appending single file to the list*/
	filename = g_strdup(g_array_index(arguments, osso_rpc_t, 0).value.s);
	files_list = g_slist_prepend(files_list, filename);
	if(creporter_ui_status == NOT_ALREADY_UP) {
	    creporter_ui_status = SEND_FILE;
	    creporter_ui_set_connec_callback_notifiers();
	    creporter_ui_ask_user_conf(FALSE);
	}
			
	return OSSO_OK;
    } else if (!strcmp(method,CREPORTER_METHOD_DUPS_NOTE)) {
	crash_upload_context *ctx = NULL;
	if((g_array_index(arguments, osso_rpc_t, 0).value.s) == NULL) {
	    osso_log(LOG_DEBUG, "NULL param passed to %s\n",
		     CRASH_REPORTER_UI);
	    return OSSO_ERROR;
	}
	/*
	 * it is bit of waste of space, since we only need to pass name,
	 * but lets keep UI stuff in single function and uniform interface to it.
	 */
	ctx = g_new0(crash_upload_context, 1);
	strncpy(ctx->filename, g_array_index(arguments, osso_rpc_t, 0).value.s, 
		sizeof(ctx->filename) - 1);
	creporter_ui_display_note(NO_MORE_DUPS, ctx);
	return OSSO_OK;
    } else {
	osso_log(LOG_DEBUG,"[%s] Unknown method is called", __FUNCTION__);
    }
    return OSSO_ERROR;
}


/**
  main:

  @param argc is a number of command line arguments.
  @param argv is Command line argument strings.
  @return gint
*/
gint 
main(int argc, char *argv[])
{
	GError *error = NULL;
	osso_hw_state_t hw_state = {TRUE, FALSE, FALSE, FALSE, 0};
	creporterSettings *settings = NULL;
	crash_upload_context * ctx = NULL;
	
	bindtextdomain(PACKAGE, "/usr/share/locale");
        bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);
	
	/* Initialization functions */
	g_thread_init(NULL);
	osso_log(LOG_DEBUG, "Crash Reporter UI is launched with %d arg\n", argc);
	context = osso_initialize(CRASH_REPORTER_UI, "1.0", TRUE, NULL);
	g_return_val_if_fail(context != NULL, 0);

	if (!gconf_init(argc, argv, &error)) {
		osso_log(LOG_ERR, "Error in Gconf init = %s  \n", error->message);
		g_error_free(error);
		return 1;
	}

	gtk_init(&argc, &argv);
	
	if(creporter_libs_init() != TRUE) {
		osso_log(LOG_ERR, "Failed to initialize Crash Reporter libs");
		return 1;
	}
	
	core_location_registry = create_core_location_registry();

	if (argc == 2) {
	    if (!strncmp(argv[1], "-a", 3)) {
		crash_reporter_send_all();
		/*
		 * UI is involved, enter gtk main loop below 
		 */
	    } else if (core_is_valid(argv[1])) {
		osso_log(LOG_DEBUG, "[%s] called with single arg [%s], send this and exit...\n", 
			 __FUNCTION__, argv[1]);
		/*
		 * no UI involved, upload and exit
		 */
		settings = creporter_get_settings();
		if (!settings) {
		    osso_log(LOG_DEBUG, "[%s]: Unable to read settings!\n", __FUNCTION__);
		    return 1;
		} 
		ctx = g_new0(crash_upload_context, 1);
		ctx->settings = settings;
		strncpy(ctx->filename, argv[1], sizeof(ctx->filename) - 1);
		return thread_creporter_upload(ctx);
	    } else {
		osso_log(LOG_DEBUG, "[%s] [%s] is not valid rcore file, exit", 
			 __FUNCTION__, argv[1]);
		return 1;
	    }
	}

	/* We want to monitor for impeding shutdown events */
	osso_hw_set_event_cb(context, &hw_state, hw_state_event_cb, NULL);

	/* registering rpc callback */
	if(osso_rpc_set_default_cb_f_with_free(context, creporter_ui_rpc_callback_handler, NULL, osso_rpc_free_val) != OSSO_OK) {
		osso_deinitialize(context);
		osso_log(LOG_DEBUG, "EXITING SINCE rpc callback register failed");
		return OSSO_ERROR;
	}

	gtk_main();
	osso_deinitialize(context);
	return 0;
}

