/*
 * @file crash-reporter-upload.c
 *   
 * This file implements the upload operation to Crash Reporter
 *  post-processing server.
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <glib.h>
#include <curl/curl.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <unistd.h>
#include <errno.h>

#include "crash-reporter-common.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-handler.h"
#include "crash-reporter-upload.h"

extern void creporter_ui_show_progress(char *str, double percent);
/**
  This function is a callback to read while uploading
  raw data to server

  @param void pointer is a file pointer
  @param size_t is the no of the bytes read from the buffer
  @param void pointer to the stream

*/
static size_t read_callback(void *ptr, size_t size, size_t nmemb, void *stream)
{
	size_t retcode;

	retcode = fread(ptr, size, nmemb, stream);

	return retcode;
}

/**
  Function to write the http response.

  @param buffer the buffer that needs to be written.
  @param size of the buffer to be written.
  @param nmemb size.
  @param userdata passed by fn callback.
  @return actual num of bytes written to file.
  */
static size_t creporter_parse_http_response(char *buffer, size_t size,
					size_t nmemb, void *userdata)
{
    crash_upload_context * ctx = (crash_upload_context *)userdata;
    size_t len = size*nmemb;

    if(ctx->http_resp_code == HTTP_RESP_OTHER_ERROR) {
	if(g_strrstr_len(buffer, len, HTTP_OK) != NULL) {
	    osso_log(LOG_DEBUG,"[%s]: HTTP header response is OK!\n",__FUNCTION__);
	    ctx->http_resp_code = HTTP_RESP_OK;
	}
	else if(g_strrstr_len(buffer, len, HTTP_UNAUTH) != NULL) {
	    osso_log(LOG_DEBUG,"[%s]: HTTP autherization failure!\n",__FUNCTION__);
	    ctx->http_resp_code = HTTP_RESP_AUTH_FAIL;
	}
    }
    if ((len > 2) && (ctx->server_response_sz + len < SERVER_RESPONSE_SZ - 1)) {
	strncpy(ctx->server_response + ctx->server_response_sz, buffer, len);
	ctx->server_response_sz += len;
    }
    return len;
}

/*
 * send visual feedback/progressbar updates about upload
 */
static int progress_callback(void *clientp,
			     double dltotal,
			     double dlnow,
			     double ultotal,
			     double ulnow)
{
     crash_upload_context * ctx = (crash_upload_context *)clientp;
     creporter_ui_show_progress(ctx->filename, ulnow/ultotal);
     return 0;
}

/**
  Function to upload raw data to server

  @param gchar * is the pointer to upload file name
  @param gchar * is the pointer to server url
  @param gchar * is the pointer to server user id
  @param gchar * is the pointer to server password
  @param ssl is TRUE if ssl is to be used else FALSE

  @return gint - returns the https response code
*/
  
gint 
upload (crash_upload_context * ctx)
{
	CURL *curl = NULL;
	CURLcode res = 0;
	FILE *hd_src;
	struct stat file_info;
	gchar *credentials = NULL;
	gchar *uploadurl = NULL;
	gchar *basename = NULL;

	hd_src = fopen(ctx->filename, "rb");
	if (hd_src == NULL) {
	    osso_log(LOG_DEBUG, "[%s]:can not open file %s", __FUNCTION__, ctx->filename);
	    return res;
	}
	if(fstat(fileno(hd_src), &file_info) < 0) {
	    /*
	     * not likely to happen after fopen success, 
	     * but lets have this check for correctness
	     */
	    osso_log(LOG_DEBUG, "[%s]:can not fstat file %s", __FUNCTION__, ctx->filename);
	    return res;
	}

	curl = curl_easy_init();
	if (curl) {
	    basename = g_path_get_basename(ctx->filename);
	    uploadurl = g_strdup_printf("%s%s", ctx->settings->server_addr, basename);
	        credentials = g_strdup_printf("%s:%s", ctx->settings->user_name, ctx->settings->passwd);
		curl_easy_setopt(curl, CURLOPT_READFUNCTION, read_callback);
		curl_easy_setopt(curl, CURLOPT_UPLOAD, 1);
		curl_easy_setopt(curl, CURLOPT_PUT, 1);
		curl_easy_setopt(curl, CURLOPT_URL, uploadurl);
		curl_easy_setopt(curl, CURLOPT_USERPWD, credentials);
		curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
		curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 60);
		curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progress_callback);
		curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, (void*)ctx);
                curl_easy_setopt(curl, CURLOPT_NOPROGRESS, FALSE);		
		if(ctx->settings->use_ssl == TRUE)
		{
			 curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
  	                 curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);
		}

		curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);
		curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

		curl_easy_setopt(curl, CURLOPT_INFILE, hd_src);
		curl_easy_setopt(curl, CURLOPT_INFILESIZE, file_info.st_size);

		/*
		 * initial value is OTHER_ERROR. Parse gets called for pieces of response,
		 * and if OK or AUHT_FAIL is seen, we record status in http_resp_code and stop noting.
		 * If none of them was seen, it was something else, i.e OTHER_ERROR remains
		 */	
		curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, creporter_parse_http_response);
		curl_easy_setopt(curl, CURLOPT_HEADERDATA, (void*)ctx);

		osso_log(LOG_DEBUG, "[%s]: Start curl upload of %s size %d B", 
			 __FUNCTION__, ctx->filename, file_info.st_size);
		ctx->http_resp_code = HTTP_RESP_OTHER_ERROR;
		ctx->server_response_sz = 0;
		ctx->server_response[0] = 0;

#ifdef RUNS_IN_SDK_HOST
		sleep(1); creporter_ui_show_progress("core", 0.01);
		sleep(1); creporter_ui_show_progress("core", 0.10);
		sleep(1); creporter_ui_show_progress("core", 0.20);
		sleep(1); creporter_ui_show_progress("core", 0.30);
		sleep(1); creporter_ui_show_progress("core", 0.40);
		sleep(30);
		sleep(1); creporter_ui_show_progress("core", 0.50);
		sleep(1); creporter_ui_show_progress("core", 0.60);
		sleep(1); creporter_ui_show_progress("core", 0.70);
		sleep(1); creporter_ui_show_progress("core", 0.80);
		sleep(1); creporter_ui_show_progress("core", 0.90);
		sleep(1); creporter_ui_show_progress("core", 0.99);
		sleep(1); creporter_ui_show_progress(NULL, 0);
		sleep(1);
		res = ctx->http_resp_code = 0;
#ifdef FAKE_SERVER_RESPONSE
		res = 0;
		ctx->curl_status = res;
		ctx->http_resp_code = HTTP_RESP_AUTH_FAIL;
		strcpy(ctx->server_response + ctx->server_response_sz, "test auth fail");
		ctx->server_response_sz += strlen("test auth fail");
#else
		ctx->curl_status = res;
#endif /* FAKE_SERVER_RESPONSE */
#else /* RUNS_IN_SDK_HOST */
		res = curl_easy_perform(curl);
		osso_log(LOG_DEBUG, "[%s]: Curl result:%d, response parse:%d server_response_sz=%d\n", __FUNCTION__, 
			 res, ctx->http_resp_code, ctx->server_response_sz);
#endif /* RUNS_IN_SDK_HOST */
		curl_easy_cleanup(curl);
		g_free(credentials);
		g_free(uploadurl);
		g_free(basename);

	} else {
	    osso_log(LOG_DEBUG, "[%s]:can not initialize curl", __FUNCTION__);
	}
	if(hd_src!=NULL)
		fclose(hd_src);

	if (res == 0) {
	    res = ctx->http_resp_code;
	}

	/* 
	 * this tells "complete" to UI
	 */
	creporter_ui_show_progress(NULL, 0);
	return res;
}
