/*
 * @file crash-reporter-handler.c
 *
 * This file contains the function defs
 * for  crash-reporter-handler.c.
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#define DBUS_API_SUBJECT_TO_CHANGE
#include <sys/stat.h>
#include <conic.h>
#include <glib.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <stdlib.h>
#include <string.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <unistd.h>
#include <curl/curl.h>
#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <gconf/gconf-client.h>

#include "crash-reporter-settings-file.h"
#include "crash-reporter-handler.h"
#include "crash-reporter-upload.h"
#include "crash-reporter-common.h"

#ifndef RUNS_IN_SDK_HOST
/* this is a connection instance */
static ConIcConnection *connection = NULL;

/**
  This function is for freeing the dbus connection
                                                                                                                      
  @param dbus_connection pointer to the DBusConnection.
  @return none.
  */
static void creporter_dbus_free(DBusConnection * dbus_connection)
{
	if (dbus_connection != NULL) {
		osso_log(LOG_DEBUG, "freeing dbus\n");
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
	}
}

/**
  This function is for obtaining/removing a dbus connection 
                                                                                                                      
  @param create TRUE to get a connection else FALSE
  @return TRUE on success else FALSE.
  */
static gboolean creporter_handle_dbus(gboolean create)
{
	DBusError error;
	static DBusConnection *dbus_connection = NULL;

	if (create == FALSE) {
		creporter_dbus_free(dbus_connection);
		return TRUE;
	}

	dbus_error_init(&error);
	if (dbus_connection == NULL) {
		dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
		if (dbus_connection == NULL) {
			osso_log(LOG_ERR, "Error conecting to system bus %s\n", error.message);
			return FALSE;

		}
	}

	return TRUE;
}

#endif /* RUNS_IN_SDK_HOST */

/** 
  This function uploads the files to the configured server

  @param filename of the file to upload
  @return CREPORTER_ERROR types in case of failure else CREPORTER_SUCCESS
  */
gint
thread_creporter_upload (crash_upload_context *ctx)
{
    gint result = 0;
    gint rv = -1;

    result = upload(ctx);

    if (result != 0) {
	rv = CREPORTER_CURL_COULD_NOT_CONN_ERROR;
    } else if (result == 0) {
	g_remove(ctx->filename);
	osso_log(LOG_DEBUG, "[%s]: Uploading succesful", 
		 __FUNCTION__);
	rv = CREPORTER_SUCCESS;
    }
    return rv;
}


#ifndef RUNS_IN_SDK_HOST
/**
  This function is the callback for connection-event.
  Uploading of the raw data file will be done after successful
  connection to IAP is made.
                                                                                                                      
  @param connection the ConIcConnection object.
  @param event pointer to the ConIcConnectionEvent.
  @param upload_file Raw data file to be uploaded
  @return none.
  */
static void creporter_connection_cb(ConIcConnection * connection,
			       ConIcConnectionEvent * event, gpointer user_data)
{
	ConIcConnectionStatus status = -1;

	status = con_ic_connection_event_get_status(event);
	switch (status) {
	case CON_IC_STATUS_CONNECTED:
		creporter_libs_gconf_set_connc_status(CONN_CONNECTED);
		osso_log(LOG_DEBUG, "[%s]: Event status is CON_IC_STATUS_CONNECTED", __FUNCTION__);
		break;

	case CON_IC_STATUS_DISCONNECTED:
		osso_log(LOG_DEBUG, "[%s]: Event status is CON_IC_STATUS_DISCONNECTED",
			 __FUNCTION__);
		creporter_libs_gconf_set_connc_status(CONN_DISCONNECTED);
		break;

	default:
		osso_log(LOG_DEBUG, "[%s]: Event status is UNKNOWN", __FUNCTION__);
		break;

	}
	return;
}

/**
  This function is for creating/destroying a conic connection 
                                                                                                                      
  @param create TRUE to create a conic connection else FALSE
  @return TRUE on success else FALSE.
  */
static gboolean creporter_handle_ic_connection(gboolean create)
{
	if (create) {
		if (connection) {
			osso_log(LOG_DEBUG, "[%s]: ConIcConnection is already available",
				 __FUNCTION__);
			return TRUE;
		}

		if (creporter_handle_dbus(TRUE) == FALSE) {
			osso_log(LOG_DEBUG, "[%s]: Failure in handling DBUS", __FUNCTION__);
			return FALSE;
		}

		connection = con_ic_connection_new();
		
		if (!connection) {
			osso_log(LOG_DEBUG, "[%s]: Failure in creating a new IC Connection",
				 __FUNCTION__);
			creporter_handle_dbus(FALSE);
			return FALSE;
		}
	}  else {
		if (!connection) {
			osso_log(LOG_DEBUG, "[%s]: ConIcConnection is not available", __FUNCTION__);
		} else {
			g_object_unref(connection);
			connection = NULL;
//modified the following line
			return FALSE;
		}
	}

	return TRUE;
}
#endif /* RUNS_IN_SDK_HOST */

/**
  This function is to initiate a connection to the default IAP.
  Uploading of the raw data file will be done in the connection
  callback function after successful connection to IAP is made.

  @param void
  @return TRUE on success else FALSE.
  */
gboolean creporter_connect_iap()
{
#ifndef RUNS_IN_SDK_HOST
	static gulong signal_handler = 0;
	gpointer user_data = NULL;
#endif /* RUNS_IN_SDK_HOST */

	creporter_libs_gconf_set_connc_status(CONN_INITIATED);

#ifdef RUNS_IN_SDK_HOST
	osso_log(LOG_DEBUG, "[%s]: connection forced in SDK/SBOX\n", __FUNCTION__);
	creporter_libs_gconf_set_connc_status(CONN_CONNECTED);
#else
	if (creporter_handle_ic_connection(TRUE) == FALSE) {
		return FALSE;
	}
	
	if (signal_handler != 0) {
		g_signal_handler_disconnect(G_OBJECT(connection), signal_handler);
	}


	signal_handler = g_signal_connect(G_OBJECT(connection),
					  "connection-event",
					   G_CALLBACK(creporter_connection_cb),
					   user_data);

	con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
#endif /* RUNS_IN_SDK_HOST */
	return TRUE;
}


/**
  This function will initialize the creporter_libs

  @param void
  @return if success TRUE  else FALSE
  */
gboolean creporter_libs_init()
{
	GConfClient *client = NULL;
	CURLcode curl_code;

	g_type_init();

	if((client = gconf_client_get_default()) == NULL) {
		return FALSE;
	}

	if(!creporter_libs_gconf_set_connc_status(CONN_NOT_INITIATED))
		return FALSE;

	if ((curl_code = curl_global_init(CURL_GLOBAL_ALL))) {
	    osso_log(LOG_DEBUG, "[%s]:curl_global_init failed with %d, can not upload cores", 
		     __FUNCTION__, curl_code);
	    return FALSE;
	}

	return TRUE;
}


/**
  This function will store the creporterConnectionState value as gconf data

  @param creporterConnectionState value
  @return TRUE if success else FALSE
  */
gboolean creporter_libs_gconf_set_connc_status(creporterConnectionState val)
{
	GConfClient *client = NULL;

	if((client = gconf_client_get_default()) == NULL)
		return FALSE;
	gconf_client_set_int(client, CREPORTER_GCONF_CONNECTION_STATUS, val, NULL);
	return TRUE;
}


/**
  This function will get the creporterConnectionState value from gconf file

   @param creporterConnectionState value
   @return TRUE if success else FALSE
   */
gboolean creporter_libs_gconf_get_connc_status(creporterConnectionState *val)
{
	GConfClient *client = NULL;

	if((client = gconf_client_get_default()) == NULL)
		return FALSE;
	*val = gconf_client_get_int(client, CREPORTER_GCONF_CONNECTION_STATUS, NULL);
	return TRUE;
}

