/**
 * @file crash-reporter-daemon-main.c
 *  
 * This file contains the main function for Crash Reporter daemon
 * and dynamically check for the availability
 * of internal/external memory cards sets monitors appropriately
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

#include <conic.h>
#include <time.h>

#include "crash-reporter-daemon-defs.h"
#include "crash-reporter-daemon-monitor.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

static gboolean mmc_change_tmo_handler(void * context);
static gint tmo_id = 0;
static gint lifelog_send_tmo_id = 0;

struct core_location_reg * core_location_registry = NULL;
gboolean sending_enabled = TRUE;
gboolean dumping_enabled = TRUE;
gboolean lifelog_enabled = FALSE;
gboolean avoid_dups = TRUE;

#define PID_FILE "/tmp/creporter.pid"
#define AFTERBOOT_WAIT (30)

/*
 * (re-)activate directory monitoring.
 */
static void
mmc_monitor_dir(int idx)
{
    GDir *dir = NULL;
    if (core_location_registry == NULL) {
	osso_log(LOG_ERR, "[%s] No core locations directory!?",__FUNCTION__);
	return;
    }
    if (is_mounted(core_location_registry[idx].mountpoint)) {
	dir = g_dir_open(core_location_registry[idx].dir, 0, NULL);
	if (dir != NULL) {
	    g_dir_close(dir);
	    if(!set_directory_monitor(idx)) {
		osso_log(LOG_ERR, "[%s] Could not set directory monitor for %s",__FUNCTION__, core_location_registry[idx].dir);
	    }
	}
    }
}

/**
  This function is a callback to detect when the device is about to
  shut down. When that occurs, it's time for the Crash Reporter to go down.
  
  @param hw_state contains the monitored hardware state changes
  @param data contains a pointer to main loop.

*/
static void 
hw_state_event_cb(osso_hw_state_t *hw_state, gpointer data)
{
	if (hw_state->shutdown_ind == TRUE) {
		sending_enabled = 0;
		osso_log(LOG_DEBUG, "[%s]: Shutting down\n", __FUNCTION__);
		g_main_loop_quit((GMainLoop *)data);
		return;
	}
}

/**
  This function is the callback called when ext/int MMC removal/insertion 
  occurs. The directory monitors are reset in such case.

  @param GConfClent for notification
  @param connection ID from gconf_client_notify_add(). 
  @param GConfEntry
  @param user_data
  @return void
  */
static void 
mmc_notifier(GConfClient * gconf_client,
	     guint cnxn_id, GConfEntry * entry,
	     gpointer user_data)
{
    int idx;

    if (entry) {
	/*
	 * called with entry != NULL when gconf values change;
	 * Order timed execution (overwriting previous timer if any)
	 */
	if (tmo_id) {
	    g_source_remove(tmo_id);
	}
	tmo_id = g_timeout_add(5000, mmc_change_tmo_handler, gconf_client);
    } else {
	/*
	 * called with entry==NULL in these cases:
	 * 1. explicitly from main() when crash_reporter_daemon starts
	 * 2. from self-ordered callback few seconds after latest MMC change;
	 * Check and start monitoring directories.
	 */
	for (idx = 0 ; dumping_enabled && (idx < MAX_CORE_DIRS) ; idx++) {
	    if (core_location_registry[idx].mountpoint[0] == 0) {
		break;
	    }
	    mmc_create_dir(idx);
	    mmc_monitor_dir(idx);
	}
    }
}

/*
 * timer based callback after latest change to MMC status, plus some more time.
 * This is needed because gconf change events come before mounting
 * but our actions need to be carried out after mounting.
 */
static gboolean 
mmc_change_tmo_handler(void * context)
{
    GConfClient * gconf_client = (GConfClient *)context;

    mmc_notifier(gconf_client, 0, NULL, NULL);	
    tmo_id = 0;
    return FALSE; /* stop timer calling this */
}

/********************* lifelog stuff ************************/
/*
 *
 * shorter times for debug
#define LIFELOG_SEND_CYCLE 5*60
#define LIFELOG_TICK 60
 */

/*
 * real values for normal use
 *
*/
#define LIFELOG_TICK 3600
#define LIFELOG_SEND_CYCLE 3600*24

#define LZOP_PROG	"/usr/bin/lzop"
#define LIFELOG_TMP_STORE 	"/tmp/lifelog_tmp"
#define LIFELOG_FNAME_BASE_CMD 	"echo -n `sysinfo-tool -g /device/production-sn`> " 
#define LIFELOG_SWREL_CMD 	"echo -n `sysinfo-tool -g /device/sw-release-ver`> " 

#define LIFELOG_STATIC_ENTRY_GEN_CMD "\
echo LL_ID_GEN  sn=`sysinfo-tool -g /device/production-sn`,\
hw=`sysinfo-tool -g /device/hw-version`,\
sw=`sysinfo-tool -g /device/sw-release-ver`,\
wlanmac=`ifconfig wlan0 | awk 'BEGIN {RS=\" *\\n\"} {print $5; exit;}'`,\
efuse=`cat /sys/power/Efuse`,"

#define LIFELOG_STATIC_ENTRY_CELL_CMD "\
echo LL_ID_CELL imei=`dbus-send --system --print-reply --dest=com.nokia.csd.Info /com/nokia/csd/info com.nokia.csd.Info.GetIMEINumber | tail -n +2 | awk '{print $2}'`,\
imsi=`dbus-send --system --print-reply --dest=com.nokia.phone.SSC /com/nokia/phone/SSC com.nokia.phone.SSC.get_imsi | tail -n +2 | awk '{print $2}'`,\
cellmosw=`dbus-send --system --print-reply --dest=com.nokia.csd.Info /com/nokia/csd/info com.nokia.csd.Info.GetMCUSWVersion | tail -n +2`,"

#define LIFELOG_TICK_ENTRY_CMD 	"\
echo LL_TICK date=`date +%s`,\
`lshal | awk '/battery.charge_level.percentage/{print \"batt_perc=\" $3} \
              /battery.rechargeable.is_charging/{print \", charging=\" $3}'`,\
uptime=`cat /proc/uptime`,\
loadavg=`awk '{print $3,$4,$5}' /proc/loadavg`,\
swapfree=`awk '/SwapFree:/{print $2}' /proc/meminfo`,"

#define LIFELOG_CELL_ENTRY_CMD 	"\
echo LL_CELL date=`date +%s`, \
netreg=`dbus-send --system --print-reply --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_registration_status | tail -n +2 |awk '{print $2}'`,\
ssc=`dbus-send --system --print-reply --dest=com.nokia.phone.SSC /com/nokia/phone/SSC com.nokia.phone.SSC.get_modem_state | tail -n +2 | awk '{print $2}'`,\
sig-strength=`dbus-send --system --print-reply --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_signal_strength | tail -n +2 | awk '{print $2}'`,\
gprs-status=`dbus-send --system --print-reply --dest=com.nokia.csd.GPRS /com/nokia/csd/gprs com.nokia.csd.GPRS.GetStatus | tail -n +2 | awk '$1~/string|boolean|uint64/ {print $2}'`,\
gprs-serv-status=`dbus-send --print-reply --system --dest=com.nokia.csd.GPRS /com/nokia/csd/gprs org.freedesktop.DBus.Properties.GetAll string: | tail -n +2 | awk '$2~/boolean|uint64/{print $3}'`,\
gprs-rx-B=`gconftool-2 --get /system/osso/connectivity/network_type/GPRS/gprs_rx_bytes`,\
gprs-tx-B=`gconftool-2 --get /system/osso/connectivity/network_type/GPRS/gprs_tx_bytes`,\
call-counters=`echo 'select * from Events where Events.service_id = 1;' | sqlite3 /home/user/.rtcom-eventlogger/el.db | \
    awk -F'|' '{c[$3]++ }END{printf(\"inbound=%d outbound=%d missed=%d\",c[1],c[2],c[3])}'`,"

#define LIFELOG_FILE 		"/home/user/.crash_reporter_lifelog"
#define LIFELOG_PREV_FILE 	"/home/user/.crash_reporter_lifelog_prev"

//static gboolean lifelog_timer_handler(void * context);
static gint lifelog_timer_id = 0;
static ConIcConnection *connection = NULL;
static int lifelog_send_countdown = LIFELOG_SEND_CYCLE;
static gboolean can_send = FALSE, want_send = FALSE;
static gchar * lifelog_file_name_base = NULL; 
static gchar * lifelog_swrel = NULL; 

static void
lifelog_start_new_file()
{
    int rv;
    osso_log(LOG_DEBUG, "[%s]: start new file", __FUNCTION__);

    rv = system(LIFELOG_STATIC_ENTRY_GEN_CMD ">" LIFELOG_FILE);
    rv = system(LIFELOG_STATIC_ENTRY_CELL_CMD ">>" LIFELOG_FILE);
}

static gint
lifelog_do_send(gchar * file)
{
    char cmd[512];
    char fname[512];
    int rv;
    guint32 rnd;
    time_t tim;
    struct tm * gtimep;

    rnd = g_random_int();
    tim = time(NULL);
    gtimep = gmtime(&tim);
    /* 
     * here we compose the final filename for lzopped file which will
     * appear uploaded in server
     */
    if (gtimep) {
	sprintf(fname, "/tmp/lifelog-%04d%02d%02d-%02d%02d%02d-%s-%08x.rcore.lzo", 
		gtimep->tm_year + 1900,gtimep->tm_mon + 1,gtimep->tm_mday,
		gtimep->tm_hour,gtimep->tm_min,gtimep->tm_sec,
		lifelog_file_name_base, rnd);
    } else {
	/*
	 * not likely, but gmtime may return NULL
	 */
	sprintf(fname, "/tmp/lifelog-%lu-%s-%08x.rcore.lzo", 
		tim,
		lifelog_file_name_base, rnd);
    }

    sprintf(cmd, "%s -c %s > %s", LZOP_PROG, file, fname);
    rv = system(cmd);
    osso_log(LOG_DEBUG, "[%s]:rv=%d from cmd=[%s]", __FUNCTION__, rv, cmd);

    sprintf(cmd, "nice /usr/bin/crash_reporter_ui %s >/dev/null 2>/dev/null", fname);
    rv = system(cmd);
    osso_log(LOG_DEBUG, "[%s]: rv=%d from cmd[%s]", 
	     __FUNCTION__, rv, cmd);

    if(!rv) {
	unlink(fname);
    }
    return rv;
}

/* 
 * Try sending lifelog report.
 * return false will stop this timer, it is once only 
 */
static gboolean
lifelog_try_send(void *context)
{
    osso_log(LOG_DEBUG, "[%s]: can_send=%d want_send=%d",
	     __FUNCTION__, can_send, want_send);

    if (can_send && want_send && (lifelog_do_send(LIFELOG_FILE) == 0)) {
	/*
	 * if send succeeded, reset state and start new countdown
	 */
	lifelog_send_countdown = LIFELOG_SEND_CYCLE;
	want_send = FALSE;
	lifelog_start_new_file();

	/* also, try to send .prev if exists */
	if (g_file_test(LIFELOG_PREV_FILE, G_FILE_TEST_EXISTS)) {
	    lifelog_do_send(LIFELOG_PREV_FILE);
	}
    }
    /* return false will stop this timer, it is once only */
    return FALSE;
}

/*
 * Order timed execution, overwriting previous timer if any;
 * 15 seconds is picked to avoid many net users rushing/jamming to same
 * moments after connection, but short enough that usually
 * connection is still expected to be there, lets hope at least.
 */
static void
lifelog_schedule_try_send()
{
    osso_log(LOG_DEBUG, "[%s]: can_send=%d want_send=%d",
	     __FUNCTION__, can_send, want_send);

    if (lifelog_send_tmo_id) {
	g_source_remove(lifelog_send_tmo_id);
    }
    lifelog_send_tmo_id = g_timeout_add(15000, lifelog_try_send, NULL);
}

/*
 * periodic-timer based callback for making one lifelog entry.
 * This is slow heartbeat, every N hours (N: 8,16 or 24 or smthng like that)
 * it keeps running all the time (return TRUE)
 * We order timed try-send here to spread load more evenly,
 * do not rush to send immediately after data collection
 */
static gboolean 
lifelog_timer_handler (void * context)
{
    int rv;

    rv = system(LIFELOG_TICK_ENTRY_CMD ">>" LIFELOG_FILE);
    rv = system(LIFELOG_CELL_ENTRY_CMD ">>" LIFELOG_FILE);

    lifelog_send_countdown -= LIFELOG_TICK;
    if (lifelog_send_countdown <= 0) {
	want_send = TRUE;
	lifelog_schedule_try_send();
    }
    return TRUE; /* timer keeps running */
}

/*
 * Here we learn about "connected or not" state
 * If getting connected, and something to send,
 * order another callback for sending
 */
static void 
lifelog_connection_cb(ConIcConnection * connection,
		      ConIcConnectionEvent * event, 
		      gpointer user_data)
{
    ConIcConnectionStatus status = con_ic_connection_event_get_status(event);

    switch (status) {
    case CON_IC_STATUS_CONNECTED:
	osso_log(LOG_DEBUG, "[%s]: Event CON_IC_STATUS_CONNECTED", 
		 __FUNCTION__);
	can_send = TRUE;
	if (want_send) {
	    lifelog_schedule_try_send();
	}
	break;

    case CON_IC_STATUS_DISCONNECTED:
	osso_log(LOG_DEBUG, "[%s]: Event CON_IC_STATUS_DISCONNECTED",
		 __FUNCTION__);
	can_send = FALSE;
	break;

    default:
	osso_log(LOG_DEBUG, "[%s]: Event status %d is not handled", 
		 __FUNCTION__, status);
	break;
    }
}

/*
 * some preparation
 */
static void
lifelog_create_name_base()
{
    int rv;
    gsize len = 0;
    GError *err = NULL;
    /*
     * create and read file name base string, as in FNAME_BASE_CMD,
     * using tmp file, then read it. bit clumsy, but works.
     */
    rv = system(LIFELOG_FNAME_BASE_CMD LIFELOG_TMP_STORE);
    /*    osso_log(LOG_DEBUG, "[%s]:rv=%d from fname_base_cmd", __FUNCTION__, rv);*/
    rv = g_file_get_contents(LIFELOG_TMP_STORE,
			     &lifelog_file_name_base,
			     &len,
			     &err);
    /*    osso_log(LOG_DEBUG, "[%s]:read basename_tmp file %s, len=%d, contents=[%s]", 
	     __FUNCTION__, LIFELOG_TMP_STORE, len, lifelog_file_name_base);
    */
    if(!rv) {
	osso_log(LOG_DEBUG, "[%s]:can not read basename_tmp file %s", __FUNCTION__, LIFELOG_TMP_STORE);
    }
    if (err) {
	osso_log(LOG_DEBUG, "[%s]:gerror:%s", __FUNCTION__, err->message);
	g_error_free(err);
    }
    /****************************/
    /*
     * similar method for getting SW release string
     */
    rv = system(LIFELOG_SWREL_CMD LIFELOG_TMP_STORE);
    /*    osso_log(LOG_DEBUG, "[%s]:rv=%d from swrel_cmd", __FUNCTION__, rv);*/
    rv = g_file_get_contents(LIFELOG_TMP_STORE,
			     &lifelog_swrel,
			     &len,
			     &err);
    /*    osso_log(LOG_DEBUG, "[%s]:read basename_tmp file %s, len=%d, contents=[%s]", 
	  __FUNCTION__, LIFELOG_TMP_STORE, len, lifelog_swrel);*/
    if(!rv) {
	osso_log(LOG_DEBUG, "[%s]:can not read basename_tmp file %s", __FUNCTION__, LIFELOG_TMP_STORE);
    }
    if (err) {
	osso_log(LOG_DEBUG, "[%s]:gerror:%s", __FUNCTION__, err->message);
	g_error_free(err);
    }
    unlink(LIFELOG_TMP_STORE);
}

/*
 * if existing log file is from different SW rel, then
 * we are reflashed. Start new file, and rename previous one to _prev,
 * it will be sent later after real file been sent
 */
static void
lifelog_check_existing_file ()
{
    int rv;
    gsize len;
    GError *err = NULL;
    gchar *curlog = NULL;
    gchar *p, *pend;
    gint records = 0;

    rv = g_file_get_contents(LIFELOG_FILE,
			     &curlog,
			     &len,
			     &err);
    if (err) {
	/*	osso_log(LOG_DEBUG, "[%s]:gerror:%s", __FUNCTION__, err->message);*/
	g_error_free(err);
    }
    if(!rv) {
	/*	osso_log(LOG_DEBUG, "[%s]:can not read current log file %s", __FUNCTION__, LIFELOG_FILE);*/
	lifelog_start_new_file();
    } else {
	if(g_strstr_len(curlog, len, lifelog_swrel)) {
	    osso_log(LOG_DEBUG, "[%s]:current swver[%s] found in existing file, will continue it", 
		     __FUNCTION__, lifelog_swrel);
	    /*
	     * count the lines
	     */
	    pend = curlog + len;
	    for (p = curlog ; p && *p && (p < pend); p++) {
		if (*p == '\n') {
		    records += 1;
		}
	    }
	    /*
	     * We have 2 lines header and 2 lines per record.
	     * XXX note that this code has 2,2 hard coded and needs to be changed
	     * whenever those numbers change!!!
	     */
	    if (records >= 4) {
		records = (records - 2)/2;
		lifelog_send_countdown -= records * LIFELOG_TICK;
		osso_log(LOG_DEBUG, "[%s]:current log has %d entries, adjust send target %d sec closer, remaining %d sec", 
			 __FUNCTION__, records, records * LIFELOG_TICK, lifelog_send_countdown);
		if (lifelog_send_countdown <= 0) {
		    want_send = TRUE;
		    lifelog_schedule_try_send();
		}
	    }

	} else {
	    osso_log(LOG_DEBUG, "[%s]:running swver[%s] does not match lifelog file, move file to prev, start a new file", 
		     __FUNCTION__, lifelog_swrel);
	    /*
	     * rename current file tp _prev. If older prev exists, tough luck for it.
	     */
	    rename(LIFELOG_FILE, LIFELOG_PREV_FILE);
	    lifelog_start_new_file();
	}
    }
    g_free(curlog);
}

static void
setup_lifelog()
{
#ifndef RUNS_IN_SDK_HOST
	static gulong signal_handler = 0;
	connection = con_ic_connection_new();
	/*	osso_log(LOG_DEBUG, "[%s]: new IC Connection = %p",
		__FUNCTION__, connection);*/
		
	if (!connection) {
	    osso_log(LOG_DEBUG, "[%s]: Failure in creating a new IC Connection",
		     __FUNCTION__);
	    return;
	}
	g_object_set(connection, "automatic-connection-events", TRUE, NULL);

	signal_handler = g_signal_connect(G_OBJECT(connection),
					  "connection-event",
					  G_CALLBACK(lifelog_connection_cb),
					  NULL);
	/*	osso_log(LOG_DEBUG, "[%s]: g_signal_connect returned %d", __FUNCTION__, signal_handler);*/
#endif /* RUNS_IN_SDK_HOST */

	lifelog_create_name_base();
	lifelog_send_countdown = LIFELOG_SEND_CYCLE;
	lifelog_check_existing_file();

	lifelog_timer_id = g_timeout_add(LIFELOG_TICK*1000, lifelog_timer_handler, NULL);

}
/********************* EO lifelog stuff ************************/

/**
  main:

  @param argc is a number of command line arguments.
  @param argv is Command line argument strings.
  @return gint
*/
gint 
main(int argc, char *argv[])
{
	GConfClient *gconf_client = NULL;
	GMainLoop *loop = NULL;
	privacySettings* privsettings = NULL;
	osso_hw_state_t hw_state = {TRUE, FALSE, FALSE, FALSE, 0};
	FILE *fp;
	int wait = 0;
	pid_t pid;
	gchar *pid_str;

	pid = getpid();
	if(access(PID_FILE, F_OK)) {
	    /*
	     * no PID file, this is first run, lets delay our start
	     */
	    wait = AFTERBOOT_WAIT;
	}

	fp = fopen(PID_FILE, "w");
	osso_log(LOG_DEBUG, "crash_reporter_daemon[%d] starts, wait=%d", pid, wait);
	if (fp) {
	    pid_str = g_strdup_printf("%d", pid);
	    fwrite(pid_str, sizeof(char), strlen(pid_str), fp);
	    fclose(fp);
	    g_free(pid_str);
	}

	if (wait) {
	    sleep(wait);
	}

	/* Initialization functions */
	g_thread_init(NULL);
	g_type_init();

	/* context for libosso */
	context = osso_initialize("crash_reporter_daemon", "1.0", TRUE, NULL);
	g_return_val_if_fail(context != NULL, 0);


	gconf_client = gconf_client_get_default();
	gconf_client_add_dir(gconf_client, GCONF_DIR,
			     GCONF_CLIENT_PRELOAD_NONE, NULL);

	core_location_registry = create_core_location_registry();

	if (core_location_registry) {
	    privsettings = creporter_read_privacy_settings();
	    if (privsettings) {
		sending_enabled = privsettings->sending_enabled;
		dumping_enabled = privsettings->dumping_enabled;
		lifelog_enabled = privsettings->lifelog_enabled;
		avoid_dups = privsettings->avoid_dups;
		creporter_free_privacy_settings(privsettings);
		privsettings = NULL;
	    } else {
		sending_enabled = TRUE;
		dumping_enabled = TRUE;
		avoid_dups = TRUE;
	    }

	    /*
	     * this creates initial inotify handle 
	     */
	    init_inotify_and_check_for_cores();	
	    /*
	     * this creates inotify handles for directories
	     */
	    mmc_notifier(gconf_client, 0, NULL, NULL);	

	    /* Order notification for some mmc events */
	    gconf_client_notify_add(gconf_client, EXTERNAL_COVER_OPEN,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, EXTERNAL_MMC_PRESENT,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, INTERNAL_MMC_PRESENT,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, MMC_CORRUPTED,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, EXTERNAL_MMC_USED_USB,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, INTERNAL_MMC_USED_USB,
				    mmc_notifier, NULL, NULL, NULL);

	    loop = g_main_loop_new(NULL, FALSE);
	    osso_hw_set_event_cb(context, &hw_state, hw_state_event_cb, loop);
	    if (lifelog_enabled && sending_enabled) {
		osso_log(LOG_DEBUG, "enable lifelog, snapshot interval %ds, send interval %ds",
			 LIFELOG_TICK, LIFELOG_SEND_CYCLE);
		setup_lifelog();
	    } else {
		osso_log(LOG_DEBUG, "lifelog disabled");
	    }
	    g_main_loop_run(loop);
	    osso_deinitialize(context);
	} else {
	    osso_log(LOG_ERR, "crash_reporter_daemon[%d] can not create core location registry", 
		     pid);
	}
	
	return 0;
}
