/*
 * @file crash-reporter-ui-utils.c
 *  
 * This file contains most of the UI operations for uploading Rich cores.
 * It uses Crash Reporter Library for uploading
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gconf/gconf-client.h>
#include <hildon/hildon-note.h>
#include <unistd.h>
#include <glib.h>
#include <stdlib.h> 
#include <string.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <glib/gstdio.h>
#include <errno.h>

/*
 * for appending to LZO archive
 */
#define RICH_CORE_TMP_NOTEFILE	"/tmp/rich-core-note.txt"
#define LZOP_PROG		"/usr/bin/lzop"

#include "crash-reporter-settings-file.h"
#include "crash-reporter-handler.h"
#include "crash-reporter-ui-utils.h"
#include "crash-reporter-ui-main.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

/*It stores the list of files to be uploaded*/
extern GSList *files_list;

/* Crash Reporter UI Application context */
extern osso_context_t *context;


/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void creporter_ui_set_connec_callback_notifiers();

/**
  When Crash Reporter UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files shud be appended
  @param TRUE if success else FALSE
*/
static gboolean creporter_ui_collect_all_corefile_names_at_location(const gchar *location);

void usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer);
void usecase_other_button_clicked (GtkWidget *button, void *arg);
void usecase_options_button_clicked (GtkWidget *button, void *arg);
static GtkWidget *creporter_ui_usecase_dialog(gchar *text);
static void creporter_ui_process_response(gboolean send_all_files_mode, gint response);
static gchar *compose_crashed_note(gchar *filename);

/*
 * counts and stats
 */
static gint tried_count = 0, succeeded_count = 0;

static gint creporter_ui_upload_files();
void creporter_ui_show_progress(char *str, double percent);

static gboolean update_progressbar (gpointer dialog);

/*
 * pointer to usecase/options/send/delete dialog,
 * so that we can call response() on it from button handlers
 */
static GtkDialog *dialog = NULL;
static GtkDialog *send_progress_dialog = NULL;
static GtkWidget* progressbar = NULL;
static GtkWidget* progresslabel = NULL;
static double progress_percent = 0.0;

#define PROGRESS_DLG_VISIBLE	(progressbar && progresslabel)
#define LIST_MAX_CORES (8)

static void 
creporter_ui_process_response(gboolean send_all_files_mode, gint response)
{
    gchar *file_path=NULL;

    switch(response) {
    case GTK_RESPONSE_CANCEL:
	osso_log(LOG_DEBUG, "[%s]: User cancelled\n", __FUNCTION__);
	creporter_ui_exit();
	break;
	
    case GTK_RESPONSE_OK:
	/* Create an internet connection and wait for gconf connec callback*/
	if (!creporter_connect_iap()) {
	    creporter_ui_display_note(CONF_CONNECTION_ERROR, NULL);
	    osso_log(LOG_DEBUG, "[%s]: could not connect\n", __FUNCTION__);
	    creporter_ui_exit();
	}
	break;
    /* GTK_RESPONSE_APPLY is the response to delete single Core file, 
     *  or "Delete All" during creporter startup 
     */	
    case GTK_RESPONSE_APPLY:
	if(files_list) {
	    osso_log(LOG_DEBUG, "[%s]: User selected Delete\n", __FUNCTION__);
	    if(send_all_files_mode) {
		GSList *temp = files_list;
		while(temp){
		    if(g_remove(temp->data) != 0)
			osso_log(LOG_DEBUG,"[%s] Could not remove the file [%s] errno=%d", 
				 __FUNCTION__, temp->data, errno);
		    temp = g_slist_next(temp);
		}
		g_slist_free(files_list);
		files_list = NULL;
	    } else {
		file_path = (gchar *)files_list->data;
		if(g_remove(file_path) != 0)
		    osso_log(LOG_DEBUG,"[%s] Could not remove the core file [%s], errno=%d",
			     __FUNCTION__, file_path, errno);
		g_free(file_path);
		files_list = g_slist_remove(files_list, files_list->data);
	    }
	}
	creporter_ui_exit();
	break;
    default:
	creporter_ui_exit();
	break;
    }
}

/**
  This function asks and takes the user confirmation
  and establishes the iap connection

  @param gboolean TRUE when need to send all files
		  FALSE when only one file is to be sent
  @param void              
*/
void creporter_ui_ask_user_conf(gboolean send_all_files_mode)
{
    gint response = 0;

    /* Display user confirmation to send rich cores */
    if(!send_all_files_mode) {
	response = creporter_ui_display_note(CONF_USER_CONFIRMATION1, NULL);
    }
    else {
	response = creporter_ui_display_note(CONF_USER_CONFIRMATION2, NULL);
    }
    creporter_ui_process_response(send_all_files_mode, response);
}

/* Callback for other buttons */
static void
send_progress_button_clicked (GtkWidget *button, void *arg)
{
    gint response = (gint)arg;

    if (send_progress_dialog) {
	gtk_dialog_response(send_progress_dialog, response);
    }
}

static GtkWidget *
creporter_ui_send_progress_dialog ()
{
    static GtkWidget *window;
    GtkWidget *cancel_button;
    
    window = gtk_dialog_new ();

    gtk_window_set_title (GTK_WINDOW (window), _("Uploading crash report"));
    //    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_widget_set_size_request(window, 600, 150);
    gtk_window_set_modal (GTK_WINDOW (window), FALSE);

    progresslabel = gtk_label_new("initial note");
    gtk_label_set_line_wrap(GTK_LABEL(progresslabel), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), GTK_WIDGET(progresslabel), 
                        TRUE, TRUE, 0);

    /*------------*/
    /* Add progress bar */
    progressbar = gtk_progress_bar_new();
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), 
			progressbar, TRUE, TRUE, 0);
    g_timeout_add(1000, update_progressbar, NULL);

    /*------------*/

    /* Add a "Stop" button to the bottom of the dialog */
    cancel_button = gtk_dialog_add_button(GTK_DIALOG(window),
					  dgettext("hildon-libs",
						   "wdgt_bd_stop"),
					  GTK_RESPONSE_CANCEL);
					  
    g_signal_connect (G_OBJECT (cancel_button), "clicked",
		      G_CALLBACK (send_progress_button_clicked),
		      (void*)GTK_RESPONSE_CANCEL); 
    gtk_widget_set_no_show_all(cancel_button, FALSE);

    gtk_widget_show_all (window);

    return window;
}

/**
  This function is called to upload all the files that stored in files_list
  This will call the creporter_libs APIs to upload the each file

  @param void
  @return void
  */
static gint 
creporter_ui_upload_files(crash_upload_context ** pctx)
{
    GSList *temp = NULL;
    temp = files_list;
    gint rv = -1;
    gint response;
    GtkWidget *wid = NULL;
    gint cur, tot;
    gchar * size_note;
    struct stat file_info;
    gchar *basename = NULL;
    GThread* send_thread = NULL;
    creporterSettings *settings = NULL;
    crash_upload_context * ctx = NULL;

    settings = creporter_get_settings();
    if (!settings) {
	osso_log(LOG_DEBUG, "[%s]: Unable to read settings!\n", __FUNCTION__);
	return CREPORTER_NO_SETTINGS_ERROR;
    } else {
	osso_log(LOG_DEBUG, "[%s]: Settings: '%s'-'%s'-'%s'\n", 
		 __FUNCTION__, settings->server_addr,
		 settings->user_name, settings->passwd);
    }

    ctx = g_new0(crash_upload_context, 1);
    ctx->settings = settings;
    tot = g_slist_length(files_list);
    tried_count = succeeded_count = 0;
    if (!send_progress_dialog) {
	wid = creporter_ui_send_progress_dialog();
	send_progress_dialog = GTK_DIALOG(wid);
	gtk_window_set_title(GTK_WINDOW(send_progress_dialog), _("Sending crash report"));
    }
    for(cur = 1, temp = files_list; temp; temp = g_slist_next(temp), cur++) {
	/*
	 * Skip processing if there is no file (means, it was just sent
	 * and deleted on previous loop round)
	 */
	if(access(temp->data, R_OK) < 0) {
	    osso_log(LOG_DEBUG, "[%s]: File %s not found, skip", __FUNCTION__, temp->data);
	    continue;
	}

	strncpy(ctx->filename, temp->data, sizeof(ctx->filename) - 1);

	if (send_thread) {
	    osso_log(LOG_DEBUG, "[%s] previous upload thread still active, wait...", __FUNCTION__);
	    rv = (gint)g_thread_join(send_thread);
	    osso_log(LOG_DEBUG, "[%s] join on previous upload thread returned %d", __FUNCTION__, rv);
	}
	send_thread = g_thread_create((GThreadFunc)thread_creporter_upload,
				      ctx, TRUE, NULL);

	tried_count += 1;
	if (PROGRESS_DLG_VISIBLE) {
	    if(stat((gchar *)temp->data, &file_info) < 0) {
		osso_log(LOG_DEBUG, "[%s]: can  not stat %s, skip", __FUNCTION__, temp->data);
		continue;
	    }
	    basename = g_filename_display_basename((gchar *)temp->data);
	    if (tot > 1) {
		gchar * sending_note = g_strdup_printf(_("Sending crash report %d / %d"), cur, tot);
		gtk_window_set_title (GTK_WINDOW (send_progress_dialog), sending_note);
		g_free(sending_note);
	    } 
	    gtk_label_set_text(GTK_LABEL(progresslabel), basename);
	    size_note = g_strdup_printf(_("%ld kB"), file_info.st_size / 1024);
	    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progressbar), size_note);
	    progress_percent = 0.0;
	    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar), progress_percent);
	    g_free(basename);
	    g_free(size_note);
	    response = gtk_dialog_run(send_progress_dialog);
	} else {
	    /*
	     * dialog was hidden, response remains CLOSE and we go waiting 
	     * on thread again. Just to be sure, and for clarity,
	     * set it CLOSE once more here
	     */
	    response = GTK_RESPONSE_CLOSE;
	}
	if (response == GTK_RESPONSE_CANCEL) {
	    /*  cancel or dialog delete */
	    osso_log(LOG_DEBUG, "[%s] cancel/delete (%d), delete dialog", __FUNCTION__, response);
	    gtk_widget_destroy(GTK_WIDGET(wid));
	    send_progress_dialog = NULL;
	    progressbar = NULL;
	    progresslabel = NULL;
	    rv = CREPORTER_CANCEL;
	    break;
	}
	else if ((response == GTK_RESPONSE_CLOSE) ||
	    (response == GTK_RESPONSE_DELETE_EVENT)) {
	    /*  hide. continue sending in background */
	    gtk_widget_hide(GTK_WIDGET(send_progress_dialog));
	    while (gtk_events_pending ()) {
		gtk_main_iteration();
	    }
	    progressbar = NULL;
	    progresslabel = NULL;
	    rv = (gint)g_thread_join(send_thread);
	    if (rv == CREPORTER_SUCCESS) {
		succeeded_count += 1;
	    }
	    send_thread = NULL;
	    continue;
	}
	/* returns with status OK if progressbar indicates "complete" by special sign.
	* Otherwise we can not get out of dialog_run easily 
	*/
	else if (response == GTK_RESPONSE_OK) {
	    rv = (gint)g_thread_join(send_thread);
	    send_thread = NULL;
	    if (rv == CREPORTER_SUCCESS) {
		succeeded_count += 1;
	    }
	}
	else {
	    osso_log(LOG_DEBUG, "[%s] unexpected ret val %d from progress dialog run", __FUNCTION__, response);
	}
    }
    if (send_progress_dialog) {
	gtk_widget_destroy(GTK_WIDGET(wid));
	send_progress_dialog = NULL;
    }
    progressbar = NULL;
    progresslabel = NULL;
    g_slist_free(files_list);
    files_list = NULL;
    *pctx = ctx;
    return rv;
}

/**
  This is callback called when there are any changes to the variable
  CREPORTER_GCONF_CONNECTION_STATUS_DIR ie, to get the connection status

  @param the GConfClient that is notifying 
  @param cnxn_id, connection ID from gconf_client_notify_add()
  @param a GConfEntry entry
  @param user data
  @retuen void 
  */
void static creporter_ui_connect_status_cb (GConfClient *gconf_client_connec_dir,
				guint cnxn_id, GConfEntry *entry,
				gpointer user_data)
{
	creporterConnectionState connec_status = -1;
	gint result = -1;
	crash_upload_context * ctx = NULL;

	creporter_libs_gconf_get_connc_status(&connec_status);

	if(connec_status == CONN_CONNECTED){
		/* Succesfully able to connect */
		result = creporter_ui_upload_files(&ctx);
	
		if(result == CREPORTER_SUCCESS) {
			/*upload success */
			creporter_ui_display_note(CONF_FILE_SENT, ctx);
			creporter_ui_exit();	
		} else if (result == CREPORTER_CANCEL) {
			osso_log(LOG_DEBUG, "[%s]: upload cancelled by user", 
				 __FUNCTION__);
			creporter_ui_exit();
		} else if ((result == -1) && (tried_count == 0)) {
			osso_log(LOG_DEBUG, "[%s]: no files were found to upload", 
				 __FUNCTION__);
			/*
			 * it was not really CURL error because we did
			 * not reach curl phase, but this shows handy generic
			 * error msg when no context given
			 */
			creporter_ui_display_note(CONF_CURL_ERROR, NULL);
			creporter_ui_exit();
		} else {
		        /* Upload operation failed */
		    if (dialog) {
			osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CURL_ERROR note", 
				 __FUNCTION__);
		    } else {
			osso_log(LOG_DEBUG,"[%s] Upload operation failed, Error=%d", __FUNCTION__, result);
			creporter_ui_display_note(CONF_CURL_ERROR, ctx);
			creporter_ui_exit();
		    }
		}
	} else if(connec_status == CONN_DISCONNECTED){
	    if (dialog) {
		osso_log(LOG_DEBUG, "[%s]: dialog already active, skip showing CONNECTION_ERROR note", 
			 __FUNCTION__);
	    } else {
		osso_log(LOG_DEBUG, "[%s] Couldnot create connection, connec_status error is %d", __FUNCTION__, connec_status);
		creporter_ui_display_note(CONF_CONNECTION_ERROR, NULL);
		creporter_ui_exit();
	    }
	}
}

/**
  This function set the connection status callback notifiers 
  
  @param void
  @param void  
*/
void creporter_ui_set_connec_callback_notifiers()
{
	GConfClient *gconf_client_connec_dir = NULL;
	gpointer user_data = NULL; 

	/* set callback notifier for gconf variable
	 * CREPORTER_GCONF_CONNECTION_STATUS*/	
	gconf_client_connec_dir = gconf_client_get_default();

	gconf_client_add_dir(gconf_client_connec_dir, 
			CREPORTER_GCONF_CONNECTION_STATUS_DIR,
			GCONF_CLIENT_PRELOAD_NONE, NULL	);

	gconf_client_notify_add(gconf_client_connec_dir, 
			CREPORTER_GCONF_CONNECTION_STATUS, 
			creporter_ui_connect_status_cb, 
			user_data, NULL, NULL);
}

static gchar *
compose_crashed_note (gchar *filename)
{
    gchar *text = NULL;
    gchar *text2 = NULL;
    gchar *sig_num = NULL;
    gchar *exe_name = NULL;
    gchar *process_id = NULL;
    gint size;

    creporter_get_rcore_fileinfo(filename, &size, &sig_num, &exe_name, &process_id);
    if (sig_num && exe_name && process_id) {
	text = g_strdup_printf(_("'%s' PID %s died to signal %s"), exe_name, process_id, sig_num);
    } else {
	text = g_strdup_printf(_("A crash file %s was generated"), (gchar *) files_list->data);
    }
    text2 = g_strdup_printf(_("%s. Send %d kB crash report for analysis?"), 
			    text, size/1024); 

    g_free(sig_num);
    g_free(exe_name);
    g_free(process_id);
    g_free(text);

    return text2;
}

static gint 
list_first_cores(gchar ** listed)
{
    gint cnt;
    gint size = 0;
    GSList *temp = NULL;
    gchar * s = NULL;
    gchar * s2 = NULL;
    struct stat file_info;
    gchar *basename;

    s = g_strdup("");
    for(cnt = 0, temp = files_list; temp ; temp = g_slist_next(temp), cnt++) {
	bzero(&file_info, sizeof(file_info));
	if(stat((gchar *)temp->data, &file_info) < 0) {
	    continue;
	}
	size += file_info.st_size;
	if (cnt <= LIST_MAX_CORES - 1) {
	    basename = g_filename_display_basename((gchar *)temp->data);
	    s2 = g_strdup_printf(_("%s%ld kB\t%s\n"), s, file_info.st_size/1024, basename);
	    g_free(s);
	    g_free(basename);
	    s = s2;
	}
    }
    if (cnt > LIST_MAX_CORES) {
	s2 = g_strdup_printf(_("%s--showing first %d of %d entries--\n"), 
			     s, LIST_MAX_CORES, cnt);
	g_free(s);
	s = s2;
    }

    *listed = s;
    return size;
}

/**
  This function is used to display all the confirmation notes in the appl.

  @param the type confirmation note to be displayed
  @return the response from the user
  */
gint 
creporter_ui_display_note (creporterConfirmationType conf_type, crash_upload_context *ctx)
{
    GtkWidget *note = NULL;
    gchar *text = NULL;
    gint response = 0;
    creporterSettings *sett = NULL;
    gint num;
    gchar * listed = NULL;
    gint size;
    gchar *basename = NULL;
	
    if (dialog) {
	osso_log(LOG_DEBUG, "[%s]: dialog already active, skip and return", 
		 __FUNCTION__);
	if (ctx) {
	    g_free(ctx);
	}
	return response;
    }
    switch(conf_type) {
    case CONF_USER_CONFIRMATION1:
	if(!files_list) {
	    osso_log(LOG_DEBUG, "[%s]: Files list is NULL",__FUNCTION__);
	    if (ctx) {
		g_free(ctx);
	    }
	    return GTK_RESPONSE_CANCEL;
	}
	text = compose_crashed_note((gchar *) files_list->data);
	note = creporter_ui_usecase_dialog(text);
	break;
    case CONF_USER_CONFIRMATION2:
	sett = creporter_get_settings();
	num = g_slist_length(files_list);
	if (num > 0) {
	    size = list_first_cores(&listed);
	    text = g_strdup_printf(_("\
This system has %u stored crash %s:\n\
%s\
Total %d kB, send for analysis?\n\
%s%s"),
				   num, (num>1)? "reports" : "report",
				   listed, size/1024,
				   sett? "\nReceiving server: " : "", 
				   sett? sett->server_addr : "");

	    g_free(listed);
	    listed = NULL;
	    note = hildon_note_new_confirmation_add_buttons(NULL, text, 
							    _("Yes"), GTK_RESPONSE_OK,
							    _("No"),GTK_RESPONSE_CANCEL, 
							    _("Delete All"), GTK_RESPONSE_APPLY,
							    NULL);
	} else {
	    text = g_strdup(_("This system has no stored crash reports."));
	    note = hildon_note_new_information(NULL, text);
	}
	break;
    case CONF_CONNECTION_ERROR:
	text = g_strdup(_("Crash Reporter failed to connect"));
	note = hildon_note_new_information(NULL, text);
	break;
    case CONF_CURL_ERROR:
	if (ctx) {
	    basename = g_filename_display_basename(ctx->filename);
	    if (ctx->server_response[0]) {
		text = g_strdup_printf(_("\
Unexpected response from crash reports server, \
file %s, send status %d, \
http status %d.\nServer response was:\n%s"), 
				       basename, ctx->curl_status, 
				       ctx->http_resp_code, ctx->server_response);
	    } else {
		text = g_strdup_printf(_("\
Crash report upload did not succeed, \
file %s, problem connecting to server, send status %d"),
				       basename, ctx->curl_status);
	    }
	    g_free(basename);
	} else {
	    text = g_strdup(_("Problem with the crash report upload."));
	}
	note = hildon_note_new_information(NULL, text);
	break;
    case CONF_FILE_SENT:
	if (tried_count == 1) {
	    if(succeeded_count == 1) {
		if (ctx) {
		    gchar *basename = g_filename_display_basename(ctx->filename);
		    text = g_strdup_printf(_("The crash report %s was successfully uploaded"), basename);
		    g_free(basename);
		} else {
		    text = g_strdup(_("The crash report was successfully uploaded"));
		}
	    } else {
		/* 
		 * this should not happen .. 
		 */
		text = g_strdup(_("The crash report sending failed"));
	    }
	} else {
	    text = g_strdup_printf(_("Crash reports upload result: %d files attempted, %d succeeded"),
				   tried_count, succeeded_count);
	}
	note = hildon_note_new_information(NULL, text);
	break;
    case NO_MORE_DUPS:
	basename = g_filename_display_basename(ctx->filename);
	text = g_strdup_printf(_("Crash Reporter detected %d duplicates for the %s crash report, the following duplicates will be silently deleted."),
				   MAX_SIMILAR_RCORES, basename);
	note = hildon_note_new_information(NULL, text);
	break;
    default:
	/*
	 * note being NULL protects us from using/showing it below
	 */
	break;
    }
        
    if (note) {
	dialog = GTK_DIALOG(note);
	response = gtk_dialog_run(dialog);
	dialog = NULL;
	gtk_widget_destroy(GTK_WIDGET(note));
	note = NULL;
    }
    if (text) {
	g_free(text);
    }
    if (ctx) {
	g_free(ctx);
    }
    return response;
}

/* Callback for close button */
void
usecase_send_button_clicked (GtkWidget *button, GtkTextBuffer *buffer)
{
    GtkTextIter start;
    GtkTextIter end;
    gchar *text;
    gchar *cmd;
    FILE *fp;
    gint len, cmd_len;
    int res;

    /* Obtain iters for the start and end of points of the buffer */
    gtk_text_buffer_get_start_iter (buffer, &start);
    gtk_text_buffer_get_end_iter (buffer, &end);

    /* Get the entire buffer text. */
    text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
    len = strlen(text);

    if (len > 0) {
	fp = fopen(RICH_CORE_TMP_NOTEFILE, "w");
	if(fp){
	    if (fwrite(text, sizeof(char), len, fp) < len) {
		osso_log(LOG_ERR, "[%s]: error writing rich-core-note to file %s", 
			 __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	    }
	    fclose(fp);
	    cmd_len = sizeof(LZOP_PROG) + sizeof(RICH_CORE_TMP_NOTEFILE) + strlen((gchar *)files_list->data) + 20;
	    cmd = g_malloc(cmd_len);
	    if (cmd) {
		sprintf(cmd, "%s -c %s >> %s", LZOP_PROG, RICH_CORE_TMP_NOTEFILE, (gchar *)files_list->data);
		res = system(cmd);
		osso_log(LOG_ERR,"[%s]: system[%s] returned %d\n", __FUNCTION__, cmd, res);
		g_free(cmd);
	    }
	    unlink(RICH_CORE_TMP_NOTEFILE);
	} else {
	    osso_log(LOG_ERR,"[%s]: error opening file %s\n", __FUNCTION__, RICH_CORE_TMP_NOTEFILE);
	}
    }
    g_free (text);

    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, GTK_RESPONSE_OK);
    }
}

/* Callback for options button */
void
usecase_options_button_clicked (GtkWidget *button, void *arg)
{
    gint res = 0;
    GtkWidget *top_window = (GtkWidget*)arg;
    
    if (context && top_window) {
	res = osso_cp_plugin_execute(context, "libcpcrashreporter.so", top_window, TRUE);
    } else {
	osso_log(LOG_DEBUG, "[%s]: no osso context or topwindow??",__FUNCTION__);
    }
}

/* Callback for other buttons */
void
usecase_other_button_clicked (GtkWidget *button, void *arg)
{
    gint response = (gint)arg;
    /*
     * If we are here, then dialog must exist and
     * point to usecase/send/delete/options dialog,
     * but lets check for being really sure 
     */
    if (dialog) {
	gtk_dialog_response(dialog, response);
    }
}

/**
  Create elements of "describe what you were doing" and 
  send/cancel/del/options dialog
  */
static GtkWidget *
creporter_ui_usecase_dialog (gchar *text)
{
    static GtkWidget *window;
    GtkWidget *scrolled_window;
    GtkWidget *label_note;
    GtkWidget *label_server;
    GtkWidget *send_button;
    GtkWidget *delete_button;
    GtkWidget *options_button;
    GtkTextBuffer *buffer;
    GtkWidget *text_view;
    GtkWidget *frame;
    creporterSettings *sett = NULL;
    
    /* Create a new dialog window for the scrolled window to be
     * packed into.  */
    window = gtk_dialog_new ();
    /***
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                        (GtkSignalFunc) destroy, NULL);
    **/
    gtk_window_set_title (GTK_WINDOW (window), _("Crash Reporter"));
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    gtk_widget_set_size_request(window, 700, 360);
    label_note = gtk_label_new(text);
    gtk_label_set_line_wrap(GTK_LABEL(label_note), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), label_note, 
                        FALSE, FALSE, 0);

    /* create a new scrolled window. */
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);

    /* the policy is one of GTK_POLICY AUTOMATIC, or GTK_POLICY_ALWAYS.
     * GTK_POLICY_AUTOMATIC will automatically decide whether you need
     * scrollbars, whereas GTK_POLICY_ALWAYS will always leave the scrollbars
     * there.  The first one is the horizontal scrollbar, the second, 
     * the vertical. */
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    frame = gtk_frame_new(_("Please describe what you were doing:"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), frame, 
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);


    /* Create a multiline text widget. */
    text_view = gtk_text_view_new ();
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_WORD);
    /* Obtaining the buffer associated with the widget. */
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));

    /* pack the text widget into the scrolled window */
    gtk_scrolled_window_add_with_viewport (
                   GTK_SCROLLED_WINDOW (scrolled_window), text_view);

    /*------------*/
    sett = creporter_get_settings();
    if (sett) {
	gchar *srv_txt;
	srv_txt = g_strdup_printf(_("Receiving server: %s"), sett->server_addr); 
	label_server = gtk_label_new(srv_txt);
	gtk_label_set_line_wrap(GTK_LABEL(label_server), TRUE);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(window)->vbox), label_server, 
			    FALSE, FALSE, 0);
    }
    /*------------*/

    /* FIXME: Now that the buttons are created with gtk_dialog_add_button,
       connecting signals to buttons should perhaps be replaced by
       local gtk_dialog_run + switch/case setup? */

    /* Add a "Send" button to the bottom of the dialog */
    send_button = gtk_dialog_add_button(GTK_DIALOG(window), _("Send"),
					GTK_RESPONSE_NONE);
    g_signal_connect (G_OBJECT (send_button), "clicked",
		      G_CALLBACK (usecase_send_button_clicked),
		      buffer);
    
    /*
     * excluding default grab (and CAN_DEFAULT above) to 
     * clarify the state with default action...
     * problem we have currently (Crash Reporter 1.20) is that
     * when pressing Enter while in multiline edit field, this warning comes:
     *
     GLIB WARNING ** GLib-GObject - gsignal.c:1019: unable to lookup signal "activate" of unloaded type `GtkEntry'
     *
     * currently no idea why is that, at least excluding those 2 things here does not help..
     */
    /*------------*/
    /* Add a "Options" button to the bottom of the dialog */
    options_button = gtk_button_new_with_label(_("Options"));
    
    hildon_gtk_widget_set_theme_size(options_button, 
				     HILDON_SIZE_FINGER_HEIGHT);

    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (window)->action_area),
		       options_button, TRUE, TRUE, 0);

    g_signal_connect (G_OBJECT (options_button), "clicked",
		      G_CALLBACK(usecase_options_button_clicked),
		      window);
    
    /*------------*/
    /* Add a "delete" button to the bottom of the dialog */
    delete_button = gtk_dialog_add_button(GTK_DIALOG(window), 
					  _("Delete"), GTK_RESPONSE_APPLY);
    g_signal_connect (G_OBJECT (delete_button), "clicked",
		      G_CALLBACK(usecase_other_button_clicked),
		      (void*)GTK_RESPONSE_APPLY);
    
    gtk_widget_show_all (window);

    return window;
}

static gboolean
update_progressbar (gpointer dialog)
{
    /* 
     * If no progressbar, stop executing and unregister this function
     */
    if ( !(progressbar && send_progress_dialog) ) {
	return 0;
    }
    /*
     * if not visible don't update
     */
    if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(progressbar))) {
	return 1;
    }

    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar), progress_percent);

    return 1;
}

void creporter_ui_show_progress(char *str, double percent)
{
    if(str == NULL) {
	/* str=NULL is end sign */
	if (send_progress_dialog && PROGRESS_DLG_VISIBLE) {
	    gtk_dialog_response(send_progress_dialog, GTK_RESPONSE_OK);
	} 
    } else if (percent >= 0.0 && percent <= 1.0) {
	progress_percent = percent;
    }
}

/**
  Create elements of "describe use case" and 
  send/cancel/del/options dialog
  */

/**
  When Crash Reporter UI application needs to send all core files, this function
  is called to append all file to 'files_list'

  @param gchar * location of dir from which all files should be appended
  @param TRUE if success else FALSE
*/
static gboolean creporter_ui_collect_all_corefile_names_at_location(const gchar *location)
{
	gchar *filename = NULL;
	GDir* dir;
	const gchar* fil;
	
	g_return_val_if_fail(location != NULL, FALSE);
	
	dir = g_dir_open(location, 0, NULL);
	if (dir != NULL) {
	    while ((fil = g_dir_read_name(dir))) {
		if(core_is_valid(fil)) {
		    /*
		     * location has '/' prepended, so no need to put
		     * it in between here
		     */
		    filename = g_strdup_printf("%s%s",location, fil);
		    files_list = g_slist_prepend(files_list, filename);
		}
	    }
	    g_dir_close(dir);
	}

	return TRUE;
}


/**
  The function will collect the names of Rich cores present in both the MMCs
  and append them to the files_list for uploading

  @param void
  @void gboolean
*/
gboolean creporter_ui_collect_all_corefile_names()
{
	gboolean rv=TRUE;
	int idx;

	for (idx = 0 ; idx < MAX_CORE_DIRS ; idx++) {
	    if (core_location_registry[idx].mountpoint[0] == 0) {
		break;
	    }
	    if(!creporter_ui_collect_all_corefile_names_at_location(core_location_registry[idx].dir)) {
		osso_log(LOG_DEBUG, "[%s]: failed to append cores from %s", 
			 __FUNCTION__, core_location_registry[idx].dir);
		rv = FALSE;
	    }
	}

	return rv;
}


/**
  This function is preforming delayed exit of the Crash Reporter UI application

  @param void
  @return void
  */
static gboolean creporter_ui_delayed_exit(void * context)
{
    if (gtk_main_level() > 0) {
	gtk_main_quit();
    }

    osso_deinitialize(context);
    osso_log(LOG_DEBUG, "Crash Reporter UI exits");
    exit(0);
}

/**
  This function is called to exit the Crash Reporter UI application.
  As it is often used from callbacks (like RPC) that work better
  when returned, we initiate a delayed exit handler here and return.

  @param void
  @return void
  */
void creporter_ui_exit()
{
    static gint exit_tmo_id = 0;

    if (exit_tmo_id == 0) {
	/*
	 * only do it once.
	 */
	exit_tmo_id = g_timeout_add(100, creporter_ui_delayed_exit, NULL);
    }
}
