/**
 * @file crash-reporter-daemon-main.c
 *  
 * This file contains the main function for Crash Reporter daemon
 * and dynamically check for the availability
 * of internal/external memory cards sets monitors appropriately
 *
 * This file is part of crash-reporter
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

#include "crash-reporter-daemon-defs.h"
#include "crash-reporter-daemon-monitor.h"
#include "crash-reporter-settings-file.h"
#include "crash-reporter-common.h"
#include "crash-reporter-utils.h"

static gboolean mmc_change_tmo_handler(void * context);
static gint tmo_id = 0;

struct core_location_reg * core_location_registry = NULL;
gboolean sending_enabled = TRUE;
gboolean dumping_enabled = TRUE;
gboolean avoid_dups = TRUE;

#define PID_FILE "/tmp/creporter.pid"
#define AFTERBOOT_WAIT (30)

/*
 * (re-)activate directory monitoring.
 */
static void
mmc_monitor_dir(int idx)
{
    GDir *dir = NULL;
    if (core_location_registry == NULL) {
	osso_log(LOG_ERR, "[%s] No core locations directory!?",__FUNCTION__);
	return;
    }
    if (is_mounted(core_location_registry[idx].mountpoint)) {
	dir = g_dir_open(core_location_registry[idx].dir, 0, NULL);
	if (dir != NULL) {
	    g_dir_close(dir);
	    if(!set_directory_monitor(idx)) {
		osso_log(LOG_ERR, "[%s] Could not set directory monitor for %s",__FUNCTION__, core_location_registry[idx].dir);
	    }
	}
    }
}

/**
  This function is a callback to detect when the device is about to
  shut down. When that occurs, it's time for the Crash Reporter to go down.
  
  @param hw_state contains the monitored hardware state changes
  @param data contains a pointer to main loop.

*/
static void 
hw_state_event_cb(osso_hw_state_t *hw_state, gpointer data)
{
	if (hw_state->shutdown_ind == TRUE) {
		sending_enabled = 0;
		osso_log(LOG_DEBUG, "[%s]: Shutting down\n", __FUNCTION__);
		g_main_loop_quit((GMainLoop *)data);
		return;
	}
}

/**
  This function is the callback called when ext/int MMC removal/insertion 
  occurs. The directory monitors are reset in such case.

  @param GConfClent for notification
  @param connection ID from gconf_client_notify_add(). 
  @param GConfEntry
  @param user_data
  @return void
  */
static void 
mmc_notifier(GConfClient * gconf_client,
	     guint cnxn_id, GConfEntry * entry,
	     gpointer user_data)
{
    int idx;

    if (entry) {
	/*
	 * called with entry != NULL when gconf values change;
	 * Order timed execution (overwriting previous timer if any)
	 */
	if (tmo_id) {
	    g_source_remove(tmo_id);
	}
	tmo_id = g_timeout_add(5000, mmc_change_tmo_handler, gconf_client);
    } else {
	/*
	 * called with entry==NULL in these cases:
	 * 1. explicitly from main() when crash_reporter_daemon starts
	 * 2. from self-ordered callback few seconds after latest MMC change;
	 * Check and start monitoring directories.
	 */
	for (idx = 0 ; dumping_enabled && (idx < MAX_CORE_DIRS) ; idx++) {
	    if (core_location_registry[idx].mountpoint[0] == 0) {
		break;
	    }
	    mmc_create_dir(idx);
	    mmc_monitor_dir(idx);
	}
    }
}

/*
 * timer based callback after latest change to MMC status, plus some more time.
 * This is needed because gconf change events come before mounting
 * but our actions need to be carried out after mounting.
 */
static gboolean 
mmc_change_tmo_handler(void * context)
{
    GConfClient * gconf_client = (GConfClient *)context;

    mmc_notifier(gconf_client, 0, NULL, NULL);	
    tmo_id = 0;
    return FALSE; /* stop timer calling this */
}

/**
  main:

  @param argc is a number of command line arguments.
  @param argv is Command line argument strings.
  @return gint
*/
gint 
main(int argc, char *argv[])
{
	GConfClient *gconf_client = NULL;
	GMainLoop *loop = NULL;
	privacySettings* privsettings = NULL;
	osso_hw_state_t hw_state = {TRUE, FALSE, FALSE, FALSE, 0};
	FILE *fp;
	int wait = 0;
	pid_t pid;
	gchar *pid_str;

	pid = getpid();
	if(access(PID_FILE, F_OK)) {
	    /*
	     * no PID file, this is first run, lets delay our start
	     */
	    wait = AFTERBOOT_WAIT;
	}

	fp = fopen(PID_FILE, "w");
	osso_log(LOG_DEBUG, "crash_reporter_daemon[%d] starts, wait=%d", pid, wait);
	if (fp) {
	    pid_str = g_strdup_printf("%d", pid);
	    fwrite(pid_str, sizeof(char), strlen(pid_str), fp);
	    fclose(fp);
	    g_free(pid_str);
	}

	if (wait) {
	    sleep(wait);
	}

	/* Initialization functions */
	g_thread_init(NULL);
	g_type_init();

	/* context for libosso */
	context = osso_initialize("crash_reporter_daemon", "1.0", TRUE, NULL);
	g_return_val_if_fail(context != NULL, 0);


	gconf_client = gconf_client_get_default();
	gconf_client_add_dir(gconf_client, GCONF_DIR,
			     GCONF_CLIENT_PRELOAD_NONE, NULL);

	core_location_registry = create_core_location_registry();

	if (core_location_registry) {
	    privsettings = creporter_read_privacy_settings();
	    if (privsettings) {
		sending_enabled = privsettings->sending_enabled;
		dumping_enabled = privsettings->dumping_enabled;
		avoid_dups = privsettings->avoid_dups;
		creporter_free_privacy_settings(privsettings);
		privsettings = NULL;
	    } else {
		sending_enabled = TRUE;
		dumping_enabled = TRUE;
		avoid_dups = TRUE;
	    }

	    /*
	     * this creates initial inotify handle 
	     */
	    init_inotify_and_check_for_cores();	
	    /*
	     * this creates inotify handles for directories
	     */
	    mmc_notifier(gconf_client, 0, NULL, NULL);	

	    /* Order notification for some mmc events */
	    gconf_client_notify_add(gconf_client, EXTERNAL_COVER_OPEN,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, EXTERNAL_MMC_PRESENT,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, INTERNAL_MMC_PRESENT,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, MMC_CORRUPTED,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, EXTERNAL_MMC_USED_USB,
				    mmc_notifier, NULL, NULL, NULL);
	    gconf_client_notify_add(gconf_client, INTERNAL_MMC_USED_USB,
				    mmc_notifier, NULL, NULL, NULL);

	    loop = g_main_loop_new(NULL, FALSE);
	    osso_hw_set_event_cb(context, &hw_state, hw_state_event_cb, loop);
	    g_main_loop_run(loop);
	    osso_deinitialize(context);
	} else {
	    osso_log(LOG_ERR, "crash_reporter_daemon[%d] can not create core location registry", 
		     pid);
	}
	
	return 0;
}
