/*
 * This file is part of control-plugin-upstart
 *
 * Copyright (C) 2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/***************************************************************
 * This is the UI version of control_plugin_upstart
 *
 ***************************************************************/

#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/reboot.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-banner.h>
#include "upstart_config_handler.h"

/* Local functions */
static void create_checkbox_view(void);
static void ui_init(GdkNativeWindow socket);
static void cb_apply_reboot(void *data);
static void cb_save(void *data);
static void cb_save_as(void *data);
static void cb_undo(void *data);
static void cb_click_cbtn(void *data);
static void give_save_note(void);
static void update_list_elements_activity(void);

/* GUI components */
static GtkWidget* main_hbox;
static GtkWidget* viewport;
static GtkWidget* scroll_window;
static GtkWidget* tick_vbox;
static GtkWidget* btn_vbox;
static GtkWidget* open_btn;
static GtkWidget* save_btn;
static GtkWidget* save_as_btn;
static GtkWidget* undo_btn;
static GtkWidget* apply_reboot_btn;
static GtkWidget* main_window;

/* Local variables */
static unsigned int plugin_socket;
static gboolean needs_save = FALSE;
static gboolean trueValue = TRUE;
static gboolean falseValue = FALSE;
static gboolean* truePtr = &trueValue;
static gboolean* falsePtr = &falseValue;

static void 
cb_apply_reboot(void *data)
{
    /* Callback function for "Apply/Reboot" button */

    g_debug("cb_apply_reboot()\n");
    if (needs_save) {
        give_save_note();
        return;
    }
    save_current_config_file_name();
    move_event_files(); 
    sync();
    g_usleep(REBOOT_WAIT);
    reboot(RB_AUTOBOOT);
}


static void 
cb_open(void *data)
{
    /* Callback function for "Open" button 
     */

    GtkWidget* dialog = NULL;
    gchar* filename = NULL;

    g_debug("cb_open()\n");
    dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(main_window), 
                                            GTK_FILE_CHOOSER_ACTION_OPEN);
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog),
                                        CONFIG_DIR);
    g_debug("  current folder is %s\n", 
            gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)));

    gtk_widget_show_all(GTK_WIDGET(dialog));
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    }
    gtk_widget_destroy(dialog);
    if (filename != NULL) {
        free_element_list();  // Free current resources before creating new ones
        config_init(filename);
        create_checkbox_view();
        g_free(filename);
        needs_save = FALSE;
    }
    gtk_widget_show_all(main_window);
}

static void 
cb_save(void *data)
{
    /* Callback function for "Save" button */

    g_debug("cb_save()\n");
    if (needs_save) {
        update_list_elements_activity();
        update_config_file();
    }
    needs_save = FALSE;
}

static void 
cb_save_as(void *data)
{
    /* Callback function for "Save as" button
     * Ask for filename and save config
     */

    GtkWidget* dialog = NULL;
    gchar* filename = NULL;

    g_debug("cb_save_as()\n");
    dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(main_window), 
                                            GTK_FILE_CHOOSER_ACTION_SAVE);
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog),
                                        CONFIG_DIR);
    g_debug("  current folder is %s\n", 
            gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)));

    gtk_widget_show_all(GTK_WIDGET(dialog));
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    }
    gtk_widget_destroy(dialog);
    if (filename != NULL) {
        g_debug("  new config file is %s\n", filename);
        set_current_config_file(filename);
        update_list_elements_activity();
        update_config_file();
        g_free(filename);
        needs_save = FALSE;
    }
    gtk_widget_show_all(main_window);
}


static void 
cb_undo(void *data)
{
    /* Callback function for "Undo" button 
     * Update checkbox buttons to match with configured activity
     */

    GList* li;
    GObject* element;
    GtkWidget* checkbutton;
    gboolean* activity;
    GString* comment_line;

    g_debug("cb_undo()\n");
    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line != NULL) {
            continue;  // Skip over comment lines
        }
        checkbutton = g_object_get_data(element, CHECK_BTN_STR);
        activity = g_object_get_data(element, ACTIVITY_STR);
        g_assert(checkbutton && activity);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), 
                                     *activity);
    }
    needs_save = FALSE;
}

static void
cb_click_cbtn(void *data)
{
    /* Callback for each click on check buttons */
    needs_save = TRUE;
}

static void 
give_save_note(void)
{
    /* Print note on UI and wait for confirm
     */
    //GtkWidget* dialog = NULL;
    GtkWidget* banner = NULL;

    g_debug("give_save_note()\n");
    banner = hildon_banner_show_information(GTK_WIDGET(main_window),
                                            NULL,
                                            "Changes to your configuration\n"
                                            "has been done.\n"
                                            "You should do 'Save' or 'Undo'\n"
                                            "before doing Reboot");

    hildon_banner_set_timeout(HILDON_BANNER(banner), 5000);
    gtk_widget_show_all(GTK_WIDGET(banner));
        /*
    dialog = hildon_note_new_confirmation(GTK_WINDOW(main_window),
                                         "Changes to your configuration\n"
                                         "has been done.\n"
                                         "You should do 'Save' or 'Undo'\n"
                                         "before doing Reboot");
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
        */
    gtk_widget_show_all(main_window);
}

static void
update_list_elements_activity(void)
{
    /* Update list elements activity to match with check buttons
     */

    GList* li;
    GObject*   element;
    gboolean*  list_activity;
    GtkWidget* checkbutton;
    gboolean   btn_activity;
    GString* comment_line;

    g_debug("update_list_elements_activity()\n");

    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line != NULL) {
            continue;  // Skip over comment lines
        }
        checkbutton = g_object_get_data(element, CHECK_BTN_STR);
        g_assert(checkbutton);
        btn_activity = 
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbutton));
        list_activity = (btn_activity ? truePtr : falsePtr);
        g_object_set_data(element, ACTIVITY_STR, list_activity);
    }
}

static void 
create_checkbox_view(void)
{
    /*
     * Go through the element list and create check-buttons for each item.
     */

    GList* li;
    GObject* element;
    GString* print_name;
    GtkWidget* checkbutton;
    gboolean* activity;
    GString* comment_line;

    g_debug("create_checkbox_view()\n");
    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line != NULL) {
            continue;  // Skip over comment lines
        }
        print_name = g_object_get_data(element, PRINT_NAME_STR);
        g_assert(print_name);
        checkbutton = gtk_check_button_new_with_label(print_name->str);
        gtk_box_pack_start(GTK_BOX(tick_vbox),checkbutton,FALSE,FALSE,0);
        activity = g_object_get_data(element, ACTIVITY_STR);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), 
                                     *activity);
        g_object_set_data(element, CHECK_BTN_STR, checkbutton);
        g_signal_connect(G_OBJECT(checkbutton),"clicked", 
                         G_CALLBACK(cb_click_cbtn),NULL);
    }
}
 
void 
ui_init(GdkNativeWindow socket)
{

    g_debug("ui_init()\n");

    /* Create window */
    main_window = gtk_plug_new(socket);
    main_hbox = gtk_hbox_new(FALSE, 10);
    tick_vbox = gtk_vbox_new(FALSE, 0);
    btn_vbox = gtk_vbox_new(TRUE, 0);
    open_btn = gtk_button_new_with_label("Open");
    save_btn = gtk_button_new_with_label("Save");
    save_as_btn = gtk_button_new_with_label("Save as...");
    undo_btn = gtk_button_new_with_label("Undo");
    apply_reboot_btn = gtk_button_new_with_label("Apply/\nReboot");
    scroll_window = gtk_scrolled_window_new(NULL, NULL);
    viewport = gtk_viewport_new(NULL, NULL);

    /* pack */
    gtk_box_pack_start (GTK_BOX(btn_vbox), open_btn, TRUE, TRUE, 2);
    gtk_box_pack_start (GTK_BOX(btn_vbox), save_btn, TRUE, TRUE, 2);
    gtk_box_pack_start (GTK_BOX(btn_vbox), save_as_btn, TRUE, TRUE, 2);
    gtk_box_pack_start (GTK_BOX(btn_vbox), undo_btn, TRUE, TRUE, 2);
    gtk_box_pack_start (GTK_BOX(btn_vbox), apply_reboot_btn, TRUE, TRUE, 2);
    gtk_container_add(GTK_CONTAINER(viewport), tick_vbox);
    gtk_container_add(GTK_CONTAINER(scroll_window), GTK_WIDGET(viewport));
    gtk_box_pack_start(GTK_BOX(main_hbox), btn_vbox, FALSE, FALSE, 5);
    gtk_box_pack_start(GTK_BOX(main_hbox), scroll_window, TRUE, TRUE, 0);
    gtk_container_add(GTK_CONTAINER(main_window), main_hbox);	
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll_window), 
                                   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

    /* Connect signals */
    g_signal_connect(G_OBJECT(open_btn), "clicked", 
                     G_CALLBACK(cb_open), NULL);
    g_signal_connect(G_OBJECT(save_btn), "clicked", 
                     G_CALLBACK(cb_save), NULL);
    g_signal_connect(G_OBJECT(save_as_btn), "clicked", 
                     G_CALLBACK(cb_save_as), NULL);
    g_signal_connect(G_OBJECT(undo_btn),"clicked", 
                     G_CALLBACK(cb_undo), NULL);
    g_signal_connect(G_OBJECT(apply_reboot_btn),"clicked", 
                     G_CALLBACK(cb_apply_reboot),NULL);

    create_checkbox_view();
    gtk_widget_show_all(main_window);
}


int main(int argc, char **argv)
{


    gtk_init (&argc, &argv);

    if (argc < 2) {
        printf("Missing socket!\n");
        return 1;
    }

    plugin_socket = atoi(argv[1]);

    config_init(NULL);
    ui_init((GdkNativeWindow)plugin_socket);

    gtk_main();

    return 0;
}
