/*
 * This file is part of control-plugin-upstart
 *
 * Copyright (C) 2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


/*
 * This file includes routines that are used both UI and command
 * line versions of control-plugin-upstart
 */

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <dirent.h>
#include "upstart_config_handler.h"


GList* element_list = NULL;

/* Local variables */
static gboolean trueValue = TRUE;
static gboolean falseValue = FALSE;
static gboolean* truePtr = &trueValue;
static gboolean* falsePtr = &falseValue;
static gchar current_config_file[NAME_MAX];

/* Local prototypes */
static gboolean parse_config_line(gchar* input_str, gchar* link_name, 
                                  gchar* print_name, gboolean* activity,
                                  gint buffer_sizes);
static int one(const struct dirent*);
static void create_element_list(gchar *config_file);
static void read_current_config_file_name(void);
static gboolean get_file_activity(gchar* file_name, gboolean* activity);


static gboolean 
parse_config_line(gchar* input_str, gchar* link_name, 
                  gchar* print_name, gboolean* activity,
                  gint buffer_sizes)
{
    /* This function parses one line of config file.
     * Each line has form like this:
     * upstart_eventfile {ON | OFF} ["free text for display"]
     * link_name is the first word on the line.
     * Second parameter is boolean type ON or OFF information.
     * ON means that eventfile is part of the upstart init.
     * OFF means that file is not handled by upstart.
     * print_name is rest of the line (optional).
     * '"' marks around free text is optional
     * Line beginning with '#' is comment line
     * Get those fields and return to caller.
     * 'buffer_sizes' tells max number of character we can write to
     * 'link_name' and 'print_name'
     * Return TRUE for good line, FALSE for empty, comment and bad lines.
     * If only link_name present, use it for both link_name and print_name
     */

    int len;
    int r;
    int w;

    /* Trim line start and end */
    input_str = g_strstrip(input_str);

    /* Skip empty and comment lines */
    if ((strlen(input_str) <= 0) || (input_str[0] == '#')) {
        return FALSE;  
    }

    /* Link name is up to first space */
    r = 0;
    w = 0;
    len = strlen(input_str);
    while((r < len) && !g_ascii_isspace(input_str[r]) &&
          (w < buffer_sizes-1)) {
        link_name[w++] = input_str[r++];
    }
    link_name[w] = 0;
    input_str += r;

    // Skip spaces 
    input_str = g_strchug(input_str);

    /* Next word is activity (ON | OFF) */
    if (strlen(input_str) <= 0) {
        return FALSE;  
    }
    if (g_ascii_strncasecmp(input_str, ON_ACTIVITY_WORD, 
                            sizeof(ON_ACTIVITY_WORD)-1) == 0) {
        *activity = TRUE;
        input_str += sizeof(ON_ACTIVITY_WORD)-1;
    } else if (g_ascii_strncasecmp(input_str, OFF_ACTIVITY_WORD, 
                                   sizeof(OFF_ACTIVITY_WORD)-1) == 0) {
        *activity = FALSE;
        input_str += sizeof(OFF_ACTIVITY_WORD)-1;
    } else {
        // ON or OFF not found
        return FALSE;  
    }

    // Skip spaces 
    input_str = g_strchug(input_str);

    /* print_name is rest of the line 
     * if there is no print_name, then use same as link_name 
     */
    len = strlen(input_str);
    if (len <= 0) {
        input_str = link_name;
        len = strlen(input_str);
    }
    r = 0;
    w = 0;
    while((r < len) && (w < buffer_sizes-1)) {
        if (input_str[r] != '"')
            print_name[w++] = input_str[r];
        r++;
    }
    print_name[w] = 0;
    return TRUE;
}

static gboolean 
get_file_activity(gchar* file_name, gboolean* activity)
{
    /* This function tries to locate upstart file either from active or
     * passive directory. If found, then writes its activity and returns TRUE.
     * If not found, then returns FALSE
     */
    gchar active_file_name[NAME_MAX];
    gchar nonactive_file_name[NAME_MAX];

    snprintf(active_file_name, sizeof(active_file_name),
             "%s/%s", EVENT_DIR, file_name);
    snprintf(nonactive_file_name, sizeof(nonactive_file_name),
             "%s/%s", NOP_DIR, file_name);

    if (g_file_test(active_file_name, 
                    G_FILE_TEST_EXISTS | 
                    G_FILE_TEST_IS_SYMLINK)) {
        *activity = TRUE;
        return TRUE;
    } else if (g_file_test(nonactive_file_name, 
                           G_FILE_TEST_EXISTS | 
                           G_FILE_TEST_IS_SYMLINK)) {
        *activity = FALSE;
        return TRUE;
    } else {
        return FALSE;
    }
}

gint
move_event_files(void)
{
    /* Move event files to new places based on info in element list.
     * Return number of moves done.
     */
    g_debug("move_event_files()\n");

    gchar old_name[NAME_MAX];
    gchar new_name[NAME_MAX];
    GString* file_name;
    GString* comment_line;
    GList* li = NULL;
    GObject* element;
    gboolean *list_activity;
    gboolean file_activity;
    gint moves = 0;

    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line != NULL) {
            continue;  // Skip over comment lines
        }
        list_activity = g_object_get_data(element, ACTIVITY_STR);
        g_assert(list_activity);
        file_name = g_object_get_data(G_OBJECT(element), FILE_NAME_STR);
        g_assert(file_name);
        if (get_file_activity(file_name->str, &file_activity)) {
            if (file_activity != *list_activity) {
                /* Activity is different, move file */
                snprintf(old_name, sizeof(old_name), "%s/%s", 
                         (file_activity ? EVENT_DIR : NOP_DIR), file_name->str);
                snprintf(new_name, sizeof(new_name), "%s/%s", 
                         (*list_activity ? EVENT_DIR : NOP_DIR), file_name->str);
                rename(old_name, new_name);
                g_debug("  Moved %s -> %s\n", old_name, new_name);
                moves++;
            }
        } else {
            g_debug("Could not find file %s\n", (file_name ? file_name->str : "NULL"));
        }
    }
    g_debug("  did %d moves\n", moves);
    return moves;
}


static void 
create_element_list(gchar *config_file)
{
    /*
     * Go through the config file and create elements for each item.
     */
    gchar buffer[NAME_MAX * 2];    
    FILE* f;
    gboolean conf_activity;
    gboolean file_activity;
    gchar print_name[NAME_MAX];
    gchar file_name[NAME_MAX];
    GString* stored_file_name;
    GString* stored_print_name;
    GString* stored_comment;
    gboolean* stored_activity;
    GObject* element;

    g_debug("create_element_list()\n");
    /* Open config file */
    g_assert(config_file != NULL);
    f = fopen(config_file, "r");
    g_assert(f != NULL);


    /* Read config file line by line and create elements for each line.
     * Config file can have also comment lines, save those as well.
     * Real config lines can be tested by checking comment string.
     * If it is NULL, then line is "real".
     */
    while(fgets(buffer, sizeof(buffer), f) != NULL) {
        element = g_object_new(G_TYPE_OBJECT, NULL);
        g_assert(element);
        element_list = g_list_append(element_list, element);
        if (!parse_config_line(buffer, &(file_name[0]), &(print_name[0]), 
                               &conf_activity, NAME_MAX)) {
            /* This was not actual config line. Save it as comment */
            buffer[0] = '#';  // Original line was comment, empty or bad one
            stored_comment = g_string_new(buffer);
            g_assert(stored_comment);
            g_object_set_data(element, COMMENT_STR, stored_comment);
        } else if (get_file_activity(file_name, &file_activity)) {
            /* This is good line, mark comment as NULL and
             * store real stuff
             */
            g_object_set_data(element, COMMENT_STR, NULL);
            stored_file_name = g_string_new(file_name);
            stored_print_name = g_string_new(print_name);
            g_assert(stored_file_name && stored_print_name);
            stored_activity = (conf_activity ? truePtr : falsePtr);
            g_object_set_data(element, FILE_NAME_STR, stored_file_name);
            g_object_set_data(element, PRINT_NAME_STR, stored_print_name);
            g_object_set_data(element, ACTIVITY_STR, stored_activity);
        } else {
            /* This is good line but file name can't be found.
             * Save this line as a comment
             */
            snprintf(buffer, sizeof(buffer),"#%-24s %-3s  \"%s\"",
                     file_name,
                     (conf_activity ? ON_ACTIVITY_WORD : OFF_ACTIVITY_WORD),
                     print_name);
            stored_comment = g_string_new(buffer);
            g_assert(stored_comment);
            g_object_set_data(element, COMMENT_STR, stored_comment);
        }
    }
    fclose(f);
    g_debug("   created %d elements\n", g_list_length(element_list));
}

void 
free_element_list(void)
{
    /* Free all elements from the element list and associated memory
     */


    GList* li;
    GObject* element;
    GtkWidget* checkbutton;
    GString* comment_line;

    g_debug("free_element_list()\n");
    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line != NULL) {
            /* Comment element has only comment line nothing else */
            g_string_free(comment_line, TRUE);
        } else {
            /* Non comment element has more data */
            g_string_free(g_object_get_data(element, FILE_NAME_STR), TRUE);
            g_string_free(g_object_get_data(element, PRINT_NAME_STR), TRUE);
            checkbutton = g_object_get_data(element, CHECK_BTN_STR);
            if (checkbutton) {
                gtk_widget_destroy(checkbutton);
            }
        }
        g_list_free(element_list);
        element_list = NULL;
    }
}


static int one(const struct dirent* notused)
{
    return 1;
}

static void
create_new_config_file(gchar* file_name)
{ 
    /* Create new config file based on current upstart
     * event file locations.
     */
    FILE* f;
    struct dirent** file_list;
    int i;
    int n;

    g_debug("create_new_config_file(%s)\n", file_name);

    g_assert(file_name);
    f = fopen(file_name, "w");
    g_assert(f != NULL);

    /* Scan first active files */
    n = scandir(EVENT_DIR, &file_list, one, alphasort);
    for (i = 0; i < n; i++) {
        if (file_list[i]->d_name[0]=='.') {
            free(file_list[i]);
            continue;
        }
        fprintf(f, "%-25s %-3s   \"%s\"\n", 
                file_list[i]->d_name,
                ON_ACTIVITY_WORD,
                file_list[i]->d_name); 
        free(file_list[i]);
    }
    if (n > 0) {
        free(file_list);
    }

    /* Then scan passive files */
    n = scandir(NOP_DIR, &file_list, one, alphasort);
    for (i = 0; i < n; i++) {
        if (file_list[i]->d_name[0]=='.') {
            free(file_list[i]);
            continue;
        }
        fprintf(f, "%-25s %-3s  \"%s\"\n", 
                file_list[i]->d_name, 
                OFF_ACTIVITY_WORD,
                file_list[i]->d_name); 
        free(file_list[i]);
    }
    if (n > 0) {
        free(file_list);
    }
    fclose(f);
}


void
update_config_file(void)
{ 
    /* Update config file to match with current settings in the element_list
     */

    FILE* f;
    GList* li = NULL;
    GObject* element;
    gboolean* list_activity;
    GString* file_name;
    GString* print_name;
    GString* comment_line;

    g_debug("update_config_file(%s)\n", current_config_file);
    /* Open config file */
    f = fopen(current_config_file, "w");
    g_assert(f != NULL);

    for (li = element_list; li != NULL; li = g_list_next(li)) {
        element = li->data;
        g_assert(element);
        comment_line = g_object_get_data(G_OBJECT(element), COMMENT_STR);
        if (comment_line == NULL) {
            /* Normal config line, update it */
            list_activity = g_object_get_data(element, ACTIVITY_STR);
            g_assert(list_activity);
            file_name = g_object_get_data(G_OBJECT(element), FILE_NAME_STR);
            g_assert(file_name);
            print_name = g_object_get_data(G_OBJECT(element), PRINT_NAME_STR);
            g_assert(print_name);
            fprintf(f, "%-25s %-3s  \"%s\"\n", 
                    file_name->str, 
                    (*list_activity ? ON_ACTIVITY_WORD : OFF_ACTIVITY_WORD),
                    print_name->str);
        } else {
            /* Comment line, write it as is */
            fprintf(f, "%s\n", comment_line->str);
        }
    }
    fclose(f);  	
}


void 
save_current_config_file_name(void)
{
    FILE* f;

    g_debug("save_current_config_file_name(%s)\n", current_config_file);
    f = fopen(CURRENT_CFILE, "w");
    g_assert(f != NULL);
    fprintf(f, "%s\n", current_config_file);
    fclose(f);
}

static void 
read_current_config_file_name(void)
{
    FILE* f;

    g_debug("read_current_config_file_name()\n");
    f = fopen(CURRENT_CFILE, "r");
    g_assert(f != NULL);
    fgets(current_config_file, sizeof(current_config_file), f);
    g_strchomp(current_config_file);
    fclose(f);
    g_debug("  got %s\n", current_config_file);
}

void
set_current_config_file(const gchar* file_name)
{
    g_debug("set_current_config_file(%s)\n", file_name);
    snprintf(current_config_file, sizeof(current_config_file),
             "%s", file_name);
}

void
config_init(gchar* config_file)
{
    /* Make sure our directories exists */
    mkdir(CONFIG_DIR, 0744);
    mkdir(NOP_DIR, 0744);

    g_debug("config_init(%s)\n", config_file ? config_file : "NULL");
    if (config_file) {
        /* Config file name was given as parameter.
         * It must exist and it becomes new "current config file".
         */
        if (g_file_test(config_file, G_FILE_TEST_EXISTS)) {
            set_current_config_file(config_file);
            snprintf(current_config_file, sizeof(current_config_file),
                     "%s", config_file);
            save_current_config_file_name();

        } else {
            /* Asked to open non existing config file.
             * Report error and quit.
             */
            g_critical("Config file does not exist %s", config_file);
            exit(1);
        }
    }


    if (!g_file_test(CURRENT_CFILE, G_FILE_TEST_EXISTS)) {
        /* Master file does not exist, create one */
        set_current_config_file(CONFIG_FILE);
        save_current_config_file_name();
    }
    /* Read config_file_name from "master file".
     * If config does not exist
     * create config file based on event files;
     */
    read_current_config_file_name();
    if (!g_file_test(current_config_file, G_FILE_TEST_EXISTS)) {
        create_new_config_file(current_config_file);
    }
    create_element_list(current_config_file);
}
